#' Bind lists of list of multiple data frames by row
#'
#' Row binds the matching innermost dataframes in a list of lists. This is
#' essentially a list inversion [`purrr::list_transpose()`] with row-binding
#' [`dplyr::bind_rows()`]
#'
#' @param list A list of lists of `data.frame`s where matching innermost
#'   elements must be bound together row-wise.
#'
#' @return
#' A list of `data.frame`s with the combined information from the inputted list.
#'
#' @examples
#' # A simple example
#' lst <- lapply(1:5, function(x) {
#'   list(
#'     "A" = data.frame("first" = x, "second" = rnorm(x)),
#'     "B" = data.frame("info" = 1, "other" = 3)
#'   )
#' })
#' braid_rows(lst)
#'
#' # An example with an additional layer and jagged innermost info
#' lapply(c(28, 186, 35), function(len) {
#'   lapply(c("a", "b"), function(x) {
#'     res <- list(
#'       "descriptive" = data.frame(
#'          risk = len,
#'          event = x,
#'          var = 1,
#'          other = 2
#'        ),
#'       "results" = data.frame(
#'          risk = len,
#'          event = x,
#'          important = 4:7,
#'          new = 3:6
#'       )
#'     )
#'     if (len < 30) {
#'       res <- c(res, list("additional" = data.frame(helps = "extra data")))
#'     }
#'     return(res)
#'   }) |> braid_rows()
#' }) |> braid_rows()
#'
#'
#' @export
braid_rows <- function(list) {

  # checking input is data frame based
  if (
    !all(
      sapply(
        unlist(list, recursive = FALSE),
        function(ls) inherits(ls, "data.frame")
      )
    )
  ) {
    stop("All elemtents must be dataframes")
  }

  if (all(diff(sapply(list, length)) == 0)) {   # the simple rectangular case
    return(
      purrr::list_transpose(list, simplify = FALSE) |>
        lapply(function(ls) ls |> dplyr::bind_rows())
    )
  } else {   # the jagged case
    # find the new list structure
    nam <- unique(unlist(lapply(list, function(ls) names(ls))))
    if (is.null(nam)) {
      nam <- 1:max(unlist(lapply(list, function(ls) length(ls))))
    } else {
      names(nam) <- nam
    }

    # loop, extract, and bind over new structure
    return(lapply(nam, function(na) {
      lapply(list, function(ls) {
        if (is.character(na)) {
          return(ls[[na]])
        } else if (na <= length(ls)) {
          return(ls[[na]])
        } else {
          return(NULL)
        }
      }) |> dplyr::bind_rows()
    })
    )
  }
}





