#' Number of clusters for each ellipsoid numbers
#'
#' \code{cluster.assign.number} generates a table and a plot for the number of
#'   eliipsoids and the number of clusters, and a list of \code{icp.torus} objects
#'   which are used for k-means to k-ellipsoids clustering
#'
#' @param data n x d matrix of toroidal data on \eqn{[0, 2\pi)^d}
#'   or \eqn{[-\pi, \pi)^d}
#' @param Jmin minimum number of ellipsoids. Default value is 3.
#' @param Jmax maximum number of ellipsoids. Default value is 35.
#' @param level a scalar between \eqn{[0,1]}. Default value
#'   is 0.1.
#' @param split.id a n-dimensional vector consisting of values 1 (estimation)
#'   and 2(evaluation)
#' @param method character which must be "homogeneous-circular",
#'  "heterogeneous-circular", "ellipsoids", or "general".
#'   If "homogeneous-circular", the radii of k-spheres are identical.
#'   If "heterogeneous-circular", the radii of k-spheres may be different.
#'   If "ellipsoids", cluster with k-ellipsoids with initial parameters.
#'   If, "general", cluster with k-ellipsoids. The parameters to construct
#'   the ellipsoids are optimized with generalized Lloyd's algorithm, which is
#'   modified for toroidal space. To see the detail, consider the references.
#' @param init determine the initial parameter of "kmeans" method,
#'   for option "general". Must be "kmeans" or "hierarchical".
#'   If "kmeans", the initial parameters are obtained with extrinsic k-means
#'   method.
#'   If "hierarchical", the initial parameters are obtained with hierarchical
#'   clustering method.
#'   Default is "kmeans".
#' @param additional.condition boolean index.
#'   If \code{TRUE}, a singular matrix will be altered to the scaled identity.
#' @param THRESHOLD number for difference between updating and
#'   updated parameters. Default is 1e-10.
#' @param maxiter the maximal number of iteration. Default is 200.
#' @param verbose boolean index, which indicates whether display
#'   additional details as to what the algorithm is doing or
#'   how many loops are done. Moreover, if \code{additional.condition} is
#'   \code{TRUE}, the warning message will also be reported.
#'   Default is \code{FALSE}.
#' @return an \code{output} object, containing a plot based on ggplot2,
#'   a \code{data.frame} for the number of clusters and \code{icp.torus} objects
#'   generated by given options, varied from \code{Jmin} to \code{Jmax}.
#' @export
#' @seealso \code{\link[ClusTorus]{icp.torus.score}}
#' @references S. Jung, K. Park, and B. Kim (2021),
#'   "Clustering on the torus by conformal prediction"
#' @examples
#' \donttest{
#' data <- ILE[1:200, 1:2]
#' cluster.assign.number(data, Jmin = 3, Jmax = 20, level = 0.1)
#'}
cluster.assign.number <- function(data, Jmin = 3, Jmax = 35, level = 0.1,
                                  split.id = NULL,
                                  method = c("homogeneous-circular",
                                             "heterogeneous-circular",
                                             "ellipsoids",
                                             "general"),
                                  init = c("kmeans", "hierarchical"),
                                  additional.condition = TRUE,
                                  THRESHOLD = 1e-10, maxiter = 200,
                                  verbose = FALSE){

  if (Jmin > Jmax) {stop("Jmin must be less than Jmax.")}
  if (level > 1 || level < 0) {stop("Level must be a non-negative number less than 1.")}
  if (is.null(method)) { method <- "homogeneous-circular" }
  if (is.null(init)) {init <- "kmeans" }
  if (length(level) != 1 || !is.numeric(level)) {stop("Level must be a numeric value.")}

  kmeansfitmethod <- match.arg(method)
  init <- match.arg(init)

  data <- as.matrix(data)
  data <- on.torus(data)

  output <- list(cluster.number = NULL, plot = NULL, icp.torus.objects = NULL)
  cluster.number <- rep(0, Jmax - Jmin)

  for (j in Jmin:Jmax){
    icp.torus <- icp.torus.score(data, split.id, method = "kmeans",
                                 kmeansfitmethod = kmeansfitmethod,
                                 init = init,
                                 additional.condition = additional.condition,
                                 param = list(J = j), THRESHOLD = THRESHOLD,
                                 maxiter = maxiter,
                                 verbose = verbose)

    output$icp.torus.objects[[j]] <- icp.torus

    c <- cluster.assign.torus(data, icp.torus, level)

    cluster.number[j - Jmin + 1] <- c$kmeans$ncluster
  }

  plot.data <- as.data.frame(cbind(Jmin:Jmax, cluster.number))
  colnames(plot.data) <- c("J", "Number")
  output$cluster.number <- plot.data

  output$plot <- ggplot2::ggplot() +
    ggplot2::geom_point(ggplot2::aes(x = .data$J, y = .data$Number),
                        data = plot.data, size = 2) + ggplot2::geom_line() +
    ggplot2::labs(title = "Cluster Number Plot") +
    ggplot2::xlab("Number of ellipsoids") +
    ggplot2::ylab("Number of clusters")

  return(output)
}
