\name{splitSpectraGroups}
\alias{splitSpectraGroups}

\title{
Create New Groups from an Existing Spectra Object
}

\description{
This function takes an existing \code{\link{Spectra}} object and uses your instructions to split the existing \code{spectra$groups} into new groups.  The new groups are added to the existing \code{\link{Spectra}} object (a list) as new elements.  This allows one to use different combinations of factors than were originally encoded in the \code{\link{Spectra}} object.  The option also exists to replace the color scheme with one which corresponds to the new factors.
}

\usage{
splitSpectraGroups(spectra, inst = NULL, rep.cols = NULL, ...)
}

\arguments{
  \item{spectra}{An object of S3 class \code{\link{Spectra}}.}
  \item{inst}{A list giving the name of the new element to be created from a set of target strings given in a character vector.  See the example for the syntax.}
  \item{rep.cols}{Optional.  A vector giving new colors which correspond to the levels of \code{inst}.  Only possible if \code{inst} has only one element, as the possible combinations of levels and colors may get complicated.}

  \item{\dots}{Additional arguments to be passed downstream. Currently not used.}
}

\value{ An object of S3 class \code{\link{Spectra}}, modified to have additional elements as specified by \code{inst}.}

\details{
The items in the character vector are grepped among the existing \code{spectra$groups} entries; when found, they are placed in a new element of \code{\link{Spectra}}.  In the example, all \code{spectra$groups} entries containing "G" are coded as "G" in a new element called \code{spectra$env}, and any entries containing "T" are handled likewise.  This amounts to a sort of recoding of factors (the example demonstrates this).  Every entry in \code{spectra$groups} should be matched by one of the entries in the character vector.  If not, you will get <NA> entries.  Also, if the targets in the character vector are not unique, your results will reflect the order of the levels.  Since this is a grep process, you can pass any valid grep string as the target.

If \code{rep.cols} is provided, these colors are mapped one for one onto the levels of the the first element of \code{inst}.  This provides a different means of changing the sample color encoding than \code{\link{conColScheme}}.
}

\references{\url{https://github.com/bryanhanson/ChemoSpec}}

\author{
Bryan A. Hanson, DePauw University. \email{hanson@depauw.edu}
}

\seealso{
\code{\link{conColScheme}}.
}

\examples{
data(metMUD2)
# Original factor encoding:
levels(metMUD2$groups)
# Split those original levels into 2 new ones (re-code them)
new.grps <- list(geneBb = c("B", "b"), geneCc = c("C", "c"))
res <- splitSpectraGroups(metMUD2, new.grps)
str(res) # note two new elements, "geneBb" and "geneCc"
#
# Note that if you want to use a newly created group in
# plotScores and other functions to drive the color scheme
# and labeling, you'll have to update the groups element:
res$groups <- as.factor(paste(res$geneBb, res$geneCc, sep = ""))
}

\keyword{ utilities }
