% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/multignk.R
\docType{data}
\name{mgnk}
\alias{mgnk}
\alias{mgnk_sim}
\alias{mgnk_sum}
\title{The multivariate G&K example}
\usage{
mgnk_sim(theta_tilde, sim_options)

mgnk_sum(y)
}
\arguments{
\item{theta_tilde}{A vector with 15 elements for the proposed model parameters.}

\item{sim_options}{A list of options for simulating data from the model. For this example, the list contains
\itemize{
\item \code{T}: The number of observations in the data.
\item \code{J}: The number of variables in the data.
\item \code{bound}: A matrix of boundaries for the uniform prior.
}}

\item{y}{A \code{T} \eqn{x} \code{J} matrix of data.}
}
\description{
Here we provide the data and tuning parameters required to reproduce
the results from the multivariate G & K (Drovandi and Pettitt, 2011) example from An et al. (2018).

The function \code{mgnk_sim(theta_tilde,sim_options)} simulates from the multivariate G & K model.

The function \code{mgnk_sum(y)} calculates the summary statistics for the multivariate G & K example.
}
\details{
It is not practical to give a reasonable explanation of this example through R documentation
given the number of equations involved. We refer the reader to the BSLasso paper (An et al., 2018)
at \url{https://eprints.qut.edu.au/102263/} for information on the model and summary statistic used in this example.
}
\section{An example dataset}{


We use the foreign currency exchange data available from \url{http://www.rba.gov.au/statistics/historical-data.html}
as in An et al. (2018).

\itemize{
 \item \code{data}:  A \code{1651} \eqn{x} \code{3} matrix of data.
 \item \code{sim_options}: Values of \code{sim_options} relevant to this example.
 \item \code{start}: A vector of suitable initial values of the parameters for MCMC.
 \item \code{cov}: Covariance matrix of the multivariate normal random walk, in the form of a \eqn{15 x 15} matrix.
}
}

\examples{
\dontrun{
require(doParallel) # You can use a different package to set up the parallel backend

# Loading the data for this example
data(mgnk)

# Opening up the parallel pools using doParallel
cl <- makeCluster(detectCores())
registerDoParallel(cl)

# Performing BSL
resultMgnkBSL <- bsl(mgnk$data, n = 60, M = 80000, start = mgnk$start, cov_rw = mgnk$cov,
                 fn_sim = mgnk_sim, fn_sum = mgnk_sum, sim_options = mgnk$sim_options, 
                 parallel = TRUE, parallel_packages = c('BSL', 'MASS', 'elliplot'),
                 theta_names = c('a1','b1','g1','k1','a2','b2','g2','k2','a3','b3','g3','k3'
                 ,'delta12','delta13','delta23'))
summary(resultMgnkBSL)
plot(resultMgnkBSL, thin = 20)

# Performing tuning for BSLasso
lambda_all <- list(exp(seq(-2.5,0.5,length.out=20)), exp(seq(-2.5,0.5,length.out=20)), 
                   exp(seq(-4,-0.5,length.out=20)), exp(seq(-5,-2,length.out=20)))

sp_mgnk <- selectPenalty(ssy = mgnk_sum(mgnk$data), n = c(15, 20, 30, 50), lambda_all,
                 theta = mgnk$start, M = 100, sigma = 1.5, fn_sim = mgnk_sim, 
                 fn_sum = mgnk_sum, sim_options = mgnk$sim_options, standardise = TRUE, 
                 parallel_sim = TRUE, parallel_sim_packages = c('BSL', 'MASS', 'elliplot'),
                 parallel_main = TRUE)
sp_mgnk
plot(sp_mgnk)

# Performing BSLasso with a fixed penalty
resultMgnkBSLasso <- bsl(mgnk$data, n = 20, M = 80000, start = mgnk$start, cov_rw = mgnk$cov,
                 fn_sim = mgnk_sim, fn_sum = mgnk_sum, sim_options = mgnk$sim_options,
                 penalty = 0.3, standardise = TRUE, parallel = TRUE,
                 parallel_packages = c('BSL', 'MASS', 'elliplot'),
                 theta_names = c('a1','b1','g1','k1','a2','b2','g2','k2','a3','b3','g3','k3',
                 'delta12','delta13','delta23'))
summary(resultMgnkBSLasso)
plot(resultMgnkBSLasso, thin = 20)

# Plotting the results together for comparison
combinePlotsBSL(resultMgnkBSL, resultMgnkBSLasso, thin = 20)

# Closing the parallel pools
stopCluster(cl)
}

}
\references{
An, Z., South, L. F., Nott, D. J. &  Drovandi, C. C. (2018). Accelerating
Bayesian synthetic likelihood with the graphical lasso. \url{https://eprints.qut.edu.au/102263/}

Drovandi, C. C. and Pettitt, A. N. (2011). Likelihood-free Bayesian estimation of multivariate
quantile distributions. Computational Statistics and Data Analysis, 55(9):2541-2556.
}
\author{
Ziwen An, Christopher C. Drovandi and Leah F. South
}
