% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitArCo.R
\name{fitArCo}
\alias{fitArCo}
\title{Estimates the ArCo using the model selected by the user}
\usage{
fitArCo(data, fn, p.fn, treated.unity, t0, lag = 0, Xreg = NULL,
  alpha = 0.05, boot.cf = FALSE, R = 100, l = 3, VCOV.type = c("iid",
  "var", "nw", "varhac"), VCOV.lag = 1, bandwidth.kernel = NULL,
  kernel.type = c("QuadraticSpectral", "Truncated", "Bartlett", "Parzen",
  "TukeyHanning"), VHAC.max.lag = 5, prewhitening.kernel = FALSE)
}
\arguments{
\item{data}{A list of matrixes or data frames of length q. Each matrix is T X n and it contains observations of a single variable for all units and all periods of time. Even in the case of a single variable (q=1), the matrix must be inside a list.}

\item{fn}{The function used to estimate the first stage model. This function must receive only two arguments in the following order: X (independent variables), y (dependent variable). If the model requires additional arguments they must be supplied inside the function fn.}

\item{p.fn}{The forecasting function used to estimate the counterfactual using the first stage model (normally a predict funtion). This function also must receive only two arguments in the following order: model (model estimated in the first stage), newdata (out of sample data to estimate the second stage). If the prediction requires additional arguments they must be supplied inside the function p.fn.}

\item{treated.unity}{Single number indicating the unity where the intervention took place.}

\item{t0}{Single number indicating the intervention period.}

\item{lag}{Number of lags in the first stage model. Default is 0, i.e. only contemporaneous variables are used.}

\item{Xreg}{Exogenous controls.}

\item{alpha}{Significance level for the delta confidence bands.}

\item{boot.cf}{Should bootstrap confidence intervals for the counterfactual be calculated (default=FALSE).}

\item{R}{Number of bootstrap replications in case boot.cf=TRUE.}

\item{l}{Block length for the block bootstrap.}

\item{VCOV.type}{Type of covariance matrix for the delta. "iid" for standard covariance matrix, "var" or "varhac" to use prewhitened covariance matrix using VAR models, "varhac" selects the order of the VAR automaticaly and "nw" for Newey West. In the last case the user may select the kernel type and combine the kernel with the VAR prewhitening. For more details see Andrews and Monahan (1992).}

\item{VCOV.lag}{Lag used on the robust covariance matrix if VCOV.type is different from "iid".}

\item{bandwidth.kernel}{Kernel bandwidth. If NULL the bandwidth is automatically calculated.}

\item{kernel.type}{Kernel to be used for VCOV.type="nw".}

\item{VHAC.max.lag}{Maximum lag of the VAR in case VCOV.type="varhac".}

\item{prewhitening.kernel}{If TRUE and VCOV.type="nw", the covariance matrix is calculated with prewhitening (default=FALSE).}
}
\value{
An object with S3 class fitArCo.
\item{cf}{estimated counterfactual}
\item{fitted.values}{In sample fitted values for the pre-treatment period.}
\item{model}{A list with q estimated models, one for each variable. Each element in the list is the output of the fn function.}
\item{delta}{The delta statistics and its confidence interval.}
\item{p.value}{ArCo p-value.}
\item{data}{The data used.}
\item{t0}{The intervention period used.}
\item{treated.unity}{The treated unity used.}
\item{boot.cf}{A list with the bootstrap result (boot.cf=TRUE) or logical FALSE (boot.cf=FALSE). In the first case, each element in the list refers to one bootstrap replication of the counterfactual, i. e. the list length is R.}
\item{call}{The matched call.}
}
\description{
Estimates the Artificial Counterfactual unsing any model supplied by the user, calculates the most relevant statistics and allows for the counterfactual confidence intervals to be estimated by block bootstrap. \cr
The model must be supplied by the user through the arguments fn and p.fn. The first determines which function will be used to estimate the model and the second determines the forecasting function. For more details see the examples and the description on the arguments.
}
\details{
This description may be useful to clarify the notation and understand how the arguments must be supplied to the functions.
\itemize{
\item{units: }{Each unity is indexed by a number between \eqn{1,\dots,n}. They are for exemple: countries, states, municipalities, firms, etc.}
\item{Variables: }{For each unity and for every time period \eqn{t=1,\dots,T} we observe \eqn{q_i \ge 1} variables. They are for example: GDP, inflation, sales, etc.}
\item{Intervention: }{The intervention took place only in the treated unity at time \eqn{t_0=\lambda_0*T}, where \eqn{\lambda_0} is in (0,1).}
}
}
\examples{
#############################
## === Example for q=1 === ##
#############################
data(data.q1)
# = First unity was treated on t=51 by adding 
# a constant equal to one standard deviation

data=list(data.q1) # = Even if q=1 the data must be in a list

## == Fitting the ArCo using linear regression == ##
# = creating fn and p.fn function = #
fn=function(X,y){
return(lm(y~X))
}
p.fn=function(model,newdata){
b=coef(model)
return(cbind(1,newdata) \%*\% b)
}

ArCo=fitArCo(data = data,fn = fn, p.fn = p.fn, treated.unity = 1 , t0 = 51)

#############################
## === Example for q=2 === ##
#############################

# = First unity was treated on t=51 by adding constants of one standard deviation
# for the first and second variables

data(data.q2) # data is already a list

## == Fitting the ArCo using the package glmnet == ##
## == Quadratic Spectral kernel weights for two lags == ##

## == Fitting the ArCo using the package glmnet == ##
## == Bartlett kernel weights for two lags == ##
require(glmnet)
set.seed(123)
ArCo2=fitArCo(data = data.q2,fn = cv.glmnet, p.fn = predict,treated.unity = 1 , t0 = 51, 
             VCOV.type = "nw",kernel.type = "QuadraticSpectral",VCOV.lag = 2)

}
\references{
Carvalho, C., Masini, R., Medeiros, M. (2016) "ArCo: An Artificial Counterfactual Approach For High-Dimensional Panel Time-Series Data.".

Andrews, D. W., & Monahan, J. C. (1992). An improved heteroskedasticity and autocorrelation consistent covariance matrix estimator. Econometrica: Journal of the Econometric Society, 953-966.
}
\seealso{
\code{\link{plot}}, \code{\link{estimate_t0}}, \code{\link{panel_to_ArCo_list}}
}
\keyword{ArCo}
