% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/wintime.R
\name{wintime}
\alias{wintime}
\title{Run a win time calculation}
\usage{
wintime(
  type,
  Time,
  Delta,
  trt,
  cov = NULL,
  model = NULL,
  resample = NULL,
  resample_num = 0,
  time_restriction = NA,
  seed = NA,
  nimp = 0
)
}
\arguments{
\item{type}{A string value indicating the desired win time method. Methods include 'ewt', 'ewtr', 'rmt', 'max', 'wtr', 'rwtr', 'pwt', 'ewtp', 'rewtp', 'ewtpr','rewtpr', and 'rpwt'.}

\item{Time}{A \verb{m x n} matrix of event times (days), where \code{m} is the number of events in the hierarchy, and \code{n} is the total number of trial participants.
Rows should represent events and columns should represent participants. Event rows should be in increasing order of clinical severity.}

\item{Delta}{A \verb{m x n} matrix of event indicators, where \code{m} is the number of events in the hierarchy, and \code{n} is the total number of trial participants.
Rows should represent events and columns should represent participants. Event rows should be in increasing order of clinical severity.}

\item{trt}{A numeric vector containing treatment arm indicators (1 for treatment, 0 for control).}

\item{cov}{Optional. A \verb{n x p} matrix of covariate values, where \code{n} is the total number of trial participants and \code{p} is the number of covariates.
Rows should represent participants and columns should represent covariate values.}

\item{model}{Optional. String value. The type of model used to calculate state distributions. Options include 'km' and 'markov'.  Default depends on \code{type}.}

\item{resample}{Optional. String value. The resampling method run after the observed data calculation. Options include 'boot' and 'perm'.  Default depends on \code{type}.}

\item{resample_num}{Optional. The number of desired resamples. Default is 0.}

\item{time_restriction}{Required only for \code{type} = 'rmt','rewtp','rewtpr', and 'rpwt'. The cutoff time (days).}

\item{seed}{Optional. Seed used for random number generation in resampling.}

\item{nimp}{Required only for \code{type} = 'ewtpr','rewtpr'. The number of random imputations for Redistribution-to-the-right.}
}
\value{
A list containing: the observed treatment effect, a vector of length \code{resample_num} containing resampled treatment effects, a message
indicating the method ran and the type of resampling done, the variance, the p-value (one-sided for treatment benefit), the total wins on treatment (pairwise methods only),
the total losses on treatment (pairwise methods only), a vector of length 'm' with the components of the treatment effect,
a vector of length 'm' with the variance of the components. A warning message will be printed for combinations of \code{type} and \code{model}/\code{resample}
that are not recommended.
}
\description{
This function runs one of the win time methods on observed and resampled data.
}
\details{
The type parameter specifies the procedure you would like to run.
'ewt' is Expected Win Time.
'ewtr' is Expected Win Time Against Reference (Control Arm).
'rmt' is Restricted Mean Time in Favor of Treatment.
'max' is the MAX procedure (max(COMP,EWTR)).
'wtr' is Win Time Ratio.
'rwtr' is Restricted Win Time Ratio.
'pwt' is Pairwise Win Time.
'ewtp' is Expected Win Time Against Trial Population.
'ewtpr' is Expected Win Time Against Trial Population With Redistribution.
'rewtp' is Time Restricted Expected Win Time Against Trial Population.
'rewtpr'is Time Restricted Expected Win Time Against Trial Population With Redistribution.
'rpwt' is Time Restricted Pairwise Win Time.
}
\examples{
# ------------------------------
# Example Inputs
# ------------------------------

# Event time vectors
TIME_1 <- c(256,44,29,186,29,80,11,380,102,33)
TIME_2 <- c(128,44,95,186,69,66,153,380,117,33)
TIME_3 <- c(435,44,95,186,69,270,1063,380,117,33)

# Event time matrix
Time <- rbind(TIME_1, TIME_2, TIME_3)

# Event indicator vectors
DELTA_1 <- c(1,0,1,0,1,1,1,0,1,0)
DELTA_2 <- c(1,0,0,0,0,1,1,0,0,0)
DELTA_3 <- c(0,0,0,0,0,0,0,0,0,0)

# Event indicator matrix
Delta <- rbind(DELTA_1, DELTA_2, DELTA_3)

# Treatment arm indicator vector
trt <- c(1,1,1,1,1,0,0,0,0,0)

# Covariate vectors
cov1 <- c(54,53,55,61,73,65,63,63,82,58,66,66)
cov2 <- c(34.4,32.1,34.7,54.1,55.7,43.6,32.1,44.8,85.2,12.5,33.4,21.4)

# Covariate vectors
cov1 <- c(66,67,54,68,77,65,55,66,77,54)
cov2 <- c(3,6,4,2,3,5,8,5,3,5)
cov3 <- c(34.6,543.6,45.8,54.7,44.3,55.6,65.9,54.7,77.9,31.2)

# Covariate matrix
cov <- cbind(cov1, cov2, cov3)

# ------------------------
# wintime Examples
# ------------------------

# Run WTR
z <- wintime("wtr", Time, Delta, trt)
print(z)

# Run EWT with default settings and 10 resamples
z <- wintime("ewt", Time, Delta, trt, resample_num = 10)
print(z)

# Run EWTR with default settings
z <- wintime("ewtr", Time, Delta, trt, cov = cov)
print(z)

# Run EWTR with KM model (This will produce a warning message)
z <- wintime("ewtr", Time, Delta, trt, cov = cov, model = "km")
print(z)

}
