% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/time_cut.R
\name{time_cut_n}
\alias{time_cut_n}
\alias{time_cut_width}
\alias{time_breaks}
\alias{time_breakpoints}
\alias{time_cut}
\title{Cut dates and datetimes into regularly spaced date or datetime intervals}
\usage{
time_cut_n(
  x,
  n = 5,
  timespan = NULL,
  from = NULL,
  to = NULL,
  time_floor = FALSE,
  week_start = getOption("lubridate.week.start", 1)
)

time_cut_width(x, timespan = granularity(x), from = NULL, to = NULL)

time_breaks(
  x,
  n = 5,
  timespan = NULL,
  from = NULL,
  to = NULL,
  time_floor = FALSE,
  week_start = getOption("lubridate.week.start", 1)
)

time_breakpoints(x, n = 10)

time_cut(
  x,
  n = 5,
  timespan = NULL,
  from = NULL,
  to = NULL,
  time_floor = FALSE,
  week_start = getOption("lubridate.week.start", 1)
)
}
\arguments{
\item{x}{Time vector. \cr
E.g. a \code{Date}, \code{POSIXt}, \code{numeric} or any time-based vector.}

\item{n}{Number of breaks.}

\item{timespan}{\link{timespan}.}

\item{from}{Start time.}

\item{to}{End time.}

\item{time_floor}{Logical. Should the initial date/datetime be
floored before building the sequence?}

\item{week_start}{day on which week starts following ISO conventions - 1
means Monday (default), 7 means Sunday.
This is only used when \code{time_floor = TRUE}.}
}
\value{
\code{time_breaks} and \code{time_breakpoints} both return a vector of breakpoints \cr
\code{time_cut_n} and \code{time_cut_width} returns a \code{time_interval}
}
\description{
Useful functions especially for when plotting time-series. \cr

\code{time_cut_n} makes approximately \code{n} groups of equal time range.
It prioritises the highest time unit possible, making axes look
less cluttered and thus prettier. \cr

\code{time_breaks} returns only the breakpoints.

\code{time_breakpoints} is a newer and faster alternative to
\code{time_breaks} which differs in
that it calls \code{range()} on the input data and therefore need only work with
a vector of 2 values, unlike \code{time_breaks} which requires more data points
to create better looking breaks.
}
\details{
To retrieve regular time breaks that simply spans the range of \code{x},
use \code{time_seq()}to manually specify the range and time width or
\code{time_grid()} to use the range of the supplied data.

By default \code{time_cut_n()} will try to find
the 'prettiest' way of cutting the interval by
trying to cut the date/date-times into
groups of the highest possible time units,
starting at years and ending at milliseconds.

\code{time_breakpoints} does the same but using a different internal method.
}
\examples{
library(timeplyr)
library(fastplyr)
library(cheapr)
library(lubridate)
library(ggplot2)
library(dplyr)

time_cut_n(1:10, n = 5)

# Easily create custom time breaks
df <- nycflights13::flights |>
  f_slice_sample(n = 100) |>
  with_local_seed(.seed = 8192821) |>
  f_select(time_hour) |>
  fastplyr::f_arrange(time_hour) |>
  mutate(date = as_date(time_hour))

# time_cut_n() and time_breaks() automatically find a
# suitable way to cut the data
time_cut_n(df$date) |>
  interval_count()
# Works with datetimes as well
time_cut_n(df$time_hour, n = 5) |>
  interval_count()
time_cut_n(df$date, timespan = "month") |>
  interval_count()
# Just the breaks
time_breaks(df$date, n = 5, timespan = "month")

cut_dates <- time_cut_n(df$date)
date_breaks <- time_breaks(df$date)

# To get exact breaks at regular intervals, use time_grid
weekly_breaks <- time_grid(
  df$date, "5 weeks",
  from = floor_date(min(df$date), "week", week_start = 1)
)
weekly_labels <- format(weekly_breaks, "\%b-\%d")
df |>
  time_by(date, "week", .name = "date") |>
  f_count() |>
  mutate(date = interval_start(date)) |>
  ggplot(aes(x = date, y = n)) +
  geom_bar(stat = "identity") +
  scale_x_date(breaks = weekly_breaks,
               labels = weekly_labels)
}
