% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/summarize_seasons.R
\name{summarize_seasons}
\alias{summarize_seasons}
\title{Seasonal and monthly value ranking}
\usage{
summarize_seasons(dates, values, season_start = 10, n_seasons = 4)
}
\arguments{
\item{dates}{A vector of dates of 'Date' class.}

\item{values}{Numeric values. The values must be in correspondence with \code{dates}, meaning the
\emph{i}th element in \code{values} must correspond to the \emph{i}th date in \code{dates}.}

\item{season_start}{The starting month of the first season, specified as an integer from 1 to
12. Default is 10 (October).}

\item{n_seasons}{The number of seasons in a year. Must be a factor of 12. Default is 4.}
}
\value{
A list with two data frames:
\describe{
\item{monthly}{\tabular{lll}{
\strong{Name} \tab \strong{Type} \tab \strong{Description} \cr
year \tab numeric \tab Year \cr
month \tab numeric \tab Month \cr
avg_value \tab numeric \tab Mean of the \code{values} for that year and month \cr
}}
\item{seasonal}{\tabular{lll}{
\strong{Name} \tab \strong{Type} \tab \strong{Description} \cr
adj_year \tab numeric \tab Adjusted year based on \code{season_start} \cr
season \tab integer \tab Season number; 1 being the first season of the year \cr
avg_value \tab numeric \tab Mean of the \code{values} for that 'adj_year' and 'season' \cr
ys_rank \tab numeric \tab Rank of the 'avg_value' for that 'adj_year' and 'season'; 1 being
the highest 'avg_value' \cr
}}
}
}
\description{
Summarize data by calculating average values for each month and season, and the year-season
rank. Allows for flexible season definitions by specifying a season start month and the number
of seasons. Intended to be used on a (near) daily water quality record.
}
\details{
The start of a season, \code{season_start}, may be any integer from 1 to 12, indicating the month
which is the start of the first season. For example \code{season_start = 1} makes the first season
start in January while \code{season_start = 10} makes the season start in October.

The seasonal average accounts for seasons split across years. For example, if \code{n_seasons = 4}
and \code{season_start = 12}, then season 1 includes December of e.g. 2020, January 2021, and
February 2021. The year is considered to begin in December and is designated by the year in
which it ends (i.e., the seasonally adjusted year). Thus, the seasonal average for season 1 of
2021 would be calculated from December 2020 to February 2021.
}
\examples{
# Four seasons starting in October
date_vec <- seq.Date(
  from = as.Date("2020-05-03"), 
  to = as.Date("2023-10-17"), 
  by = "day"
)
set.seed(123)
num_vec <- sample(30:3000, length(date_vec), replace = TRUE)
# Four seasons starting in October
results <- summarize_seasons(
  dates = date_vec, 
  values = num_vec,  
  season_start = 10, 
  n_seasons = 4
)
print(head(results$monthly))
print(head(results$seasonal))
# Three seasons starting in January
results <- summarize_seasons(
  dates = date_vec, 
  values = num_vec, 
  season_start = 1, 
  n_seasons = 3
)
print(head(results$monthly))
print(head(results$seasonal))

}
