# This is a stripped-down port of the ui.Chat feature in py-shiny. The main
# things it's missing are server-side state management, i.e. the py-shiny
# version will keep the list of messages for you, and will handle the
# trimming of the message history to fit within the context window; these
# are left for the caller to handle in the R version.

chat_deps <- function() {
  htmltools::htmlDependency(
    "shinychat",
    utils::packageVersion("shinychat"),
    package = "shinychat",
    src = "lib/shiny",
    script = list(
      list(src = "chat/chat.js", type = "module"),
      list(src = "markdown-stream/markdown-stream.js", type = "module")
    ),
    stylesheet = c(
      "chat/chat.css",
      "markdown-stream/markdown-stream.css"
    )
  )
}

#' Create a chat UI element
#'
#' @description
#' Inserts a chat UI element into a Shiny UI, which includes a scrollable
#' section for displaying chat messages, and an input field for the user to
#' enter new messages.
#'
#' To respond to user input, listen for `input$ID_user_input` (for example, if
#' `id="my_chat"`, user input will be at `input$my_chat_user_input`), and use
#' [chat_append()] to append messages to the chat.
#'
#' @param id The ID of the chat element
#' @param ... Extra HTML attributes to include on the chat element
#' @param messages A list of messages to prepopulate the chat with. Each
#'   message can be one of the following:
#'
#'   * A string, which is interpreted as markdown and rendered to HTML on
#'     the client.
#'     * To prevent interpreting as markdown, mark the string as
#'       [htmltools::HTML()].
#'   * A UI element.
#'     * This includes [htmltools::tagList()], which take UI elements
#'       (including strings) as children. In this case, strings are still
#'       interpreted as markdown as long as they're not inside HTML.
#'   * A named list of `content` and `role`. The `content` can contain content
#'     as described above, and the `role` can be "assistant" or "user".
#'
#' @param placeholder The placeholder text for the chat's user input field
#' @param width The CSS width of the chat element
#' @param height The CSS height of the chat element
#' @param fill Whether the chat element should try to vertically fill its
#'   container, if the container is
#'   [fillable](https://rstudio.github.io/bslib/articles/filling/index.html)
#' @param icon_assistant The icon to use for the assistant chat messages.
#'   Can be HTML or a tag in the form of [htmltools::HTML()] or
#'   [htmltools::tags()]. If `None`, a default robot icon is used.
#' @returns A Shiny tag object, suitable for inclusion in a Shiny UI
#'
#' @examplesIf interactive()
#' library(shiny)
#' library(bslib)
#' library(shinychat)
#'
#' ui <- page_fillable(
#'   chat_ui("chat", fill = TRUE)
#' )
#'
#' server <- function(input, output, session) {
#'   observeEvent(input$chat_user_input, {
#'     # In a real app, this would call out to a chat client or API,
#'     # perhaps using the 'ellmer' package.
#'     response <- paste0(
#'       "You said:\n\n",
#'       "<blockquote>",
#'       htmltools::htmlEscape(input$chat_user_input),
#'       "</blockquote>"
#'     )
#'     chat_append("chat", response)
#'     chat_append("chat", stream)
#'   })
#' }
#'
#' shinyApp(ui, server)
#'
#' @export
chat_ui <- function(
  id,
  ...,
  messages = NULL,
  placeholder = "Enter a message...",
  width = "min(680px, 100%)",
  height = "auto",
  fill = TRUE,
  icon_assistant = NULL
) {
  attrs <- rlang::list2(...)
  if (!all(nzchar(rlang::names2(attrs)))) {
    rlang::abort("All arguments in ... must be named.")
  }

  message_tags <- lapply(messages, function(x) {
    role <- "assistant"
    content <- x
    if (is.list(x) && ("content" %in% names(x))) {
      content <- x[["content"]]
      role <- x[["role"]] %||% role
    }

    # `content` is most likely a string, so avoid overhead in that case
    # (it's also important that we *don't escape HTML* here).
    if (is.character(content)) {
      ui <- list(html = paste(content, collapse = "\n"))
    } else {
      ui <- with_current_theme(htmltools::renderTags(content))
    }

    tag(
      "shiny-chat-message",
      rlang::list2(
        `data-role` = role,
        content = ui[["html"]],
        icon = if (!is.null(icon_assistant)) as.character(icon_assistant),
        ui[["dependencies"]],
      )
    )
  })

  res <- tag(
    "shiny-chat-container",
    rlang::list2(
      id = id,
      style = css(
        width = width,
        height = height
      ),
      placeholder = placeholder,
      fill = if (isTRUE(fill)) NA else NULL,
      # Also include icon on the parent so that when messages are dynamically added,
      # we know the default icon has changed
      `icon-assistant` = if (!is.null(icon_assistant)) {
        as.character(icon_assistant)
      },
      ...,
      tag("shiny-chat-messages", message_tags),
      tag(
        "shiny-chat-input",
        list(id = paste0(id, "_user_input"), placeholder = placeholder)
      ),
      chat_deps(),
      htmltools::findDependencies(icon_assistant)
    )
  )

  if (isTRUE(fill)) {
    res <- bslib::as_fill_carrier(res)
  }

  tag_require(res, version = 5, caller = "chat_ui")
}

#' Append an assistant response (or user message) to a chat control
#'
#' @description
#' The `chat_append` function appends a message to an existing [chat_ui()]. The
#' `response` can be a string, string generator, string promise, or string
#' promise generator (as returned by the 'ellmer' package's `chat`, `stream`,
#' `chat_async`, and `stream_async` methods, respectively).
#'
#' This function should be called from a Shiny app's server. It is generally
#' used to append the client's response to the chat, while user messages are
#' added to the chat UI automatically by the front-end. You'd only need to use
#' `chat_append(role="user")` if you are programmatically generating queries
#' from the server and sending them on behalf of the user, and want them to be
#' reflected in the UI.
#'
#' # Error handling
#'
#' If the `response` argument is a generator, promise, or promise generator, and
#' an error occurs while producing the message (e.g., an iteration in
#' `stream_async` fails), the promise returned by `chat_append` will reject with
#' the error. If the `chat_append` call is the last expression in a Shiny
#' observer, shinychat will log the error message and show a message that the
#' error occurred in the chat UI.
#'
#' @param id The ID of the chat element
#' @param response The message or message stream to append to the chat element.
#'   The actual message content can one of the following:
#'
#'   * A string, which is interpreted as markdown and rendered to HTML on
#'     the client.
#'     * To prevent interpreting as markdown, mark the string as
#'       [htmltools::HTML()].
#'   * A UI element.
#'     * This includes [htmltools::tagList()], which take UI elements
#'       (including strings) as children. In this case, strings are still
#'       interpreted as markdown as long as they're not inside HTML.
#'
#' @param role The role of the message (either "assistant" or "user"). Defaults
#'   to "assistant".
#' @param icon An optional icon to display next to the message, currently only
#'   used for assistant messages. The icon can be any HTML element (e.g., an
#'   [htmltools::img()] tag) or a string of HTML.
#' @param session The Shiny session object
#'
#' @returns Returns a promise that resolves to the contents of the stream, or an
#'   error. This promise resolves when the message has been successfully sent to
#'   the client; note that it does not guarantee that the message was actually
#'   received or rendered by the client. The promise rejects if an error occurs
#'   while processing the response (see the "Error handling" section).
#'
#' @examplesIf interactive()
#' library(shiny)
#' library(coro)
#' library(bslib)
#' library(shinychat)
#'
#' # Dumbest chatbot in the world: ignores user input and chooses
#' # a random, vague response.
#' fake_chatbot <- async_generator(function(input) {
#'   responses <- c(
#'     "What does that suggest to you?",
#'     "I see.",
#'     "I'm not sure I understand you fully.",
#'     "What do you think?",
#'     "Can you elaborate on that?",
#'     "Interesting question! Let's examine thi... **See more**"
#'   )
#'
#'   await(async_sleep(1))
#'   for (chunk in strsplit(sample(responses, 1), "")[[1]]) {
#'     yield(chunk)
#'     await(async_sleep(0.02))
#'   }
#' })
#'
#' ui <- page_fillable(
#'   chat_ui("chat", fill = TRUE)
#' )
#'
#' server <- function(input, output, session) {
#'   observeEvent(input$chat_user_input, {
#'     response <- fake_chatbot(input$chat_user_input)
#'     chat_append("chat", response)
#'   })
#' }
#'
#' shinyApp(ui, server)
#'
#' @export
chat_append <- function(
  id,
  response,
  role = c("assistant", "user"),
  icon = NULL,
  session = getDefaultReactiveDomain()
) {
  check_active_session(session)
  role <- match.arg(role)

  stream <- as_generator(response)
  chat_append_stream(id, stream, role = role, icon = icon, session = session)
}

#' Low-level function to append a message to a chat control
#'
#' For advanced users who want to control the message chunking behavior. Most
#' users should use [chat_append()] instead.
#'
#' @param id The ID of the chat element
#' @param msg The message to append. Should be a named list with `role` and
#'   `content` fields. The `role` field should be either "user" or "assistant".
#'   The `content` field should be a string containing the message content, in
#'   Markdown format.
#' @param chunk Whether `msg` is just a chunk of a message, and if so, what
#'   type. If `FALSE`, then `msg` is a complete message. If `"start"`, then
#'   `msg` is the first chunk of a multi-chunk message. If `"end"`, then `msg`
#'   is the last chunk of a multi-chunk message. If `TRUE`, then `msg` is an
#'   intermediate chunk of a multi-chunk message. Default is `FALSE`.
#' @param operation The operation to perform on the message. If `"append"`,
#'   then the new content is appended to the existing message content. If
#'   `"replace"`, then the existing message content is replaced by the new
#'   content. Ignored if `chunk` is `FALSE`.
#' @param icon An optional icon to display next to the message, currently only
#'   used for assistant messages. The icon can be any HTML element (e.g.,
#'   [htmltools::img()] tag) or a string of HTML.
#' @param session The Shiny session object
#'
#' @returns Returns nothing (\code{invisible(NULL)}).
#'
#' @importFrom shiny getDefaultReactiveDomain
#'
#' @examplesIf interactive()
#' library(shiny)
#' library(coro)
#' library(bslib)
#' library(shinychat)
#'
#' # Dumbest chatbot in the world: ignores user input and chooses
#' # a random, vague response.
#' fake_chatbot <- async_generator(function(id, input) {
#'   responses <- c(
#'     "What does that suggest to you?",
#'     "I see.",
#'     "I'm not sure I understand you fully.",
#'     "What do you think?",
#'     "Can you elaborate on that?",
#'     "Interesting question! Let's examine thi... **See more**"
#'   )
#'
#'   # Use low-level chat_append_message() to temporarily set a progress message
#'   chat_append_message(id, list(role = "assistant", content = "_Thinking..._ "))
#'   await(async_sleep(1))
#'   # Clear the progress message
#'   chat_append_message(id, list(role = "assistant", content = ""), operation = "replace")
#'
#'   for (chunk in strsplit(sample(responses, 1), "")[[1]]) {
#'     yield(chunk)
#'     await(async_sleep(0.02))
#'   }
#' })
#'
#' ui <- page_fillable(
#'   chat_ui("chat", fill = TRUE)
#' )
#'
#' server <- function(input, output, session) {
#'   observeEvent(input$chat_user_input, {
#'     response <- fake_chatbot("chat", input$chat_user_input)
#'     chat_append("chat", response)
#'   })
#' }
#'
#' shinyApp(ui, server)
#'
#' @export
chat_append_message <- function(
  id,
  msg,
  chunk = TRUE,
  operation = c("append", "replace"),
  icon = NULL,
  session = getDefaultReactiveDomain()
) {
  check_active_session(session)

  if (!is.list(msg)) {
    rlang::abort("`msg` must be a named list with 'role' and 'content' fields")
  }
  if (!isTRUE(msg[["role"]] %in% c("user", "assistant"))) {
    warning("Invalid role argument; must be 'user' or 'assistant'")
    return(invisible(NULL))
  }

  if (!isFALSE(chunk)) {
    msg_type <- "shiny-chat-append-message-chunk"
    if (chunk == "start") {
      chunk_type <- "message_start"
    } else if (chunk == "end") {
      chunk_type <- "message_end"
    } else if (isTRUE(chunk)) {
      chunk_type <- NULL
    } else {
      rlang::abort("Invalid chunk argument")
    }
  } else {
    msg_type <- "shiny-chat-append-message"
    chunk_type <- NULL
  }

  content <- msg[["content"]]
  is_html <- inherits(
    content,
    c(
      "shiny.tag",
      "shiny.tag.list",
      "html",
      "htmlwidget",
      "shinychat_tool_card"
    )
  )
  content_type <- if (is_html) "html" else "markdown"

  operation <- match.arg(operation)
  if (identical(operation, "replace")) {
    operation <- NULL
  }

  if (is.character(content)) {
    # content is most likely a string, so avoid overhead in that case
    ui <- list(html = content, deps = "[]")
  } else {
    # process_ui() does *not* render markdown->HTML, but it does:
    # 1. Extract and register HTMLdependency()s with the session.
    # 2. Returns a HTML string representation of the TagChild
    #    (i.e., `div()` -> `"<div>"`).
    ui <- process_ui(content, session)
  }

  msg_content <- ui[["html"]]
  if (is_html) {
    # Code blocks with `{=html}` infostrings are rendered as-is by a custom
    # rendering method in markdown-stream.ts
    msg_content <- sprintf(
      "\n\n````````{=html}\n%s\n````````\n\n",
      msg_content
    )
  }

  msg <- list(
    content = msg_content,
    role = msg[["role"]],
    content_type = content_type,
    html_deps = ui[["deps"]],
    chunk_type = chunk_type,
    operation = operation
  )

  if (!is.null(icon)) {
    msg$icon <- as.character(icon)
  }

  session$sendCustomMessage(
    "shinyChatMessage",
    list(
      id = resolve_id(id, session),
      handler = msg_type,
      obj = msg
    )
  )

  invisible(NULL)
}

chat_append_stream <- function(
  id,
  stream,
  role = "assistant",
  icon = NULL,
  session = getDefaultReactiveDomain()
) {
  result <- chat_append_stream_impl(id, stream, role, icon, session)
  result <- chat_update_bookmark(id, result, session = session)
  # Handle erroneous result...
  result <- promises::catch(result, function(reason) {
    # ...but rethrow the error as a silent error, so the caller can also handle
    # it if they want, but it won't bring down the app.
    class(reason) <- c("shiny.silent.error", class(reason))
    cnd_signal(reason)
  })

  promises::catch(result, function(reason) {
    chat_append_message(
      id,
      list(
        role = role,
        content = sanitized_chat_error(reason)
      ),
      chunk = "end",
      operation = "append",
      session = session
    )
    rlang::warn(
      sprintf(
        "ERROR: An error occurred in `chat_append_stream(id=\"%s\")`",
        session$ns(id)
      ),
      parent = reason
    )
  })

  # Note that we're not returning the result of `promises::catch()`, because we
  # want to return a rejected promise so the caller can see the error. But we
  # use the `catch()` both to make the error visible to the user *and* to ensure
  # there's no "unhandled promise error" warning if the caller chooses not to do
  # anything with it.
  result
}

utils:::globalVariables(c("generator_env", "exits", "yield"))

chat_append_stream_impl <- NULL
rlang::on_load(
  chat_append_stream_impl <- coro::async(function(
    id,
    stream,
    role = "assistant",
    icon = NULL,
    session = shiny::getDefaultReactiveDomain()
  ) {
    chat_append_ <- function(content, chunk = TRUE, ...) {
      chat_append_message(
        id,
        msg = list(role = role, content = content),
        operation = "append",
        chunk = chunk,
        session = session,
        ...
      )
    }

    chat_append_("", chunk = "start", icon = icon)

    res <- fastmap::fastqueue(200)

    for (msg in stream) {
      if (promises::is.promising(msg)) {
        msg <- await(msg)
      }
      if (coro::is_exhausted(msg)) {
        break
      }

      res$add(msg)

      if (S7::S7_inherits(msg, ellmer::ContentToolResult)) {
        if (!is.null(msg@request)) {
          session$sendCustomMessage("shiny-tool-request-hide", msg@request@id)
        }
      }

      if (S7::S7_inherits(msg, ellmer::Content)) {
        msg <- contents_shinychat(msg)
      }

      chat_append_(msg)
    }

    chat_append_("", chunk = "end")

    res <- res$as_list()
    if (every(res, is.character)) {
      paste(unlist(res), collapse = "")
    } else {
      res
    }
  })
)


#' Clear all messages from a chat control
#'
#' @param id The ID of the chat element
#' @param session The Shiny session object
#'
#' @export
#' @examplesIf interactive()
#'
#' library(shiny)
#' library(bslib)
#'
#' ui <- page_fillable(
#'   chat_ui("chat", fill = TRUE),
#'   actionButton("clear", "Clear chat")
#' )
#'
#' server <- function(input, output, session) {
#'   observeEvent(input$clear, {
#'     chat_clear("chat")
#'   })
#'
#'   observeEvent(input$chat_user_input, {
#'     response <- paste0("You said: ", input$chat_user_input)
#'     chat_append("chat", response)
#'   })
#' }
#'
#' shinyApp(ui, server)
chat_clear <- function(id, session = getDefaultReactiveDomain()) {
  check_active_session(session)

  session$sendCustomMessage(
    "shinyChatMessage",
    list(
      id = resolve_id(id, session),
      handler = "shiny-chat-clear-messages",
      obj = NULL
    )
  )
}


#' Update the user input of a chat control
#'
#' @param id The ID of the chat element
#' @param ... Currently unused, but reserved for future use.
#' @param value The value to set the user input to. If `NULL`, the input will not be updated.
#' @param placeholder The placeholder text for the user input
#' @param submit Whether to automatically submit the text for the user. Requires `value`.
#' @param focus Whether to move focus to the input element. Requires `value`.
#' @param session The Shiny session object
#'
#' @export
#' @examplesIf interactive()
#' library(shiny)
#' library(bslib)
#' library(shinychat)
#'
#' ui <- page_fillable(
#'   chat_ui("chat"),
#'   layout_columns(
#'     fill = FALSE,
#'     actionButton("update_placeholder", "Update placeholder"),
#'     actionButton("update_value", "Update user input")
#'   )
#' )
#'
#' server <- function(input, output, session) {
#'   observeEvent(input$update_placeholder, {
#'     update_chat_user_input("chat", placeholder = "New placeholder text")
#'   })
#'
#'   observeEvent(input$update_value, {
#'     update_chat_user_input("chat", value = "New user input", focus = TRUE)
#'   })
#'
#'   observeEvent(input$chat_user_input, {
#'     response <- paste0("You said: ", input$chat_user_input)
#'     chat_append("chat", response)
#'   })
#' }
#'
#' shinyApp(ui, server)

update_chat_user_input <- function(
  id,
  ...,
  value = NULL,
  placeholder = NULL,
  submit = FALSE,
  focus = FALSE,
  session = getDefaultReactiveDomain()
) {
  rlang::check_dots_empty()
  check_active_session(session)

  if (is.null(value) && (submit || focus)) {
    rlang::abort(
      "An input `value` must be provided when `submit` or `focus` are `TRUE`."
    )
  }

  vals <- drop_nulls(
    list(
      value = value,
      placeholder = placeholder,
      submit = submit,
      focus = focus
    )
  )

  session$sendCustomMessage(
    "shinyChatMessage",
    list(
      id = resolve_id(id, session),
      handler = "shiny-chat-update-user-input",
      obj = vals
    )
  )
}
