% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tt_dotabulation.R
\name{build_table}
\alias{build_table}
\title{Create a table from a layout and data}
\usage{
build_table(
  lyt,
  df,
  alt_counts_df = NULL,
  col_counts = NULL,
  col_total = if (is.null(alt_counts_df)) nrow(df) else nrow(alt_counts_df),
  topleft = NULL,
  hsep = default_hsep(),
  round_type = if (is(lyt, "PreDataTableLayouts")) obj_round_type(lyt) else
    valid_round_type,
  ...
)
}
\arguments{
\item{lyt}{(\code{PreDataTableLayouts})\cr layout object pre-data used for tabulation.}

\item{df}{(\code{data.frame} or \code{tibble})\cr dataset.}

\item{alt_counts_df}{(\code{data.frame} or \code{tibble})\cr alternative full dataset the rtables framework will use
\emph{only} when calculating column counts.}

\item{col_counts}{(\code{numeric} or \code{NULL})\cr \ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#deprecated}{\figure{lifecycle-deprecated.svg}{options: alt='[Deprecated]'}}}{\strong{[Deprecated]}} if non-\code{NULL}, column counts
\emph{for leaf-columns only} which override those calculated automatically during tabulation. Must specify
"counts" for \emph{all} leaf-columns if non-\code{NULL}. \code{NA} elements will be replaced with the automatically
calculated counts. Turns on display of leaf-column counts when non-\code{NULL}.}

\item{col_total}{(\code{integer(1)})\cr the total observations across all columns. Defaults to \code{nrow(df)}.}

\item{topleft}{(\code{character})\cr override values for the "top left" material to be displayed during printing.}

\item{hsep}{(\code{string})\cr set of characters to be repeated as the separator between the header and body of
the table when rendered as text. Defaults to a connected horizontal line (unicode 2014) in locals that use a UTF
charset, and to \code{-} elsewhere (with a once per session warning). See \code{\link[formatters:default_horizontal_sep]{formatters::set_default_hsep()}} for further
information.}

\item{round_type}{(\code{"iec"} (default), \code{"iec_mod"} or \code{"sas"})\cr the type of rounding to perform.
See \code{\link[formatters:format_value]{formatters::format_value()}} for details.}

\item{...}{ignored.}
}
\value{
A \code{TableTree} or \code{ElementaryTable} object representing the table created by performing the tabulations
declared in \code{lyt} to the data \code{df}.
}
\description{
Layouts are used to describe a table pre-data. \code{build_table} is used to create a table
using a layout and a dataset.
}
\details{
When \code{alt_counts_df} is specified, column counts are calculated by applying the exact column subsetting
expressions determined when applying column splitting to the main data (\code{df}) to \code{alt_counts_df} and
counting the observations in each resulting subset.

In particular, this means that in the case of splitting based on cuts of the data, any dynamic cuts will have
been calculated based on \code{df} and simply re-used for the count calculation.
}
\note{
When overriding the column counts or totals care must be taken that, e.g., \code{length()} or \code{nrow()} are not called
within tabulation functions, because those will NOT give the overridden counts. Writing/using tabulation
functions which accept \code{.N_col} and \code{.N_total} or do not rely on column counts at all (even implicitly) is the
only way to ensure overridden counts are fully respected.
}
\examples{
lyt <- basic_table() \%>\%
  split_cols_by("Species") \%>\%
  analyze("Sepal.Length", afun = function(x) {
    list(
      "mean (sd)" = rcell(c(mean(x), sd(x)), format = "xx.xx (xx.xx)"),
      "range" = diff(range(x))
    )
  })
lyt

tbl <- build_table(lyt, iris)
tbl

# analyze multiple variables
lyt2 <- basic_table() \%>\%
  split_cols_by("Species") \%>\%
  analyze(c("Sepal.Length", "Petal.Width"), afun = function(x) {
    list(
      "mean (sd)" = rcell(c(mean(x), sd(x)), format = "xx.xx (xx.xx)"),
      "range" = diff(range(x))
    )
  })

tbl2 <- build_table(lyt2, iris)
tbl2

# an example more relevant for clinical trials with column counts
lyt3 <- basic_table(show_colcounts = TRUE) \%>\%
  split_cols_by("ARM") \%>\%
  analyze("AGE", afun = function(x) {
    setNames(as.list(fivenum(x)), c(
      "minimum", "lower-hinge", "median",
      "upper-hinge", "maximum"
    ))
  })

tbl3 <- build_table(lyt3, DM)
tbl3

tbl4 <- build_table(lyt3, subset(DM, AGE > 40))
tbl4

# with column counts calculated based on different data
miniDM <- DM[sample(1:NROW(DM), 100), ]
tbl5 <- build_table(lyt3, DM, alt_counts_df = miniDM)
tbl5

tbl6 <- build_table(lyt3, DM, col_counts = 1:3)
tbl6

}
\author{
Gabriel Becker
}
