% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ee_as_sf.R
\name{ee_as_sf}
\alias{ee_as_sf}
\title{Convert an Earth Engine table into a sf object}
\usage{
ee_as_sf(
  x,
  dsn,
  overwrite = TRUE,
  via = "getInfo",
  container = "rgee_backup",
  crs = NULL,
  maxFeatures = 5000,
  selectors = NULL,
  lazy = FALSE,
  public = FALSE,
  add_metadata = TRUE,
  timePrefix = TRUE,
  quiet = FALSE
)
}
\arguments{
\item{x}{Earth Engine table (ee$FeatureCollection) to be converted into a sf
object.}

\item{dsn}{Character. Output filename. In case \code{dsn} is missing,
a shapefile is created in the \code{tmp()} directory.}

\item{overwrite}{Logical. Delete data source \code{dsn} before attempting
to write?.}

\item{via}{Character. Method to export the image. Three method are
implemented: "getInfo", "drive", "gcs". See details.}

\item{container}{Character. Name of the folder ('drive') or bucket ('gcs')
to be exported into (ignored if \code{via} is not defined as "drive" or
"gcs").}

\item{crs}{Integer or Character. Coordinate Reference System (CRS)
for the EE table. If it is NULL, \code{ee_as_sf} will take the CRS of
the first element.}

\item{maxFeatures}{Numeric. The maximum number of features allowed for export (ignore if \code{via} is not set as "getInfo"). The task will fail
if the exported region covers more features than the specified in
\code{maxFeatures}. Default is 5000.}

\item{selectors}{List of properties to include in the output, as a
list/vector of strings or a comma-separated string. By default, all properties are
included.}

\item{lazy}{Logical. If TRUE, a \code{\link[future:sequential]{
future::sequential}} object is created to evaluate the task in the future.
Ignore if \code{via} is set as "getInfo". See details.}

\item{public}{Logical. If TRUE, a public link to the file is created.
See details.}

\item{add_metadata}{Add metadata to the sf object. See details.}

\item{timePrefix}{Logical. Add current date and time (\code{Sys.time()}) as
a prefix to export files. This parameter helps to prevent exported files from
having the same name. By default TRUE.}

\item{quiet}{Logical. Suppress info message.}
}
\value{
An sf object.
}
\description{
Convert an Earth Engine table into a sf object
}
\details{
\code{ee_as_sf} supports the download of \code{ee$Geometry}, \code{ee$Feature},
and \code{ee$FeatureCollection} by three different options:
"getInfo" (which make an REST call to retrieve the data), "drive"
(which use \href{https://CRAN.R-project.org/package=googledrive}{Google Drive})
and "gcs" (which use \href{https://CRAN.R-project.org/package=googleCloudStorageR}{
Google Cloud Storage}). The advantage of using "getInfo" is a
direct and faster download. However, there is a limit of 5000 features by
request, which makes it not recommendable for large FeatureCollection. Instead of
"getInfo", the "drive" and "gcs" options are suitable for large FeatureCollections
because they use an intermediate container. When \code{via} is set as "drive" or "gcs"
\code{ee_as_sf} performs the following steps:
\describe{
\item{1. }{A task is started (i.e., \code{ee$batch$Task$start()}) to
move the EE Table from Earth Engine to the file storage system (Google Drive
or Google Cloud Storage) specified in the \code{via} argument.}
\item{2. }{If the argument \code{lazy} is TRUE, the task will not be
monitored. This is useful for launching several tasks simultaneously and
calling them later using \code{\link{ee_utils_future_value}} or
\code{\link[future:value]{future::value}}. At the end of this step,
the EE Table is stored under the path specified by the argument
\code{dsn}.}
\item{3. }{Finally, if the argument \code{add_metadata} is TRUE, a list
with the following elements is added to the sf object.
\describe{
\item{\bold{if via is "drive":}}{
\describe{
\item{\bold{ee_id: }}{Earth Engine task name.}
\item{\bold{drive_name: }}{Google Drive table name}
\item{\bold{drive_id: }}{Google Drive table ID}
\item{\bold{drive_download_link: }}{Link to download the table}
}}
}
\describe{
\item{\bold{if via is "gcs":}}{
\describe{
\item{\bold{ee_id: }}{Earth Engine task name.}
\item{\bold{gcs_name: }}{Google Cloud Storage table name}
\item{\bold{gcs_bucket: }}{Bucket name}
\item{\bold{gcs_fileFormat: }}{Table format}
\item{\bold{gcs_public_link: }}{Link to download the table.}
\item{\bold{gcs_URI: }}{gs:// link to the table.}
}}
}
Run \code{attr(sf, "metadata")} to get the list.
}
}

To get more information about exporting data from Earth Engine, take
a look at the
\href{https://developers.google.com/earth-engine/guides/exporting}{Google
Earth Engine Guide - Export data}.
}
\examples{
\dontrun{
library(rgee)

ee_Initialize(drive = TRUE, gcs = TRUE)

# Region of interest
roi <- ee$Geometry$Polygon(list(
  c(-122.275, 37.891),
  c(-122.275, 37.868),
  c(-122.240, 37.868),
  c(-122.240, 37.891)
))

# TIGER: US Census Blocks Dataset
blocks <- ee$FeatureCollection("TIGER/2010/Blocks")
subset <- blocks$filterBounds(roi)
sf_subset <- ee_as_sf(x = subset)
plot(sf_subset)

# Create Random points in Earth Engine
region <- ee$Geometry$Rectangle(-119.224, 34.669, -99.536, 50.064)
ee_help(ee$FeatureCollection$randomPoints)
ee_randomPoints <- ee$FeatureCollection$randomPoints(region, 100)

# Download via GetInfo
sf_randomPoints <- ee_as_sf(ee_randomPoints)
plot(sf_randomPoints)

# Download via drive
sf_randomPoints_drive <- ee_as_sf(
  x = ee_randomPoints,
  via = 'drive'
)

# Download via GCS
sf_randomPoints_gcs <- ee_as_sf(
 x = subset,
 via = 'gcs',
 container = 'rgee_dev' #GCS bucket name
)
}
}
\concept{vector download functions}
