#' Save a Visualization to File (PDF, PNG, or SVG)
#'
#' This function saves a visualization object to a file in the specified
#' format and directory. It supports visualizations generated by
#' `chea3_visualizeRank()`, `ggplot2`, or any other plot object that
#' can be rendered using `print()` inside a graphics device. Optionally,
#' the current date (stored in the `today` variable) can be prepended to
#' the filename.
#'
#' @param viz A visualization object typically created by
#'   `visualizeRank()`, but can also be a `ggplot2` plot or
#'   any other plot object printable with `print()`.
#' @param output_dir A string specifying the output directory. This parameter
#' is required and has no default.
#' @param output_file A string specifying the base filename (without
#'   extension). Defaults to `"viz_rChEA3"`.
#' @param format Output format. One of `"pdf"`, `"png"`, or `"svg"`.
#'   Defaults to `"pdf"`.
#' @param with_date Logical (default `TRUE`). Whether to prepend the
#'   current date (from `today`) to the filename.
#' @param width Width of the output file in inches. Default is 8.
#' @param height Height of the output file in inches. Default is 5.
#' @param resolution Resolution in DPI (only used for PNG). Default is
#'   300.
#' @param verbose Logical. If `TRUE`, print a message with the saved
#'   path. Default `TRUE`.
#'
#' @return The visualization is saved to a file on disk. Invisibly
#'   returns the full path to the saved file.
#' @export
#'
#' @examples
#' \donttest{
#'     genes <- c("TP53", "MYC", "STAT3")
#'     results <- queryChEA3(genes, verbose = FALSE)
#'     meanRank_res <- results[["Integrated--meanRank"]]
#'
#'     # Create visualization
#'     viz <- visualizeRank(meanRank_res)
#'
#'     # Save as PDF
#'     saveViz(viz, output_dir = tempdir(), output_file = "chea3_results")
#'
#'     # Save as PNG with custom dimensions
#'     saveViz(viz, output_dir = tempdir(), output_file = "chea3_results",
#'             format = "png", width = 10, height = 6)
#' }
saveViz <- function(viz,
                    output_dir,
                    output_file = "figure_rChEA3",
                    format = "pdf",
                    with_date = TRUE,
                    width = 8,
                    height = 5,
                    resolution = 300,
                    verbose = TRUE) {
    format <- match.arg(format, choices = c("pdf", "png", "svg"))

    # Check that output_dir is provided
    if (missing(output_dir)) {
        stop("'output_dir' must be specified. Please specify your desired directory.",
             call. = FALSE)
    }

    if (!dir.exists(output_dir)) {
        dir.create(output_dir, recursive = TRUE)
    }

    if (isTRUE(with_date)) {
        output_file <- paste0(get_today(), "_", output_file)
    }

    filepath <- file.path(output_dir, paste0(output_file, ".", format))

    switch(format,
           pdf = grDevices::cairo_pdf(file = filepath, width = width, height = height),
           png = grDevices::png(file = filepath, width = width, height = height, units = "in", res = resolution),
           svg = grDevices::svg(file = filepath, width = width, height = height))

    print(viz)
    grDevices::dev.off()

    if (isTRUE(verbose)) {
        message(" > Visualization (", format, ") saved in ", filepath)
    }
    invisible(filepath)
}
