#' @title Korsmeyer-Peppas Drug Release Kinetic Model
#' @name korsmeyer_peppas_model
#' @description
#' Fits experimental cumulative drug release data to the Korsmeyer-Peppas model
#' using log10-transformed fraction released vs. log10-transformed time. The function
#' automatically normalizes cumulative percent drug release to fraction (0-1) by default
#' and removes t = 0. In addition, the function supports optional grouping variables
#' (e.g., formulation, batch) and optional pH-dependent analysis. It generates
#' publication-quality plots with experimental curves, fitted Korsmeyer-Peppas
#' straight lines, mechanism table, and annotations for Korsmeyer-Peppas release
#' constant (kKP), release exponent (n), intercept, coefficient of determination
#' (R^2), and the time required for 50-percent drug release (t50).
#'
#' Users can toggle 'normalize = TRUE/FALSE' to use fraction (0-1) or raw percent drug release.
#'
#' Model:
#' log10(Mt/MInf) = log10(k) + n * log10(t)
#'
#' The release exponent, n, indicates the drug release mechanism:
#'      - n = 0.5 : Fickian diffusion
#'      - 0.5 < n < 1 : Non-Fickian (anomalous) diffusion
#'      - n = 1   : Case II transport (zero-order release)
#'      - n > 1   : Super Case II transport
#'
#' @param data A data frame containing experimental cumulative percent drug release.
#' @param time_col Character string specifying the column name for time (minutes).
#' @param release_col Character string specifying the column name for
#' cumulative percent drug release.
#' @param group_col Optional character string specifying a grouping variable
#' (e.g., formulation, batch).
#' @param pH_col Optional character string specifying a column containing pH values.
#' @param plot Logical; if TRUE, generates a plot with fitted Korsmeyer-Peppas lines.
#' @param annotate Logical; if TRUE, annotates the plot with kKP, n, intercept,
#' R^2, and t50 (only if <= 2 groups).
#' @param normalize Logical; if TRUE (default), normalizes cumulative percent release
#' to fraction (0-1). If FALSE, retains original percent values.
#'
#' @import stats
#' @import gridExtra
#' @import ggplot2
#' @importFrom stats lm na.omit
#' @importFrom gridExtra tableGrob ttheme_default grid.arrange
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_smooth geom_text
#' labs theme theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with kKP, n, intercept, R^2, and t50 for each group.}
#'   \item{\code{data}}{Processed data used for fitting and plotting.}
#' }
#' @examples
#' # Example I: Single formulation
#' df1 <- data.frame(
#'   time = c(0, 15, 30, 45, 60, 90, 120, 150, 180),
#'   release = c(0, 11.4, 20.8, 30.8, 39.8, 57.8, 72, 84.8, 93.5)
#' )
#' korsmeyer_peppas_model(
#'   data = df1,
#'   time_col = "time",
#'   release_col = "release",
#'   normalize = TRUE  # default
#' )
#'
#' # Example II: Two formulations (grouped, not pH-dependent)
#' df2 <- data.frame(
#'   time = rep(c(0, 30, 60, 90, 120, 150), 2),
#'   release = c(
#'     0, 18, 35, 55, 72, 88,  # Formulation A
#'     0, 12, 26, 40, 58, 70   # Formulation B
#'   ),
#'   formulation = rep(c("Formulation A", "Formulation B"), each = 6)
#' )
#' korsmeyer_peppas_model(
#'   data = df2,
#'   time_col = "time",
#'   release_col = "release",
#'   group_col = "formulation"
#' )
#'
#' # Example III: pH-dependent release
#' df_pH <- data.frame(
#'   time = rep(c(0, 60, 120, 180), 2),
#'   release = c(0, 40, 75, 95, 0, 30, 60, 80),
#'   pH = rep(c(7.4, 4.5), each = 4)
#' )
#' korsmeyer_peppas_model(
#'   data = df_pH,
#'   time_col = "time",
#'   release_col = "release",
#'   pH_col = "pH"
#' )
#'
#' # Example IV: Two formulations under two pH conditions
#' df1 <- data.frame(
#'   time = rep(c(0, 20, 40, 60, 80, 100, 120, 140, 160, 180), 2),
#'   release = c(
#'     0, 12, 25, 38, 50, 62, 73, 82, 89, 95,  # pH 4.5
#'     0, 15, 30, 45, 59, 70, 79, 86, 91, 97   # pH 7.6
#'   ),
#'   pH = rep(c(4.5, 7.6), each = 10)
#' )
#' df2 <- data.frame(
#'   time = rep(c(0, 15, 30, 45, 60, 75, 90, 105, 120, 135), 2),
#'   release = c(
#'     0, 10, 22, 34, 46, 57, 67, 76, 84, 91,  # pH 4.5
#'     0, 13, 27, 41, 55, 67, 77, 85, 92, 98   # pH 7.6
#'   ),
#'   pH = rep(c(4.5, 7.6), each = 10)
#' )
#' df_all <- rbind(
#'   cbind(formulation = "Dataset 1", df1),
#'   cbind(formulation = "Dataset 2", df2)
#' )
#' korsmeyer_peppas_model(
#'   data = df_all,
#'   time_col = "time",
#'   release_col = "release",
#'   group_col = "formulation",
#'   pH_col = "pH"
#' )
#' @references Korsmeyer, R. W., Gurny, R., Doelker, E., Buri, P., & Peppas, N. A.
#' (1983) <doi:10.1016/0378-5173(83)90064-9> Mechanisms of solute release from
#' porous hydrophilic polymers. International Journal of Pharmaceutics, 15(1), 25–35.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "release", "log_time", "log_release", "group",
                         "kKP", "n", "intercept", "R2", "t50", "label", "x_pos",
                         "y_pos", "hjust", "vjust", "Mechanism"))

korsmeyer_peppas_model <- function(data,
                                   time_col = "time",
                                   release_col = "release",
                                   group_col = NULL,
                                   pH_col = NULL,
                                   plot = TRUE,
                                   annotate = TRUE,
                                   normalize = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE)) stop("Package 'ggplot2' is required.")
  if (!requireNamespace("gridExtra", quietly = TRUE)) stop("Package 'gridExtra' is required.")

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(time_col, release_col, group_col, pH_col), drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("time", "release")

  if (normalize) df$release <- df$release / 100

  df <- df[df$time > 0 & df$release > 0, ]
  df$log_time <- log10(df$time)
  df$log_release <- log10(df$release)

  # -------------------------
  # Grouping
  # -------------------------
  if (!is.null(group_col) && !is.null(pH_col)) {
    df$group <- paste0(df[[group_col]], " | pH ", df[[pH_col]])
  } else if (!is.null(pH_col)) {
    df$group <- paste0("pH ", df[[pH_col]])
  } else if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- "Experimental"
  }
  df$group <- as.factor(df$group)

  # -------------------------
  # Korsmeyer-Peppas fitting
  # -------------------------
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {
    fit <- stats::lm(log_release ~ log_time, data = d)
    s <- summary(fit)

    n <- coef(fit)[2]
    intercept <- coef(fit)[1]
    kKP <- 10^intercept

    t50 <- if (!is.na(n) && n != 0) {
      10^((log10(0.5) - intercept) / n)
    } else NA_real_

    data.frame(
      group = unique(d$group),
      kKP = kKP,
      n = n,
      intercept = intercept,
      R2 = s$r.squared,
      t50 = t50
    )
  }))

  # -------------------------
  # Mechanism table
  # -------------------------
  mechanism_table <- data.frame(
    n = c("n = 0.5", "0.5 < n < 1", "n = 1", "n > 1"),
    Mechanism = c(
      "Fickian diffusion",
      "Non-Fickian diffusion",
      "Case II transport",
      "Super Case II transport"
    )
  )

  table_grob <- gridExtra::tableGrob(
    mechanism_table,
    rows = NULL,
    theme = gridExtra::ttheme_default(
      core = list(fg_params = list(cex = 0.6)),
      colhead = list(fg_params = list(cex = 0.6))
    )
  )

  # -------------------------
  # Plot
  # -------------------------
  if (plot) {
    p <- ggplot2::ggplot(df, ggplot2::aes(x = log_time, y = log_release, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(linewidth = 1) +
      ggplot2::geom_smooth(method = "lm", se = FALSE, color = "black", linewidth = 1) +
      ggplot2::labs(
        title = "Korsmeyer-Peppas Drug Release Kinetics",
        subtitle = ifelse(normalize,
                          "Log10(fraction released) vs Log10(time)",
                          "Log10(% release) vs Log10(time)"),
        x = "log10(Time)",
        y = ifelse(normalize,
                   "log10(Fraction Drug Released)",
                   "log10(Cumulative % Drug Released)"),
        color = "Condition"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    num_groups <- nlevels(df$group)
    if (annotate && num_groups <= 2) {
      ann <- fit_results
      ann$label <- paste0(
        "kKP = ", signif(ann$kKP, 3), "\n",
        "n = ", round(ann$n, 3), "\n",
        "intercept = ", round(ann$intercept, 3), "\n",
        "R^2 = ", round(ann$R2, 3), "\n",
        "t50 = ", round(ann$t50, 1)
      )

      x_min <- min(df$log_time)
      x_max <- max(df$log_time)
      y_min <- min(df$log_release)
      y_max <- max(df$log_release)

      ann$x_pos <- c(x_min + 0.05 * (x_max - x_min),
                     x_max - 0.05 * (x_max - x_min))[seq_len(nrow(ann))]
      ann$y_pos <- c(y_max - 0.05 * (y_max - y_min),
                     y_min + 0.05 * (y_max - y_min))[seq_len(nrow(ann))]

      ann$hjust <- c(0, 1)[seq_len(nrow(ann))]
      ann$vjust <- c(1, 0)[seq_len(nrow(ann))]

      p <- p + ggplot2::geom_text(
        data = ann,
        ggplot2::aes(x = x_pos, y = y_pos, label = label, color = group),
        hjust = ann$hjust,
        vjust = ann$vjust,
        size = 4,
        show.legend = FALSE
      )
    } else if (annotate && num_groups > 2) {
      message("More than 2 groups detected - annotations disabled to prevent overlap.")
    }

    gridExtra::grid.arrange(
      p, table_grob,
      ncol = 2,
      widths = c(3, 1)
    )
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
