% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clustered_neurovec.R
\name{ClusteredNeuroVec}
\alias{ClusteredNeuroVec}
\title{ClusteredNeuroVec: Cluster-aware 4D neuroimaging data}
\usage{
ClusteredNeuroVec(x, cvol, FUN = mean, weights = NULL, label = "")
}
\arguments{
\item{x}{Either a `NeuroVec` object to be reduced by clusters, or a pre-computed 
numeric matrix of cluster time-series (T x K, where T=time points, K=clusters)}

\item{cvol}{A `ClusteredNeuroVol` object defining the cluster assignments}

\item{FUN}{Reduction function to aggregate voxels within clusters (default: mean).
Common choices include \code{mean}, \code{median}, or custom functions.}

\item{weights}{Optional numeric vector of per-voxel weights for weighted aggregation.
Must have length equal to the number of non-zero voxels in the mask.}

\item{label}{Optional character label for the object (default: "")}
}
\value{
A \code{ClusteredNeuroVec} object containing:
\describe{
  \item{cvol}{The input ClusteredNeuroVol defining cluster structure}
  \item{ts}{A TxK matrix of cluster time-series (T=timepoints, K=clusters)}
  \item{cl_map}{Integer vector mapping linear voxel indices to cluster IDs}
  \item{label}{Character label for the object}
}
}
\description{
`ClusteredNeuroVec` creates a 4D array-like object where voxels are grouped into clusters,
with one time-series per cluster. All voxels within a cluster share the same time-series,
making it ideal for parcellated analyses (e.g., Schaefer-Yeo parcellations).
}
\details{
This class implements array-like 4D access while storing data efficiently as a TxK matrix
instead of the full voxel x time representation. Each cluster's time-series is computed by
applying the aggregation function (\code{FUN}) to all voxels within that cluster.

The object supports standard NeuroVec operations:
\itemize{
  \item Indexing: \code{x[,,,t]} to extract 3D volumes at time t
  \item Series extraction: \code{series(x, i, j, k)} for time-series at voxel (i,j,k)
  \item Matrix conversion: \code{as.matrix(x)} to get the TxK cluster matrix
}

Single-voxel clusters are handled efficiently without aggregation overhead.
}
\examples{
# Create synthetic 4D data (10x10x10 volume, 20 timepoints)
sp4 <- NeuroSpace(c(10,10,10,20), c(1,1,1))
arr <- array(rnorm(10*10*10*20), dim=c(10,10,10,20))
vec <- NeuroVec(arr, sp4)

# Create a mask covering the central region
sp3 <- NeuroSpace(c(10,10,10), c(1,1,1))
mask_arr <- array(FALSE, dim=c(10,10,10))
mask_arr[3:8, 3:8, 3:8] <- TRUE
mask <- LogicalNeuroVol(mask_arr, sp3)

# Assign voxels to 5 random clusters
n_voxels <- sum(mask_arr)
clusters <- sample(1:5, n_voxels, replace=TRUE)
cvol <- ClusteredNeuroVol(mask, clusters)

# Create clustered representation
cv <- ClusteredNeuroVec(vec, cvol)

# Access like a regular NeuroVec
vol_t1 <- cv[,,,1]  # 3D volume at time 1
ts <- series(cv, 5, 5, 5)  # time-series at voxel (5,5,5)

# Get cluster time-series matrix
cluster_matrix <- as.matrix(cv)  # T x K matrix
dim(cluster_matrix)  # 20 x 5
}
\seealso{
\code{\link{ClusteredNeuroVol}} for creating cluster assignments,
\code{\link{cluster_searchlight_series}} for cluster-based searchlight analysis,
\code{\link{series}} for extracting time-series
}
