% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/netsplit.R
\name{netsplit}
\alias{netsplit}
\alias{print.netsplit}
\title{Split direct and indirect evidence in network meta-analysis}
\usage{
netsplit(
  x,
  method,
  upper = TRUE,
  reference.group = x$reference.group,
  baseline.reference = x$baseline.reference,
  show = gs("show"),
  overall = TRUE,
  direct = TRUE,
  indirect = TRUE,
  only.reference = FALSE,
  ci = FALSE,
  test = show \%in\% c("all", "with.direct", "both"),
  order = NULL,
  nchar.trts = x$nchar.trts,
  sep.trts = x$sep.trts,
  quote.trts = "",
  tol.direct = 5e-04,
  common = x$common,
  random = x$random,
  backtransf = x$backtransf,
  warn = FALSE,
  warn.deprecated = gs("warn.deprecated"),
  verbose = FALSE,
  ...
)

\method{print}{netsplit}(
  x,
  common = x$x$common,
  random = x$x$random,
  show = x$show,
  overall = x$overall,
  direct = x$direct,
  indirect = x$indirect,
  ci = x$ci,
  test = x$test,
  only.reference = x$only.reference,
  sortvar = NULL,
  subset = NULL,
  nchar.trts = x$nchar.trts,
  digits = gs("digits"),
  digits.stat = gs("digits.stat"),
  digits.pval = gs("digits.pval"),
  digits.prop = max(gs("digits.pval") - 2, 2),
  text.NA = gs("lab.NA"),
  backtransf = x$backtransf,
  scientific.pval = gs("scientific.pval"),
  big.mark = gs("big.mark"),
  legend = gs("legend"),
  indent = TRUE,
  warn.deprecated = gs("warn.deprecated"),
  ...
)
}
\arguments{
\item{x}{An object of class \code{netmeta} or \code{netsplit}.}

\item{method}{A character string indicating which method to split
direct and indirect evidence is to be used. Either
\code{"Back-calculation"} or \code{"SIDDE"}, can be abbreviated.
See Details.}

\item{upper}{A logical indicating whether treatment comparisons
should be selected from the lower or upper triangle of the
treatment effect matrices (see list elements \code{TE.common} and
\code{TE.random} in the \code{netmeta} object). Ignored if
argument \code{order} is provided.}

\item{reference.group}{Reference treatment. Ignored if argument
\code{order} is provided.}

\item{baseline.reference}{A logical indicating whether results
should be expressed as comparisons of other treatments versus the
reference treatment or vice versa. This argument is only
considered if \code{reference.group} is not equal to \code{""}
and argument\code{order} is not provided.}

\item{show}{A character string indicating which comparisons should
be printed (see Details).}

\item{overall}{A logical indicating whether estimates from network
meta-analysis should be printed.}

\item{direct}{A logical indicating whether direct estimates should
be printed.}

\item{indirect}{A logical indicating whether indirect estimates
should be printed.}

\item{only.reference}{A logical indicating whether only comparisons
with the reference group should be printed.}

\item{ci}{A logical indicating whether confidence intervals should
be printed in addition to treatment estimates.}

\item{test}{A logical indicating whether results of a test
comparing direct and indirect estimates should be printed.}

\item{order}{A optional character or numerical vector specifying
the order of treatments in comparisons.}

\item{nchar.trts}{A numeric defining the minimum number of
characters used to create unique treatment names.}

\item{sep.trts}{A character string used in comparison names as
separator between treatment labels, e.g., " vs ".}

\item{quote.trts}{A character used to print around treatment
labels.}

\item{tol.direct}{A numeric defining the maximum deviation of the
direct evidence proportion from 0 or 1 to classify a comparison
as providing only indirect or direct evidence, respectively.}

\item{common}{A logical indicating whether results for the common
effects network meta-analysis should be printed.}

\item{random}{A logical indicating whether results for the random
effects network meta-analysis should be printed.}

\item{backtransf}{A logical indicating whether printed results
should be back transformed. For example, if \code{backtransf =
TRUE}, results for \code{sm = "OR"} are printed as odds ratios
rather than log odds ratios.}

\item{warn}{A logical indicating whether warnings should be
printed.}

\item{warn.deprecated}{A logical indicating whether warnings should
be printed if deprecated arguments are used.}

\item{verbose}{A logical indicating whether progress information
should be printed.}

\item{\dots}{Additional arguments.}

\item{sortvar}{An optional vector used to sort comparisons (must be
of same length as the total number of comparisons).}

\item{subset}{An optional logical vector specifying a subset of
comparisons to print (must be of same length as the total number of
comparisons) .}

\item{digits}{Minimal number of significant digits, see
\code{print.default}.}

\item{digits.stat}{Minimal number of significant digits for z-value
of test of agreement between direct and indirect evidence, see
\code{print.default}.}

\item{digits.pval}{Minimal number of significant digits for p-value
of test of agreement between direct and indirect evidence, see
\code{print.default}.}

\item{digits.prop}{Minimal number of significant digits for direct
evidence proportions, see \code{print.default}.}

\item{text.NA}{A character string specifying text printed for
missing values.}

\item{scientific.pval}{A logical specifying whether p-values should
be printed in scientific notation, e.g., 1.2345e-01 instead of
0.12345.}

\item{big.mark}{A character used as thousands separator.}

\item{legend}{A logical indicating whether a legend should be
printed.}

\item{indent}{A logical indicating whether items in the legend
should be indented.}
}
\value{
An object of class \code{netsplit} with corresponding \code{print}
and \code{forest} functions. The object is a list containing the
following components:
\item{common, random}{As defined above.}
\item{comparison}{A vector with treatment comparisons.}
\item{prop.common, prop.random}{A vector with direct evidence
  proportions (common / random effects model).}
\item{common, random}{Results of network meta-analysis (common /
  random effects model), i.e., data frame with columns comparison,
  TE, seTE, lower, upper, z, and p.}
\item{direct.common, direct.random}{Network meta-analysis results
  based on direct evidence (common / random effects model), i.e.,
  data frame with columns comparison, TE, seTE, lower, upper, z,
  and p.}
\item{indirect.common, indirect.random}{Network meta-analysis
  results based on indirect evidence (common / random effects
  model), i.e., data frame with columns comparison, TE, seTE,
  lower, upper, z, and p.}
\item{compare.common, compare.random}{Comparison of direct and
  indirect evidence in network meta-analysis (common / random
  effects model), i.e., data frame with columns comparison, TE,
  seTE, lower, upper, z, and p.}
\item{sm}{A character string indicating underlying summary measure}
\item{level.ma}{The level used to calculate confidence intervals
  for pooled estimates.}
\item{tictoc}{Computation times for node-splitting method or SIDDE
  (if R package \bold{tictoc} is installed).}
\item{version}{Version of R package netmeta used to create object.}
}
\description{
Methods to split network estimates into the contribution of direct
and indirect evidence and to test for local inconsistency in
network meta-analysis.
}
\details{
A comparison of direct and indirect treatment estimates can serve
as check for consistency of network meta-analysis (Dias et al.,
2010).

This function provides two methods to derive indirect estimates:
\itemize{
\item Separate Indirect from Direct Evidence (SIDE) using a
  back-calculation method (\code{method = "Back-calculation"})
  based on the \emph{direct evidence proportion} to calculate the
  indirect evidence (König et al., 2013);
\item Separate Indirect from Direct Design Evidence (SIDDE) as
  described in Efthimiou et al. (2019).
}

Note, for the back-calculation method, indirect treatment estimates
are already calculated in \code{\link{netmeta}} and this function
combines and prints these estimates in a user-friendly
way. Furthermore, this method is not available for the
Mantel-Haenszel and non-central hypergeometric distribution
approach implemented in \code{\link{netmetabin}}.

For the random-effects model, the direct treatment estimates are
based on the common between-study variance \eqn{\tau^2} from the
network meta-analysis, i.e. the square of list element
\code{x$tau}.

Argument \code{show} determines which comparisons are printed:
\tabular{ll}{
\dQuote{all} \tab All comparisons \cr
\dQuote{both} \tab Only comparisons contributing both direct and
  indirect evidence \cr
\dQuote{with.direct} \tab Comparisons providing direct evidence \cr
\dQuote{direct.only} \tab Comparisons providing only direct
  evidence \cr
\dQuote{indirect.only} \tab Comparisons providing only indirect
  evidence
}

The node-splitting method and SIDDE can be compute-intensive in
large networks. Crude information on the computation progress is
printed if argument \code{verbose = TRUE}. In addition, computation
times are printed if R package \bold{tictoc} is installed.
}
\examples{
#
# 1) COPD example
#

pw1 <- pairwise(treatment, event = r, n = N,
  studlab = author, data = dat.woods2010, sm = "OR")
#
net1 <- netmeta(pw1)
#
print(netsplit(net1), digits = 2)

\donttest{
print(netsplit(net1), digits = 2,
  backtransf = FALSE, common = FALSE)

# Sort by increasing number of studies in direct comparisons
print(netsplit(net1), digits = 2, sortvar = k)
# Sort by decreasing number of studies in direct comparisons
print(netsplit(net1), digits = 2, sortvar = -k)

# Sort by increasing evidence proportion under common effects model
print(netsplit(net1), digits = 2, sortvar = prop.common)
# Sort by decreasing evidence proportion under common effects model
print(netsplit(net1), digits = 2, sortvar = -prop.common)

# Sort by decreasing evidence proportion under common effects model
# and number of studies
print(netsplit(net1), digits = 2, sortvar = cbind(-prop.common, -k))

#
# 2) Diabetes example
#

data(Senn2013)
#
net2 <- netmeta(TE, seTE, treat1.long, treat2.long,
  studlab, data = Senn2013)
#
ns2 <- netsplit(net2)
#
print(ns2, digits = 2)
# Layout of Puhan et al. (2014), Table 1
print(ns2, digits = 2, ci = TRUE, test = FALSE)

# Forest plot showing comparisons contributing both direct and
# indirect evidence
#
forest(ns2, fontsize = 6, spacing = 0.5, addrow.subgroups = FALSE)

# Forest plot showing comparisons contributing direct evidence
#
forest(ns2, fontsize = 6, spacing = 0.5, addrow.subgroups = FALSE,
  show = "with.direct")

# Forest plot only showing network estimates compared to reference
# group and prediction intervals
#
forest(ns2, fontsize = 8, spacing = 0.75, show = "all",
  only.reference = TRUE, prediction = TRUE,
  direct = FALSE, indirect = FALSE)

#
# 3) Another COPD example
#

pw3 <- pairwise(treatment, death, randomized, studlab = id,
  data = dat.dong2013, sm = "OR")
net3 <- netmetabin(pw3)
netsplit(net3)
}

}
\references{
Dias S, Welton NJ, Caldwell DM, Ades AE (2010):
Checking consistency in mixed treatment comparison meta-analysis.
\emph{Statistics in Medicine},
\bold{29}, 932--44

Efthimiou O, Rücker G, Schwarzer G, Higgins J, Egger M, Salanti G
(2019):
A Mantel-Haenszel model for network meta-analysis of rare events.
\emph{Statistics in Medicine},
\bold{38}, 2992--3012

König J, Krahn U, Binder H (2013):
Visualizing the flow of evidence in network meta-analysis and
characterizing mixed treatment comparisons.
\emph{Statistics in Medicine},
\bold{32}, 5414--29

Puhan MA, Schünemann HJ, Murad MH, et al. (2014):
A GRADE working group approach for rating the quality of treatment
effect estimates from network meta-analysis.
\emph{British Medical Journal},
\bold{349}, g5630
}
\seealso{
\code{\link{forest.netsplit}}, \code{\link{netmeta}},
  \code{\link{netmetabin}}, \code{\link{netmeasures}},
  \code{\link[metadat]{dat.woods2010}}, \code{\link{Senn2013}},
  \code{\link[metadat]{dat.dong2013}}
}
\author{
Guido Schwarzer \email{guido.schwarzer@uniklinik-freiburg.de}, Gerta
  Rücker \email{gerta.ruecker@uniklinik-freiburg.de}, Orestis Efthimiou
  \email{oremiou@gmail.com}
}
