#' Local mark correlation functions for homogeneous point patterns on Euclidean spaces.
#'
#' Local mark correlation functions for homogeneous point patterns on Euclidean spaces.
#'
#' @usage \method{lmcorr}{ppp}(X,
#' ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart",
#'  "isham", "stoyancov", "schlather"),
#' r = NULL,
#' method = c("density","loess"),
#' normalise = TRUE,
#' f = NULL,
#' tol = 0.01,
#' ...)
#'
#' @param X An object of class ppp.
#' @param ftype Type of the test function \eqn{t_f}. Currently any selection of \code{"variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"}.
#' @param r Optional. The values of the argument \eqn{r} at which the mark correlation function should be evaluated.
#' @param method Type of smoothing, either \code{density} or \code{loess}. See details.
#' @param normalise If \code{normalise=FALSE}, only the numerator of the expression for the mark correlation function will be computed.
#' @param f  Optional. Test function \eqn{t_f} used in the definition of the mark correlation function. If \code{ftype} is given, \eqn{t_f} should be \code{NULL}.
#' @param tol Tolerance used in the calculation of the conditional mean of the marks. This is used only if \code{ftype} is \code{schlather}.
#' @param ... Arguments passed to \code{\link[spatstat.univar]{unnormdensity}} or \code{\link[stats]{loess}}.
#' @details
#' This function computes local mark correlation functions for a stationary point pattern in \eqn{\mathbb{R}^2}. See the details of test functions used in \code{\link[markstat]{mcorr.ppp}}. Technical details are given in Eckardt and Moradi (2025).
#' @examples
#'  library(spatstat.geom)
#'  library(spatstat.random)
#'  X <- rpoispp(100)
#'  marks(X) <- runif(npoints(X),10,11)
#'  lmcorr.ppp(X, ftype = "stoyan", method = "density")

#' @return A \code{data.frame} where the first column contains the values of the argument \eqn{r} at which the mark correlation function is evaluated, and the remaining columns contain the estimated values of the mark correlation function for each data point; column names correspond to the IDs of the data points.
#' If there are multiple numeric marks, the result will instead be a list, with each element corresponding to one of the marks.
#' 
#' @author Mehdi Moradi \email{m2.moradi@yahoo.com} and Matthias Eckardt
#' @seealso \code{\link[markstat]{mcorr.ppp}}.
#' @references Eckardt, M., & Moradi, M. (2025). Local indicators of mark association for marked spatial point processes.


#' @import spatstat.explore
#' @import spatstat.univar
#' @import spatstat.random
#' @import spatstat.geom
#' @import spatstat.utils
#' @import stats
#' @export

lmcorr.ppp <- function(X,
                       ftype = c("variogram", "stoyan", "rcorr", "shimatani", "beisbart", "isham", "stoyancov", "schlather"),
                       r = NULL,
                       method = c("density","loess"),
                       normalise = TRUE,
                       f = NULL,
                       tol = 0.01,
                       ...){

  if (all(class(X) != "ppp")) stop("object X should be of class ppp.")
  
  if (is.null(f) & missing(ftype)) stop("ftype must be provided if 'f' is NULL.")
  
  if (missing(method)) stop("smoothing method should be chosen.")

  m <- marks(X)

  if (any(class(m) == "hyperframe" | class(m) == "data.frame")){
    m <- as.data.frame(m)
    num_cols <- unlist(sapply(m, is.numeric))
    s <- which(num_cols)
    
    out <- list()
    for (i in 1:length(s)) {
      marks(X) <- as.numeric(m[,s[i]])
      out[[i]] <- lmcorr.ppp(X, ftype = ftype, r = r, method = method,
                               normalise = normalise, f = f, tol = tol, ...)
    }

    names(out) <- names(s)
    
    class(out) <- "mc"
    attr(out, "mtype") <- "real-valued"
    attr(out, "type") <- "local"
    attr(out, "ftype") <- ftype
    attr(out, "method") <- method
    attr(out, "normalise") <- normalise
    attr(out, "nmark") <- length(s)
    
    return(out)
  }

  if (is.null(f)) {
    if (ftype == "variogram") {
      f <- function(m1, m2, mu = NULL) 0.5 * ((m1 - m2)^2)
    } else if (ftype == "stoyan") {
      f <- function(m1, m2, mu = NULL) m1 * m2
    } else if (ftype == "rcorr") {
      f <- function(m1, m2, mu = NULL) m1
    } else if (ftype == "shimatani") {
      f <- function(m1, m2, mu = NULL) (m1 - mean(m)) * (m2 - mean(m))
    } else if (ftype == "beisbart") {
      f <- function(m1, m2, mu = NULL) m1 + m2
    } else if (ftype == "isham") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - (mean(m))^2
    } else if (ftype == "stoyancov") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - (mean(m))^2
    } else if (ftype == "schlather") {
      f <- function(m1, m2, mu = NULL) m1 * m2 - mu * (m1 + m2) + mu^2
    } else {
      stop("Your ftype is not supported!")
    }
  } else {
    warning("Your given test function is not among the default ones; only unnormalised version will be calculated.")
  }

  n <- npoints(X)
  d <- pairdist(X)

  if(is.null(r)){
    W <- X$window
    rmaxdefault <- rmax.rule("K", W, n/area(W))
    if(length(rmaxdefault)==0) {rmaxdefault <- 0.5 * max(d)}
    breaks <- handle.r.b.args(r, NULL, W, rmaxdefault = rmaxdefault)
    r <- breaks$r
  }

  rmax <- max(r)

  Eff <- list()
  out <- list()

  df <- cbind(dist=as.vector(d),
              id.row=rep(c(1:n),each=n),
              id.col=rep(c(1:n),n))

  df.filter <- df[df[,1]< rmax & df[,1]>0,]

  m1 <- m[df.filter[,2]]
  m2 <- m[df.filter[,3]]

  if (ftype=="schlather"){
    df.filter <- cbind(df.filter,
                       mu = as.numeric(unlist(sapply(df.filter[,1], function(d) {
                         matched <- df.filter[,3][abs(df.filter[,1] - d) <= tol]
                         paste(mean(m[matched]), collapse = ",")
                       }))))
    mu <- df.filter[,4]
    dfvario <- data.frame(d=df.filter[,1], ff=(f(m1, m2, mu)), id.row = df.filter[,"id.row"], id.col = df.filter[,"id.col"])
  }else{
    dfvario <- data.frame(d=df.filter[,1], ff=(f(m1, m2)), id.row = df.filter[,"id.row"], id.col = df.filter[,"id.col"])
  }

  for(i in 1:n){

    data <- dfvario[dfvario$id.row==i,]

    if(method=="density"){
      Kf <- unnormdensity(data$d, weights = data$ff,
                          from=min(r), to=max(r), n=length(r), ...
      )$y
      ## smooth estimate of kappa_1
      K1 <- unnormdensity(data$d, weights=rep(1,nrow(data)),
                          from=min(r), to=max(r), n=length(r), ...
      )$y
      Eff[[i]] <-  Kf/K1
    }
    else if(method=="loess"){
      lo <- loess(ff~d, data = data,
                  control = loess.control(surface="direct"), ...)
      Eff[[i]] <- predict(lo, newdata=data.frame(d=r))
    }


    if(normalise){
      if(ftype=="stoyan"){
        mean.i <- mean(m[data[,"id.row"][1]] * m[data[,"id.col"]])
        out[[i]] <- Eff[[i]]/mean.i
      } else if(ftype=="variogram" | ftype=="isham" | ftype=="schlather" | ftype=="shimatani"){
        var.i <- var(c(m[data[,"id.row"][1]], m[data[,"id.col"]]))
        out[[i]] <- Eff[[i]]/var.i
      }else if(ftype=="rcorr"){
        mean.i <- mean(c(m[data[,"id.row"][1]], m[data[,"id.col"]]))
        out[[i]] <- Eff[[i]]/mean.i
      }else if(ftype=="beisbart"){
        mean.i <- 2*mean(c(m[data[,"id.row"][1]], m[data[,"id.col"]]))
        out[[i]] <- Eff[[i]]/mean.i
      }else if(ftype=="stoyancov"){
        out[[i]] <- Eff[[i]]
      }
      else{
        stop("your ftype is not supported!!")
      }
    }else{
      out[[i]] <- Eff[[i]]
    }
  }

  out <- do.call(cbind, out)
  colnames(out) <- as.character(c(1:n))
  out <- cbind(r, out)
  
  if(ncol(out) == npoints(X) + 1 ) type <- "local" else type <- "global"
  
  class(out) <- "mc"
  attr(out, "mtype") <- "real-valued"
  attr(out, "type") <- type
  attr(out, "ftype") <- ftype
  attr(out, "method") <- method
  attr(out, "normalise") <- normalise
  
  return(out)
}
