#' Methods for `ifit` objects
#'
#' Methods and functions usable fo extracting information from an object
#' of class `ifit`
#'
#' @name ifit-methods
#' @keywords methods
NULL

#' @rdname ifit-methods
#' @param object an object returned by `ifit`
#' @param ... additional arguments, currently not used
#' @returns
#'    `coef` returns a numeric vector with the estimated parameters
#' @exportS3Method coef ifit
coef.ifit <- function(object, ...) object$guess

#' @rdname ifit-methods
#' @param type should the (estimated) covariance matrix of the
#'        parameters or of summary statistics returned?
#' @returns
#'    `vcov` returns a numeric matrix containing either the
#'     estimates or summary statistics variance-covariance matrix.
#' @exportS3Method vcov ifit
vcov.ifit <- function(object, type = c("parameters", "statistics"), ...) {
    type <- match.arg(type)
    if (type == "parameters") object$cov else object$V
}

#' @rdname ifit-methods
#' @param level the confidence level required.
#' @param parm a specification of which parameters are to be given
#'        confidence intervals.
#' @returns
#'    `confint` returns a matrix (or vector) with columns giving lower and upper
#'    confidence limits for each parameter. The intervals assume the 
#'    asymptotic normality of the summary statistics.
#' @exportS3Method confint ifit
confint.ifit <- function(object, parm, level = 0.95, ...) {
    stats::confint.default(object, level = level)
}

#' @rdname ifit-methods
#' @returns
#'    `controlIFIT` return a list containing the values of the constants
#'    `Ninit`, `NFitlocal`,... used to estimate the model.
#' @export controlIFIT
controlIFIT <- function(object) object$ctrl

#' @rdname ifit-methods
#' @returns
#'    `globalIFIT` return the vector of the estimated parameters after
#'    the global search
#' @export globalIFIT
globalIFIT <- function(object) object$global



#' @rdname ifit-methods
#' @returns
#'    `jacobianIFIT` returns a numeric matrix containing the estimated
#'    jacobian of the summary statistcs mean.
#' @export jacobianIFIT
jacobianIFIT <- function(object) object$B

#' @rdname ifit-methods
#' @returns
#'    `numsimIFIT` returns an integer vector of length 2 containing the
#'    number of simulations used during the global and local search
#'    phases.
#' @export numsimIFIT
numsimIFIT <- function(object) object$nsim

#' @rdname ifit-methods
#' @param plot if TRUE the summary statistics are plotted and the object
#' is returned as invisible; otherwise, the statistics are not plotted
#' and the object is returned as visible.
#' @returns
#'    `diagIFIT` returns, and optionally plot,  a numeric vector containing
#'    the observed summary statistics standardized with their means and
#'    standard deviances estimated at the final parameters; and, as attributes, 
#'    the Sargan-Hansen test statistic, its degrees of freedom and
#'    the corresponding p-value. The 
#'    class of the returned object is `ifit.diag` for which suitable
#'    print and plot methods are available.
#' @export diagIFIT
diagIFIT <- function(object, plot=TRUE) {
    sh <- sum((object$isqmV %*% object$a)^2)
    df <- object$data$q-object$data$p
    ans <- structure(
        object$a / sqrt(diag(object$V)),
        name = "sstat",
        SH = sh, 
        df = df,
        pv = stats::pchisq(sh, df, lower.tail=FALSE), 
        class = c("ifit.diag")
    )
    if (plot) { 
        plot.ifit.diag(ans)
        invisible(ans)
    } else {
        ans
    }
}

#' @rdname ifit-methods
#' @returns
#'    `estfunIFIT` returns a numeric vector containing the estimating function evaluated
#'     at the final parameters; as attributes, the function also returns
#'     the estimated standard errors of the estimating function, its Mahalanobis norm
#'     and the number of degree of freedom. The
#'     class of the returned object is `ifit.estfun` for which a suitable
#'     print method is available.
#' @export estfunIFIT
estfunIFIT <- function(object) {
    structure(object$grad,
        name = "estfun",
        se = object$grad.se,
        mahalanobis = object$grnorm2,
        df = object$lfree,
        act = if (object$data$p == object$lfree) "none" else object$data$p - object$lfree,
        class = c("ifit.estfun")
    )
}


#' @exportS3Method print ifit
print.ifit <- function(x, digits = getOption("digits"), ...) {
    cat("\nCall:",
        deparse(x$call, width.cutoff = floor(getOption("width") * 0.85)),
        "",
        sep = "\n"
    )
    a <- rbind(x$guess, sqrt(diag(x$cov)))
    rownames(a) <- c("", "s.e")
    colnames(a) <- names(x$guess)
    cat("Estimates:\n")
    print.default(format(a, digits = digits), print.gap = 2, quote = FALSE)
    invisible(x)
}


#' @exportS3Method print ifit.diag
print.ifit.diag <- function(x, digits = getOption("digits"), ...) {
    print.default(format(x, digits = digits), print.gap = 2, quote = FALSE)
    cat(
        "\nSargan-Hansen: ", format(attr(x, "SH"), digits = 2),
        ", df: ", attr(x, "df"), 
        ", p-value: ", format(attr(x, "pv"), digits=2), "\n", sep="")
    invisible(x)
}

#' @exportS3Method plot ifit.diag
plot.ifit.diag <- function(x, ...) {
    pv <- attr(x, "pv")
    pv <- if (pv < 1E0-6) ", p < 1E0-6" else paste0(", p=", format(pv, digits=2))
    tit <- paste0("Sargan-Hansen=", format(attr(x, "SH"), digits=2),
                  ", df=", attr(x, "df"), pv)
    z <- as.numeric(x)
    ll <- 1.1*max(abs(z))
    plot(z, type="h", main = tit, ylim=c(-ll, ll),   
         ylab="Standardized summary statistics", xlab="") 
    graphics::grid()
    invisible(x)
}


#' @exportS3Method print ifit.estfun
print.ifit.estfun <- function(x, digits = getOption("digits"), ...) {
    a <- rbind(x, attr(x, "se"), x / attr(x, "se"))
    rownames(a) <- c("est. fun.", "s.e.", "(est. fun.)/s.e.")
    colnames(a) <- names(x)
    print.default(format(a, digits = digits), print.gap = 2, quote = FALSE)
    cat(
        "\nSquared (Mahalanobis) norm: ", format(attr(x, "mahalanobis"), digits = 2),
        ", df: ", attr(x, "df"),
        if (!is.null(attr(x, "act"))) {
            paste0(",  parameters at the bounds: ", attr(x, "act"), "\n")
        } else {
            "\n"
        },
        sep = ""
    )
    invisible(x)
}



