% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/apply-threshold.R
\name{apply_threshold}
\alias{apply_threshold}
\title{Identify urban areas by applying a threshold on grid cells}
\usage{
apply_threshold(
  grid,
  type = "predefined",
  threshold_value = NULL,
  fun = NULL,
  ...,
  regions = NULL,
  operator = "greater_than",
  smoothing = TRUE
)
}
\arguments{
\item{grid}{SpatRaster with the data}

\item{type}{character. Either \code{"predefined"} or \code{"data-driven"}.}

\item{threshold_value}{numeric or vector. The threshold value used to identify urban areas when \code{type="predefined"}. If \code{regions} is not \code{NULL}, a vector of threshold values can be provided, where each value corresponds to a specific region (the respective values are linked to regions in alphabetical order based on their IDs; see examples). In addition, ensure that the threshold values are in the same unit as the grid values.}

\item{fun}{character or function. This function is used to derive the threshold value from the data when \code{type="data-driven"}. Either as character: \code{"min"}, \code{"max"}, \code{"mean"}, \code{"median"}, or \code{"pX"} where X denotes a percentile value (e.g., p95 for the 95\% percentile value"). It is also possible to provide a custom function for relatively small grids.}

\item{...}{additional arguments passed to fun}

\item{regions}{character, SpatRaster or sf object. If not \code{NULL}, a different threshold value is applied in the separate regions (i.e. a relative thresholding approach). The argument can either be:
\itemize{
\item a SpatRaster with a separate value for each region
\item the path to the region data (as character)
\item an sf polygon layer
In the latter two cases, the function \code{\link[=convert_regions_to_grid]{convert_regions_to_grid()}} will be used to convert the regions to a gridded format.
}}

\item{operator}{character. Operator used to enforce the threshold. Either \code{"greater_than"}, \code{"greater_or_equal"}, \code{"smaller_than"}, \code{"smaller_or_equal"} or \code{"equals"}.}

\item{smoothing}{logical. Whether to smooth the edges of the boundaries. If \code{TRUE}, boundaries will be smoothed with the function \code{\link[=apply_majority_rule]{apply_majority_rule()}}.}
}
\value{
named list with the following elements:
\itemize{
\item \code{rboundaries}: SpatRaster in which cells that are part of an urban area have a value of '1'
\item \code{vboundaries}: sf object with the urban areas as separate polygons
\item \code{threshold}: dataframe with per region the threshold value that is applied
\item \code{regions}: SpatRaster with the gridded version of the regions that is employed
}
}
\description{
The function identifies urban areas by applying a threshold to individual grid cells. Two key decisions must be made regarding the thresholding approach:
\enumerate{
\item \strong{How is the threshold value determined?}
\itemize{
\item The threshold can be \emph{predefined by the user} (\code{type="predefined"}) or \emph{derived from the data} (\code{type="data-driven"}).
}
\item \strong{How and where is the threshold enforced?}
\itemize{
\item The threshold can be enforced \emph{consistently across the study area} (= absolute approach, \code{regions=NULL}) or \emph{tailored within specific regions} (= relative approach, \code{regions} not \code{NULL}).
}
}

For more details on these thresholding approaches, including their advantages and limitations, see the \code{vignette("vig8-apply-thresholds")} The table below outlines the appropriate combination of function arguments for each approach:\tabular{lll}{
    \tab \strong{Absolute Approach} \tab \strong{Relative Approach} \cr
   \strong{Predefined Value} \tab \code{type="predefined"} with \code{threshold_value} not \code{NULL}, and \code{regions=NULL} \tab \code{type="predefined"} with \code{threshold_value} not \code{NULL}, and \code{regions} not \code{NULL} \cr
   \strong{Data-Driven Value} \tab \code{type="data-driven"} with \code{fun} not \code{NULL}, and \code{regions=NULL} \tab \code{type="data-driven"} with \code{fun} not \code{NULL}, and \code{regions} not \code{NULL} \cr
}
}
\examples{
proxies <- load_proxies_belgium()

# option 1: predefined - absolute threshold
predefined_absolute <- apply_threshold(proxies$pop,
  type = "predefined",
  threshold_value = 1500
)
terra::plot(predefined_absolute$rboundaries)

# option 2: data-driven - absolute threshold
datadriven_absolute <- apply_threshold(proxies$pop,
  type = "data-driven",
  fun = "p90"
)
terra::plot(datadriven_absolute$rboundaries)

# in the examples below we will use 'Bruxelles', 'Vlaanderen' and 'Wallonie' as separate regions
regions <- convert_regions_to_grid(flexurba::units_belgium, proxies$pop, "NAME_1")
terra::plot(regions)

# option 3: predefined - relative threshold
# note that the threshold values are linked to the regions in alphabetical
# order based on their IDs. So, the threshold of 1500 is applied to
# 'Bruxelles', # 1200 to 'Vlaanderen', and 1000 to 'Wallonie'.
predefined_relative <- apply_threshold(proxies$pop,
  type = "predefined",
  threshold_value = c(1500, 1200, 1000),
  regions = regions
)
terra::plot(predefined_relative$rboundaries)

# option 4: data-driven - relative threshold
datadriven_relative <- apply_threshold(proxies$pop,
  type = "data-driven",
  fun = "p95",
  regions = regions
)
terra::plot(datadriven_relative$rboundaries)

}
