#' Calculate Composite Multifunctionality Index (MFK)
#'
#' @description
#' Calculates a composite multifunctionality index based on the K index approach
#' (Wojcik 2024), integrating three facets of multifunctionality: richness (MFric),
#' regularity (MFreg), and divergence (MFdiv) using their geometric mean.
#'
#' @param data A data frame or matrix where rows represent observations and columns
#'        represent ecosystem functions. Function values should be normalized to a
#'        common scale (e.g., 0-1).
#' @param weights A numeric vector of weights for each function (column). If NULL,
#'        equal weights are assigned to all functions. Default is NULL.
#'
#' @details
#' The composite index MFK is calculated as the geometric mean of three facets:
#' \deqn{MFK = \sqrt[3]{MFric \times MFreg \times MFdiv}}{MFK = (MFric * MFreg * MFdiv)^(1/3)}
#'
#' This approach follows the K index methodology (Wojcik 2024), where the geometric
#' mean combines multiple facets of diversity into a single integrated measure.
#' The three facets represent:
#' \itemize{
#'   \item MFric: Multifunctionality richness
#'   \item MFreg: Multifunctionality regularity
#'   \item MFdiv: Multifunctionality divergence
#' }
#'
#' @return A data frame containing:
#' \itemize{
#'   \item MFric: Multifunctionality richness values
#'   \item MFreg: Multifunctionality regularity values
#'   \item MFdiv: Multifunctionality divergence values
#'   \item MFK: Composite multifunctionality index values
#' }
#'
#' @examples
#' # Example with sample data
#' data <- data.frame(
#'   func1 = c(0.5, 0.3, 0.2),
#'   func2 = c(0.2, 0.4, 0.4),
#'   func3 = c(0.3, 0.3, 0.4)
#' )
#' rownames(data) <- c("Site1", "Site2", "Site3")
#'
#' # Calculate MFK with equal weights
#' MFK(data)
#'
#' # Calculate MFK with custom weights (emphasizing func2)
#' MFK(data, weights = c(1, 2, 1))
#'
#'
#' @references
#' Wojcik, L. A., Gaedke, U., van Velzen, E., & Klauschies, T. (2025). Measuring overall functional
#' diversity by aggregating its multiple facets: Functional richness, biomass evenness, trait evenness
#' and dispersion. Methods in Ecology and Evolution, 16, 215–227.
#'
#' @seealso \code{\link{MFric}}, \code{\link{MFreg}}, \code{\link{MFdiv}}
#'
#' @export
MFK <- function(data, weights = NULL) {
  # Input validation
  if (ncol(data) <= 2) {
    stop("The number of functions must be greater than 2!")
  }
  # Set equal weights if not provided
  if (is.null(weights)) {
    weights <- rep(1, ncol(data))
  }
  if (length(weights) != ncol(data)) {
    stop("The length of the weight vector must be equal to the number of columns in the data frame")
  }
  # Calculate each facet using their respective functions
  mfric_result <- MFric(data, weights = weights)
  mfreg_result <- MFreg(data, weights = weights)
  mfdiv_result <- MFdiv(data, weights = weights)
  # Extract values from result data frames
  mfric_values <- mfric_result$MFric
  mfreg_values <- mfreg_result$MFreg
  mfdiv_values <- mfdiv_result$MFdiv
  # Calculate MFK as geometric mean (cube root of product)
  mfk_values <- (mfric_values * mfreg_values * mfdiv_values)^(1/3)
  # Build result data frame
  result <- data.frame(
    MFric = mfric_values,
    MFdiv = mfdiv_values,
    MFreg = mfreg_values,
    MFK = mfk_values
  )
  rownames(result) <- rownames(data)
  return(result)
}
