% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/deviation_metrics.R
\name{deviation_metrics}
\alias{deviation_metrics}
\alias{resistance}
\alias{amplitude}
\alias{recovery}
\alias{net_change}
\title{Metrics of trajectory deviation with respect to a reference trajectory}
\usage{
resistance(
  d,
  trajectories,
  states,
  disturbed_trajectories,
  disturbed_states,
  predisturbed_states = disturbed_states - 1
)

amplitude(
  d,
  trajectories,
  states,
  disturbed_trajectories,
  disturbed_states,
  predisturbed_states = disturbed_states - 1,
  reference,
  index = c("absolute", "relative"),
  method = "nearest_state"
)

recovery(
  d,
  trajectories,
  states,
  disturbed_trajectories,
  disturbed_states,
  reference,
  index = c("absolute", "relative"),
  method = "nearest_state"
)

net_change(
  d,
  trajectories,
  states,
  disturbed_trajectories,
  disturbed_states,
  predisturbed_states = disturbed_states - 1,
  reference,
  index = c("absolute", "relative"),
  method = "nearest_state"
)
}
\arguments{
\item{d}{Either a symmetric matrix or an object of class \code{\link{dist}} containing the
dissimilarities between each pair of states.}

\item{trajectories}{Vector indicating the trajectory or site to which each
state in \code{d} belongs.}

\item{states}{Vector of integers indicating the order of the states in \code{d} for
each trajectory.}

\item{disturbed_trajectories}{Vector of the same class as \code{trajectories} indicating
the identifier of the disturbed trajectories.}

\item{disturbed_states}{Vector of integers included in \code{states}indicating the
first state after the release of the disturbance for each value in
\code{disturbed_trajectories}.}

\item{predisturbed_states}{Vector of integers included in \code{states} indicating
the last undisturbed state of each \code{disturbed_trajectories}. The previous states
to \code{disturbed_states} are considered by default.}

\item{reference}{Object of class \code{RETRA} indicating the representative trajectory
taken as the reference to compute the amplitude, recovery, and net_change of
the disturbed trajectories (see Details).}

\item{index}{Method to calculate amplitude, recovery, or net change (\code{"absolute"},
\code{"relative"}; see Details).}

\item{method}{Method to calculate the distance between the \code{disturbed_states}
or \code{predisturbed_states} and the \code{reference} trajectory. One of \code{"nearest_state"},
\code{"projection"} or \code{"mixed"} (see Details).}
}
\value{
\itemize{
\item \code{resistance()} returns a data frame of two columns indicating the resistance
value (\code{Rt}) for each \code{disturbed_trajectory}.
\item \code{amplitude()} returns a data frame of three columns indicating the amplitude
value (\code{A_abs}; \code{A_rel}) for each \code{disturbed_trajectory} and \code{reference}.
If \code{index = c("absolute", "relative")}, both values are included in a data
frame of four columns.
\item \code{recovery()} returns a data frame of four columns indicating the recovery
value (\code{Rc_abs}; \code{Rc_rel}) for each \code{disturbed_trajectory}, post-disturbance
state (\code{state}) and \code{reference}. If \code{index = c("absolute", "relative")}, both
values are included in a data frame of five columns.
\item \code{net_change} returns a data frame of four columns indicating the net change
value (\code{NC_abs}; \code{NC_rel}) for each \code{disturbed_trajectory}, post-disturbance
state (\code{state}), and \code{reference}. If \code{index = c("absolute", "relative")}, both
values are included in a data frame of five columns.
}
}
\description{
Set of metrics to analyze the deviation of disturbed trajectories from an
ecological dynamic regime (EDR) considering a representative trajectory as the
reference. These metrics include the resistance to the disturbance, amplitude,
recovery, and net change.
}
\details{
\strong{Resistance (\code{resistance()})}

\emph{Resistance} captures the immediate impact of the disturbance as a function
of the changes in the state variables (Sánchez-Pinillos et al., 2019).

\eqn{
Rt = 1 - d_{pre,dist}
}

\strong{Amplitude (\code{amplitude()})}

\emph{Amplitude} indicates the direction in which the system is displaced during the
disturbance in relation to the \code{reference} (Sánchez-Pinillos et al., 2024).
Positive values indicate that the disturbance displaces the system towards the
boundaries of the dynamic regime. Negative values indicate that the disturbance
displaces the system towards the representative trajectory.

Two indices can be calculated:

If \code{index = "absolute"},

\eqn{
A = d_{dist,RT} - d_{pre,RT}
}

If \code{index = "relative"},

\eqn{
A = \frac{d_{dist,RT} - d_{pre,RT}}{d_{pre,dist}}
}

\strong{Recovery (\code{recovery()})}

\emph{Recovery} quantifies the ability of the system to evolve towards the \code{reference}
following the relief of the disturbance (if positive) or move in the direction
of the boundaries of the dynamic regime (if negative) (Sánchez-Pinillos et al.,
2024).

Two indices can be calculated:

If \code{index = "absolute"},

\eqn{
Rc = d_{dist,RT} - d_{post,RT}
}

If \code{index = "relative"},

\eqn{
Rc = \frac{d_{dist,RT} - d_{post,RT}}{d_{dist,post}}
}

\strong{Net change (\code{net_change()})}

\emph{Net change} quantifies the proximity of the system to the \code{reference} relative to
the pre-disturbed state (Sánchez-Pinillos et al., 2024). Positive values indicate
that the system eventually evolves towards the boundaries of the dynamic regime.
Negative values indicate that the system eventually evolves towards the
\code{reference}.

Two indices can be calculated:

If \code{index = "absolute"},

\eqn{
NC = d_{post,RT} - d_{pre,RT}
}

If \code{index = "relative"},

\eqn{
NC = \frac{d_{post,RT} - d_{pre,RT}}{d_{pre,post}}
}

In all cases:
\itemize{
\item \eqn{d_{pre,RT}} is the dissimilarity between the \code{predisturbed_states} and
the \code{reference}.
\item \eqn{d_{dist,RT}} is the dissimilarity between the \code{disturbed_states} and
the \code{reference}.
\item \eqn{d_{post,RT}} is the dissimilarity between the states after \code{disturbed_states}
and the \code{reference}.
\item \eqn{d_{pre,dist}} is the dissimilarity contained in \code{d} between the
\code{predisturbed_states} and the \code{disturbed_states}.
\item \eqn{d_{dist,post}} is the dissimilarity contained in \code{d} between the
\code{disturbed_states} and the post-disturbed states.
\item \eqn{d_{pre,post}} is the dissimilarity contained in \code{d} between the
\code{predisturbed_states} and the post-disturbed states.
}

\eqn{d_{pre,RT}}, \eqn{d_{dist,RT}}, and \eqn{d_{post,RT}} are calculated using
the function \code{\link[=state_to_trajectory]{state_to_trajectory()}} by three different methods:
\itemize{
\item If \code{method = "nearest_state"}, \eqn{d_{pre,RT}}, \eqn{d_{dist,RT}}, and
\eqn{d_{post,RT}} are calculated as the dissimilarity between the pre-disturbance,
disturbed, or post-disturbance states and their nearest state in the  \code{reference}.
\item If \code{method = "projection"}, \eqn{d_{pre,RT}}, \eqn{d_{dist,RT}}, and
\eqn{d_{post,RT}} are calculated as the dissimilarity between the pre-disturbance,
disturbed, or post-disturbance states and their projection onto the \code{reference}.
\item If \code{method = "mixed"}, \eqn{d_{pre,RT}}, \eqn{d_{dist,RT}}, and \eqn{d_{post,RT}}
are calculated in the same way than \code{method = "projection"} whenever the
pre-disturbance, disturbed and post-disturbance states can be projected onto
any segment of the \code{reference}. Otherwise, \eqn{d_{pre,RT}}, \eqn{d_{dist,RT}},
and \eqn{d_{post,RT}} are calculated using the nearest state of the \code{reference}.
}
}
\examples{
# Identify the representative trajectories of the EDR from undisturbed trajectories
RT <- retra_edr(d = EDR_data$EDR3$state_dissim,
                trajectories = EDR_data$EDR3$abundance$traj,
                states = as.integer(EDR_data$EDR3$abundance$state),
                minSegs = 5)

# Abundance matrix including disturbed and undisturbed trajectories
abundance <- rbind(EDR_data$EDR3$abundance,
                   EDR_data$EDR3_disturbed$abundance, fill = TRUE)

# State dissimilarities (Bray-Curtis) for disturbed and undisturbed trajectories
d <- vegan::vegdist(abundance[, paste0("sp", 1:12)], method = "bray")

# Resistance
Rt <- resistance(d = d, trajectories = abundance$traj, states = abundance$state,
                 disturbed_trajectories = unique(abundance[!is.na(disturbed_states)]$traj),
                 disturbed_states = abundance[disturbed_states == 1]$state)

# Amplitude
A <- amplitude(d = d, trajectories = abundance$traj, states = abundance$state,
               disturbed_trajectories = unique(abundance[!is.na(disturbed_states)]$traj),
               disturbed_states = abundance[disturbed_states == 1]$state, reference = RT)

# Recovery
Rc <- recovery(d = d, trajectories = abundance$traj, states = abundance$state,
               disturbed_trajectories = unique(abundance[!is.na(disturbed_states)]$traj),
               disturbed_states = abundance[disturbed_states == 1]$state, reference = RT)

# Net change
NC <- net_change(d = d, trajectories = abundance$traj, states = abundance$state,
                 disturbed_trajectories = unique(abundance[!is.na(disturbed_states)]$traj),
                 disturbed_states = abundance[disturbed_states == 1]$state, reference = RT)

}
\references{
Sánchez-Pinillos, M., Leduc, A., Ameztegui, A., Kneeshaw, D., Lloret, F., & Coll, L.
(2019). Resistance, resilience or change: Post-disturbance dynamics of boreal
forests after insect outbreaks. \emph{Ecosystems} 22, 1886-1901
https://doi.org/10.1007/s10021-019-00378-6

Sánchez-Pinillos, M., Dakos, V., & Kéfi, S. (2024). Ecological dynamic regimes:
A key concept for assessing ecological resilience. \emph{Biological Conservation}
289, 110409 https://doi.org/10.1016/j.biocon.2023.110409
}
\seealso{
\code{\link[=retra_edr]{retra_edr()}} to identify representative trajectories in an ecological dynamic
regime.

\code{\link[=define_retra]{define_retra()}} to generate an object of class\code{RETRA}.

\code{\link[=state_to_trajectory]{state_to_trajectory()}} to calculate the position of a state with respect to
a trajectory.
}
\author{
Martina Sánchez-Pinillos
}
