% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/VaR.R
\name{ES}
\alias{ES}
\alias{ES.default}
\alias{ES.numeric}
\alias{ES.matrix}
\title{Compute expected shortfall (ES)}
\usage{
ES(dist, p_loss = 0.05, ...)

\method{ES}{default}(
  dist,
  p_loss = 0.05,
  dist.type = "qf",
  qf,
  ...,
  intercept = 0,
  slope = 1,
  control = list(),
  transf = FALSE,
  x
)

\method{ES}{numeric}(dist, p_loss = 0.05, ..., intercept = 0, slope = 1, transf = FALSE, x)

\method{ES}{matrix}(dist, p_loss = 0.05, ..., intercept = 0, slope = 1, transf = FALSE, x)
}
\arguments{
\item{dist}{specifies the distribution whose ES is computed, usually a function or a name
of a function computing quantiles, cdf, pdf, or a random number generator. Can also be
a numeric vector or matrix, representing data, see section \sQuote{Details}.}

\item{p_loss}{level, default is 0.05.}

\item{...}{passed on to \code{dist}.}

\item{dist.type}{a character string specifying what is computed by \code{dist}, such as
"qf" or "cdf".}

\item{qf}{quantile function, only used if \code{dist.type = "pdf"}.}

\item{intercept, slope}{compute ES for the linear transformation \code{intercept +
slope*X}, where \code{X} has distribution specified by \code{dist}, see Details.}

\item{control}{additional control parameters for the numerical integration routine.}

\item{transf}{use only if \code{"dist"} represents log-returns. If \code{TRUE}, give the
result for the returns (not log-returns), see section \sQuote{Details}.}

\item{x}{deprecated and will soon be removed. \code{x} was renamed to \code{p_loss},
please use the latter.}
}
\value{
a numeric vector or matrix
}
\description{
Compute expected shortfall (ES).
}
\details{
\code{ES} is a generic function for computation of expected shortfall. The default
    method computes ES corresponding to a distribution, usually fitted or implied by a
    fitted model.  There are also methods for data (numeric or matrix).

    Alternative terms for ES include CVaR (conditional value at risk), AVaR (average value
    at risk), and ETL (expected tail loss).

    The default method of \code{ES} computes the expected shortfall for distributions
    specified by the arguments. \code{dist} is typically a function (or the name of
    one). What \code{dist} computes is determined by \code{dist.type}, whose default
    setting is \code{"qf"} (the quantile function). Other possible settings of
    \code{dist.type} include \code{"cdf"} and \code{"pdf"}.  Additional arguments for
    \code{dist} can be given with the \code{"..."} arguments.

    Argument \code{dist} can also be a numeric vector. In that case the ES is computed,
    effectively, for the empirical cumulative distribution function (ecdf) of the
    vector. The ecdf is not created explicitly and the \code{\link[stats]{quantile}}
    function is used instead for the computation of VaR. Arguments in \code{"..."} are
    passed eventually to \code{quantile()} and can be used, for example, to select a
    non-defult method for the computation of quantiles.

    If \code{dist} is a matrix, the numeric method is applied to each of its columns.

    Except for the exceptions discussed below, a function computing VaR for the specified
    distribution is constructed and the expected shortfall is computed by numerically
    integrating it. The numerical integration can be fine-tuned with argument
    \code{control}, which should be a named list, see \code{\link{integrate}} for the
    available options.

    If \code{dist.type} is \code{"pdf"}, VaR is not computed, Instead, the partial
    expectation of the lower tail is computed by numerical integration of \code{x *
    pdf(x)}.  Currently the quantile function is required anyway, via argument \code{qf},
    to compute the upper limit of the integral. So, this case is mainly for testing and
    comparison purposes.


    A bunch of expected shortfalls is computed if argument \code{p_loss} or any of the
    arguments in \code{"..."} are of length greater than one. They are recycled to equal
    length, if necessary, using the normal \R recycling rules.

    \code{intercept} and \code{slope} can be used to compute the expected shortfall for
    the location-scale transformation \code{Y = intercept + slope * X}, where the
    distribution of \code{X} is as specified by the other parameters and \code{Y} is the
    variable of interest. The expected shortfall of \code{X} is calculated and then
    transformed to that of \code{Y}. Note that the distribution of \code{X} doesn't need
    to be standardised, although it typically will.

    The \code{intercept} and the \code{slope} can be vectors. Using them may be
    particularly useful for cheap calculations in, for example, forecasting, where the
    predictive distributions are often from the same family, but with different location
    and scale parameters. Conceptually, the described treatment of \code{intercept} and
    \code{slope} is equivalent to recycling them along with the other arguments, but more
    efficiently.

    The names, \code{intercept} and \code{slope}, for the location and scale parameters
    were chosen for their expressiveness and to minimise the possibility for a clash with
    parameters of \code{dist} (e.g., the Gamma distribution has parameter \code{scale}).

    When argument \code{dist} represents log-returns, ES is for the log-returns. Use
    \code{transf = TRUE} to return its value for the returns. Note that the ES of the
    returns cannot be obtained by exponentiating the ES for the log-returns.
}
\examples{
ES(qnorm)

## Gaussian
ES(qnorm, dist.type = "qf")
ES(pnorm, dist.type = "cdf")

## t-dist
ES(qt, dist.type = "qf", df = 4)
ES(pt, dist.type = "cdf", df = 4)

ES(pnorm, 0.95, dist.type = "cdf")
ES(qnorm, 0.95, dist.type = "qf")
## - VaRES::esnormal(0.95, 0, 1)
## - PerformanceAnalytics::ETL(p=0.05, method = "gaussian", mu = 0,
##                             sigma = 1, weights = 1)             # same

cvar::ES(pnorm, dist.type = "cdf")
cvar::ES(qnorm, dist.type = "qf")
cvar::ES(pnorm, 0.05, dist.type = "cdf")
cvar::ES(qnorm, 0.05, dist.type = "qf")

## this uses "pdf"
cvar::ES(dnorm, 0.05, dist.type = "pdf", qf = qnorm)


## this gives warning (it does more than simply computing ES):
## PerformanceAnalytics::ETL(p=0.95, method = "gaussian", mu = 0, sigma = 1, weights = 1)

## run this if VaRRES is present
\dontrun{
x <- seq(0.01, 0.99, length = 100)
y <- sapply(x, function(p) cvar::ES(qnorm, p, dist.type = "qf"))
yS <- sapply(x, function(p) - VaRES::esnormal(p))
plot(x, y)
lines(x, yS, col = "blue")
}

}
\seealso{
\code{\link{VaR}} for VaR,

    \code{\link[=predict.garch1c1]{predict}} for examples with fitted models
}
\concept{AVaR}
\concept{CVaR}
\concept{ETL}
\concept{average value at risk}
\concept{conditional value at risk}
\concept{expected shortfall}
\concept{expected tail loss}
