% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/brand_use_logo.R
\name{brand_use_logo}
\alias{brand_use_logo}
\title{Extract a logo resource from a brand}
\usage{
brand_use_logo(
  brand,
  name,
  variant = c("auto", "light", "dark", "light-dark"),
  ...,
  .required = !name \%in\% c("small", "medium", "large", "smallest", "largest"),
  .allow_fallback = TRUE
)
}
\arguments{
\item{brand}{A brand object from \code{\link[=read_brand_yml]{read_brand_yml()}} or \code{\link[=as_brand_yml]{as_brand_yml()}}.}

\item{name}{The name of the logo to use. Either a size (\code{"small"},
\code{"medium"}, \code{"large"}) or an image name from \code{brand.logo.images}.
Alternatively, you can also use \code{"smallest"} or \code{"largest"} to select the
smallest or largest available logo size, respectively.}

\item{variant}{Which variant to use, only used when \code{name} is one of the
brand.yml fixed logo sizes (\code{"small"}, \code{"medium"}, or \code{"large"}). Can be
one of:
\itemize{
\item \code{"auto"}: Auto-detect, returns a light/dark logo resource if both
variants are present, otherwise it returns a single logo resource, either
the value for \verb{brand.logo.\{name\}} or the single light or dark variant if
only one is present.
\item \code{"light"}: Returns only the light variant. If no light variant is
present, but \verb{brand.logo.\{name\}} is a single logo resource and
\code{allow_fallback} is \code{TRUE}, \code{brand_use_logo()} falls back to the single
logo resource.
\item \code{"dark"}: Returns only the dark variant, or, as above, falls back to the
single logo resource if no dark variant is present and \code{allow_fallback}
is \code{TRUE}.
\item \code{"light-dark"}: Returns a light/dark object with both variants. If a
single logo resource is present for \verb{brand.logo.\{name\}} and
\code{allow_fallback} is \code{TRUE}, the single logo resource is promoted to a
light/dark logo resource with identical light and dark variants.
}}

\item{...}{Additional named attributes to be added to the image HTML or
markdown when created via \code{format()}, \code{\link[knitr:knit_print]{knitr::knit_print()}}, or
\code{\link[htmltools:as.tags]{htmltools::as.tags()}}.}

\item{.required}{Logical or character string. If \code{TRUE}, an error is thrown if
the requested logo is not found. If a string, it is used to describe why
the logo is required in the error message and completes the phrase
\code{"is required ____"}. Defaults to \code{FALSE} when \code{name} is one of the fixed
sizes -- \code{"small"}, \code{"medium"}, \code{"large"} or \code{"smallest"} or \code{"largest"}.
Otherwise, an error is thrown by default if the requested logo is not
found.}

\item{.allow_fallback}{If \code{TRUE} (the default), allows falling back to a
non-variant-specific logo when a specific variant is requested. Only used
when \code{name} is one of the fixed logo sizes (\code{"small"}, \code{"medium"}, or
\code{"large"}).}
}
\value{
A \code{brand_logo_resource} object, a \code{brand_logo_resource_light_dark}
object, or \code{NULL} if the requested logo doesn't exist and \code{.required} is
\code{FALSE}.
}
\description{
Returns a brand logo resource specified by name and variant from a brand
object. The image paths in the returned object are adjusted to be absolute,
relative to the location of the brand YAML file, if \code{brand} was read from a
file, or the local working directory otherwise.
}
\section{Shiny apps and HTML documents}{

You can use \code{brand_use_logo()} to include logos in \link[shiny:shinyApp]{Shiny apps} or in HTML documents produced by
\href{https://quarto.org/docs/output-formats/html-basics.html}{Quarto} or \link[rmarkdown:html_document]{R Markdown}.

In Shiny apps, logos returned by \code{brand_use_logo()} will automatically be
converted into HTML tags using \code{\link[htmltools:as.tags]{htmltools::as.tags()}}, so you can include
them directly in your UI code.

\if{html}{\out{<div class="sourceCode r">}}\preformatted{library(shiny)
library(bslib)

brand <- read_brand_yml()

ui <- page_navbar(
  title = tagList(
    brand_use_logo(brand, "small"),
    "Brand Use Logo Test"
  ),
  nav_panel(
    "Page 1",
    card(
      card_header("My Brand"),
      brand_use_logo(brand, "medium", variant = "dark")
    )
  )
  # ... The rest of your app
)
}\if{html}{\out{</div>}}

If your brand includes a light/dark variant for a specific size, both images
will be included in the app, but only the appropriate image will be shown
based on the user's system preference of the app's current theme mode if you
are using \code{\link[bslib:input_dark_mode]{bslib::input_dark_mode()}}.

To include additional classes or attributes in the \verb{<img>} tag, you can call
\code{\link[htmltools:as.tags]{htmltools::as.tags()}} directly and provide those attributes:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{brand <- as_brand_yml(list(
  logo = list(
    images = list(
      "cat-light" = list(
        path = "https://placecats.com/louie/300/300",
        alt = "A light-colored tabby cat on a purple rug"
      ),
      "cat-dark" = list(
        path = "https://placecats.com/millie/300/300",
        alt = "A dark-colored cat looking into the camera"
      ),
      "cat-med" = "https://placecats.com/600/600"
    ),
    small = list(light = "cat-light", dark = "cat-dark"),
    medium = "cat-med"
  )
))

brand_use_logo(brand, "small") |>
  htmltools::as.tags(class = "my-logo", height = 32)
}\if{html}{\out{</div>}}\if{html}{\out{
<!--html_preserve--><span class="brand-logo-light-dark">
}}
\if{html}{\out{
<img alt="A light-colored tabby cat on a purple rug" class="brand-logo light-content my-logo" height="32" src="https://placecats.com/louie/300/300"/>
<img alt="A dark-colored cat looking into the camera" class="brand-logo dark-content my-logo" height="32" src="https://placecats.com/millie/300/300"/>
</span><!--/html_preserve-->
}}


The same applies to HTML documents produced by Quarto or R Markdown, where
images can be used in-line:

\if{html}{\out{<div class="sourceCode markdown">}}\preformatted{```\{r\}
brand_use_logo(brand, "small")
```

This is my brand's medium sized logo: `r brand_use_logo(brand, "medium")`
}\if{html}{\out{</div>}}

Finally, you can use \code{format()} to convert the logo to raw HTML or markdown:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{cat(format(brand_use_logo(brand, "small", variant = "light")))
#> <img src="https://placecats.com/louie/300/300" alt="A light-colored tabby cat on a purple rug" class="brand-logo"/>

cat(format(
  brand_use_logo(brand, "medium"),
  .format = "markdown",
  class = "my-logo",
  height = 500
))
#> ![](https://placecats.com/600/600)\{.brand-logo .my-logo alt="" height="500"\}
}\if{html}{\out{</div>}}
}

\examples{
brand <- as_brand_yml(list(
  logo = list(
    images = list(
      small = "logos/small.png",
      huge = list(path = "logos/huge.png", alt = "Huge Logo")
    ),
    small = "small",
    medium = list(
      light = list(
        path = "logos/medium-light.png",
        alt = "Medium Light Logo"
      ),
      dark = list(path = "logos/medium-dark.png")
    )
  )
))

brand_use_logo(brand, "small")
brand_use_logo(brand, "medium")
brand_use_logo(brand, "large")
brand_use_logo(brand, "huge")

brand_use_logo(brand, "small", variant = "light")
brand_use_logo(brand, "small", variant = "light", allow_fallback = FALSE)
brand_use_logo(brand, "small", variant = "light-dark")
brand_use_logo(
  brand,
  "small",
  variant = "light-dark",
  allow_fallback = FALSE
)

brand_use_logo(brand, "medium", variant = "light")
brand_use_logo(brand, "medium", variant = "dark")
brand_use_logo(brand, "medium", variant = "light-dark")

}
