% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SEMml.R
\name{SEMml}
\alias{SEMml}
\title{Nodewise SEM train using Machine Learning (ML)}
\usage{
SEMml(
  graph,
  data,
  outcome = NULL,
  algo = "sem",
  thr = NULL,
  nboot = 0,
  ncores = 2,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{graph}{An igraph object.}

\item{data}{A matrix with rows corresponding to subjects, and columns 
to graph nodes (variables).}

\item{outcome}{A character vector (as.fctor) of labels for a categorical
output (target). If NULL (default), the categorical output (target) will
not be considered.}

\item{algo}{ML method used for nodewise-based predictions.
Four algorithms can be specified:
\itemize{
\item \code{algo="sem"} (default) for a linear SEM, see \code{\link[SEMgraph]{SEMrun}}. 
\item \code{algo="tree"} for a CART model, see \code{\link[rpart]{rpart}}.
\item \code{algo="rf"} for a random forest model, see \code{\link[ranger]{ranger}}.
\item \code{algo="xgb"} for a XGBoost model, see \code{\link[xgboost]{xgboost}}.
}}

\item{thr}{A numeric value [0-1] indicating the threshold to apply to the
variable importance values to color the graph. If thr = NULL (default), the
threshold is set to thr = 0.5*max(abs(variable importance values)).}

\item{nboot}{number of bootstrap samples that will be used to compute cheap
(lower, upper) CIs for all input variable weights. As a default, nboot = 0.}

\item{ncores}{number of cpu cores (default = 2)}

\item{verbose}{A logical value. If FALSE (default), the processed graph
will not be plotted to screen.}

\item{...}{Currently ignored.}
}
\value{
An S3 object of class "ML" is returned. It is a list of 5 objects:
\enumerate{
\item "fit", a list of ML model objects, including: the estimated covariance 
matrix (Sigma), the estimated model errors (Psi), the fitting indices (fitIdx),
and the parameterEstimates, i.e., the variable importance measures (VarImp).
\item "gest", the data.frame of variable importances (parameterEstimates)
of outcome levels, if outcome != NULL.
\item "model", a list of all the fitted non-linear nodewise-based models 
(tree, rf, xgb, nn or dnn).
\item "graph", the induced DAG of the input graph  mapped on data variables. 
The DAG with colored edge/nodes based on the variable importance measures,
i.e., if abs(VarImp) > thr will be highlighted in red (VarImp > 0) or blue
(VarImp < 0). If the outcome vector is given, nodes with variable importances
summed over the outcome levels, i.e. sum(VarImp[outcome levels])) > thr,
will be highlighted in pink.
\item "data", input data subset mapping graph nodes.
}
Using the default \code{algo="sem"}, the usual output of a linear nodewise-based
SEM, see \code{\link[SEMgraph]{SEMrun}} (algo="cggm"), will be returned.
}
\description{
The function converts a graph to a collection of 
nodewise-based models: each mediator or sink variable can be expressed as 
a function of its parents. Based on the assumed type of relationship, 
i.e. linear or non-linear, \code{SEMml()} fits a ML model to each
node (variable) with non-zero incoming connectivity. 
The model fitting is performed equation-by equation (r=1,...,R) 
times, where R is the number of mediators and sink nodes.
}
\details{
By mapping data onto the input graph, \code{SEMml()} creates
a set of nodewise models based on the directed links, i.e., 
a set of edges pointing in the same direction, between two nodes 
in the input graph that are causally relevant to each other. 
The mediator or sink variables are defined as functions of their parents.
Then, an ML model (sem, tree, rf, xgb) can be fitted to each variable with
non-zero inbound connectivity. The model fitting process is performed
equation-by-equation (r=1,...,R) times, where R represents the number of
mediators and sink nodes in the input graph.

If boot != 0, the function will implement the cheap bootstrapping proposed by
Lam (2002) to generate uncertainties (i.e., bootstrap \code{90\%CIs}) for ML
parameters. Bootstrapping can be enabled by setting a small number (1 to 10) of
bootstrap samples. Note, however, that the computation can be time-consuming for
massive MLs, even with cheap bootstrapping!
}
\examples{

\donttest{
# Load Amyotrophic Lateral Sclerosis (ALS)
ig<- alsData$graph
data<- alsData$exprs
data<- transformData(data)$data
group<- alsData$group

#...with train-test (0.5-0.5) samples
set.seed(123)
train<- sample(1:nrow(data), 0.5*nrow(data))
ncores <- parallel::detectCores(logical = FALSE)

start<- Sys.time()
# ... tree
res1<- SEMml(ig, data[train, ], algo="tree")

# ... rf
res2<- SEMml(ig, data[train, ], algo="rf", ncores = ncores)

# ... xgb
res3<- SEMml(ig, data[train, ], algo="xgb", ncores = ncores)

# ... sem
res4<- SEMml(ig, data[train, ], algo="sem")

end<- Sys.time()
print(end-start)

#visualizaation of the colored dag for algo="sem"
gplot(res4$graph, l="dot", main="sem")

#Comparison of fitting indices (in train data)
res1$fit$fitIdx #tree
res2$fit$fitIdx #rf
res3$fit$fitIdx #xgb
res4$fit$fitIdx #sem

#Comparison of parameter estimates (in train data)
parameterEstimates(res1$fit) #tree
parameterEstimates(res2$fit) #rf
parameterEstimates(res3$fit) #xgb
parameterEstimates(res4$fit) #sem

#Comparison of VarImp (in train data)
table(E(res1$graph)$color) #tree
table(E(res2$graph)$color) #rf
table(E(res3$graph)$color) #xgb
table(E(res4$graph)$color) #sem

#Comparison of AMSE, R2, SRMR (in test data)
print(predict(res1, data[-train, ])$PE) #tree
print(predict(res2, data[-train, ])$PE) #rf
print(predict(res3, data[-train, ])$PE) #xgb
print(predict(res4, data[-train, ])$PE) #sem

#...with a categorical (as.factor) outcome
outcome <- factor(ifelse(group == 0, "control", "case")); table(outcome) 

res5 <- SEMml(ig, data[train, ], outcome[train], algo="tree")
gplot(res5$graph)
table(E(res5$graph)$color)
table(V(res5$graph)$color)

pred <- predict(res5, data[-train, ], outcome[-train], verbose=TRUE)
yhat <- pred$Yhat[ ,levels(outcome)]; head(yhat)
yobs <- outcome[-train]; head(yobs)
classificationReport(yobs, yhat, verbose=TRUE)$stats
}

}
\references{
Grassi M., Palluzzi F., and Tarantino B. (2022). SEMgraph: An R Package for Causal 
Network Analysis of High-Throughput Data with Structural Equation Models. 
Bioinformatics, 38 (20), 4829–4830 <https://doi.org/10.1093/bioinformatics/btac567>

Breiman L., Friedman J.H., Olshen R.A., and Stone, C.J. (1984) Classification
and Regression Trees. Chapman and Hall/CRC.

Breiman L. (2001). Random Forests, Machine Learning 45(1), 5-32.

Chen T., and Guestrin C. (2016). XGBoost: A Scalable Tree Boosting System. 
Proceedings of the 22nd ACM SIGKDD International Conference on Knowledge 
Discovery and Data Mining.

Lam, H. (2022). Cheap bootstrap for input uncertainty quantification. WSC '22:
Proceedings of the Winter Simulation Conference, 2318-2329.
}
\author{
Mario Grassi \email{mario.grassi@unipv.it}
}
