#' @title Mean Productivity (MP)
#' @description
#' This function computes Mean Productivity (MP) using any traits (like yield) under stress and non-stress conditions. The lower values of MP indicates greater tolerance. For more details see Rosielle and Hamblin (1981) <doi:10.2135/cropsci1981.0011183X002100060033x>.
#'
#' @param Gen Character vector of genotype identifiers.
#' @param YN Numeric vector: yield (any trait) under non-stress (normal) environment.
#' @param YS Numeric vector: yield (any trait) under stress environment.
#' @return A data frame with \code{Gen}, \code{MP}.
#'
#' \deqn{MP = \frac{YN + YS}{2}}
#'
#' @references Rosielle, A.A. and Hamblin, J. (1981). <10.2135/cropsci1981.0011183X002100060033x>.
#' @examples
#' out = MP(
#' Gen=c("G1","G2","G3"),
#' YN=c(10,8,5),
#' YS=c(7,5,3)
#' )
#' print(out)
#' @export
MP <- function(Gen, YN, YS) {
  .check_lengths(Gen, YN, YS)
  YN <- as.numeric(YN); YS <- as.numeric(YS)
  out <- (YN + YS) / 2
  return(data.frame(Gen = as.character(Gen), MP = out, row.names = NULL))
}
