#' Identifies the roots of a univariate function over a given interval by
#' subdividing the interval into smaller subintervals and applying
#' \code{\link[stats]{uniroot}} on those subintervals where a sign change is detected.
#'
#' @param intersection_function A univariate numeric function whose roots are
#'   to be located.
#' @param interval Numeric vector of length 2 specifying the lower and upper
#'   bounds of the search interval.
#' @param n_subintervals Integer. Number of subintervals used to partition
#'   \code{interval}.
#'
#' @return
#' A numeric vector containing the distinct roots found within \code{interval}.
#' If no roots are detected, an empty numeric vector is returned.
#' @keywords internal
#' @importFrom stats uniroot
encontrar_raices <- function(intersection_function, interval, n_subintervals = 10) {

  n_subintervals <- as.integer(n_subintervals)

  subintervals <- seq(interval[1], interval[2], length.out = n_subintervals + 1)
  roots <- rep(NA_real_, n_subintervals)
  n_found <- 0L

  for (i in 1:n_subintervals) {
    subint <- c(subintervals[i], subintervals[i + 1])

    # Evito uniroot si no hay cambio de signo
    f1 <- intersection_function(subint[1])
    f2 <- intersection_function(subint[2])
    if (!is.finite(f1) || !is.finite(f2) || f1 * f2 > 0) next

    result <- tryCatch({
      uniroot(intersection_function, subint)$root
    }, error = function(e) {
      return(NULL)
    })

    if (!is.null(result) && is.finite(result)) {
      n_found <- n_found + 1L
      roots[n_found] <- result
    }
  }

  roots <- roots[seq_len(n_found)]

  # Para evitar que la misma raíz aparezca repetida varias veces
  if (length(roots) > 1L) {
    roots <- sort(roots)
    roots <- roots[c(TRUE, diff(roots) > 1e-10)]
  }

  return(roots)
}

