# Functions for generating synthetic and mixture data for stable distributions
# =============================================================================
# MIXTURE DATA GENERATION
# =============================================================================


#' Simulates a mixture of alpha-stable distributions with randomly sampled parameters.
#'
#' @param K Integer. Number of mixture components.
#' @param N Integer. Total number of samples to generate.
#'
#' @return A list containing:
#' \describe{
#'   \item{data}{Numeric vector of generated samples.}
#'   \item{params}{List of parameters for each component (alpha, beta, gamma, delta, pi).}
#' }
#' @importFrom stabledist rstable
#' @export
generate_mixture_data <- function(K = 2, N = 1000) {
  weights <- rgamma(K, 1)
  weights <- weights / sum(weights)
  params <- list()
  data <- c()
  for (i in 1:K) {
    alpha <- runif(1, 1.3, 1.95)
    beta  <- runif(1, -1, 1)
    gamma <- runif(1, 0.5, 2.0)
    delta <- runif(1, -2, 2)
    n_i   <- round(weights[i] * N)

    # FIX: Use stabledist::rstable directly to avoid parameter confusion
    sample <- stabledist::rstable(n_i, alpha = alpha, beta = beta,
                                  gamma = gamma, delta = delta, pm = 1)
    data <- c(data, sample)
    params[[i]] <- list(
      alpha = alpha,
      beta  = beta,
      gamma = gamma,
      delta = delta,
      pi    = weights[i]
    )
  }
  return(list(data = data, params = params))
}

# =============================================================================
# ALPHA-STABLE MIXTURE SAMPLING
# =============================================================================


#' Generate samples from a predefined alpha-stable mixture
#'
#' Simulates samples from a mixture of alpha-stable distributions using specified parameters.
#'
#' @param weights Numeric vector of mixture weights.
#' @param alphas Numeric vector of alpha parameters.
#' @param betas Numeric vector of beta parameters.
#' @param gammas Numeric vector of gamma parameters.
#' @param deltas Numeric vector of delta parameters.
#' @param size Integer. Number of samples to generate.
#'
#' @return A list with:
#' \describe{
#'   \item{samples}{Numeric vector of generated samples.}
#'   \item{labels}{Integer vector indicating the component each sample came from.}
#' }
#' @importFrom stabledist rstable
#' @export
generate_alpha_stable_mixture <- function(weights, alphas, betas, gammas, deltas,
                                          size = 1000) {
  n_components <- length(weights)
  weights <- weights / sum(weights)
  samples <- numeric(size)
  labels  <- numeric(size)

  for (i in 1:size) {
    component <- sample(1:n_components, 1, prob = weights)

    # FIX: Use stabledist::rstable directly
    samples[i] <- stabledist::rstable(1, alpha = alphas[component],
                                      beta = betas[component],
                                      gamma = gammas[component],
                                      delta = deltas[component], pm = 1)
    labels[i] <- component
  }
  return(list(samples = samples, labels = labels))
}

# =============================================================================
# SYNTHETIC DATA GENERATION
# =============================================================================

#' Generate synthetic data from two alpha-stable components
#'
#' Creates a synthetic dataset composed of two alpha-stable distributions.
#'
#' @param n Integer. Total number of samples to generate.
#'
#' @return Numeric vector of shuffled synthetic data.
#' @importFrom stabledist rstable
#' @export
generate_synthetic_data <- function(n = 1000) {
  comp1 <- stabledist::rstable(n %/% 2, alpha = 1.2, beta = 0.9,
                               gamma = 0.8, delta = -2, pm = 1)
  comp2 <- stabledist::rstable(n %/% 2, alpha = 1.8, beta = -0.5,
                               gamma = 1.5, delta = 5, pm = 1)

  data <- c(comp1, comp2)
  data <- sample(data)  # Shuffle
  return(data)
}

# =============================================================================
# SERIAL INTERVAL COMPUTATION
# =============================================================================


#' Compute serial interval from CSV file
#'
#' Calculates the serial interval from a CSV file containing date columns.
#'
#' @param filepath Character. Path to the CSV file.
#'
#' @return Numeric vector of serial intervals.
#' @importFrom graphics hist grid
#' @importFrom utils read.csv
#' @export
compute_serial_interval <- function(filepath) {
  data <- read.csv(filepath, sep = ";", dec = ".", header = TRUE)
  data$x.lb <- as.Date(data$x.lb, format = "%d/%m/%Y")
  data$x.ub <- as.Date(data$x.ub, format = "%d/%m/%Y")
  data$y    <- as.Date(data$y, format = "%d/%m/%Y")
  reference_date <- as.Date("2020-01-01")
  data$x.lb_days <- as.numeric(data$x.lb - reference_date)
  data$x.ub_days <- as.numeric(data$x.ub - reference_date)
  data$y_days    <- as.numeric(data$y - reference_date)
  data$SI <- data$y_days - (data$x.ub_days + data$x.lb_days) / 2
  hist(data$SI, breaks = 30, freq = FALSE, col = rgb(0, 1, 0, 0.6),
       main = "Histogramme de l'intervalle serial",
       xlab = "Intervalle Serial (jours)",
       ylab = "Frequence")
  grid()
  return(data$SI)
}
