% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aggregate_Date.R
\name{aggregate_Date}
\alias{aggregate_Date}
\title{Aggregate dates to a single day}
\usage{
aggregate_Date(
  dataset,
  Datetime.colname = Datetime,
  unit = "none",
  type = c("round", "floor", "ceiling"),
  date.handler = stats::median,
  numeric.handler = mean,
  character.handler = function(x) names(which.max(table(x, useNA = "ifany"))),
  logical.handler = function(x) mean(x) >= 0.5,
  factor.handler = function(x) factor(names(which.max(table(x, useNA = "ifany")))),
  datetime.handler = stats::median,
  duration.handler = function(x) lubridate::duration(mean(x)),
  time.handler = function(x) hms::as_hms(mean(x)),
  ...
)
}
\arguments{
\item{dataset}{A light logger dataset. Expects a \code{dataframe}. If not imported
by \link{LightLogR}, take care to choose a sensible variable for the
\code{Datetime.colname}.}

\item{Datetime.colname}{column name that contains the datetime. Defaults to
\code{"Datetime"} which is automatically correct for data imported with
\link{LightLogR}. Expects a \code{symbol}. Needs to be part of the \code{dataset}. Must
be of type \code{POSIXct}.}

\item{unit}{Unit of binning. See \code{\link[lubridate:round_date]{lubridate::round_date()}} for examples. The
default is \code{"none"}, which will not aggregate the data at all, but is only
recommended for regular data, as the condensation across different days
will be performed by time. Another option is \code{"dominant.epoch"}, which
means everything will be aggregated to the most common interval. This is
especially useful for slightly irregular data, but can be computationally
expensive.}

\item{type}{One of \code{"round"}(the default), \code{"ceiling"} or \code{"floor"}. Setting
chooses the relevant function from \pkg{lubridate}.}

\item{date.handler}{A function that calculates the aggregated day for each
group. By default, this is set to \code{median}.}

\item{numeric.handler, character.handler, logical.handler, factor.handler, datetime.handler, duration.handler, time.handler}{functions that handle the respective data types. The default handlers
calculate the \code{mean} or \code{median} for \code{numeric}, \code{POSIXct}, \code{duration}, and
\code{hms}, and the \code{mode} for \code{character}, \code{factor} and \code{logical} types.}

\item{...}{arguments given over to \code{\link[dplyr:summarise]{dplyr::summarize()}} to handle columns
that do not fall into one of the categories above. Be careful with partial
matching of argument names. E.g., creating a new column \code{n}, without having
changed the default for \code{numeric.handler} will match \code{n} to that argument.
You can avoid this be either explicitly supplying the \code{numeric.handler}
argument (using the default again), or you change the variable name, e.g,
to \code{n.}, which avoids the partial matching.}
}
\value{
A \code{tibble} with aggregated \code{Datetime} data, at maximum one day per
group. If the handler arguments capture all column types, the number of
columns will be the same as in the input \code{dataset}.
}
\description{
Condenses a \code{dataset} by aggregating the data to a single day per group, with
a resolution of choice \code{unit}. \code{\link[=aggregate_Date]{aggregate_Date()}} is opinionated in the sense
that it sets default handlers for each data type of \code{numeric}, \code{character},
\code{logical}, and \code{factor}. These can be overwritten by the user. Columns that
do not fall into one of these categories need to be handled individually by
the user (\code{...} argument) or will be removed during aggregation. If no unit
is specified the data will simply be aggregated to the most common interval
(\code{dominant.epoch}) in every group. \code{\link[=aggregate_Date]{aggregate_Date()}} is especially useful
for summary plots that show an average day.
}
\details{
Summary values for type \code{POSIXct} are calculated as the median, because the
mean can be nonsensical at times (e.g., the mean of Day1 18:00 and Day2
18:00, is Day2 6:00, which can be the desired result, but if the focus is on
time, rather then on datetime, it is recommended that values are converted to
times via \code{\link[hms:hms]{hms::as_hms()}} before applying the function (the mean of 18:00 and
18:00 is still 18:00, not 6:00). Using the median as a default handler
ensures a more sensible datetime.

\code{\link[=aggregate_Date]{aggregate_Date()}} splits the \code{Datetime} column into a \code{Date.data}
and a \code{Time} column. It will create subgroups for each \code{Time}
present in a group and aggregate each group into a single day, then remove
the sub grouping.

Use the \code{...} to create summary statistics for each group, e.g. maximum or
minimum values for each time point group.

Performing \code{\link[=aggregate_Datetime]{aggregate_Datetime()}} with any \code{unit} and then
\code{\link[=aggregate_Date]{aggregate_Date()}} with a \code{unit} of \code{"none"} is equivalent to just using
\code{\link[=aggregate_Date]{aggregate_Date()}} with that \code{unit} directly (provided the other arguments
are set the same between the functions). Disentangling the two functions
can be useful to split the computational cost for very small instances of
\code{unit} in large datasets. It can also be useful to apply different handlers
when aggregating data to the desired \code{unit} of time, before further
aggregation to a single day, as these handlers as well as \code{...} are used
twice if the \code{unit} is not set to \code{"none"}.
}
\examples{
library(ggplot2)
#gg_days without aggregation
sample.data.environment \%>\%
 gg_days()

#with daily aggregation
sample.data.environment \%>\%
 aggregate_Date() \%>\%
 gg_days()

#with daily aggregation and a different time aggregation
sample.data.environment \%>\%
 aggregate_Date(unit = "15 mins", type = "floor") \%>\%
 gg_days()

#adding further summary statistics about the range of MEDI
 sample.data.environment \%>\%
 aggregate_Date(unit = "15 mins", type = "floor",
                MEDI_max = max(MEDI),
                MEDI_min = min(MEDI)) \%>\%
 gg_days() +
 geom_ribbon(aes(ymin = MEDI_min, ymax = MEDI_max), alpha = 0.5)
}
