#' Add states to gg_day() or gg_days() plots
#'
#' [gg_states()] is a helper function to add state information to plots
#' generated with [gg_day()], [gg_days()], or [gg_doubleplot()]. The function
#' can draw on any column in the dataset, but factor-like or logical columns
#' make the most sense.
#'
#' @param ggplot_obj A `ggplot` object generated with [gg_day()] or [gg_days()]
#'   (or [gg_doubleplot()].
#' @param alpha A numerical value between 0 and 1 representing the transparency
#'   of the states. Default is 0.2.
#' @param ... Additional arguments given to the [ggplot2::geom_rect()] used to
#'   construct the state shading. Can be used to change the fill color or other
#'   aesthetic properties.
#' @param on.top Logical scalar. If `TRUE`, the states will be plotted on top of
#'   the existing plot. If `FALSE`, the states will be plotted underneath the
#'   existing plot. Default is `FALSE`.
#' @param State.colname,Datetime.colname The colnames of the state to add to the
#'   plot, and the column containing the datetimes. Must be part of the dataset.
#'   Expects a `symbol`.
#' @param aes_fill,aes_col conditional aesthetics for [ggplot2::geom_rect()].
#'   The default (`NULL`) will be ignored, so that `col` and `fill` arguments
#'   can be set through the `...` arguments. As the states work from a
#'   summarized dataset, only a few columns are available for filling/coloring:
#'   The `State.colname`, Grouping variables, and variables created by using
#'   [extract_states()].
#' @param ignore.FALSE Logical that drops `FALSE` values of a logical state
#'   column, so that only `TRUE` values are recognized as a state. Is only
#'   relevant for logical state columns and will be ignored otherwise. Default
#'   is `TRUE`.
#' @param ymin,ymax customize the height of the state rectangle. By default it
#'   will cover the whole vertical range (-Inf, Inf), but can be set to any
#'   value, including variables generated by [extract_states()], which generates
#'   the data for the geom. New variables are accessible through the
#'   `extract_metrics` argument.
#' @param extract.metric Expression to be used in [extract_metric()]. Can be
#'   used to extract additional information that is then available for
#'   `aes_fill`, `aes_col`, `ymin`, and `ymax`. Requires a list containing
#'   variable names and formulas to be converted into calls. E.g., calculating
#'   the median during each state would require `list(new_metric =
#'   ~median(MEDI))`. Then `new_metric` is available for the conditional
#'   arguments.
#' @param return.state.data Used for debugging. Returns a tibble with the
#'   extracted state data. If you are uncertain what information is available,
#'   set this to `TRUE`.
#'
#' @returns a modified `ggplot` object with the states added.
#' @export
#'
#' @examples
#' #creating a simple TRUE/FALSE state in the sample data: Light above 250 lx mel EDI
#' #and a second state that cuts data into chunks relating to the Brown et al. 2022 thresholds
#' #(+aggregating Data to 5 minute intervals & reducing it to three days)
#' state_data <-
#'   sample.data.environment |>
#'    dplyr::mutate(state = MEDI > 250) |>
#'    Brown_cut(MEDI, state2) |>
#'    aggregate_Datetime(unit = "5 mins") |>
#'    filter_Datetime(length = "3 days")
#'
#' state_data |>
#'  gg_days() |>
#'  gg_states(state)
#'
#' #state 2 has more than one valid state, thus we need to assign a fill aesthetic
#' state_data |>
#'  gg_days() |>
#'  gg_states(state2, aes_fill = state2) +
#'  ggplot2::scale_fill_manual(values=c("#868686FF", "#EFC000FF", "#0073C2FF"))
#'  #this line is simply for sensible colors
#'
#' #same, but with gg_day()
#' state_data |>
#'  dplyr::filter(Id == "Participant") |>
#'  gg_day(geom = "line") |>
#'  gg_states(state, fill = "red")
#'
#'  #more complex state
#'  state_data |>
#'  dplyr::filter(Id == "Participant") |>
#'  gg_day(geom = "line") |>
#'  gg_states(state2, aes_fill = state2)
#'
#'  #with gg_doubleplot
#'  state_data |>
#'  dplyr::filter(Id == "Participant") |>
#'  gg_doubleplot() |>
#'  gg_states(state2, aes_fill = state2)
#'
#'  #setting the height of the bars 
#'  state_data |>
#'  dplyr::filter(Id == "Participant") |>
#'  gg_day(geom = "line") |>
#'  gg_states(state2, aes_fill = state2, ymin= -0.5, ymax = 0, alpha = 1) +
#'  ggplot2::scale_fill_manual(values=c("#868686FF", "#EFC000FF", "#0073C2FF"))
#'
#'  #setting the height of the bars conditionally based on the median value
#'  state_data |>
#'  dplyr::filter(Id == "Participant") |>
#'  gg_day(geom = "line") |>
#'  gg_states(state2, aes_fill = state2, ymin= -Inf, ymax = median, alpha = 0.75,
#'            extract.metric = list(median = ~median(MEDI))) +
#'  ggplot2::scale_fill_manual(values=c("#868686FF", "#EFC000FF", "#0073C2FF"))

gg_states <- function(ggplot_obj, 
                      State.colname,
                      aes_fill = NULL,
                      aes_col = NULL,
                      ymin = -Inf,
                      ymax = Inf,
                      Datetime.colname = Datetime,
                      alpha = 0.2,
                      on.top = FALSE,
                      ignore.FALSE = TRUE,
                      extract.metric = NULL,
                      return.state.data = FALSE,
                      ...) {
  
  # Initial Checks ----------------------------------------------------------
  
  #ggplot must be a ggplot object
  stopifnot(
    "ggplot_obj must be a ggplot object" = inherits(ggplot_obj, "gg")
  )
  
  #determine whether the ggplot has a hms or a POSIXct x-axis
  x_axis_type <- 
    get_ggplot_axis_type(ggplot_obj, "x")
  
  extract.metric <- rlang::as_quosures(extract.metric, 
                                        env = rlang::caller_env())
  
  # Function ----------------------------------------------------------
  
  #remove FALSE states, if argument is set
  
  state_logical <- 
    ggplot_obj$data |> dplyr::pull({{ State.colname }}) |> is.logical()
  
  if(ignore.FALSE & state_logical) {
    ggplot_obj$data <- 
      ggplot_obj$data |> 
      dplyr::mutate({{ State.colname }} := dplyr::na_if({{ State.colname }},0))
  }
  
  #if the y_axis_type is time
  if(x_axis_type == "time") {
    #create a table of states, by date
    rlang::inject(
    state_data <-
      ggplot_obj$data |>
      # dplyr::group_by(date.grouper = lubridate::date({{ Datetime.colname }}), .add = TRUE) |>
      extract_states({{ State.colname }},
                     Datetime.colname = {{ Datetime.colname }}) |> 
      extract_metric(ggplot_obj$data, 
                     Datetime.colname = {{ Datetime.colname }},
                     midnight.before =
                       min({{ Datetime.colname }}) |> lubridate::floor_date("day"),
                     midnight.after =
                       max({{ Datetime.colname }}) |> lubridate::ceiling_date("day"),
                     !!!extract.metric
      ) |> 
      tidyr::drop_na({{ State.colname }})
    )
    
    state_data <-
      state_data |>
      dplyr::mutate(
        start = dplyr::case_when(midnight.before > start ~ midnight.before,
                                .default = start),
        end = dplyr::case_when(midnight.after < (end+epoch/2) ~ midnight.after,
                                .default = end)
      )
  }
  
  #if the x_axis_type is hms
  if(x_axis_type == "hms") {
    rlang::inject(
      state_data <-
      ggplot_obj$data |>
      dplyr::group_by(Day.data, .add = TRUE) |>
      extract_states({{ State.colname }},
                     Datetime.colname = {{ Datetime.colname }}) |>
      extract_metric(ggplot_obj$data |> dplyr::group_by(Day.data, .add = TRUE),
                     Datetime.colname = {{ Datetime.colname }},
                     !!!extract.metric
                     ) |> 
      tidyr::drop_na({{ State.colname }})
    )
    
    state_data <-
      state_data |>
      dplyr::mutate(
        dplyr::across(c(start, end), hms::as_hms),
        start = ifelse(end < start, 0, start) |> hms::as_hms(),
        end = ifelse(end < start, 24*3600, end) |> hms::as_hms()
      )
  }
  
  if(return.state.data){
    return(state_data)
  }

  #create the geoms for the states
  state_geoms <- 
    list(
      ggplot2::geom_rect(
        inherit.aes = FALSE,
        data = state_data,
        ggplot2::aes(
          xmin = start,
          xmax = end,
          ymin = {{ ymin }},
          ymax = {{ ymax }}, 
          fill = {{ aes_fill }},
          col = {{ aes_col }}
        ),
        alpha = alpha,
        ...
      )
    )
  
  #add the geoms to the ggplot
  new_plot_obj <- ggplot_obj + state_geoms
  
  if(on.top) {
    return(new_plot_obj)
  }
  
  #reorder the layers so that the new geoms are at the very bottom
  new_plot_obj$layers <-
    c(new_plot_obj$layers |> utils::tail(1), 
      new_plot_obj$layers |> utils::head(-1))
  
  #return
  new_plot_obj
  
}


#' Add states to gg_day() or gg_days() plots
#'
#' @description
#' `r lifecycle::badge("deprecated")`
#' `gg_state()` has been deprecated. Use `gg_states()` instead
#'
#' @param ... arguments given to [gg_states()]
#'
#' @returns a ggplot object
#' @export
#'
#' @examples
#' sample.data.irregular|> 
#' dplyr::mutate(movement = dplyr::na_if(movement, 0)) |>  
#' gg_days() |> gg_state(movement)

gg_state <- function(...) {
  lifecycle::deprecate_soft("0.10.0", "gg_state()", "gg_states()")
  gg_states(...)
}