\name{feasible}
\alias{feasible}
\title{Estimate Feasible Intervention Strategies}
\description{
Estimate a family of feasible intervention strategies for a continuous treatment
(and optionally time-varying covariates). The method returns, for each intervention
strategy, the corresponding \dQuote{feasible} intervention values and a summary of
overlap and positivity-violation diagnostics.
}
\usage{
feasible(
  X, Anodes = NULL, Ynodes = NULL, Lnodes = NULL, Cnodes = NULL,
  abar = NULL,
  alpha = 0.95, grid.size = 0.5, tol = 1e-2,
  left.boundary = NULL, right.boundary = NULL,
  screen = FALSE, survival = FALSE,
  d.method = c("hazardbinning", "binning", "parametric", "hal_density"),
  verbose = TRUE, ...
)
}
\arguments{
  \item{X}{A data frame containing all nodes in temporal order. Columns must include
    the treatment, outcome, covariate and censoring nodes specified in
    \code{Anodes}, \code{Ynodes}, \code{Lnodes}, and \code{Cnodes}.}

  \item{Anodes}{Character vector giving the column names in \code{X} of the
    (possibly time-varying) treatment nodes. These define the treatment history.}

  \item{Ynodes}{Character vector giving the column names in \code{X} of the
    outcome nodes. At least one outcome node must be specified, and all of them
    must occur after the first treatment node in the column ordering of \code{X}.}

  \item{Lnodes}{Optional character vector of confounder nodes. May be \code{NULL} if there are no such nodes.}

  \item{Cnodes}{Optional character vector of censoring (or competing event) nodes.
    May be \code{NULL} if there is no censoring.}

  \item{abar}{Numeric vector or matrix specifying the target interventions.
    \itemize{
      \item If a \emph{vector}, each element defines a static intervention that
            sets the treatment to that value at \emph{all} time points. In this
            case, each strategy corresponds to a single scalar target value.
      \item If a \emph{matrix}, rows index intervention strategies and columns
            index time points (one column per element in \code{Anodes}). Then
            \code{abar[k, t]} is the target treatment value for strategy
            \code{k} at time \code{t}.
    }
    The argument must be numeric; \code{NULL} is not allowed.}

  \item{alpha}{Numeric scalar in \code{(0, 1)} controlling the density-truncation
    level. For each observation and time point, the method finds the smallest
    density threshold \eqn{f_\alpha} such that at most \code{alpha} of the total
    mass lies above this threshold. Cells with density below \eqn{f_\alpha} are
    treated as \dQuote{infeasible}. Default is \code{0.95}.}

  \item{grid.size}{Positive numeric scalar giving the spacing of the grid used to
    approximate the treatment density. If \code{NULL}, no internal grid is
    constructed and \code{abar} itself is used as the grid of evaluation points
    (this is only allowed when \code{abar} is a vector). Default is \code{0.5}.}

  \item{tol}{Non-negative numeric tolerance used when combining \code{abar} with
    the grid. Points closer than \code{tol} to an element of \code{abar} are
    considered duplicates and are dropped from the internal grid before
    merging with \code{abar}. Default is \code{1e-2}.}

  \item{left.boundary}{Optional numeric scalar setting the left boundary of the
    grid used to approximate the treatment density. If \code{NULL}, the minimum
    of the observed treatment values and \code{abar} is used.}

  \item{right.boundary}{Optional numeric scalar setting the right boundary of the
    density grid. If \code{NULL}, the maximum of the observed treatment values
    and \code{abar} is used.}

  \item{screen}{Logical; if \code{TRUE}, use variable-screening (via
    internal functions from \pkg{CICI}) for the treatment models, otherwise
    use the full treatment model formulae. Default is \code{FALSE}.}

  \item{survival}{Logical; indicates whether the outcome nodes correspond to a
    survival-type structure. Passed to the internal model-building function.
    Default is \code{FALSE}.}

  \item{d.method}{Character string specifying the density-estimation method used
    to estimate the conditional treatment density. Must be one of
    \code{"hazardbinning"}, \code{"binning"}, \code{"parametric"}, or
    \code{"hal_density"}. }

  \item{verbose}{Logical; if \code{TRUE}, print warnings about ignored arguments
    passed via \code{...} and other diagnostic messages. Default is \code{TRUE}.}

  \item{\dots}{Additional arguments passed to the underlying density-estimation
    function determined by \code{d.method}. For example, these may include
    \code{SL.library} for Super Learner-based methods, or tuning parameters for
    specific density estimators. Arguments not recognised by the chosen
    \code{d.method} are silently ignored when \code{verbose = FALSE} and produce
    a warning when \code{verbose = TRUE}.}
}
\details{
The main steps of the algorithm are:

\enumerate{
  \item \strong{Model specification:}
    Treatment models for each time point are constructed via helper routines from
    \pkg{CICI}. If \code{screen = TRUE}, a screening step updates the treatment
    formulas before density estimation (only recommended to address computational constraints).

  \item \strong{Grid construction:}
    A grid of treatment values, \code{query_abar}, is formed by combining:
    \itemize{
      \item observed treatment values in \code{X[, Anodes]},
      \item the target values in \code{abar}, and
      \item a regular grid from \code{left.boundary} to \code{right.boundary}
            with spacing \code{grid.size} (when \code{grid.size} is not \code{NULL}).
    }
    If \code{grid.size = NULL}, the grid is restricted to the unique values in
    \code{abar} (only allowed when \code{abar} is a vector).

  \item \strong{Density estimation:}
    For each time point, the conditional treatment density is evaluated on the
    grid for each observation using the specified \code{d.method}. The resulting
    matrices are normalised so that each row integrates to one over the grid
    (accounting for bin width).

  \item \strong{Feasibility threshold:}
    For each observation and time point, a density threshold \eqn{f_\alpha} is
    computed such that the cumulative mass below the sorted densities first
    exceeds \code{1 - alpha}. Cells with density below \eqn{f_\alpha} are flagged
    as \dQuote{infeasible}.

  \item \strong{Feasible mapping:}
    For each grid cell with density below \eqn{f_\alpha}, the algorithm finds the
    closest grid cell with density at or above \eqn{f_\alpha} (in terms of grid
    index) and maps its value to that cell. This defines a \dQuote{feasible}
    intervention that avoids low-density regions.

  \item \strong{Summary:}
    For each time point \eqn{t} and each intervention strategy (row of
    \code{abar}), the method collects:
    \itemize{
      \item the mean feasible value across individuals (column \code{Feasible}),
      \item the proportion of cells below the density threshold (column
            \code{Low}, interpreted as \code{\%infeasible} in the associated
            S3 methods),
      \item the corresponding target value \code{Abar} at time \eqn{t}, and
      \item the strategy index \code{Strategy}.
    }
    These are combined into a data frame stored as the \code{"summary"} attribute
    of the returned object.
}

Plotting and printing methods are available for visual and tabular diagnostics;
see \code{\link{plot.feasible}}, \code{\link{print.feasible}}, and
\code{\link{summary.feasible}}.
}
\value{
An object of class \code{"feasible"} with the following components:
\itemize{
  \item \code{feasible}: a list of length equal to the number of strategies
        (rows of \code{abar}). Each element is a matrix with one column per
        time point, containing the feasible intervention values for that
        strategy and time point across observations.
  \item \code{low_matrix}: a list of length equal to the number of time points.
        Each element is a logical matrix indicating, on the internal grid, which
        cells were marked as below the density threshold.
}
The object additionally has a \code{"summary"} attribute, a data frame with at
least the columns \code{time}, \code{Strategy}, \code{Abar}, \code{Feasible},
and \code{Low}, which is accessed and formatted by \code{\link{summary.feasible}}
and \code{\link{print.feasible}}.
}
\seealso{
\code{\link{plot.feasible}}, \code{\link{print.feasible}},
\code{\link{summary.feasible}}
}
\examples{
data(EFV)

\donttest{
Lnodes <- c("adherence.1","weight.1",
            "adherence.2","weight.2",
            "adherence.3","weight.3",
            "adherence.4","weight.4")
Ynodes <- c("VL.0","VL.1","VL.2","VL.3","VL.4")
Anodes <- c("efv.0","efv.1","efv.2","efv.3","efv.4")

## ------------------------------------------------------------------
## Example 1: Hazard binning with default grid
## Static grid of targets (vector abar) over the full support of efv.*
## ------------------------------------------------------------------

abar_static <- seq(0, 10, by = 1)

m_hazard <- feasible(
  X      = EFV,
  Anodes = Anodes,
  Lnodes = Lnodes,
  Ynodes = Ynodes,
  d.method      = "hazardbinning", # long computation, but appropriate
  abar          = abar_static,
  grid.size     = 0.5,
  left.boundary = 0,
  right.boundary = 10
)


## Individual-level feasible values (one matrix per strategy):
## rows = individuals, columns = time points
feasible_matrix <- m_hazard$feasible # pass on to gofrmula/sgf
lapply(feasible_matrix, head)

## Inspect feasability of strategies
m_hazard             # see also ?print.feasible
summary(m_hazard)    # see also ?summary.feasible


## ------------------------------------------------------------------
## Example 2: Parametric density, using abar as the grid
## Here grid.size = NULL, so only the target values are used as grid
## ------------------------------------------------------------------

abar_param <- seq(0, 10, by = 2)

m_param <- feasible(
  X      = EFV,
  Anodes = Anodes,
  Lnodes = Lnodes,
  Ynodes = Ynodes,
  # fast, but useful for reasonably symmetric distributions
  d.method      = "parametric", 
  abar          = abar_param,
  grid.size     = NULL,
  left.boundary = 0,
  right.boundary = 10
)

## Inspect feasability of strategies
m_param             # see also ?print.feasible
summary(m_param)    # see also ?summary.feasible


## ------------------------------------------------------------------
## Example 3: Matrix abar with non-constant strategies over time
## Each row is a strategy, each column corresponds to efv.0, ..., efv.4
## ------------------------------------------------------------------

abar_matrix <- rbind(
  c(0, 2, 4, 6, 8),  # strategy 1
  c(9, 6, 2, 1, 0),  # strategy 2
  c(1, 3, 5, 7, 9)   # strategy 3
)

m_matrix <- feasible(
  X      = EFV,
  Anodes = Anodes,
  Lnodes = Lnodes,
  Ynodes = Ynodes,
  d.method      = "parametric",
  abar          = abar_matrix,
  grid.size     = 1,
  left.boundary = 0,
  right.boundary = 10
)

## Inspect feasability of strategies
m_matrix             # see also ?print.feasible
summary(m_matrix)    # see also ?summary.feasible

}
}