#' Linter configurations for `{admiral}` family of packages
#'
#' This function lists out the linter configurations for the `{admiral}`
#' family of packages.
#'
#' @param line_length Line length linter
#' @param object_usage Object usage linter
#' @param cyclocomp Cyclomatic complexity linter
#' @param indentation Indentation linter
#' @param undesirable_function Undesirable function linter
#' @param pipe_consistency Pipe consistency linter
#' @param ... Any additional default linters to specify for use within the package
#'
#' @details
#' Default configurations of special note:
#'   - Enforcing the use of `cli::cli_abort()` over `stop()`
#'   - Enforcing the use of `cli::cli_warn()` over `warning()`
#'   - Enforcing the use of `cli::cli_inform()` over `message()`
#'   - Enforcing the use of  vectorized `dplyr::if_else()`  over `ifelse()`
#'   - Enforcing the use of `stringr::str_detect()` over `grepl()`
#'
#' @keywords internal
#'
#' @examples
#' library(lintr)
#'
#' # The code below should reside in .lintr.R file
#' linters <- admiral_linters(
#'   line_length = line_length_linter(80),
#'   indentation = NULL
#' )
#'
admiral_linters <- function(
  line_length = lintr::line_length_linter(100),
  object_usage = NULL,
  cyclocomp = lintr::cyclocomp_linter(complexity_limit = 20),
  pipe_consistency = lintr::pipe_consistency_linter("auto"),
  indentation = NULL,
  undesirable_function = lintr::undesirable_function_linter(
    fun = c(
      lintr::default_undesirable_functions,
      "stop" = "please use cli_abort() from the {cli} package",
      "warning" = "please use cli_warning() from the {cli} package",
      "message" = "please use cli_inform() from the {cli} package",
      "ifelse" = "please use vectorized if_else() from the {dplyr} package",
      "grepl" = "please use str_detect() from the {stringr} package",
      "stopifnot" = "please consider combination of any() and cli_abort() from the {cli} package"
    ),
    symbol_is_undesirable = FALSE
  ),
  ...
) {
  c(
    lintr::linters_with_defaults(
      # Add default linters below
      line_length_linter = line_length,
      object_usage_linter = object_usage,
      cyclocomp_linter = cyclocomp,
      pipe_consistency_linter = pipe_consistency,
      indentation_linter = indentation,
      undesirable_function_linter = undesirable_function,
      ...
    )
    # Add custom linters below
  )
}
