## ----include = FALSE----------------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)

## ----setup--------------------------------------------------------------------
library(rLifting)

if (!requireNamespace("dplyr", quietly = TRUE) ||
    !requireNamespace("ggplot2", quietly = TRUE)) {
  knitr::opts_chunk$set(eval = FALSE)
  message("Required packages 'dplyr' and 'ggplot2' are missing. Vignette code will not run.")
} else {
  library(dplyr)
  library(ggplot2)
}

data("doppler_example", package = "rLifting")

# Visualize the noisy input
ggplot(doppler_example, aes(x = index, y = noisy)) +
  geom_line(color = "grey60", alpha = 0.8) +
  geom_line(aes(y = original), color = "black", alpha = 0.5, linetype = "dashed") +
  theme_minimal() +
  labs(title = "Input signal (Doppler + noise)")

## ----offline------------------------------------------------------------------
# Define the Wavelet Scheme (Haar for this example)
scheme = lifting_scheme("haar")

# Denoise using strict offline mode
# levels = floor(log2(n)) ensures maximum decomposition depth
offline_clean = denoise_signal_offline(
  doppler_example$noisy,
  scheme,
  levels = floor(log2(nrow(doppler_example))),
  method = "semisoft"
)

## ----causal-------------------------------------------------------------------
# Causal Denoising with a window of 256 points
causal_clean = denoise_signal_causal(
  doppler_example$noisy,
  scheme,
  window_size = 256,
  levels = floor(log2(256)), 
  method = "semisoft"
)

## ----online-------------------------------------------------------------------
# Initialize the Stream Processor
processor = new_wavelet_stream(
  scheme,
  window_size = 256,
  levels = floor(log2(256)),
  method = "semisoft"
)

# Simulate streaming (vectorized loop for demonstration)
online_clean = numeric(nrow(doppler_example))
for (i in 1:nrow(doppler_example)) {
  online_clean[i] = processor(doppler_example$noisy[i])
}

## ----comparison---------------------------------------------------------------
# Combine results
df_plot = doppler_example |>
  mutate(
    Offline = offline_clean,
    Causal = causal_clean,
    Online = online_clean
  )

# Plot focusing on a specific section to see the lag
ggplot(df_plot, aes(x = index)) +
  # Add Truth curve
  geom_line(aes(y = original, color = "Truth"), linewidth = 0.8, alpha = 0.6) +
  geom_line(aes(y = Offline, color = "Offline"), alpha = 0.8) +
  geom_line(aes(y = Causal, color = "Causal"), alpha = 0.8) +
  scale_color_manual(values = c("Truth" = "black", "Offline" = "blue", "Causal" = "red")) +
  theme_minimal() +
  labs(title = "Trade-off: smoothness vs. causality") +
  coord_cartesian(xlim = c(500, 1000)) # Zoom in

