## ----setup, include=FALSE-----------------------------------------------------
knitr::opts_chunk$set(echo = TRUE)

library(gt)
library(glue)
library(gluedown)
library(kableExtra)

## ----loadData, include=FALSE--------------------------------------------------
# formatTable <- function(tableName, tableSTOICH){
#   tableNEON <- select(filter(tableNEON, STOICH_db_Table == tableName), !c(STOICH_db_Table)) %>%
#     mutate(NEON_Data=if_else(is.na(Set_Value),
#                              if_else(is.na(NEON_Table) & is.na(NEON_Table_Variable), 
#                                      NEON_Table, 
#                                      paste(NEON_Table, NEON_Table_Variable, Operation_Using_NEON_Variable, sep=" -> ")),
#                              if_else(is.na(NEON_Table), Set_Value, paste(Set_Value, " (", NEON_Table, ")", sep=""))),
#            .keep="unused")
#   return(tableNEON)
# }

library(tidyverse)
# library(tibble)
# library(dplyr)
# library(tidyr)
# library(purrr)
# library(stringr)
# library(readr)
library(lubridate)
library(sf)
library(units)
library(jsonlite)
library(stoichUtilities)

basePath <- system.file("testdata", "STOICH_NEON_Only_2025-09-10", package="stoichUtilities")
stoichData <- stoichUtilities::loadSTOICH(dataPath=basePath)
stoichCols <- names(stoichData)
stoichTbl <- tibble(index = 1:100)
for (i in stoichCols){
  stoichTbl <- left_join(stoichTbl, tibble(x = colnames(stoichData[[i]])) %>% mutate(index=seq(n())), by="index")
  
  colnames(stoichTbl) <- c(colnames(stoichTbl)[1:length(colnames(stoichTbl))-1], i)
}

stoichTbl <- stoichTbl %>%
  select(c("tbl_InputFile", "tbl_Contact", "tbl_Source", "tbl_Site", "tbl_SampleEvent", "tbl_OrganismStoichiometry", "tbl_WaterChemistry")) %>%
  distinct()

stoichTbl <- stoichTbl %>% 
  slice_head(n=nrow(stoichTbl)-1) %>% 
  mutate(across(everything(), ~if_else(is.na(.x), "", .x)))

stoichSample <- stoichUtilities::joinSTOICH(stoichData) %>%
  slice_sample(n=5)

## ----stoich-install, eval=FALSE-----------------------------------------------
# install.packages(“stoichUtilties”)

## ----package-install, eval=FALSE----------------------------------------------
# install.packages(“tidyverse”)
# install.packages(“lubridate”)
# install.packages(“sf”)
# install.packages(“units”)

## ----load-packages, eval=FALSE------------------------------------------------
# # Tidyverse and other support libraries
# library(tidyverse)
# library(lubridate)
# library(sf)
# library(units)
# 
# # Load stoichUtilities
# library(stoichUtilities)

## ----using-utilities, eval=FALSE----------------------------------------------
# # Load the STOICH data (assumed to be in a directory named "data" inside the home directory)
# stoichData <- loadSTOICH(dataPath=file.path(path.expand("~"), "data"))
# 
# # Filter the STOICH data
# stoichFiltered <- filterSTOICH(dataTables=stoichData,
#                                var="State",
#                                val="North Dakota",
#                                condition="Equal")
# 
# # Match organism stoichiometry data with water chemsitry data for samples that weren't taken at the exact same time
# stoichPaired <- locateDataPairsSTOICH(stoichData,
#                                       timeDiff=2,
#                                       timeUnits="weeks",
#                                       distance=2,
#                                       pairMethod="Min Time",
#                                       ignoreExisting=TRUE)
# 
# # Join all the tables into one large wide table
# stoichTable <- joinSTOICH(stoichPaired)
# 

## ----mapping STOICH data, eval=TRUE-------------------------------------------
# Count how many sites are in each state.
stateCounts <- stoichData$tbl_Site |>
  select(c("State")) |> 
  drop_na(State) |>
  # Change the data to lowercase to allow the table to merge with the Tidyverse state data
  mutate(State=str_to_lower(State)) |>
  group_by(State) |>
  # Summarise to get the number of sites for each state.
  summarise(nSites=n(), .groups="keep")

# Use state data from Tidyverse and join the site data to it.
us_states <- map_data("state") |>
  left_join(stateCounts, by=c("region"="State"))

# Create a list of states in the Tidyverse dataset so only data from states with boundaries are used.
stateList <- unique(us_states$region)

# Split out the GPS data to map individual sites.
gpsSites <- stoichData$tbl_Site |>
  filter(str_to_lower(State) %in% stateList) |>
  select(c("Latitude", "Longitude")) |>
  drop_na(Latitude, Longitude) |>
  # Round the gps coordinates so points that are close will be merged into a single point.
  mutate(across(everything(), ~round(.x, 2))) |>
  unique()

# Start building the map.
ggplot(data=us_states, mapping=aes(x=long, y=lat, group=group, fill=nSites)) +
  geom_polygon(color="black") +
  theme(legend.position = "bottom") +
  geom_point(data=gpsSites, aes(x=Longitude, y=Latitude, group=NA), size=2, shape=23, fill="orange")


## ----help-functions, eval=FALSE-----------------------------------------------
# help(loadSTOICH)

## ----STOICH-structure, echo = FALSE-------------------------------------------
gt(stoichTbl) %>%
  tab_header(
    title = "STOICH Data Table Structure",
    subtitle = "(variables for each table in the data set)"
  ) %>%
  tab_options(
    table.align = "left",
    table.font.size = "small",
    column_labels.font.weight = "bold",
    data_row.padding.horizontal = px(10),
    column_labels.padding.horizontal = px(10)
  ) %>%
  tab_style(
    style = list(
      cell_borders(
        sides = "all",
        color = "#000000",
        style = "solid",
        weight = px(1)
      ),
      cell_borders(
        sides = "all",
        color = "#000000",
        style = "solid",
        weight = px(1)
      )
    ),
    locations = list(
      cells_body(
        columns = everything(),
        rows = everything()
      ),
      cells_column_labels(
        columns = everything()
      )
    )
  )

## ----STOICH-sample, echo = FALSE----------------------------------------------
gt(stoichSample) %>%
  tab_header(
    title = "Sample of the STOICH Data",
    subtitle = ""
  )%>%
  tab_options(
    table.align = "left",
    table.font.size = "small",
    column_labels.font.weight = "bold",
    data_row.padding.horizontal = px(10),
    column_labels.padding.horizontal = px(10)
  ) %>%
  tab_style(
    style = list(
      cell_borders(
        sides = "all",
        color = "#000000",
        style = "solid",
        weight = px(1)
      ),
      cell_borders(
        sides = "all",
        color = "#000000",
        style = "solid",
        weight = px(1)
      )
    ),
    locations = list(
      cells_body(
        columns = everything(),
        rows = everything()
      ),
      cells_column_labels(
        columns = everything()
      )
    )
  )

