import logging
import os
import shutil
import socket
import sys
import webbrowser
from pathlib import Path

script_dir_name = str(Path(__file__).parent / "nsys_cpu_stats")
sys.path.append(script_dir_name)
from cli import process_nsys_rep_cli
from nsys_cpu_stats.trace_progress_indicator import g_progress_indicator as progress

from nsys_recipe import log
from nsys_recipe.data_service import DataService
from nsys_recipe.lib import data_utils, helpers, recipe, summary
from nsys_recipe.lib.args import Option
from nsys_recipe.log import logger

default_output_root_folder = "./output"
server_folder_rel = "/server"
data_folder_rel = "/data"
output_sqlite_folder_rel = "/sqlite"
output_root_folder = default_output_root_folder
server_folder = default_output_root_folder + server_folder_rel
data_folder = default_output_root_folder + data_folder_rel
output_sqlite_folder = default_output_root_folder + output_sqlite_folder_rel


def is_server_port_open(server_hostname, server_port):
    sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
    sock.settimeout(0.1)
    err = sock.connect_ex((server_hostname, server_port))
    sock.close()
    return 0 == err


def set_output_folders(root_folder):
    global output_root_folder
    global server_folder
    global data_folder
    global output_sqlite_folder
    output_root_folder = root_folder
    server_folder = root_folder + server_folder_rel
    data_folder = root_folder + data_folder_rel
    output_sqlite_folder = root_folder + output_sqlite_folder_rel


def verify_folder_created(dirpath):
    path = Path(dirpath)
    if path.exists():
        return
    path.mkdir(parents=True, exist_ok=True)


def verify_output_folders_created(sqlite=None):
    verify_folder_created(server_folder)
    verify_folder_created(data_folder)
    if sqlite:
        verify_folder_created(sqlite)
    else:
        verify_folder_created(output_sqlite_folder)


def SetLoggerEnv(log_arg):
    # Apply the values to the logger if it was already configured:
    logging_handlers = [logging.StreamHandler(sys.stdout)]
    logging_format = "[%(levelname)-7s][%(asctime)s] %(message)s"
    log_arg = log_arg.lower()

    # Set logger output to stdout if enabled
    if log_arg in ("d", "debug"):
        os.environ["PNR_LOGGER_STDOUT"] = "True"
        os.environ["PNR_LOGGER_SEVERITY"] = "DEBUG"
        logging.basicConfig(
            level=logging.DEBUG,
            handlers=logging_handlers,
            format=logging_format,
            force=True,
        )
        progress.printer = progress.NewlinePrinter
    elif log_arg in ("i", "info"):
        os.environ["PNR_LOGGER_STDOUT"] = "True"
        os.environ["PNR_LOGGER_SEVERITY"] = "INFO"
        logging.basicConfig(
            level=logging.INFO,
            handlers=logging_handlers,
            format=logging_format,
            force=True,
        )
        progress.printer = progress.NewlinePrinter
    elif log_arg in ("w", "warn", "warning"):
        os.environ["PNR_LOGGER_STDOUT"] = "True"
        os.environ["PNR_LOGGER_SEVERITY"] = "WARNING"
        logging.basicConfig(
            level=logging.WARNING,
            handlers=logging_handlers,
            format=logging_format,
            force=True,
        )
        progress.printer = progress.NewlinePrinter
    elif log_arg in ("e", "error"):
        os.environ["PNR_LOGGER_STDOUT"] = "True"
        os.environ["PNR_LOGGER_SEVERITY"] = "ERROR"
        logging.basicConfig(
            level=logging.ERROR,
            handlers=logging_handlers,
            format=logging_format,
            force=True,
        )
        progress.printer = progress.NewlinePrinter
    else:  # log_arg in ('n', 'none')
        os.environ["PNR_LOGGER_STDOUT"] = "False"
        os.environ["PNR_LOGGER_SEVERITY"] = "NOTSET"
        logging.basicConfig(level=logging.NOTSET, format=logging_format, force=True)
        progress.printer = progress.SameLinePrinter
        # By default, the Windows console does not support escape codes
        if "win32" == sys.platform:
            try:
                import colorama

                colorama.just_fix_windows_console()
            except ImportError:
                pass


def SetSqlEnv(sqlite_arg):
    if not sqlite_arg or not isinstance(sqlite_arg, str):
        return
    sqlite_arg = sqlite_arg.lower()
    if sqlite_arg in ("n", "no"):
        os.environ["PNR_DO_SQL"] = "False"
        os.environ["PNR_DO_SQL_FORCE"] = "False"
        process_nsys_rep_cli.g_do_sql = False
        process_nsys_rep_cli.g_do_sql_force = False
    elif sqlite_arg in ("y", "yes", "s", "skip"):
        os.environ["PNR_DO_SQL"] = "True"
        os.environ["PNR_DO_SQL_FORCE"] = "False"
        process_nsys_rep_cli.g_do_sql = True
        process_nsys_rep_cli.g_do_sql_force = False
    elif sqlite_arg in ("f", "force"):
        os.environ["PNR_DO_SQL"] = "True"
        os.environ["PNR_DO_SQL_FORCE"] = "True"
        process_nsys_rep_cli.g_do_sql = True
        process_nsys_rep_cli.g_do_sql_force = True


class GraphicsHotspotAnalysis(recipe.Recipe):
    def run(self, context):
        super().run(context)

        SetLoggerEnv(super().get_parsed_arg("log", "NONE"))
        if super().get_parsed_arg("silent", False):
            progress.printer = None

        # Clear data_folder
        output_path_exists = os.path.lexists(data_folder)
        if output_path_exists:
            output_path_is_directory = os.path.isdir(data_folder)
            if super().get_parsed_arg("force_overwrite", False):
                try:
                    if output_path_is_directory:
                        for file in os.listdir(data_folder):
                            os.remove(os.path.join(data_folder, file))
                    else:
                        print(
                            f"Warning! output path '{data_folder}' exists and "
                            "is not a directory, it will now be removed."
                        )
                        os.remove(data_folder)
                except (OSError, FileNotFoundError) as e:
                    print(f"Error while attempting to remove '{data_folder}':{e}")
            else:  # !force_overwrite
                if output_path_is_directory:
                    print(
                        f"Output directory '{data_folder}' already exists. "
                        "Please run with '--force-overwrite' to clear its "
                        "contents and proceed"
                    )
                    return
                else:
                    print(
                        f"Output path '{data_folder}' already exists and is "
                        "not a directory. Please run with '--force-overwrite' to "
                        "remove it and proceed"
                    )
                    return

        set_output_folders(super().get_output_dir() or default_output_root_folder)

        SetSqlEnv(super().get_parsed_arg("export_cache", "yes"))
        sqlite_path_arg_dir = None
        sqlite_path_arg_filename = None
        sqlite_path_arg = super().get_parsed_arg("export_cache_path", None)
        if sqlite_path_arg:
            if os.path.isdir(sqlite_path_arg):
                sqlite_path_arg_dir = sqlite_path_arg
            else:
                sqlite_path_arg_dir, sqlite_path_arg_filename = os.path.split(
                    sqlite_path_arg
                )
            if not os.path.isabs(sqlite_path_arg_dir):
                sqlite_path_arg_dir = os.path.join(
                    output_root_folder, sqlite_path_arg_dir
                )

        verify_output_folders_created(sqlite_path_arg_dir)
        viewer_src_folder = os.path.join(script_dir_name, "viewer")
        server_src_folder = os.path.join(script_dir_name, "server")
        shutil.copytree(viewer_src_folder, output_root_folder, dirs_exist_ok=True)
        shutil.copytree(server_src_folder, server_folder, dirs_exist_ok=True)

        input_files = []
        for input_path in self._parsed_args.input:
            if os.path.isfile(input_path):
                input_files.append(input_path)
            elif os.path.isdir(input_path):
                for f in os.listdir(input_path):
                    input_file = None
                    if f.endswith(".nsys-rep"):
                        input_file = os.path.join(input_path, f)
                    elif f.endswith(".etl"):
                        input_file = os.path.join(input_path, f)

        if not input_files:
            return  # print an error?

        nsys_path = str(Path(__file__).parent.parent.parent.parent.parent.parent)

        output_sqlite_file = None
        for input_file in input_files:
            if input_file:
                print(f"Processing {input_file}")

                # Need to add all of the optional args and set them as None,
                # otherwise there is an issue writing the meta_info df to sql
                sqlite_dir = output_sqlite_folder
                sqlite_filename = "gfx_hotspot.sqlite"
                if sqlite_path_arg_dir:
                    sqlite_dir = sqlite_path_arg_dir
                process_nsys_rep_cli.process(
                    nsys_dir_path=nsys_path,
                    input_file=input_file,
                    output_path=data_folder,
                    output_sqlite_path=sqlite_dir,
                    output_sqlite_name=sqlite_filename,
                    start_time_sec=None,
                    end_time_sec=None,
                )
                sqlite_filepath = os.path.join(data_folder, sqlite_filename)
                if (
                    sqlite_filepath
                    and os.path.exists(sqlite_filepath)
                    and not output_sqlite_file
                ):
                    output_sqlite_file = sqlite_filepath
                break
        if not output_sqlite_file:
            return

        run_viewer = super().get_parsed_arg("run_viewer", False)
        if run_viewer:
            server_port = 4200
            server_port_arg = super().get_parsed_arg("viewer_server_port", server_port)
            if 0 <= server_port_arg <= 65535:
                server_port = server_port_arg
            server_hostname = "127.0.0.1"
            if is_server_port_open(server_hostname, server_port):
                print(
                    f"Connection at {server_hostname}:{server_port} is already "
                    "open, aborting."
                )
            else:
                viewer_url = f"http://{server_hostname}:{server_port}"
                print(f"Serving report viewer at {viewer_url}/ ...")

                viewer_shown = False
                if super().get_parsed_arg("open_viewer", False):
                    viewer_shown = webbrowser.open_new(viewer_url)
                if not viewer_shown:
                    print("Please open this URL in a web browser to view the report")

                print("Press <Ctrl>+C to exit")

                try:
                    process_nsys_rep_cli.run_viewer(
                        report_path=output_sqlite_file, server_port=server_port
                    )
                except KeyboardInterrupt as e:
                    pass
        run_script_path = os.path.join(output_root_folder, "run_viewer.py")
        again_suffix = " again" if run_viewer else ""
        print(f"To run the viewer{again_suffix}, use {run_script_path}")

    @classmethod
    def get_argument_parser(cls):
        parser = super().get_argument_parser()

        parser.add_recipe_argument(Option.INPUT, required=True)
        parser.add_recipe_argument(
            "--log",
            type=str,
            metavar="NONE|ERROR|WARNING|INFO|DEBUG",
            default="NONE",
            help="Display logging for the recipe processing\n"
            "steps in the console.\n"
            "Possible values: NONE (default), ERROR, WARNING,\n"
            "                 INFO, DEBUG",
        )
        parser.add_recipe_argument(
            "--export-cache",
            type=str,
            default="yes",
            metavar="yes|no|force",
            help="Cache the exported sqlite report from the nsys-rep file.\n"
            "Possible values:\n"
            "  yes   - (default) Will generate a sqlite report if\n"
            "          one does not exist.\n"
            "  no    - Will not generate a sqlite report.\n"
            "          Requires '--sqlite-path' to be set.\n"
            "  force - Will always generate a report.\n",
        )
        parser.add_recipe_argument(
            "--export-cache-path",
            type=str,
            default=None,
            help="Path of cached sqlite report.\n"
            "If '--export-cache' is 'no', this value is required.\n",
        )
        parser.add_recipe_argument(
            "--run-viewer",
            action="store_true",
            help="Will run the viewer in a localhost web server.",
        )
        parser.add_recipe_argument(
            "--viewer-server-port",
            type=int,
            default=4200,
            metavar="PORT",
            help="Server port for the localhost viewer.",
        )
        parser.add_recipe_argument(
            "--open-viewer",
            action="store_true",
            help="Will automaticaly open the viewer in the\n"
            "default web browser.\n"
            "Requires '--run-viewer' to be set.",
        )

        return parser
