﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/comprehend/ComprehendRequest.h>
#include <aws/comprehend/Comprehend_EXPORTS.h>
#include <aws/comprehend/model/UpdateDataSecurityConfig.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace Comprehend {
namespace Model {

/**
 */
class UpdateFlywheelRequest : public ComprehendRequest {
 public:
  AWS_COMPREHEND_API UpdateFlywheelRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateFlywheel"; }

  AWS_COMPREHEND_API Aws::String SerializePayload() const override;

  AWS_COMPREHEND_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The Amazon Resource Number (ARN) of the flywheel to update.</p>
   */
  inline const Aws::String& GetFlywheelArn() const { return m_flywheelArn; }
  inline bool FlywheelArnHasBeenSet() const { return m_flywheelArnHasBeenSet; }
  template <typename FlywheelArnT = Aws::String>
  void SetFlywheelArn(FlywheelArnT&& value) {
    m_flywheelArnHasBeenSet = true;
    m_flywheelArn = std::forward<FlywheelArnT>(value);
  }
  template <typename FlywheelArnT = Aws::String>
  UpdateFlywheelRequest& WithFlywheelArn(FlywheelArnT&& value) {
    SetFlywheelArn(std::forward<FlywheelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Number (ARN) of the active model version.</p>
   */
  inline const Aws::String& GetActiveModelArn() const { return m_activeModelArn; }
  inline bool ActiveModelArnHasBeenSet() const { return m_activeModelArnHasBeenSet; }
  template <typename ActiveModelArnT = Aws::String>
  void SetActiveModelArn(ActiveModelArnT&& value) {
    m_activeModelArnHasBeenSet = true;
    m_activeModelArn = std::forward<ActiveModelArnT>(value);
  }
  template <typename ActiveModelArnT = Aws::String>
  UpdateFlywheelRequest& WithActiveModelArn(ActiveModelArnT&& value) {
    SetActiveModelArn(std::forward<ActiveModelArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon Resource Name (ARN) of the IAM role that grants Amazon Comprehend
   * permission to access the flywheel data.</p>
   */
  inline const Aws::String& GetDataAccessRoleArn() const { return m_dataAccessRoleArn; }
  inline bool DataAccessRoleArnHasBeenSet() const { return m_dataAccessRoleArnHasBeenSet; }
  template <typename DataAccessRoleArnT = Aws::String>
  void SetDataAccessRoleArn(DataAccessRoleArnT&& value) {
    m_dataAccessRoleArnHasBeenSet = true;
    m_dataAccessRoleArn = std::forward<DataAccessRoleArnT>(value);
  }
  template <typename DataAccessRoleArnT = Aws::String>
  UpdateFlywheelRequest& WithDataAccessRoleArn(DataAccessRoleArnT&& value) {
    SetDataAccessRoleArn(std::forward<DataAccessRoleArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Flywheel data security configuration.</p>
   */
  inline const UpdateDataSecurityConfig& GetDataSecurityConfig() const { return m_dataSecurityConfig; }
  inline bool DataSecurityConfigHasBeenSet() const { return m_dataSecurityConfigHasBeenSet; }
  template <typename DataSecurityConfigT = UpdateDataSecurityConfig>
  void SetDataSecurityConfig(DataSecurityConfigT&& value) {
    m_dataSecurityConfigHasBeenSet = true;
    m_dataSecurityConfig = std::forward<DataSecurityConfigT>(value);
  }
  template <typename DataSecurityConfigT = UpdateDataSecurityConfig>
  UpdateFlywheelRequest& WithDataSecurityConfig(DataSecurityConfigT&& value) {
    SetDataSecurityConfig(std::forward<DataSecurityConfigT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_flywheelArn;

  Aws::String m_activeModelArn;

  Aws::String m_dataAccessRoleArn;

  UpdateDataSecurityConfig m_dataSecurityConfig;
  bool m_flywheelArnHasBeenSet = false;
  bool m_activeModelArnHasBeenSet = false;
  bool m_dataAccessRoleArnHasBeenSet = false;
  bool m_dataSecurityConfigHasBeenSet = false;
};

}  // namespace Model
}  // namespace Comprehend
}  // namespace Aws
