/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.repositories;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRunnable;
import org.elasticsearch.action.admin.cluster.repositories.delete.DeleteRepositoryRequest;
import org.elasticsearch.action.admin.cluster.repositories.put.PutRepositoryRequest;
import org.elasticsearch.cluster.AckedClusterStateUpdateTask;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateApplier;
import org.elasticsearch.cluster.RepositoryCleanupInProgress;
import org.elasticsearch.cluster.RestoreInProgress;
import org.elasticsearch.cluster.SnapshotDeletionsInProgress;
import org.elasticsearch.cluster.SnapshotsInProgress;
import org.elasticsearch.cluster.ack.AckedRequest;
import org.elasticsearch.cluster.ack.ClusterStateUpdateResponse;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.metadata.RepositoriesMetadata;
import org.elasticsearch.cluster.metadata.RepositoryMetadata;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.node.DiscoveryNodeRole;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.component.AbstractLifecycleComponent;
import org.elasticsearch.common.regex.Regex;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.repositories.RepositoriesStatsArchive;
import org.elasticsearch.repositories.Repository;
import org.elasticsearch.repositories.RepositoryData;
import org.elasticsearch.repositories.RepositoryException;
import org.elasticsearch.repositories.RepositoryMissingException;
import org.elasticsearch.repositories.RepositoryStatsSnapshot;
import org.elasticsearch.repositories.VerifyNodeRepositoryAction;
import org.elasticsearch.repositories.blobstore.MeteredBlobStoreRepository;
import org.elasticsearch.threadpool.ThreadPool;
import org.elasticsearch.transport.TransportService;

public class RepositoriesService
extends AbstractLifecycleComponent
implements ClusterStateApplier {
    private static final Logger logger = LogManager.getLogger(RepositoriesService.class);
    public static final Setting<TimeValue> REPOSITORIES_STATS_ARCHIVE_RETENTION_PERIOD = Setting.positiveTimeSetting("repositories.stats.archive.retention_period", TimeValue.timeValueHours((long)2L), Setting.Property.NodeScope);
    public static final Setting<Integer> REPOSITORIES_STATS_ARCHIVE_MAX_ARCHIVED_STATS = Setting.intSetting("repositories.stats.archive.max_archived_stats", 100, 0, Setting.Property.NodeScope);
    private final Map<String, Repository.Factory> typesRegistry;
    private final Map<String, Repository.Factory> internalTypesRegistry;
    private final ClusterService clusterService;
    private final ThreadPool threadPool;
    private final VerifyNodeRepositoryAction verifyAction;
    private final Map<String, Repository> internalRepositories = ConcurrentCollections.newConcurrentMap();
    private volatile Map<String, Repository> repositories = Collections.emptyMap();
    private final RepositoriesStatsArchive repositoriesStatsArchive;

    public RepositoriesService(Settings settings, ClusterService clusterService, TransportService transportService, Map<String, Repository.Factory> typesRegistry, Map<String, Repository.Factory> internalTypesRegistry, ThreadPool threadPool) {
        this.typesRegistry = typesRegistry;
        this.internalTypesRegistry = internalTypesRegistry;
        this.clusterService = clusterService;
        this.threadPool = threadPool;
        if ((DiscoveryNode.isDataNode(settings) || DiscoveryNode.isMasterNode(settings)) && !RepositoriesService.isDedicatedVotingOnlyNode(DiscoveryNode.getRolesFromSettings(settings))) {
            clusterService.addHighPriorityApplier(this);
        }
        this.verifyAction = new VerifyNodeRepositoryAction(transportService, clusterService, this);
        this.repositoriesStatsArchive = new RepositoriesStatsArchive(REPOSITORIES_STATS_ARCHIVE_RETENTION_PERIOD.get(settings), REPOSITORIES_STATS_ARCHIVE_MAX_ARCHIVED_STATS.get(settings), threadPool::relativeTimeInMillis);
    }

    public void registerRepository(final PutRepositoryRequest request, ActionListener<ClusterStateUpdateResponse> listener) {
        assert (this.lifecycle.started()) : "Trying to register new repository but service is in state [" + (Object)((Object)this.lifecycle.state()) + "]";
        final RepositoryMetadata newRepositoryMetadata = new RepositoryMetadata(request.name(), request.type(), request.settings());
        RepositoriesService.validate(request.name());
        ActionListener<ClusterStateUpdateResponse> registrationListener = request.verify() ? ActionListener.delegateFailure(listener, (delegatedListener, clusterStateUpdateResponse) -> {
            if (clusterStateUpdateResponse.isAcknowledged()) {
                this.verifyRepository(request.name(), ActionListener.delegateFailure(delegatedListener, (innerDelegatedListener, discoveryNodes) -> innerDelegatedListener.onResponse(clusterStateUpdateResponse)));
            } else {
                delegatedListener.onResponse(clusterStateUpdateResponse);
            }
        }) : listener;
        try {
            this.closeRepository(this.createRepository(newRepositoryMetadata, this.typesRegistry));
        }
        catch (Exception e) {
            registrationListener.onFailure(e);
            return;
        }
        this.clusterService.submitStateUpdateTask("put_repository [" + request.name() + "]", new AckedClusterStateUpdateTask<ClusterStateUpdateResponse>((AckedRequest)request, registrationListener){

            @Override
            protected ClusterStateUpdateResponse newResponse(boolean acknowledged) {
                return new ClusterStateUpdateResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) {
                RepositoriesService.ensureRepositoryNotInUse(currentState, request.name());
                Metadata metadata = currentState.metadata();
                Metadata.Builder mdBuilder = Metadata.builder(currentState.metadata());
                RepositoriesMetadata repositories = (RepositoriesMetadata)metadata.custom("repositories");
                if (repositories == null) {
                    logger.info("put repository [{}]", (Object)request.name());
                    repositories = new RepositoriesMetadata(Collections.singletonList(new RepositoryMetadata(request.name(), request.type(), request.settings())));
                } else {
                    boolean found = false;
                    ArrayList<RepositoryMetadata> repositoriesMetadata = new ArrayList<RepositoryMetadata>(repositories.repositories().size() + 1);
                    for (RepositoryMetadata repositoryMetadata : repositories.repositories()) {
                        if (repositoryMetadata.name().equals(newRepositoryMetadata.name())) {
                            if (newRepositoryMetadata.equalsIgnoreGenerations(repositoryMetadata)) {
                                return currentState;
                            }
                            found = true;
                            repositoriesMetadata.add(newRepositoryMetadata);
                            continue;
                        }
                        repositoriesMetadata.add(repositoryMetadata);
                    }
                    if (!found) {
                        logger.info("put repository [{}]", (Object)request.name());
                        repositoriesMetadata.add(new RepositoryMetadata(request.name(), request.type(), request.settings()));
                    } else {
                        logger.info("update repository [{}]", (Object)request.name());
                    }
                    repositories = new RepositoriesMetadata(repositoriesMetadata);
                }
                mdBuilder.putCustom("repositories", repositories);
                return ClusterState.builder(currentState).metadata(mdBuilder).build();
            }

            @Override
            public void onFailure(String source, Exception e) {
                logger.warn(() -> new ParameterizedMessage("failed to create repository [{}]", (Object)request.name()), (Throwable)e);
                super.onFailure(source, e);
            }

            @Override
            public boolean mustAck(DiscoveryNode discoveryNode) {
                return discoveryNode.isMasterNode() || discoveryNode.isDataNode();
            }
        });
    }

    public void unregisterRepository(final DeleteRepositoryRequest request, ActionListener<ClusterStateUpdateResponse> listener) {
        this.clusterService.submitStateUpdateTask("delete_repository [" + request.name() + "]", new AckedClusterStateUpdateTask<ClusterStateUpdateResponse>((AckedRequest)request, listener){

            @Override
            protected ClusterStateUpdateResponse newResponse(boolean acknowledged) {
                return new ClusterStateUpdateResponse(acknowledged);
            }

            @Override
            public ClusterState execute(ClusterState currentState) {
                Metadata metadata = currentState.metadata();
                Metadata.Builder mdBuilder = Metadata.builder(currentState.metadata());
                RepositoriesMetadata repositories = (RepositoriesMetadata)metadata.custom("repositories");
                if (repositories != null && repositories.repositories().size() > 0) {
                    ArrayList<RepositoryMetadata> repositoriesMetadata = new ArrayList<RepositoryMetadata>(repositories.repositories().size());
                    boolean changed = false;
                    for (RepositoryMetadata repositoryMetadata : repositories.repositories()) {
                        if (Regex.simpleMatch(request.name(), repositoryMetadata.name())) {
                            RepositoriesService.ensureRepositoryNotInUse(currentState, repositoryMetadata.name());
                            logger.info("delete repository [{}]", (Object)repositoryMetadata.name());
                            changed = true;
                            continue;
                        }
                        repositoriesMetadata.add(repositoryMetadata);
                    }
                    if (changed) {
                        repositories = new RepositoriesMetadata(repositoriesMetadata);
                        mdBuilder.putCustom("repositories", repositories);
                        return ClusterState.builder(currentState).metadata(mdBuilder).build();
                    }
                }
                if (Regex.isMatchAllPattern(request.name())) {
                    return currentState;
                }
                throw new RepositoryMissingException(request.name());
            }

            @Override
            public boolean mustAck(DiscoveryNode discoveryNode) {
                return discoveryNode.isMasterNode() || discoveryNode.isDataNode();
            }
        });
    }

    public void verifyRepository(final String repositoryName, ActionListener<List<DiscoveryNode>> listener) {
        final Repository repository = this.repository(repositoryName);
        this.threadPool.executor("snapshot").execute(new ActionRunnable<List<DiscoveryNode>>(listener){

            @Override
            protected void doRun() {
                String verificationToken = repository.startVerification();
                if (verificationToken != null) {
                    try {
                        RepositoriesService.this.verifyAction.verify(repositoryName, verificationToken, ActionListener.delegateFailure(this.listener, (delegatedListener, verifyResponse) -> RepositoriesService.this.threadPool.executor("snapshot").execute(() -> {
                            try {
                                repository.endVerification(verificationToken);
                            }
                            catch (Exception e) {
                                logger.warn(() -> new ParameterizedMessage("[{}] failed to finish repository verification", (Object)repositoryName), (Throwable)e);
                                delegatedListener.onFailure(e);
                                return;
                            }
                            delegatedListener.onResponse(verifyResponse);
                        })));
                    }
                    catch (Exception e) {
                        RepositoriesService.this.threadPool.executor("snapshot").execute(() -> {
                            try {
                                repository.endVerification(verificationToken);
                            }
                            catch (Exception inner) {
                                inner.addSuppressed(e);
                                logger.warn(() -> new ParameterizedMessage("[{}] failed to finish repository verification", (Object)repositoryName), (Throwable)inner);
                            }
                            this.listener.onFailure(e);
                        });
                    }
                } else {
                    this.listener.onResponse(Collections.emptyList());
                }
            }
        });
    }

    static boolean isDedicatedVotingOnlyNode(Set<DiscoveryNodeRole> roles) {
        return roles.contains(DiscoveryNodeRole.MASTER_ROLE) && !roles.contains(DiscoveryNodeRole.DATA_ROLE) && roles.stream().anyMatch(role -> role.roleName().equals("voting_only"));
    }

    @Override
    public void applyClusterState(ClusterChangedEvent event) {
        try {
            ClusterState state = event.state();
            RepositoriesMetadata oldMetadata = (RepositoriesMetadata)event.previousState().getMetadata().custom("repositories");
            RepositoriesMetadata newMetadata = (RepositoriesMetadata)state.getMetadata().custom("repositories");
            if (oldMetadata == null && newMetadata == null || oldMetadata != null && oldMetadata.equalsIgnoreGenerations(newMetadata)) {
                for (Repository repo : this.repositories.values()) {
                    repo.updateState(state);
                }
                return;
            }
            logger.trace("processing new index repositories for state version [{}]", (Object)event.state().version());
            HashMap<String, Repository> survivors = new HashMap<String, Repository>();
            for (Map.Entry<String, Repository> entry : this.repositories.entrySet()) {
                if (newMetadata == null || newMetadata.repository(entry.getKey()) == null) {
                    logger.debug("unregistering repository [{}]", (Object)entry.getKey());
                    Repository repository = entry.getValue();
                    this.closeRepository(repository);
                    this.archiveRepositoryStats(repository, state.version());
                    continue;
                }
                survivors.put(entry.getKey(), entry.getValue());
            }
            HashMap<String, Repository> builder = new HashMap<String, Repository>();
            if (newMetadata != null) {
                for (RepositoryMetadata repositoryMetadata : newMetadata.repositories()) {
                    Repository repository = (Repository)survivors.get(repositoryMetadata.name());
                    if (repository != null) {
                        RepositoryMetadata previousMetadata = repository.getMetadata();
                        if (!previousMetadata.type().equals(repositoryMetadata.type()) || !previousMetadata.settings().equals(repositoryMetadata.settings())) {
                            logger.debug("updating repository [{}]", (Object)repositoryMetadata.name());
                            this.closeRepository(repository);
                            this.archiveRepositoryStats(repository, state.version());
                            repository = null;
                            try {
                                repository = this.createRepository(repositoryMetadata, this.typesRegistry);
                            }
                            catch (RepositoryException ex) {
                                logger.warn(() -> new ParameterizedMessage("failed to change repository [{}]", (Object)repositoryMetadata.name()), (Throwable)ex);
                            }
                        }
                    } else {
                        try {
                            repository = this.createRepository(repositoryMetadata, this.typesRegistry);
                        }
                        catch (RepositoryException ex) {
                            logger.warn(() -> new ParameterizedMessage("failed to create repository [{}]", (Object)repositoryMetadata.name()), (Throwable)ex);
                        }
                    }
                    if (repository == null) continue;
                    logger.debug("registering repository [{}]", (Object)repositoryMetadata.name());
                    builder.put(repositoryMetadata.name(), repository);
                }
            }
            for (Repository repo : builder.values()) {
                repo.updateState(state);
            }
            this.repositories = Collections.unmodifiableMap(builder);
        }
        catch (Exception ex) {
            assert (false) : new AssertionError((Object)ex);
            logger.warn("failure updating cluster state ", (Throwable)ex);
        }
    }

    public void getRepositoryData(String repositoryName, ActionListener<RepositoryData> listener) {
        try {
            Repository repository = this.repository(repositoryName);
            assert (repository != null);
            repository.getRepositoryData(listener);
        }
        catch (Exception e) {
            listener.onFailure(e);
        }
    }

    public Repository repository(String repositoryName) {
        Repository repository = this.repositories.get(repositoryName);
        if (repository != null) {
            return repository;
        }
        repository = this.internalRepositories.get(repositoryName);
        if (repository != null) {
            return repository;
        }
        throw new RepositoryMissingException(repositoryName);
    }

    public List<RepositoryStatsSnapshot> repositoriesStats() {
        List<RepositoryStatsSnapshot> archivedRepoStats = this.repositoriesStatsArchive.getArchivedStats();
        List<RepositoryStatsSnapshot> activeRepoStats = this.getRepositoryStatsForActiveRepositories();
        ArrayList<RepositoryStatsSnapshot> repositoriesStats = new ArrayList<RepositoryStatsSnapshot>(archivedRepoStats);
        repositoriesStats.addAll(activeRepoStats);
        return repositoriesStats;
    }

    private List<RepositoryStatsSnapshot> getRepositoryStatsForActiveRepositories() {
        return Stream.concat(this.repositories.values().stream(), this.internalRepositories.values().stream()).filter(r -> r instanceof MeteredBlobStoreRepository).map(r -> (MeteredBlobStoreRepository)r).map(MeteredBlobStoreRepository::statsSnapshot).collect(Collectors.toList());
    }

    public List<RepositoryStatsSnapshot> clearRepositoriesStatsArchive(long maxVersionToClear) {
        return this.repositoriesStatsArchive.clear(maxVersionToClear);
    }

    public void registerInternalRepository(String name, String type) {
        RepositoryMetadata metadata = new RepositoryMetadata(name, type, Settings.EMPTY);
        Repository repository = this.internalRepositories.computeIfAbsent(name, n -> {
            logger.debug("put internal repository [{}][{}]", (Object)name, (Object)type);
            return this.createRepository(metadata, this.internalTypesRegistry);
        });
        if (!type.equals(repository.getMetadata().type())) {
            logger.warn((Message)new ParameterizedMessage("internal repository [{}][{}] already registered. this prevented the registration of internal repository [{}][{}].", new Object[]{name, repository.getMetadata().type(), name, type}));
        } else if (this.repositories.containsKey(name)) {
            logger.warn((Message)new ParameterizedMessage("non-internal repository [{}] already registered. this repository will block the usage of internal repository [{}][{}].", new Object[]{name, metadata.type(), name}));
        }
    }

    public void unregisterInternalRepository(String name) {
        Repository repository = this.internalRepositories.remove(name);
        if (repository != null) {
            RepositoryMetadata metadata = repository.getMetadata();
            logger.debug(() -> new ParameterizedMessage("delete internal repository [{}][{}].", (Object)metadata.type(), (Object)name));
            this.closeRepository(repository);
        }
    }

    private void closeRepository(Repository repository) {
        logger.debug("closing repository [{}][{}]", (Object)repository.getMetadata().type(), (Object)repository.getMetadata().name());
        repository.close();
    }

    private void archiveRepositoryStats(Repository repository, long clusterStateVersion) {
        RepositoryStatsSnapshot stats;
        if (repository instanceof MeteredBlobStoreRepository && !this.repositoriesStatsArchive.archive(stats = ((MeteredBlobStoreRepository)repository).statsSnapshotForArchival(clusterStateVersion))) {
            logger.warn("Unable to archive the repository stats [{}] as the archive is full.", (Object)stats);
        }
    }

    private Repository createRepository(RepositoryMetadata repositoryMetadata, Map<String, Repository.Factory> factories) {
        logger.debug("creating repository [{}][{}]", (Object)repositoryMetadata.type(), (Object)repositoryMetadata.name());
        Repository.Factory factory = factories.get(repositoryMetadata.type());
        if (factory == null) {
            throw new RepositoryException(repositoryMetadata.name(), "repository type [" + repositoryMetadata.type() + "] does not exist");
        }
        Repository repository = null;
        try {
            repository = factory.create(repositoryMetadata, factories::get);
            repository.start();
            return repository;
        }
        catch (Exception e) {
            IOUtils.closeWhileHandlingException(repository);
            logger.warn((Message)new ParameterizedMessage("failed to create repository [{}][{}]", (Object)repositoryMetadata.type(), (Object)repositoryMetadata.name()), (Throwable)e);
            throw new RepositoryException(repositoryMetadata.name(), "failed to create repository", e);
        }
    }

    private static void validate(String repositoryName) {
        if (!Strings.hasLength(repositoryName)) {
            throw new RepositoryException(repositoryName, "cannot be empty");
        }
        if (repositoryName.contains("#")) {
            throw new RepositoryException(repositoryName, "must not contain '#'");
        }
        if (!Strings.validFileName(repositoryName)) {
            throw new RepositoryException(repositoryName, "must not contain the following characters " + Strings.INVALID_FILENAME_CHARS);
        }
    }

    private static void ensureRepositoryNotInUse(ClusterState clusterState, String repository) {
        if (RepositoriesService.isRepositoryInUse(clusterState, repository)) {
            throw new IllegalStateException("trying to modify or unregister repository that is currently used");
        }
    }

    private static boolean isRepositoryInUse(ClusterState clusterState, String repository) {
        SnapshotsInProgress snapshots = clusterState.custom("snapshots", SnapshotsInProgress.EMPTY);
        for (SnapshotsInProgress.Entry entry : snapshots.entries()) {
            if (!repository.equals(entry.snapshot().getRepository())) continue;
            return true;
        }
        for (SnapshotDeletionsInProgress.Entry entry : clusterState.custom("snapshot_deletions", SnapshotDeletionsInProgress.EMPTY).getEntries()) {
            if (!entry.repository().equals(repository)) continue;
            return true;
        }
        for (RepositoryCleanupInProgress.Entry entry : clusterState.custom("repository_cleanup", RepositoryCleanupInProgress.EMPTY).entries()) {
            if (!entry.repository().equals(repository)) continue;
            return true;
        }
        for (RestoreInProgress.Entry entry : clusterState.custom("restore", RestoreInProgress.EMPTY)) {
            if (!repository.equals(entry.snapshot().getRepository())) continue;
            return true;
        }
        return false;
    }

    @Override
    protected void doStart() {
    }

    @Override
    protected void doStop() {
    }

    @Override
    protected void doClose() throws IOException {
        this.clusterService.removeApplier(this);
        ArrayList<Repository> repos = new ArrayList<Repository>();
        repos.addAll(this.internalRepositories.values());
        repos.addAll(this.repositories.values());
        IOUtils.close(repos);
    }
}

