/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.unit;

import java.io.IOException;
import java.util.Locale;
import java.util.Objects;
import org.elasticsearch.ElasticsearchParseException;
import org.elasticsearch.Version;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.io.stream.Writeable;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.xcontent.ToXContent;
import org.elasticsearch.common.xcontent.ToXContentFragment;
import org.elasticsearch.common.xcontent.XContentBuilder;

public class ByteSizeValue
implements Writeable,
Comparable<ByteSizeValue>,
ToXContentFragment {
    public static final ByteSizeValue ZERO = new ByteSizeValue(0L, ByteSizeUnit.BYTES);
    private final long size;
    private final ByteSizeUnit unit;

    public ByteSizeValue(StreamInput in) throws IOException {
        if (in.getVersion().before(Version.V_6_2_0)) {
            this.size = in.readVLong();
            this.unit = ByteSizeUnit.BYTES;
        } else {
            this.size = in.readZLong();
            this.unit = ByteSizeUnit.readFrom(in);
        }
    }

    @Override
    public void writeTo(StreamOutput out) throws IOException {
        if (out.getVersion().before(Version.V_6_2_0)) {
            out.writeVLong(this.getBytes());
        } else {
            out.writeZLong(this.size);
            this.unit.writeTo(out);
        }
    }

    public ByteSizeValue(long bytes) {
        this(bytes, ByteSizeUnit.BYTES);
    }

    public ByteSizeValue(long size, ByteSizeUnit unit) {
        if (size < -1L || size == -1L && unit != ByteSizeUnit.BYTES) {
            throw new IllegalArgumentException("Values less than -1 bytes are not supported: " + size + unit.getSuffix());
        }
        if (size > Long.MAX_VALUE / unit.toBytes(1L)) {
            throw new IllegalArgumentException("Values greater than 9223372036854775807 bytes are not supported: " + size + unit.getSuffix());
        }
        this.size = size;
        this.unit = unit;
    }

    long getSize() {
        return this.size;
    }

    ByteSizeUnit getUnit() {
        return this.unit;
    }

    @Deprecated
    public int bytesAsInt() {
        long bytes = this.getBytes();
        if (bytes > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("size [" + this.toString() + "] is bigger than max int");
        }
        return (int)bytes;
    }

    public long getBytes() {
        return this.unit.toBytes(this.size);
    }

    public long getKb() {
        return this.unit.toKB(this.size);
    }

    public long getMb() {
        return this.unit.toMB(this.size);
    }

    public long getGb() {
        return this.unit.toGB(this.size);
    }

    public long getTb() {
        return this.unit.toTB(this.size);
    }

    public long getPb() {
        return this.unit.toPB(this.size);
    }

    public double getKbFrac() {
        return (double)this.getBytes() / 1024.0;
    }

    public double getMbFrac() {
        return (double)this.getBytes() / 1048576.0;
    }

    public double getGbFrac() {
        return (double)this.getBytes() / 1.073741824E9;
    }

    public double getTbFrac() {
        return (double)this.getBytes() / 1.099511627776E12;
    }

    public double getPbFrac() {
        return (double)this.getBytes() / 1.125899906842624E15;
    }

    public String getStringRep() {
        if (this.size <= 0L) {
            return String.valueOf(this.size);
        }
        return this.size + this.unit.getSuffix();
    }

    public String toString() {
        long bytes = this.getBytes();
        double value = bytes;
        String suffix = ByteSizeUnit.BYTES.getSuffix();
        if (bytes >= 0x4000000000000L) {
            value = this.getPbFrac();
            suffix = ByteSizeUnit.PB.getSuffix();
        } else if (bytes >= 0x10000000000L) {
            value = this.getTbFrac();
            suffix = ByteSizeUnit.TB.getSuffix();
        } else if (bytes >= 0x40000000L) {
            value = this.getGbFrac();
            suffix = ByteSizeUnit.GB.getSuffix();
        } else if (bytes >= 0x100000L) {
            value = this.getMbFrac();
            suffix = ByteSizeUnit.MB.getSuffix();
        } else if (bytes >= 1024L) {
            value = this.getKbFrac();
            suffix = ByteSizeUnit.KB.getSuffix();
        }
        return Strings.format1Decimals(value, suffix);
    }

    public static ByteSizeValue parseBytesSizeValue(String sValue, String settingName) throws ElasticsearchParseException {
        return ByteSizeValue.parseBytesSizeValue(sValue, null, settingName);
    }

    public static ByteSizeValue parseBytesSizeValue(String sValue, ByteSizeValue defaultValue, String settingName) throws ElasticsearchParseException {
        settingName = Objects.requireNonNull(settingName);
        if (sValue == null) {
            return defaultValue;
        }
        String lowerSValue = sValue.toLowerCase(Locale.ROOT).trim();
        if (lowerSValue.endsWith("k")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "k", ByteSizeUnit.KB, settingName);
        }
        if (lowerSValue.endsWith("kb")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "kb", ByteSizeUnit.KB, settingName);
        }
        if (lowerSValue.endsWith("m")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "m", ByteSizeUnit.MB, settingName);
        }
        if (lowerSValue.endsWith("mb")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "mb", ByteSizeUnit.MB, settingName);
        }
        if (lowerSValue.endsWith("g")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "g", ByteSizeUnit.GB, settingName);
        }
        if (lowerSValue.endsWith("gb")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "gb", ByteSizeUnit.GB, settingName);
        }
        if (lowerSValue.endsWith("t")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "t", ByteSizeUnit.TB, settingName);
        }
        if (lowerSValue.endsWith("tb")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "tb", ByteSizeUnit.TB, settingName);
        }
        if (lowerSValue.endsWith("p")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "p", ByteSizeUnit.PB, settingName);
        }
        if (lowerSValue.endsWith("pb")) {
            return ByteSizeValue.parse(sValue, lowerSValue, "pb", ByteSizeUnit.PB, settingName);
        }
        if (lowerSValue.endsWith("b")) {
            return new ByteSizeValue(Long.parseLong(lowerSValue.substring(0, lowerSValue.length() - 1).trim()), ByteSizeUnit.BYTES);
        }
        if (lowerSValue.equals("-1")) {
            return new ByteSizeValue(-1L, ByteSizeUnit.BYTES);
        }
        if (lowerSValue.equals("0")) {
            return new ByteSizeValue(0L, ByteSizeUnit.BYTES);
        }
        throw new ElasticsearchParseException("failed to parse setting [{}] with value [{}] as a size in bytes: unit is missing or unrecognized", settingName, sValue);
    }

    private static ByteSizeValue parse(String initialInput, String normalized, String suffix, ByteSizeUnit unit, String settingName) {
        String s = normalized.substring(0, normalized.length() - suffix.length()).trim();
        try {
            try {
                return new ByteSizeValue(Long.parseLong(s), unit);
            }
            catch (NumberFormatException e) {
                try {
                    double doubleValue = Double.parseDouble(s);
                    DeprecationLoggerHolder.deprecationLogger.deprecate("fractional_byte_values", "Fractional bytes values are deprecated. Use non-fractional bytes values instead: [{}] found for setting [{}]", initialInput, settingName);
                    return new ByteSizeValue((long)(doubleValue * (double)unit.toBytes(1L)));
                }
                catch (NumberFormatException ignored) {
                    throw new ElasticsearchParseException("failed to parse [{}]", (Throwable)e, initialInput);
                }
            }
        }
        catch (IllegalArgumentException e) {
            throw new ElasticsearchParseException("failed to parse setting [{}] with value [{}] as a size in bytes", (Throwable)e, settingName, initialInput);
        }
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        return this.compareTo((ByteSizeValue)o) == 0;
    }

    public int hashCode() {
        return Long.hashCode(this.size * this.unit.toBytes(1L));
    }

    @Override
    public int compareTo(ByteSizeValue other) {
        long thisValue = this.size * this.unit.toBytes(1L);
        long otherValue = other.size * other.unit.toBytes(1L);
        return Long.compare(thisValue, otherValue);
    }

    public XContentBuilder toXContent(XContentBuilder builder, ToXContent.Params params) throws IOException {
        return builder.value(this.toString());
    }

    static class DeprecationLoggerHolder {
        static DeprecationLogger deprecationLogger = DeprecationLogger.getLogger(ByteSizeValue.class);

        DeprecationLoggerHolder() {
        }
    }
}

