/*
 * Decompiled with CFR 0.152.
 */
package io.trino.operator;

import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.LinkedListMultimap;
import com.google.common.collect.ListMultimap;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.SettableFuture;
import io.airlift.log.Logger;
import io.airlift.slice.Slice;
import io.airlift.units.DataSize;
import io.trino.execution.TaskId;
import io.trino.operator.ExchangeClientBuffer;
import io.trino.operator.RetryPolicy;
import io.trino.spi.ErrorCodeSupplier;
import io.trino.spi.StandardErrorCode;
import io.trino.spi.TrinoException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.Executor;
import javax.annotation.concurrent.GuardedBy;

public class DeduplicationExchangeClientBuffer
implements ExchangeClientBuffer {
    private static final Logger log = Logger.get(DeduplicationExchangeClientBuffer.class);
    private final Executor executor;
    private final long bufferCapacityInBytes;
    private final RetryPolicy retryPolicy;
    private final SettableFuture<Void> blocked = SettableFuture.create();
    @GuardedBy(value="this")
    private final Set<TaskId> allTasks = new HashSet<TaskId>();
    @GuardedBy(value="this")
    private boolean noMoreTasks;
    @GuardedBy(value="this")
    private final Set<TaskId> successfulTasks = new HashSet<TaskId>();
    @GuardedBy(value="this")
    private final Map<TaskId, Throwable> failedTasks = new HashMap<TaskId, Throwable>();
    @GuardedBy(value="this")
    private boolean inputFinished;
    @GuardedBy(value="this")
    private Throwable failure;
    @GuardedBy(value="this")
    private final ListMultimap<TaskId, Slice> pageBuffer = LinkedListMultimap.create();
    @GuardedBy(value="this")
    private Iterator<Slice> pagesIterator;
    @GuardedBy(value="this")
    private volatile long bufferRetainedSizeInBytes;
    @GuardedBy(value="this")
    private volatile long maxBufferRetainedSizeInBytes;
    @GuardedBy(value="this")
    private int maxAttemptId;
    @GuardedBy(value="this")
    private boolean closed;

    public DeduplicationExchangeClientBuffer(Executor executor, DataSize bufferCapacity, RetryPolicy retryPolicy) {
        this.executor = Objects.requireNonNull(executor, "executor is null");
        this.bufferCapacityInBytes = Objects.requireNonNull(bufferCapacity, "bufferCapacity is null").toBytes();
        Objects.requireNonNull(retryPolicy, "retryPolicy is null");
        Preconditions.checkArgument((retryPolicy == RetryPolicy.QUERY ? 1 : 0) != 0, (String)"retryPolicy is expected to be QUERY: %s", (Object)((Object)retryPolicy));
        this.retryPolicy = retryPolicy;
    }

    @Override
    public ListenableFuture<Void> isBlocked() {
        return Futures.nonCancellationPropagating(this.blocked);
    }

    @Override
    public synchronized Slice pollPage() {
        this.throwIfFailed();
        if (this.closed) {
            return null;
        }
        if (!this.inputFinished) {
            return null;
        }
        if (this.pagesIterator == null) {
            this.pagesIterator = this.pageBuffer.values().iterator();
        }
        if (!this.pagesIterator.hasNext()) {
            return null;
        }
        Slice page = this.pagesIterator.next();
        this.pagesIterator.remove();
        this.bufferRetainedSizeInBytes -= page.getRetainedSize();
        return page;
    }

    @Override
    public synchronized void addTask(TaskId taskId) {
        if (this.closed) {
            return;
        }
        Preconditions.checkState((!this.noMoreTasks ? 1 : 0) != 0, (Object)"no more tasks expected");
        Preconditions.checkState((boolean)this.allTasks.add(taskId), (String)"task already registered: %s", (Object)taskId);
        if (taskId.getAttemptId() > this.maxAttemptId) {
            this.maxAttemptId = taskId.getAttemptId();
            if (this.retryPolicy == RetryPolicy.QUERY) {
                this.removePagesForPreviousAttempts(taskId.getAttemptId());
            }
        }
    }

    @Override
    public synchronized void addPages(TaskId taskId, List<Slice> pages) {
        if (this.closed) {
            return;
        }
        Preconditions.checkState((boolean)this.allTasks.contains(taskId), (String)"task is not registered: %s", (Object)taskId);
        Preconditions.checkState((!this.successfulTasks.contains(taskId) ? 1 : 0) != 0, (String)"task is finished: %s", (Object)taskId);
        Preconditions.checkState((!this.failedTasks.containsKey(taskId) ? 1 : 0) != 0, (String)"task is failed: %s", (Object)taskId);
        if (this.failure != null) {
            return;
        }
        if (this.retryPolicy == RetryPolicy.QUERY && taskId.getAttemptId() < this.maxAttemptId) {
            return;
        }
        long pagesRetainedSizeInBytes = 0L;
        for (Slice page : pages) {
            pagesRetainedSizeInBytes += page.getRetainedSize();
        }
        this.bufferRetainedSizeInBytes += pagesRetainedSizeInBytes;
        if (this.bufferRetainedSizeInBytes > this.bufferCapacityInBytes) {
            this.failure = new TrinoException((ErrorCodeSupplier)StandardErrorCode.NOT_SUPPORTED, "Retries for queries with large result set currently unsupported");
            this.pageBuffer.clear();
            this.bufferRetainedSizeInBytes = 0L;
            this.unblock(this.blocked);
            return;
        }
        this.maxBufferRetainedSizeInBytes = Math.max(this.maxBufferRetainedSizeInBytes, this.bufferRetainedSizeInBytes);
        this.pageBuffer.putAll((Object)taskId, pages);
    }

    @Override
    public synchronized void taskFinished(TaskId taskId) {
        if (this.closed) {
            return;
        }
        Preconditions.checkState((boolean)this.allTasks.contains(taskId), (String)"task is not registered: %s", (Object)taskId);
        Preconditions.checkState((!this.failedTasks.containsKey(taskId) ? 1 : 0) != 0, (String)"task is failed: %s", (Object)taskId);
        Preconditions.checkState((boolean)this.successfulTasks.add(taskId), (String)"task is finished: %s", (Object)taskId);
        if (this.retryPolicy == RetryPolicy.TASK) {
            throw new UnsupportedOperationException("task level retry policy is unsupported");
        }
        this.checkInputFinished();
    }

    @Override
    public synchronized void taskFailed(TaskId taskId, Throwable t) {
        if (this.closed) {
            return;
        }
        Preconditions.checkState((boolean)this.allTasks.contains(taskId), (String)"task is not registered: %s", (Object)taskId);
        Preconditions.checkState((!this.successfulTasks.contains(taskId) ? 1 : 0) != 0, (String)"task is finished: %s", (Object)taskId);
        Preconditions.checkState((this.failedTasks.put(taskId, t) == null ? 1 : 0) != 0, (String)"task is already failed: %s", (Object)taskId);
        this.checkInputFinished();
    }

    @Override
    public synchronized void noMoreTasks() {
        if (this.closed) {
            return;
        }
        this.noMoreTasks = true;
        this.checkInputFinished();
    }

    private synchronized void checkInputFinished() {
        if (this.failure != null) {
            return;
        }
        if (this.inputFinished) {
            return;
        }
        if (!this.noMoreTasks) {
            return;
        }
        if (this.allTasks.isEmpty()) {
            this.inputFinished = true;
            this.unblock(this.blocked);
            return;
        }
        switch (this.retryPolicy) {
            case TASK: {
                throw new UnsupportedOperationException("task level retry policy is unsupported");
            }
            case QUERY: {
                Set latestAttemptTasks = (Set)this.allTasks.stream().filter(taskId -> taskId.getAttemptId() == this.maxAttemptId).collect(ImmutableSet.toImmutableSet());
                if (this.successfulTasks.containsAll(latestAttemptTasks)) {
                    this.removePagesForPreviousAttempts(this.maxAttemptId);
                    this.inputFinished = true;
                    this.unblock(this.blocked);
                    return;
                }
                Throwable failure = null;
                for (Map.Entry<TaskId, Throwable> entry : this.failedTasks.entrySet()) {
                    TaskId taskId2 = entry.getKey();
                    Throwable taskFailure = entry.getValue();
                    if (taskId2.getAttemptId() != this.maxAttemptId) continue;
                    if (taskFailure instanceof TrinoException && StandardErrorCode.REMOTE_TASK_FAILED.toErrorCode().equals((Object)((TrinoException)taskFailure).getErrorCode())) {
                        log.debug("Task failure discovered while fetching task results: %s", new Object[]{taskId2});
                        continue;
                    }
                    if (failure == null) {
                        failure = taskFailure;
                        continue;
                    }
                    if (failure == taskFailure) continue;
                    failure.addSuppressed(taskFailure);
                }
                if (failure == null) break;
                this.pageBuffer.clear();
                this.bufferRetainedSizeInBytes = 0L;
                this.failure = failure;
                this.unblock(this.blocked);
                break;
            }
            default: {
                throw new UnsupportedOperationException("unexpected retry policy: " + this.retryPolicy);
            }
        }
    }

    private synchronized void removePagesForPreviousAttempts(int currentAttemptId) {
        long pagesRetainedSizeInBytes = 0L;
        Iterator iterator = this.pageBuffer.entries().iterator();
        while (iterator.hasNext()) {
            Map.Entry entry = (Map.Entry)iterator.next();
            if (((TaskId)entry.getKey()).getAttemptId() >= currentAttemptId) continue;
            pagesRetainedSizeInBytes += ((Slice)entry.getValue()).getRetainedSize();
            iterator.remove();
        }
        this.bufferRetainedSizeInBytes -= pagesRetainedSizeInBytes;
    }

    @Override
    public synchronized boolean isFinished() {
        return this.failure == null && (this.closed || this.inputFinished && this.pageBuffer.isEmpty());
    }

    @Override
    public synchronized boolean isFailed() {
        return this.failure != null;
    }

    @Override
    public long getRemainingCapacityInBytes() {
        return Math.max(this.bufferCapacityInBytes - this.bufferRetainedSizeInBytes, 0L);
    }

    @Override
    public long getRetainedSizeInBytes() {
        return this.bufferRetainedSizeInBytes;
    }

    @Override
    public long getMaxRetainedSizeInBytes() {
        return this.maxBufferRetainedSizeInBytes;
    }

    @Override
    public synchronized int getBufferedPageCount() {
        return this.pageBuffer.size();
    }

    @Override
    public synchronized void close() {
        if (this.closed) {
            return;
        }
        this.closed = true;
        this.pageBuffer.clear();
        this.bufferRetainedSizeInBytes = 0L;
        this.unblock(this.blocked);
    }

    private synchronized void throwIfFailed() {
        if (this.failure != null) {
            Throwables.throwIfUnchecked((Throwable)this.failure);
            throw new RuntimeException(this.failure);
        }
    }

    private void unblock(SettableFuture<Void> blocked) {
        this.executor.execute(() -> blocked.set(null));
    }
}

