/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hugegraph.backend.store.hbase;

import java.io.IOException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import org.apache.hadoop.hbase.NamespaceExistException;
import org.apache.hadoop.hbase.TableExistsException;
import org.apache.hadoop.hbase.TableNotFoundException;
import org.apache.hugegraph.backend.BackendException;
import org.apache.hugegraph.backend.id.Id;
import org.apache.hugegraph.backend.query.Query;
import org.apache.hugegraph.backend.store.AbstractBackendStore;
import org.apache.hugegraph.backend.store.BackendAction;
import org.apache.hugegraph.backend.store.BackendEntry;
import org.apache.hugegraph.backend.store.BackendFeatures;
import org.apache.hugegraph.backend.store.BackendMutation;
import org.apache.hugegraph.backend.store.BackendSession;
import org.apache.hugegraph.backend.store.BackendStoreProvider;
import org.apache.hugegraph.backend.store.BackendTable;
import org.apache.hugegraph.backend.store.hbase.HbaseFeatures;
import org.apache.hugegraph.backend.store.hbase.HbaseMetrics;
import org.apache.hugegraph.backend.store.hbase.HbaseOptions;
import org.apache.hugegraph.backend.store.hbase.HbaseSessions;
import org.apache.hugegraph.backend.store.hbase.HbaseTable;
import org.apache.hugegraph.backend.store.hbase.HbaseTables;
import org.apache.hugegraph.config.HugeConfig;
import org.apache.hugegraph.exception.ConnectionException;
import org.apache.hugegraph.type.HugeType;
import org.apache.hugegraph.util.E;
import org.apache.hugegraph.util.Log;
import org.slf4j.Logger;

public abstract class HbaseStore
extends AbstractBackendStore<HbaseSessions.Session> {
    private static final Logger LOG = Log.logger(HbaseStore.class);
    private final BackendFeatures features;
    private final String store;
    private final String namespace;
    private final BackendStoreProvider provider;
    private final Map<HugeType, HbaseTable> tables = new HashMap<HugeType, HbaseTable>();
    private short vertexLogicPartitions;
    private short edgeLogicPartitions;
    private HbaseSessions sessions;

    public HbaseStore(BackendStoreProvider provider, String namespace, String store, boolean enablePartition) {
        this.provider = provider;
        this.namespace = namespace;
        this.store = store;
        this.sessions = null;
        this.features = new HbaseFeatures(enablePartition);
        this.registerMetaHandlers();
        LOG.debug("Store loaded: {}", (Object)store);
    }

    private void registerMetaHandlers() {
        this.registerMetaHandler("metrics", (session, meta, args) -> {
            HbaseMetrics metrics = new HbaseMetrics(this.sessions);
            return metrics.metrics();
        });
        this.registerMetaHandler("compact", (session, meta, args) -> {
            HbaseMetrics metrics = new HbaseMetrics(this.sessions);
            return metrics.compact(this.tableNames());
        });
    }

    protected void registerTableManager(HugeType type, HbaseTable table) {
        this.tables.put(type, table);
    }

    protected final HbaseTable table(HugeType type) {
        assert (type != null);
        HbaseTable table = this.tables.get(HbaseStore.convertTaskOrServerToVertex((HugeType)type));
        if (table == null) {
            throw new BackendException("Unsupported table type: %s", new Object[]{type});
        }
        return table;
    }

    protected HbaseSessions.Session session(HugeType type) {
        this.checkOpened();
        return this.sessions.session();
    }

    protected List<String> tableNames() {
        return this.tables.values().stream().map(BackendTable::table).collect(Collectors.toList());
    }

    public String namespace() {
        return this.namespace;
    }

    public String store() {
        return this.store;
    }

    public String database() {
        return this.namespace;
    }

    public BackendStoreProvider provider() {
        return this.provider;
    }

    public BackendFeatures features() {
        return this.features;
    }

    public synchronized void open(HugeConfig config) {
        E.checkNotNull((Object)config, (String)"config");
        this.vertexLogicPartitions = ((Integer)config.get(HbaseOptions.HBASE_VERTEX_PARTITION)).shortValue();
        this.edgeLogicPartitions = ((Integer)config.get(HbaseOptions.HBASE_EDGE_PARTITION)).shortValue();
        if (this.sessions == null) {
            this.sessions = new HbaseSessions(config, this.namespace, this.store);
        }
        assert (this.sessions != null);
        if (!this.sessions.closed()) {
            LOG.debug("Store {} has been opened before", (Object)this.store);
            this.sessions.useSession();
            return;
        }
        try {
            this.sessions.open();
        }
        catch (Throwable e) {
            LOG.error("Failed to open HBase '{}'", (Object)this.store, (Object)e);
            throw new ConnectionException("Failed to connect to HBase", e);
        }
        this.sessions.session().open();
        LOG.debug("Store opened: {}", (Object)this.store);
    }

    public void close() {
        this.checkOpened();
        this.sessions.close();
        LOG.debug("Store closed: {}", (Object)this.store);
    }

    public boolean opened() {
        this.checkConnectionOpened();
        return this.sessions.session().opened();
    }

    public void mutate(BackendMutation mutation) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Store {} mutation: {}", (Object)this.store, (Object)mutation);
        }
        this.checkOpened();
        HbaseSessions.Session session = this.sessions.session();
        Iterator it = mutation.mutation();
        while (it.hasNext()) {
            this.mutate(session, (BackendAction)it.next());
        }
    }

    private void mutate(HbaseSessions.Session session, BackendAction item) {
        BackendEntry entry = item.entry();
        HbaseTable table = this.table(entry.type());
        switch (item.action()) {
            case INSERT: {
                table.insert(session, entry);
                break;
            }
            case DELETE: {
                table.delete(session, entry);
                break;
            }
            case APPEND: {
                table.append(session, entry);
                break;
            }
            case ELIMINATE: {
                table.eliminate(session, entry);
                break;
            }
            case UPDATE_IF_PRESENT: {
                table.updateIfPresent((BackendSession)session, entry);
                break;
            }
            case UPDATE_IF_ABSENT: {
                table.updateIfAbsent((BackendSession)session, entry);
                break;
            }
            default: {
                throw new AssertionError((Object)String.format("Unsupported mutate action: %s", item.action()));
            }
        }
    }

    public Iterator<BackendEntry> query(Query query) {
        this.checkOpened();
        HbaseSessions.Session session = this.sessions.session();
        HbaseTable table = this.table(HbaseTable.tableType((Query)query));
        return table.query(session, query);
    }

    public Number queryNumber(Query query) {
        this.checkOpened();
        HbaseSessions.Session session = this.sessions.session();
        HbaseTable table = this.table(HbaseTable.tableType((Query)query));
        return table.queryNumber(session, query);
    }

    public void init() {
        this.checkConnectionOpened();
        try {
            this.sessions.createNamespace();
        }
        catch (NamespaceExistException namespaceExistException) {
        }
        catch (IOException e) {
            throw new BackendException("Failed to create namespace '%s' for '%s' store", (Throwable)e, new Object[]{this.namespace, this.store});
        }
        for (String table : this.tableNames()) {
            try {
                if (table.equals("g_oe") || table.equals("g_ie")) {
                    this.sessions.createPreSplitTable(table, HbaseTable.cfs(), this.edgeLogicPartitions);
                    continue;
                }
                if (table.equals("g_v")) {
                    this.sessions.createPreSplitTable(table, HbaseTable.cfs(), this.vertexLogicPartitions);
                    continue;
                }
                this.sessions.createTable(table, HbaseTable.cfs());
            }
            catch (TableExistsException ignored) {
            }
            catch (IOException e) {
                throw new BackendException("Failed to create table '%s' for '%s' store", (Throwable)e, new Object[]{table, this.store});
            }
        }
        LOG.debug("Store initialized: {}", (Object)this.store);
    }

    public void clear(boolean clearSpace) {
        this.checkConnectionOpened();
        try {
            if (!this.sessions.existsNamespace()) {
                return;
            }
        }
        catch (IOException e) {
            throw new BackendException("Exception when checking for the existence of '%s'", (Throwable)e, new Object[]{this.namespace});
        }
        if (!clearSpace) {
            for (String table : this.tableNames()) {
                try {
                    this.sessions.dropTable(table);
                }
                catch (TableNotFoundException e) {
                    LOG.warn("The table '{}' of '{}' store does not exist when trying to drop", (Object)table, (Object)this.store);
                }
                catch (IOException e) {
                    throw new BackendException("Failed to drop table '%s' of '%s' store", (Throwable)e, new Object[]{table, this.store});
                }
            }
        } else {
            try {
                this.sessions.dropNamespace();
            }
            catch (IOException e) {
                String notEmpty = "Only empty namespaces can be removed";
                if (e.getCause().getMessage().contains(notEmpty)) {
                    LOG.debug("Can't drop namespace '{}': {}", (Object)this.namespace, (Object)e);
                }
                throw new BackendException("Failed to drop namespace '%s' of '%s' store", (Throwable)e, new Object[]{this.namespace, this.store});
            }
        }
        LOG.debug("Store cleared: {}", (Object)this.store);
    }

    public boolean initialized() {
        this.checkConnectionOpened();
        try {
            if (!this.sessions.existsNamespace()) {
                return false;
            }
            for (String table : this.tableNames()) {
                if (this.sessions.existsTable(table)) continue;
                return false;
            }
        }
        catch (IOException e) {
            throw new BackendException("Failed to obtain table info", (Throwable)e);
        }
        return true;
    }

    public void truncate() {
        this.checkOpened();
        long timeout = (Long)this.sessions.config().get(HbaseOptions.TRUNCATE_TIMEOUT);
        long start = System.currentTimeMillis();
        BiFunction<String, Future, Void> wait = (table, future) -> {
            long elapsed = System.currentTimeMillis() - start;
            long remainingTime = timeout - elapsed / 1000L;
            try {
                return (Void)future.get(remainingTime, TimeUnit.SECONDS);
            }
            catch (Exception e) {
                throw new BackendException("Error when truncating table '%s' of '%s' store: %s", new Object[]{table, this.store, e.toString()});
            }
        };
        List<String> tables = this.tableNames();
        HashMap<String, Future<Void>> futures = new HashMap<String, Future<Void>>(tables.size());
        try {
            for (String string : tables) {
                futures.put(string, this.sessions.disableTableAsync(string));
            }
            for (Map.Entry entry : futures.entrySet()) {
                wait.apply((String)entry.getKey(), (Future)entry.getValue());
            }
        }
        catch (Exception e) {
            this.enableTables();
            throw new BackendException("Failed to disable table for '%s' store", (Throwable)e, new Object[]{this.store});
        }
        try {
            for (String string : tables) {
                futures.put(string, this.sessions.truncateTableAsync(string));
            }
            for (Map.Entry entry : futures.entrySet()) {
                wait.apply((String)entry.getKey(), (Future)entry.getValue());
            }
        }
        catch (Exception e) {
            this.enableTables();
            throw new BackendException("Failed to truncate table for '%s' store", (Throwable)e, new Object[]{this.store});
        }
        LOG.debug("Store truncated: {}", (Object)this.store);
    }

    private void enableTables() {
        for (String table : this.tableNames()) {
            try {
                this.sessions.enableTable(table);
            }
            catch (Exception e) {
                LOG.warn("Failed to enable table '{}' of '{}' store", new Object[]{table, this.store, e});
            }
        }
    }

    public void beginTx() {
    }

    public void commitTx() {
        this.checkOpened();
        HbaseSessions.Session session = this.sessions.session();
        session.commit();
    }

    public void rollbackTx() {
        this.checkOpened();
        HbaseSessions.Session session = this.sessions.session();
        session.rollback();
    }

    private void checkConnectionOpened() {
        E.checkState((this.sessions != null && this.sessions.opened() ? 1 : 0) != 0, (String)"HBase store has not been initialized", (Object[])new Object[0]);
    }

    public static class HbaseSystemStore
    extends HbaseGraphStore {
        private final HbaseTables.Meta meta = new HbaseTables.Meta();

        public HbaseSystemStore(HugeConfig config, BackendStoreProvider provider, String namespace, String store) {
            super(config, provider, namespace, store);
        }

        @Override
        protected List<String> tableNames() {
            List<String> tableNames = super.tableNames();
            tableNames.add(this.meta.table());
            return tableNames;
        }

        @Override
        public void init() {
            super.init();
            HbaseSessions.Session session = super.session(null);
            String driverVersion = this.provider().driverVersion();
            this.meta.writeVersion(session, driverVersion);
            LOG.info("Write down the backend version: {}", (Object)driverVersion);
        }

        public String storedVersion() {
            HbaseSessions.Session session = super.session(null);
            return this.meta.readVersion(session);
        }
    }

    public static class HbaseGraphStore
    extends HbaseStore {
        private boolean enablePartition;

        public HbaseGraphStore(HugeConfig config, BackendStoreProvider provider, String namespace, String store) {
            super(provider, namespace, store, (Boolean)config.get(HbaseOptions.HBASE_ENABLE_PARTITION));
            this.enablePartition = (Boolean)config.get(HbaseOptions.HBASE_ENABLE_PARTITION);
            this.registerTableManager(HugeType.VERTEX, new HbaseTables.Vertex(store, this.enablePartition));
            this.registerTableManager(HugeType.EDGE_OUT, HbaseTables.Edge.out(store, this.enablePartition));
            this.registerTableManager(HugeType.EDGE_IN, HbaseTables.Edge.in(store, this.enablePartition));
            this.registerTableManager(HugeType.SECONDARY_INDEX, new HbaseTables.SecondaryIndex(store));
            this.registerTableManager(HugeType.VERTEX_LABEL_INDEX, new HbaseTables.VertexLabelIndex(store));
            this.registerTableManager(HugeType.EDGE_LABEL_INDEX, new HbaseTables.EdgeLabelIndex(store));
            this.registerTableManager(HugeType.RANGE_INT_INDEX, HbaseTables.RangeIndex.rangeInt(store));
            this.registerTableManager(HugeType.RANGE_FLOAT_INDEX, HbaseTables.RangeIndex.rangeFloat(store));
            this.registerTableManager(HugeType.RANGE_LONG_INDEX, HbaseTables.RangeIndex.rangeLong(store));
            this.registerTableManager(HugeType.RANGE_DOUBLE_INDEX, HbaseTables.RangeIndex.rangeDouble(store));
            this.registerTableManager(HugeType.SEARCH_INDEX, new HbaseTables.SearchIndex(store));
            this.registerTableManager(HugeType.SHARD_INDEX, new HbaseTables.ShardIndex(store));
            this.registerTableManager(HugeType.UNIQUE_INDEX, new HbaseTables.UniqueIndex(store));
        }

        public boolean isSchemaStore() {
            return false;
        }

        public Id nextId(HugeType type) {
            throw new UnsupportedOperationException("HbaseGraphStore.nextId()");
        }

        public void increaseCounter(HugeType type, long num) {
            throw new UnsupportedOperationException("HbaseGraphStore.increaseCounter()");
        }

        public long getCounter(HugeType type) {
            throw new UnsupportedOperationException("HbaseGraphStore.getCounter()");
        }
    }

    public static class HbaseSchemaStore
    extends HbaseStore {
        private final HbaseTables.Counters counters = new HbaseTables.Counters();

        public HbaseSchemaStore(HugeConfig config, BackendStoreProvider provider, String namespace, String store) {
            super(provider, namespace, store, (Boolean)config.get(HbaseOptions.HBASE_ENABLE_PARTITION));
            this.registerTableManager(HugeType.VERTEX_LABEL, new HbaseTables.VertexLabel());
            this.registerTableManager(HugeType.EDGE_LABEL, new HbaseTables.EdgeLabel());
            this.registerTableManager(HugeType.PROPERTY_KEY, new HbaseTables.PropertyKey());
            this.registerTableManager(HugeType.INDEX_LABEL, new HbaseTables.IndexLabel());
            this.registerTableManager(HugeType.SECONDARY_INDEX, new HbaseTables.SecondaryIndex(store));
        }

        @Override
        protected List<String> tableNames() {
            List<String> tableNames = super.tableNames();
            tableNames.add(this.counters.table());
            return tableNames;
        }

        public void increaseCounter(HugeType type, long increment) {
            super.checkOpened();
            this.counters.increaseCounter(this.sessions.session(), type, increment);
        }

        public long getCounter(HugeType type) {
            super.checkOpened();
            return this.counters.getCounter(this.sessions.session(), type);
        }

        public boolean isSchemaStore() {
            return true;
        }
    }
}

