/*
 * Decompiled with CFR 0.152.
 */
package org.apache.orc.tools.convert;

import com.google.gson.JsonArray;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonPrimitive;
import com.google.gson.JsonStreamParser;
import java.io.IOException;
import java.io.Reader;
import java.nio.charset.StandardCharsets;
import java.sql.Timestamp;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.OffsetDateTime;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAccessor;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.hive.common.type.HiveDecimal;
import org.apache.hadoop.hive.ql.exec.vector.BytesColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.ColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.DateColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.DecimalColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.DoubleColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.ListColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.LongColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.MapColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.StructColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.TimestampColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.UnionColumnVector;
import org.apache.hadoop.hive.ql.exec.vector.VectorizedRowBatch;
import org.apache.orc.RecordReader;
import org.apache.orc.TypeDescription;

public class JsonReader
implements RecordReader {
    private final TypeDescription schema;
    private final Iterator<JsonElement> parser;
    private final JsonConverter[] converters;
    private final long totalSize;
    private final FSDataInputStream input;
    private long rowNumber = 0L;
    private final DateTimeFormatter dateTimeFormatter;
    private String unionTag = "tag";
    private String unionValue = "value";

    JsonConverter createConverter(TypeDescription schema) {
        switch (schema.getCategory()) {
            case BYTE: 
            case SHORT: 
            case INT: 
            case LONG: {
                return new LongColumnConverter();
            }
            case FLOAT: 
            case DOUBLE: {
                return new DoubleColumnConverter();
            }
            case CHAR: 
            case VARCHAR: 
            case STRING: {
                return new StringColumnConverter();
            }
            case DECIMAL: {
                return new DecimalColumnConverter();
            }
            case DATE: {
                return new DateColumnConverter();
            }
            case TIMESTAMP: 
            case TIMESTAMP_INSTANT: {
                return new TimestampColumnConverter();
            }
            case BINARY: {
                return new BinaryColumnConverter();
            }
            case BOOLEAN: {
                return new BooleanColumnConverter();
            }
            case STRUCT: {
                return new StructColumnConverter(schema);
            }
            case LIST: {
                return new ListColumnConverter(schema);
            }
            case MAP: {
                return new MapColumnConverter(schema);
            }
            case UNION: {
                return new UnionColumnConverter(schema);
            }
        }
        throw new IllegalArgumentException("Unhandled type " + String.valueOf(schema));
    }

    public JsonReader(Reader reader, FSDataInputStream underlying, long size, TypeDescription schema, String timestampFormat, String unionTag, String unionValue) throws IOException {
        this((Iterator<JsonElement>)new JsonStreamParser(reader), underlying, size, schema, timestampFormat);
        this.unionTag = unionTag;
        this.unionValue = unionValue;
    }

    public JsonReader(Reader reader, FSDataInputStream underlying, long size, TypeDescription schema, String timestampFormat) throws IOException {
        this((Iterator<JsonElement>)new JsonStreamParser(reader), underlying, size, schema, timestampFormat);
    }

    public JsonReader(Iterator<JsonElement> parser, FSDataInputStream underlying, long size, TypeDescription schema, String timestampFormat) throws IOException {
        this.schema = schema;
        if (schema.getCategory() != TypeDescription.Category.STRUCT) {
            throw new IllegalArgumentException("Root must be struct - " + String.valueOf(schema));
        }
        this.input = underlying;
        this.totalSize = size;
        this.parser = parser;
        this.dateTimeFormatter = DateTimeFormatter.ofPattern(timestampFormat);
        List fieldTypes = schema.getChildren();
        this.converters = new JsonConverter[fieldTypes.size()];
        for (int c = 0; c < this.converters.length; ++c) {
            this.converters[c] = this.createConverter((TypeDescription)fieldTypes.get(c));
        }
    }

    public boolean nextBatch(VectorizedRowBatch batch) throws IOException {
        batch.reset();
        int maxSize = batch.getMaxSize();
        List fieldNames = this.schema.getFieldNames();
        while (this.parser.hasNext() && batch.size < maxSize) {
            JsonObject elem = this.parser.next().getAsJsonObject();
            for (int c = 0; c < this.converters.length; ++c) {
                JsonElement field = elem.get((String)fieldNames.get(c));
                if (field == null) {
                    batch.cols[c].noNulls = false;
                    batch.cols[c].isNull[batch.size] = true;
                    continue;
                }
                this.converters[c].convert(field, batch.cols[c], batch.size);
            }
            ++batch.size;
        }
        this.rowNumber += (long)batch.size;
        return batch.size != 0;
    }

    public long getRowNumber() throws IOException {
        return this.rowNumber;
    }

    public float getProgress() throws IOException {
        long pos = this.input.getPos();
        return this.totalSize != 0L && pos < this.totalSize ? (float)pos / (float)this.totalSize : 1.0f;
    }

    public void close() throws IOException {
        this.input.close();
    }

    public void seekToRow(long rowCount) throws IOException {
        throw new UnsupportedOperationException("Seek is not supported by JsonReader");
    }

    static class LongColumnConverter
    implements JsonConverter {
        LongColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                LongColumnVector vector = (LongColumnVector)vect;
                vector.vector[row] = value.getAsLong();
            }
        }
    }

    static class DoubleColumnConverter
    implements JsonConverter {
        DoubleColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                DoubleColumnVector vector = (DoubleColumnVector)vect;
                vector.vector[row] = value.getAsDouble();
            }
        }
    }

    static class StringColumnConverter
    implements JsonConverter {
        StringColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                BytesColumnVector vector = (BytesColumnVector)vect;
                byte[] bytes = value.getAsString().getBytes(StandardCharsets.UTF_8);
                vector.setRef(row, bytes, 0, bytes.length);
            }
        }
    }

    static class DecimalColumnConverter
    implements JsonConverter {
        DecimalColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                DecimalColumnVector vector = (DecimalColumnVector)vect;
                vector.vector[row].set(HiveDecimal.create((String)value.getAsString()));
            }
        }
    }

    static class DateColumnConverter
    implements JsonConverter {
        DateColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                DateColumnVector vector = (DateColumnVector)vect;
                LocalDate dt = LocalDate.parse(value.getAsString());
                if (dt != null) {
                    vector.vector[row] = dt.toEpochDay();
                } else {
                    vect.noNulls = false;
                    vect.isNull[row] = true;
                }
            }
        }
    }

    class TimestampColumnConverter
    implements JsonConverter {
        TimestampColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                TimestampColumnVector vector = (TimestampColumnVector)vect;
                TemporalAccessor temporalAccessor = JsonReader.this.dateTimeFormatter.parseBest(value.getAsString(), ZonedDateTime::from, OffsetDateTime::from, LocalDateTime::from);
                if (temporalAccessor instanceof ZonedDateTime) {
                    ZonedDateTime zonedDateTime = (ZonedDateTime)temporalAccessor;
                    Timestamp timestamp = Timestamp.from(zonedDateTime.toInstant());
                    vector.set(row, timestamp);
                } else if (temporalAccessor instanceof OffsetDateTime) {
                    OffsetDateTime offsetDateTime = (OffsetDateTime)temporalAccessor;
                    Timestamp timestamp = Timestamp.from(offsetDateTime.toInstant());
                    vector.set(row, timestamp);
                } else if (temporalAccessor instanceof LocalDateTime) {
                    ZonedDateTime tz = ((LocalDateTime)temporalAccessor).atZone(ZoneId.systemDefault());
                    Timestamp timestamp = Timestamp.from(tz.toInstant());
                    vector.set(row, timestamp);
                } else {
                    vect.noNulls = false;
                    vect.isNull[row] = true;
                }
            }
        }
    }

    static class BinaryColumnConverter
    implements JsonConverter {
        BinaryColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                BytesColumnVector vector = (BytesColumnVector)vect;
                String binStr = value.getAsString();
                byte[] bytes = new byte[binStr.length() / 2];
                for (int i = 0; i < bytes.length; ++i) {
                    bytes[i] = (byte)Integer.parseInt(binStr.substring(i * 2, i * 2 + 2), 16);
                }
                vector.setRef(row, bytes, 0, bytes.length);
            }
        }
    }

    static class BooleanColumnConverter
    implements JsonConverter {
        BooleanColumnConverter() {
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                LongColumnVector vector = (LongColumnVector)vect;
                vector.vector[row] = value.getAsBoolean() ? 1L : 0L;
            }
        }
    }

    class StructColumnConverter
    implements JsonConverter {
        private JsonConverter[] childrenConverters;
        private List<String> fieldNames;

        StructColumnConverter(TypeDescription schema) {
            List kids = schema.getChildren();
            this.childrenConverters = new JsonConverter[kids.size()];
            for (int c = 0; c < this.childrenConverters.length; ++c) {
                this.childrenConverters[c] = JsonReader.this.createConverter((TypeDescription)kids.get(c));
            }
            this.fieldNames = schema.getFieldNames();
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                StructColumnVector vector = (StructColumnVector)vect;
                JsonObject obj = value.getAsJsonObject();
                for (int c = 0; c < this.childrenConverters.length; ++c) {
                    JsonElement elem = obj.get(this.fieldNames.get(c));
                    this.childrenConverters[c].convert(elem, vector.fields[c], row);
                }
            }
        }
    }

    class ListColumnConverter
    implements JsonConverter {
        private JsonConverter childrenConverter;

        ListColumnConverter(TypeDescription schema) {
            this.childrenConverter = JsonReader.this.createConverter((TypeDescription)schema.getChildren().get(0));
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                ListColumnVector vector = (ListColumnVector)vect;
                JsonArray obj = value.getAsJsonArray();
                vector.lengths[row] = obj.size();
                vector.offsets[row] = vector.childCount;
                vector.childCount = (int)((long)vector.childCount + vector.lengths[row]);
                vector.child.ensureSize(vector.childCount, true);
                for (int c = 0; c < obj.size(); ++c) {
                    this.childrenConverter.convert(obj.get(c), vector.child, (int)vector.offsets[row] + c);
                }
            }
        }
    }

    class MapColumnConverter
    implements JsonConverter {
        private JsonConverter keyConverter;
        private JsonConverter valueConverter;

        MapColumnConverter(TypeDescription schema) {
            TypeDescription keyType = (TypeDescription)schema.getChildren().get(0);
            if (keyType.getCategory() != TypeDescription.Category.STRING) {
                throw new IllegalArgumentException("JSON can only support MAP key in STRING type: " + String.valueOf(schema));
            }
            this.keyConverter = JsonReader.this.createConverter(keyType);
            this.valueConverter = JsonReader.this.createConverter((TypeDescription)schema.getChildren().get(1));
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                MapColumnVector vector = (MapColumnVector)vect;
                JsonObject obj = value.getAsJsonObject();
                vector.lengths[row] = obj.entrySet().size();
                vector.offsets[row] = vector.childCount;
                vector.childCount = (int)((long)vector.childCount + vector.lengths[row]);
                vector.keys.ensureSize(vector.childCount, true);
                vector.values.ensureSize(vector.childCount, true);
                int cnt = 0;
                for (Map.Entry entry : obj.entrySet()) {
                    int offset = (int)vector.offsets[row] + cnt++;
                    this.keyConverter.convert((JsonElement)new JsonPrimitive((String)entry.getKey()), vector.keys, offset);
                    this.valueConverter.convert((JsonElement)entry.getValue(), vector.values, offset);
                }
            }
        }
    }

    class UnionColumnConverter
    implements JsonConverter {
        private JsonConverter[] childConverter;

        UnionColumnConverter(TypeDescription schema) {
            int size = schema.getChildren().size();
            this.childConverter = new JsonConverter[size];
            for (int i = 0; i < size; ++i) {
                this.childConverter[i] = JsonReader.this.createConverter((TypeDescription)schema.getChildren().get(i));
            }
        }

        @Override
        public void convert(JsonElement value, ColumnVector vect, int row) {
            if (value == null || value.isJsonNull()) {
                vect.noNulls = false;
                vect.isNull[row] = true;
            } else {
                int tag;
                UnionColumnVector vector = (UnionColumnVector)vect;
                JsonObject obj = value.getAsJsonObject();
                vector.tags[row] = tag = obj.get(JsonReader.this.unionTag).getAsInt();
                this.childConverter[tag].convert(obj.get(JsonReader.this.unionValue), vector.fields[tag], row);
            }
        }
    }

    static interface JsonConverter {
        public void convert(JsonElement var1, ColumnVector var2, int var3);
    }
}

