/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.segment.file;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import com.google.common.primitives.Ints;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import javax.annotation.Nullable;
import org.apache.druid.error.DruidException;
import org.apache.druid.java.util.common.ByteBufferUtils;
import org.apache.druid.java.util.common.io.Closer;
import org.apache.druid.segment.data.CompressionStrategy;
import org.apache.druid.segment.file.SegmentFileContainerMetadata;
import org.apache.druid.segment.file.SegmentFileMapper;
import org.apache.druid.segment.file.SegmentFileMetadata;
import org.apache.druid.segment.file.SegmentInternalFileMetadata;
import org.apache.druid.utils.CloseableUtils;

public class SegmentFileMapperV10
implements SegmentFileMapper {
    private final File segmentFile;
    private final SegmentFileMetadata segmentFileMetadata;
    private final List<MappedByteBuffer> containers;
    private final Map<String, SegmentFileMapperV10> externalMappers;
    private final AtomicBoolean closed = new AtomicBoolean(false);

    public static SegmentFileMapperV10 create(File segmentFile, ObjectMapper mapper, List<String> externals) throws IOException {
        SegmentFileMapperV10 entryPoint = SegmentFileMapperV10.create(segmentFile, mapper);
        HashMap<String, SegmentFileMapperV10> externalMappers = new HashMap<String, SegmentFileMapperV10>();
        try {
            for (String filename : externals) {
                File externalFile = new File(segmentFile.getParentFile(), filename);
                if (!externalFile.exists()) continue;
                externalMappers.put(filename, SegmentFileMapperV10.create(externalFile, mapper));
            }
        }
        catch (Throwable t) {
            Closer closer = Closer.create();
            closer.registerAll(externalMappers.values());
            throw CloseableUtils.closeAndWrapInCatch(t, closer);
        }
        return new SegmentFileMapperV10(entryPoint.segmentFile, entryPoint.segmentFileMetadata, entryPoint.containers, externalMappers);
    }

    public static SegmentFileMapperV10 create(File segmentFile, ObjectMapper mapper) throws IOException {
        try (FileInputStream fis = new FileInputStream(segmentFile);){
            int startOffset;
            byte[] header = new byte[6];
            int read = fis.read(header);
            if (read < header.length) {
                throw DruidException.defensive("expected at least [%s] bytes, but only read [%s]", header.length, read);
            }
            ByteBuffer headerBuffer = ByteBuffer.wrap(header).order(ByteOrder.LITTLE_ENDIAN);
            if (headerBuffer.get(0) != 10) {
                throw DruidException.defensive("not v10, got[%s] instead", headerBuffer.get(0));
            }
            byte compression = headerBuffer.get(1);
            CompressionStrategy compressionStrategy = CompressionStrategy.forId(compression);
            int metaLength = headerBuffer.getInt(2);
            byte[] meta = new byte[metaLength];
            if (CompressionStrategy.NONE == compressionStrategy) {
                startOffset = header.length + meta.length;
                read = fis.read(meta);
                if (read < meta.length) {
                    throw DruidException.defensive("read[%s] which is less than expected metadata length[%s]", read, metaLength);
                }
            } else {
                byte[] compressedLengthBytes = new byte[4];
                read = fis.read(compressedLengthBytes);
                if (read != 4) {
                    throw DruidException.defensive("read[%s] which is less than expected [%s]", read, 4);
                }
                ByteBuffer compressedLengthBuffer = ByteBuffer.wrap(compressedLengthBytes).order(ByteOrder.LITTLE_ENDIAN);
                int compressedLength = compressedLengthBuffer.getInt(0);
                startOffset = header.length + 4 + compressedLength;
                byte[] compressed = new byte[compressedLength];
                read = fis.read(compressed);
                if (read < compressed.length) {
                    throw DruidException.defensive("read[%s] which is less than expected compressed metadata length[%s]", read, compressedLength);
                }
                ByteBuffer inBuffer = ByteBuffer.wrap(compressed).order(ByteOrder.LITTLE_ENDIAN);
                ByteBuffer outBuffer = ByteBuffer.wrap(meta).order(ByteOrder.LITTLE_ENDIAN);
                CompressionStrategy.Decompressor decompressor = compressionStrategy.getDecompressor();
                decompressor.decompress(inBuffer, compressedLength, outBuffer);
            }
            SegmentFileMetadata metadata = (SegmentFileMetadata)mapper.readValue(meta, SegmentFileMetadata.class);
            ArrayList containers = Lists.newArrayListWithCapacity((int)metadata.getContainers().size());
            try (RandomAccessFile f = new RandomAccessFile(segmentFile, "r");
                 FileChannel channel = f.getChannel();){
                for (SegmentFileContainerMetadata containerMetadata : metadata.getContainers()) {
                    containers.add(channel.map(FileChannel.MapMode.READ_ONLY, (long)startOffset + containerMetadata.getStartOffset(), containerMetadata.getSize()));
                }
            }
            catch (IOException e) {
                Closer closer = Closer.create();
                for (MappedByteBuffer buffer : containers) {
                    closer.register(() -> ByteBufferUtils.unmap(buffer));
                }
                CloseableUtils.closeAndWrapExceptions(closer);
                throw DruidException.defensive(e, "Problem mapping segment file[%s]", segmentFile.getAbsolutePath());
            }
            SegmentFileMapperV10 segmentFileMapperV10 = new SegmentFileMapperV10(segmentFile, metadata, List.copyOf(containers), Map.of());
            return segmentFileMapperV10;
        }
    }

    public SegmentFileMapperV10(File segmentFile, SegmentFileMetadata segmentFileMetadata, List<MappedByteBuffer> containers, Map<String, SegmentFileMapperV10> externalMappers) {
        this.segmentFile = segmentFile;
        this.segmentFileMetadata = segmentFileMetadata;
        this.containers = containers;
        this.externalMappers = externalMappers;
    }

    public SegmentFileMetadata getSegmentFileMetadata() {
        return this.segmentFileMetadata;
    }

    @Override
    public Set<String> getInternalFilenames() {
        return this.segmentFileMetadata.getFiles().keySet();
    }

    @Override
    @Nullable
    public ByteBuffer mapFile(String name) throws IOException {
        this.checkClosed();
        SegmentInternalFileMetadata fileMetadata = this.segmentFileMetadata.getFiles().get(name);
        if (fileMetadata == null) {
            return null;
        }
        MappedByteBuffer container = this.containers.get(fileMetadata.getContainer());
        if (container == null) {
            throw DruidException.defensive("invalid container[%s]", fileMetadata.getContainer());
        }
        ByteBuffer view = container.asReadOnlyBuffer();
        view.position(Ints.checkedCast((long)fileMetadata.getStartOffset())).limit(Ints.checkedCast((long)(fileMetadata.getStartOffset() + fileMetadata.getSize())));
        return view.slice();
    }

    @Override
    @Nullable
    public ByteBuffer mapExternalFile(String filename, String name) throws IOException {
        this.checkClosed();
        SegmentFileMapperV10 externalMapper = this.externalMappers.get(filename);
        if (externalMapper == null) {
            throw DruidException.defensive("external file[%s] containing[%s] not found", filename, name);
        }
        return externalMapper.mapFile(name);
    }

    private void checkClosed() {
        if (this.closed.get()) {
            throw DruidException.defensive("Segment file[%s] is closed", this.segmentFile);
        }
    }

    @Override
    public void close() {
        if (this.closed.compareAndSet(false, true)) {
            Closer closer = Closer.create();
            for (MappedByteBuffer buffer : this.containers) {
                closer.register(() -> ByteBufferUtils.unmap(buffer));
            }
            closer.registerAll(this.externalMappers.values());
            CloseableUtils.closeAndWrapExceptions(closer);
        }
    }
}

