/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.codec.binary;

import java.math.BigInteger;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Random;
import java.util.stream.Stream;
import org.apache.commons.codec.CodecPolicy;
import org.apache.commons.codec.DecoderException;
import org.apache.commons.codec.EncoderException;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.codec.binary.BaseNCodec;
import org.apache.commons.codec.binary.BaseNCodecTest;
import org.apache.commons.codec.binary.BaseNTestData;
import org.apache.commons.codec.binary.Hex;
import org.apache.commons.codec.binary.StringUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.Assumptions;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.Arguments;
import org.junit.jupiter.params.provider.MethodSource;
import org.junit.jupiter.params.provider.ValueSource;

class Base64Test {
    private static final String FOX_BASE64 = "VGhlIH@$#$@%F1aWN@#@#@@rIGJyb3duIGZve\n\r\t%#%#%#%CBqd##$#$W1wZWQgb3ZlciB0aGUgbGF6eSBkb2dzLg==";
    private static final String FOX_TEXT = "The quick brown fox jumped over the lazy dogs.";
    private static final Charset CHARSET_UTF8 = StandardCharsets.UTF_8;
    static final String[] BASE64_IMPOSSIBLE_CASES = new String[]{"ZE==", "ZmC=", "Zm9vYE==", "Zm9vYmC=", "AB"};
    private static final byte[] STANDARD_ENCODE_TABLE = new byte[]{65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 43, 47};
    private final Random random = new Random();

    Base64Test() {
    }

    private static void assertBase64DecodingOfTrailingBits(int nbits) {
        Base64 codec = new Base64(0, null, false, CodecPolicy.STRICT);
        Assertions.assertTrue((boolean)codec.isStrictDecoding());
        Assertions.assertEquals((Object)CodecPolicy.STRICT, (Object)codec.getCodecPolicy());
        Base64 defaultCodec = new Base64();
        Assertions.assertFalse((boolean)defaultCodec.isStrictDecoding());
        Assertions.assertEquals((Object)CodecPolicy.LENIENT, (Object)defaultCodec.getCodecPolicy());
        int length = nbits / 6;
        byte[] encoded = new byte[4];
        Arrays.fill(encoded, 0, length, STANDARD_ENCODE_TABLE[0]);
        Arrays.fill(encoded, length, encoded.length, (byte)61);
        int discard = nbits % 8;
        int emptyBitsMask = (1 << discard) - 1;
        boolean invalid = length == 1;
        int last = length - 1;
        for (int i = 0; i < 64; ++i) {
            byte[] decoded;
            encoded[last] = STANDARD_ENCODE_TABLE[i];
            if (invalid || (i & emptyBitsMask) != 0) {
                Assertions.assertThrows(IllegalArgumentException.class, () -> codec.decode(encoded), (String)"Final base-64 digit should not be allowed");
                decoded = defaultCodec.decode(encoded);
                Assertions.assertFalse((boolean)Arrays.equals(encoded, defaultCodec.encode(decoded)));
                continue;
            }
            decoded = codec.decode(encoded);
            int bitsEncoded = i >> discard;
            Assertions.assertEquals((int)bitsEncoded, (int)decoded[decoded.length - 1], (String)"Invalid decoding of last character");
            Assertions.assertArrayEquals((byte[])encoded, (byte[])codec.encode(decoded));
        }
    }

    static Stream<Object> testIsBase64() {
        return Stream.of(Arguments.of((Object[])new Object[]{new byte[]{1, 2, 3}, false}), Arguments.of((Object[])new Object[]{new byte[]{-128}, false}), Arguments.of((Object[])new Object[]{new byte[]{-125}, false}), Arguments.of((Object[])new Object[]{new byte[]{-10}, false}), Arguments.of((Object[])new Object[]{new byte[]{0}, false}), Arguments.of((Object[])new Object[]{new byte[]{64, 127}, false}), Arguments.of((Object[])new Object[]{new byte[]{127}, false}), Arguments.of((Object[])new Object[]{new byte[]{65}, true}), Arguments.of((Object[])new Object[]{new byte[]{65, -128}, false}), Arguments.of((Object[])new Object[]{new byte[]{65, 90, 97}, true}), Arguments.of((Object[])new Object[]{new byte[]{47, 61, 43}, true}), Arguments.of((Object[])new Object[]{new byte[]{36}, false}));
    }

    public Random getRandom() {
        return this.random;
    }

    @Test
    void testBase64() {
        String content = "Hello World";
        byte[] encodedBytes = Base64.encodeBase64((byte[])StringUtils.getBytesUtf8((String)"Hello World"));
        String encodedContent = StringUtils.newStringUtf8((byte[])encodedBytes);
        Assertions.assertEquals((Object)"SGVsbG8gV29ybGQ=", (Object)encodedContent, (String)"encoding hello world");
        Base64 b64 = new Base64(76, null);
        encodedBytes = b64.encode(StringUtils.getBytesUtf8((String)"Hello World"));
        encodedContent = StringUtils.newStringUtf8((byte[])encodedBytes);
        Assertions.assertEquals((Object)"SGVsbG8gV29ybGQ=", (Object)encodedContent, (String)"encoding hello world");
        b64 = new Base64(0, null);
        encodedBytes = b64.encode(StringUtils.getBytesUtf8((String)"Hello World"));
        encodedContent = StringUtils.newStringUtf8((byte[])encodedBytes);
        Assertions.assertEquals((Object)"SGVsbG8gV29ybGQ=", (Object)encodedContent, (String)"encoding hello world");
        byte[] decode = b64.decode("SGVsbG{\u00e9\u00e9\u00e9\u00e9\u00e9\u00e9}8gV29ybGQ=");
        String decodeString = StringUtils.newStringUtf8((byte[])decode);
        Assertions.assertEquals((Object)"Hello World", (Object)decodeString, (String)"decode hello world");
    }

    @Test
    void testBase64AtBufferEnd() {
        this.testBase64InBuffer(100, 0);
    }

    @Test
    void testBase64AtBufferMiddle() {
        this.testBase64InBuffer(100, 100);
    }

    @Test
    void testBase64AtBufferStart() {
        this.testBase64InBuffer(0, 100);
    }

    @Test
    void testBase64DecodingOfTrailing12Bits() {
        Base64Test.assertBase64DecodingOfTrailingBits(12);
    }

    @Test
    void testBase64DecodingOfTrailing18Bits() {
        Base64Test.assertBase64DecodingOfTrailingBits(18);
    }

    @Test
    void testBase64DecodingOfTrailing6Bits() {
        Base64Test.assertBase64DecodingOfTrailingBits(6);
    }

    @Test
    void testBase64ImpossibleSamples() {
        Base64 codec = new Base64(0, null, false, CodecPolicy.STRICT);
        for (String s : BASE64_IMPOSSIBLE_CASES) {
            Assertions.assertThrows(IllegalArgumentException.class, () -> codec.decode(s));
        }
    }

    private void testBase64InBuffer(int startPasSize, int endPadSize) {
        String content = "Hello World";
        byte[] bytesUtf8 = StringUtils.getBytesUtf8((String)"Hello World");
        byte[] buffer = ArrayUtils.addAll((byte[])bytesUtf8, (byte[])new byte[endPadSize]);
        buffer = ArrayUtils.addAll((byte[])new byte[startPasSize], (byte[])buffer);
        byte[] encodedBytes = new Base64().encode(buffer, startPasSize, bytesUtf8.length);
        String encodedContent = StringUtils.newStringUtf8((byte[])encodedBytes);
        Assertions.assertEquals((Object)"SGVsbG8gV29ybGQ=", (Object)encodedContent, (String)"encoding hello world");
    }

    @Test
    void testBuilderCodecPolicy() {
        Assertions.assertEquals((Object)CodecPolicy.LENIENT, (Object)Base64.builder().get().getCodecPolicy());
        Assertions.assertEquals((Object)CodecPolicy.LENIENT, (Object)((Base64.Builder)Base64.builder().setDecodingPolicy(CodecPolicy.LENIENT)).get().getCodecPolicy());
        Assertions.assertEquals((Object)CodecPolicy.STRICT, (Object)((Base64.Builder)Base64.builder().setDecodingPolicy(CodecPolicy.STRICT)).get().getCodecPolicy());
        Assertions.assertEquals((Object)CodecPolicy.LENIENT, (Object)((Base64.Builder)((Base64.Builder)Base64.builder().setDecodingPolicy(CodecPolicy.STRICT)).setDecodingPolicy(null)).get().getCodecPolicy());
        Assertions.assertEquals((Object)CodecPolicy.LENIENT, (Object)((Base64.Builder)Base64.builder().setDecodingPolicy(null)).get().getCodecPolicy());
    }

    @Test
    void testBuilderLineAttributes() {
        Assertions.assertNull((Object)Base64.builder().get().getLineSeparator());
        Assertions.assertNull((Object)((Base64.Builder)Base64.builder().setLineSeparator(BaseNCodec.CHUNK_SEPARATOR)).get().getLineSeparator());
        Assertions.assertArrayEquals((byte[])BaseNCodec.CHUNK_SEPARATOR, (byte[])((Base64.Builder)((Base64.Builder)Base64.builder().setLineLength(4)).setLineSeparator(BaseNCodec.CHUNK_SEPARATOR)).get().getLineSeparator());
        Assertions.assertArrayEquals((byte[])BaseNCodec.CHUNK_SEPARATOR, (byte[])((Base64.Builder)((Base64.Builder)Base64.builder().setLineLength(4)).setLineSeparator(null)).get().getLineSeparator());
        Assertions.assertArrayEquals((byte[])BaseNCodec.CHUNK_SEPARATOR, (byte[])((Base64.Builder)((Base64.Builder)Base64.builder().setLineLength(10)).setLineSeparator(null)).get().getLineSeparator());
        Assertions.assertNull((Object)((Base64.Builder)((Base64.Builder)Base64.builder().setLineLength(-1)).setLineSeparator(null)).get().getLineSeparator());
        Assertions.assertNull((Object)((Base64.Builder)((Base64.Builder)Base64.builder().setLineLength(0)).setLineSeparator(null)).get().getLineSeparator());
        Assertions.assertArrayEquals((byte[])new byte[]{1}, (byte[])((Base64.Builder)((Base64.Builder)Base64.builder().setLineLength(4)).setLineSeparator(new byte[]{1})).get().getLineSeparator());
        Assertions.assertEquals((Object)"Zm94\r\n", (Object)((Base64.Builder)Base64.builder().setLineLength(4)).get().encodeToString("fox".getBytes(CHARSET_UTF8)));
    }

    @Test
    void testBuilderPadingByte() {
        Assertions.assertNull((Object)Base64.builder().get().getLineSeparator());
        Assertions.assertNull((Object)((Base64.Builder)Base64.builder().setLineSeparator(BaseNCodec.CHUNK_SEPARATOR)).get().getLineSeparator());
        Assertions.assertArrayEquals((byte[])BaseNCodec.CHUNK_SEPARATOR, (byte[])((Base64.Builder)((Base64.Builder)Base64.builder().setLineLength(4)).setLineSeparator(BaseNCodec.CHUNK_SEPARATOR)).get().getLineSeparator());
        Assertions.assertArrayEquals((byte[])BaseNCodec.CHUNK_SEPARATOR, (byte[])((Base64.Builder)((Base64.Builder)Base64.builder().setLineLength(4)).setLineSeparator(null)).get().getLineSeparator());
        Assertions.assertArrayEquals((byte[])BaseNCodec.CHUNK_SEPARATOR, (byte[])((Base64.Builder)((Base64.Builder)Base64.builder().setLineLength(10)).setLineSeparator(null)).get().getLineSeparator());
        Assertions.assertNull((Object)((Base64.Builder)((Base64.Builder)Base64.builder().setLineLength(-1)).setLineSeparator(null)).get().getLineSeparator());
        Assertions.assertNull((Object)((Base64.Builder)((Base64.Builder)Base64.builder().setLineLength(0)).setLineSeparator(null)).get().getLineSeparator());
        Assertions.assertArrayEquals((byte[])new byte[]{1}, (byte[])((Base64.Builder)((Base64.Builder)Base64.builder().setLineLength(4)).setLineSeparator(new byte[]{1})).get().getLineSeparator());
        Assertions.assertEquals((Object)"VGhlIGJyb3duIGZveA==", (Object)Base64.builder().get().encodeToString("The brown fox".getBytes(CHARSET_UTF8)));
        Assertions.assertEquals((Object)"VGhlIGJyb3duIGZveA__", (Object)((Base64.Builder)Base64.builder().setPadding((byte)95)).get().encodeToString("The brown fox".getBytes(CHARSET_UTF8)));
    }

    @Test
    void testBuilderUrlSafe() {
        Assertions.assertFalse((boolean)Base64.builder().get().isUrlSafe());
        Assertions.assertFalse((boolean)Base64.builder().setUrlSafe(false).get().isUrlSafe());
        Assertions.assertFalse((boolean)Base64.builder().setUrlSafe(true).setUrlSafe(false).get().isUrlSafe());
        Assertions.assertTrue((boolean)Base64.builder().setUrlSafe(false).setUrlSafe(true).get().isUrlSafe());
    }

    @Test
    void testByteToStringVariations() throws DecoderException {
        Base64 base64 = new Base64(0);
        byte[] b1 = StringUtils.getBytesUtf8((String)"Hello World");
        byte[] b2 = new byte[]{};
        byte[] b3 = null;
        byte[] b4 = Hex.decodeHex((String)"2bf7cc2701fe4397b49ebeed5acc7090");
        Assertions.assertEquals((Object)"SGVsbG8gV29ybGQ=", (Object)base64.encodeToString(b1), (String)"byteToString Hello World");
        Assertions.assertEquals((Object)"SGVsbG8gV29ybGQ=", (Object)Base64.encodeBase64String((byte[])b1), (String)"byteToString static Hello World");
        Assertions.assertEquals((Object)"", (Object)base64.encodeToString(b2), (String)"byteToString \"\"");
        Assertions.assertEquals((Object)"", (Object)Base64.encodeBase64String((byte[])b2), (String)"byteToString static \"\"");
        Assertions.assertNull((Object)base64.encodeToString(b3), (String)"byteToString null");
        Assertions.assertNull((Object)Base64.encodeBase64String(b3), (String)"byteToString static null");
        Assertions.assertEquals((Object)"K/fMJwH+Q5e0nr7tWsxwkA==", (Object)base64.encodeToString(b4), (String)"byteToString UUID");
        Assertions.assertEquals((Object)"K/fMJwH+Q5e0nr7tWsxwkA==", (Object)Base64.encodeBase64String((byte[])b4), (String)"byteToString static UUID");
        Assertions.assertEquals((Object)"K_fMJwH-Q5e0nr7tWsxwkA", (Object)Base64.encodeBase64URLSafeString((byte[])b4), (String)"byteToString static-url-safe UUID");
    }

    @Test
    void testChunkedEncodeMultipleOf76() {
        byte[] expectedEncode = Base64.encodeBase64((byte[])BaseNTestData.DECODED, (boolean)true);
        String actualResult = "9IPNKwUvdLiIAp6ctz12SiQmOGstWyYvSPeevufDhrzaws65voykKjbIj33YWTa9xA7c/FHypWcl\nrZhQ7onfc3JE93BJ5fT4R9zAEdjbjy1hv4ZYNnET4WJeXMLJ/5p+qBpTsPpepW8DNVYy1c02/1wy\nC+kgA6CvRUd9cSr/lt88AEdsTV4GMCn1+EwuAiYdivxuzn+cLM8q2jewqlI52tP9J7Cs8vqG71s6\n+WAELKvm/UovvyaOi+OdMUfjQ0JLiLkHu6p9OwUgvQqiDKzEv/Augo0dTPZzYGEyCP5GVrle3QQd\ngciIHnpdd4VUTPGRUbXeKbh++U3fbJIng/sQXM3IYByMZ7xt9HWS1LUcRdQ7Prwn/IlQWxOMeq+K\nZJSoAviWtdserXyHbIEa//hmr4p/j80k0g9q35hq1ayGM9984ALTSaZ8WeyFbZx1CxC/Qoqf92UH\n/ylBRnSJNn4sS0oa3uUbNvOnpkB4D9V7Ut9atinCJrw+wiJcMl+9kp251IUxBGA4cUxh0eaxk3OD\nWnwI95EktmWOKwCSP0xjWwIMxDjygwAG5R8fk9H9bVi1thMavm4nDc4vaNoSE1RnZNYwbiUVlVPM\n9EclvJWTWd6igWeA0MxHAA8iOM5Vnmqp/WGM7UDq59rBIdNQCoeTJaAkEtAuLL5zogOa5e+MzVjv\nB5MYQlOlaaTtQrRApXa5Z4VfEanu9UK2fi1T8jJPFC2PmXebxp0bnO+VW+bgyEdIIkIQCaZq1MKW\nC3KuiOS9BJ1t7O0A2JKJKvoE4UNulzV2TGCC+KAnmjRqQBqXlJmgjHQAoHNZKOma/uIQOsvfDnqi\ncYdDmfyCYuV89HjA1H8tiDJ85VfsrFHdcbPAoNCpi65awJSHfdPO1NDONOK++S7Y0VXUgoYYrBV4\nY7YbC8wg/nqcimr3lm3tRyp+QsgKzdREbfNRk0F5PLyLfsUElepjs1QdV3fEV1LJtiywA3ubVNQJ\nRxhbYxa/C/Xy2qxpm6vvdL92l3q1ccev35IcaOiSx7Im+/GxV2lVKdaOvYVGDD1zBRe6Y2CwQb9p\n088l3/93qGR5593NCiuPPWcsDWwUShM1EyW0FNX1F8bnzHnYijoyE/jf4s/l9bBd7yJdRWRCyih2\nWcypAiOIEkBsH+dCTgalu8sRDoMh4ZIBBdgHfoZUycLqReQFLZZ4Sl4zSmzt5vQxQFhEKb9+ff/4\nrb1KAo6wifengxVfIsa2b5ljXzAqXs7JkPvmC6fa7X4ZZndRokaxYlu3cg8OV+uG/6YAHZilo8at\n0OpkkNdNFuhwuGlkBqrZKNUj/gSiYYc06gF/r/z6iWAjpXJRW1qq3CLZXdZFZ/VrqXeVjtOAu2A=\n".replace("\n", "\r\n");
        byte[] actualEncode = StringUtils.getBytesUtf8((String)actualResult);
        Assertions.assertArrayEquals((byte[])expectedEncode, (byte[])actualEncode, (String)"chunkedEncodeMultipleOf76");
    }

    @Test
    void testCodec112() {
        byte[] in = new byte[]{0};
        byte[] out = Base64.encodeBase64((byte[])in);
        Base64.encodeBase64((byte[])in, (boolean)false, (boolean)false, (int)out.length);
    }

    @Test
    void testCodec263() {
        Base64.decodeBase64((String)"publishMessage");
        Assertions.assertTrue((boolean)Base64.isBase64((String)"publishMessage"));
    }

    @Test
    void testCodec265() {
        long estimatedMemory;
        int size1GiB = 0x40000000;
        int blocks = (int)Math.ceil(3.579139413333333E8);
        int expectedLength = 4 * blocks;
        long presumableFreeMemory = BaseNCodecTest.getPresumableFreeMemory();
        Assumptions.assumeTrue((presumableFreeMemory > (estimatedMemory = 0x100000000L + (long)expectedLength + 32768L) ? 1 : 0) != 0, (String)"Not enough free memory for the test");
        byte[] bytes = new byte[0x40000000];
        byte[] encoded = Base64.encodeBase64((byte[])bytes);
        Assertions.assertEquals((int)expectedLength, (int)encoded.length);
    }

    @Test
    void testCodec68() {
        byte[] x = new byte[]{110, 65, 61, 61, -100};
        Base64.decodeBase64((byte[])x);
    }

    @Test
    void testCodeInteger1() {
        String encodedInt1 = "li7dzDacuo67Jg7mtqEm2TRuOMU=";
        BigInteger bigInt1 = new BigInteger("857393771208094202104259627990318636601332086981");
        Assertions.assertEquals((Object)"li7dzDacuo67Jg7mtqEm2TRuOMU=", (Object)new String(Base64.encodeInteger((BigInteger)bigInt1)));
        Assertions.assertEquals((Object)bigInt1, (Object)Base64.decodeInteger((byte[])"li7dzDacuo67Jg7mtqEm2TRuOMU=".getBytes(CHARSET_UTF8)));
    }

    @Test
    void testCodeInteger2() {
        String encodedInt2 = "9B5ypLY9pMOmtxCeTDHgwdNFeGs=";
        BigInteger bigInt2 = new BigInteger("1393672757286116725466646726891466679477132949611");
        Assertions.assertEquals((Object)"9B5ypLY9pMOmtxCeTDHgwdNFeGs=", (Object)new String(Base64.encodeInteger((BigInteger)bigInt2)));
        Assertions.assertEquals((Object)bigInt2, (Object)Base64.decodeInteger((byte[])"9B5ypLY9pMOmtxCeTDHgwdNFeGs=".getBytes(CHARSET_UTF8)));
    }

    @Test
    void testCodeInteger3() {
        String encodedInt3 = "FKIhdgaG5LGKiEtF1vHy4f3y700zaD6QwDS3IrNVGzNp2rY+1LFWTK6D44AyiC1n8uWz1itkYMZF0/aKDK0Yjg==";
        BigInteger bigInt3 = new BigInteger("1080654815409387346195174854511969891364164488058190793635243098977490449581124171362405574495062430572478766856090958495998158114332651671116876320938126");
        Assertions.assertEquals((Object)"FKIhdgaG5LGKiEtF1vHy4f3y700zaD6QwDS3IrNVGzNp2rY+1LFWTK6D44AyiC1n8uWz1itkYMZF0/aKDK0Yjg==", (Object)new String(Base64.encodeInteger((BigInteger)bigInt3)));
        Assertions.assertEquals((Object)bigInt3, (Object)Base64.decodeInteger((byte[])"FKIhdgaG5LGKiEtF1vHy4f3y700zaD6QwDS3IrNVGzNp2rY+1LFWTK6D44AyiC1n8uWz1itkYMZF0/aKDK0Yjg==".getBytes(CHARSET_UTF8)));
    }

    @Test
    void testCodeInteger4() {
        String encodedInt4 = "ctA8YGxrtngg/zKVvqEOefnwmViFztcnPBYPlJsvh6yKI4iDm68fnp4Mi3RrJ6bZAygFrUIQLxLjV+OJtgJAEto0xAs+Mehuq1DkSFEpP3oDzCTOsrOiS1DwQe4oIb7zVk/9l7aPtJMHW0LVlMdwZNFNNJoqMcT2ZfCPrfvYvQ0=";
        BigInteger bigInt4 = new BigInteger("80624726256040348115552042320696813500187275370942441977258669395023235020055564647117594451929708788598704081077890850726227289270230377442285367559774800853404089092381420228663316324808605521697655145608801533888071381819208887705771753016938104409283940243801509765453542091716518238707344493641683483917");
        Assertions.assertEquals((Object)"ctA8YGxrtngg/zKVvqEOefnwmViFztcnPBYPlJsvh6yKI4iDm68fnp4Mi3RrJ6bZAygFrUIQLxLjV+OJtgJAEto0xAs+Mehuq1DkSFEpP3oDzCTOsrOiS1DwQe4oIb7zVk/9l7aPtJMHW0LVlMdwZNFNNJoqMcT2ZfCPrfvYvQ0=", (Object)new String(Base64.encodeInteger((BigInteger)bigInt4)));
        Assertions.assertEquals((Object)bigInt4, (Object)Base64.decodeInteger((byte[])"ctA8YGxrtngg/zKVvqEOefnwmViFztcnPBYPlJsvh6yKI4iDm68fnp4Mi3RrJ6bZAygFrUIQLxLjV+OJtgJAEto0xAs+Mehuq1DkSFEpP3oDzCTOsrOiS1DwQe4oIb7zVk/9l7aPtJMHW0LVlMdwZNFNNJoqMcT2ZfCPrfvYvQ0=".getBytes(CHARSET_UTF8)));
    }

    @Test
    void testCodeIntegerEdgeCases() {
    }

    @Test
    void testCodeIntegerNull() {
        Assertions.assertThrows(NullPointerException.class, () -> Base64.encodeInteger(null), (String)"Exception not thrown when passing in null to encodeInteger(BigInteger)");
    }

    @Test
    void testConstructor_Int_ByteArray_Boolean() {
        Base64 base64 = new Base64(65, new byte[]{9}, false);
        byte[] encoded = base64.encode(BaseNTestData.DECODED);
        String expectedResult = "9IPNKwUvdLiIAp6ctz12SiQmOGstWyYvSPeevufDhrzaws65voykKjbIj33YWTa9\nxA7c/FHypWclrZhQ7onfc3JE93BJ5fT4R9zAEdjbjy1hv4ZYNnET4WJeXMLJ/5p+\nqBpTsPpepW8DNVYy1c02/1wyC+kgA6CvRUd9cSr/lt88AEdsTV4GMCn1+EwuAiYd\nivxuzn+cLM8q2jewqlI52tP9J7Cs8vqG71s6+WAELKvm/UovvyaOi+OdMUfjQ0JL\niLkHu6p9OwUgvQqiDKzEv/Augo0dTPZzYGEyCP5GVrle3QQdgciIHnpdd4VUTPGR\nUbXeKbh++U3fbJIng/sQXM3IYByMZ7xt9HWS1LUcRdQ7Prwn/IlQWxOMeq+KZJSo\nAviWtdserXyHbIEa//hmr4p/j80k0g9q35hq1ayGM9984ALTSaZ8WeyFbZx1CxC/\nQoqf92UH/ylBRnSJNn4sS0oa3uUbNvOnpkB4D9V7Ut9atinCJrw+wiJcMl+9kp25\n1IUxBGA4cUxh0eaxk3ODWnwI95EktmWOKwCSP0xjWwIMxDjygwAG5R8fk9H9bVi1\nthMavm4nDc4vaNoSE1RnZNYwbiUVlVPM9EclvJWTWd6igWeA0MxHAA8iOM5Vnmqp\n/WGM7UDq59rBIdNQCoeTJaAkEtAuLL5zogOa5e+MzVjvB5MYQlOlaaTtQrRApXa5\nZ4VfEanu9UK2fi1T8jJPFC2PmXebxp0bnO+VW+bgyEdIIkIQCaZq1MKWC3KuiOS9\nBJ1t7O0A2JKJKvoE4UNulzV2TGCC+KAnmjRqQBqXlJmgjHQAoHNZKOma/uIQOsvf\nDnqicYdDmfyCYuV89HjA1H8tiDJ85VfsrFHdcbPAoNCpi65awJSHfdPO1NDONOK+\n+S7Y0VXUgoYYrBV4Y7YbC8wg/nqcimr3lm3tRyp+QsgKzdREbfNRk0F5PLyLfsUE\nlepjs1QdV3fEV1LJtiywA3ubVNQJRxhbYxa/C/Xy2qxpm6vvdL92l3q1ccev35Ic\naOiSx7Im+/GxV2lVKdaOvYVGDD1zBRe6Y2CwQb9p088l3/93qGR5593NCiuPPWcs\nDWwUShM1EyW0FNX1F8bnzHnYijoyE/jf4s/l9bBd7yJdRWRCyih2WcypAiOIEkBs\nH+dCTgalu8sRDoMh4ZIBBdgHfoZUycLqReQFLZZ4Sl4zSmzt5vQxQFhEKb9+ff/4\nrb1KAo6wifengxVfIsa2b5ljXzAqXs7JkPvmC6fa7X4ZZndRokaxYlu3cg8OV+uG\n/6YAHZilo8at0OpkkNdNFuhwuGlkBqrZKNUj/gSiYYc06gF/r/z6iWAjpXJRW1qq\n3CLZXdZFZ/VrqXeVjtOAu2A=\n";
        expectedResult = expectedResult.replace('\n', '\t');
        String result = StringUtils.newStringUtf8((byte[])encoded);
        Assertions.assertEquals((Object)expectedResult, (Object)result, (String)"new Base64(65, \\t, false)");
    }

    @Test
    void testConstructor_Int_ByteArray_Boolean_UrlSafe() {
        Base64 base64 = new Base64(64, new byte[]{9}, true);
        byte[] encoded = base64.encode(BaseNTestData.DECODED);
        String expectedResult = "9IPNKwUvdLiIAp6ctz12SiQmOGstWyYvSPeevufDhrzaws65voykKjbIj33YWTa9\nxA7c/FHypWclrZhQ7onfc3JE93BJ5fT4R9zAEdjbjy1hv4ZYNnET4WJeXMLJ/5p+\nqBpTsPpepW8DNVYy1c02/1wyC+kgA6CvRUd9cSr/lt88AEdsTV4GMCn1+EwuAiYd\nivxuzn+cLM8q2jewqlI52tP9J7Cs8vqG71s6+WAELKvm/UovvyaOi+OdMUfjQ0JL\niLkHu6p9OwUgvQqiDKzEv/Augo0dTPZzYGEyCP5GVrle3QQdgciIHnpdd4VUTPGR\nUbXeKbh++U3fbJIng/sQXM3IYByMZ7xt9HWS1LUcRdQ7Prwn/IlQWxOMeq+KZJSo\nAviWtdserXyHbIEa//hmr4p/j80k0g9q35hq1ayGM9984ALTSaZ8WeyFbZx1CxC/\nQoqf92UH/ylBRnSJNn4sS0oa3uUbNvOnpkB4D9V7Ut9atinCJrw+wiJcMl+9kp25\n1IUxBGA4cUxh0eaxk3ODWnwI95EktmWOKwCSP0xjWwIMxDjygwAG5R8fk9H9bVi1\nthMavm4nDc4vaNoSE1RnZNYwbiUVlVPM9EclvJWTWd6igWeA0MxHAA8iOM5Vnmqp\n/WGM7UDq59rBIdNQCoeTJaAkEtAuLL5zogOa5e+MzVjvB5MYQlOlaaTtQrRApXa5\nZ4VfEanu9UK2fi1T8jJPFC2PmXebxp0bnO+VW+bgyEdIIkIQCaZq1MKWC3KuiOS9\nBJ1t7O0A2JKJKvoE4UNulzV2TGCC+KAnmjRqQBqXlJmgjHQAoHNZKOma/uIQOsvf\nDnqicYdDmfyCYuV89HjA1H8tiDJ85VfsrFHdcbPAoNCpi65awJSHfdPO1NDONOK+\n+S7Y0VXUgoYYrBV4Y7YbC8wg/nqcimr3lm3tRyp+QsgKzdREbfNRk0F5PLyLfsUE\nlepjs1QdV3fEV1LJtiywA3ubVNQJRxhbYxa/C/Xy2qxpm6vvdL92l3q1ccev35Ic\naOiSx7Im+/GxV2lVKdaOvYVGDD1zBRe6Y2CwQb9p088l3/93qGR5593NCiuPPWcs\nDWwUShM1EyW0FNX1F8bnzHnYijoyE/jf4s/l9bBd7yJdRWRCyih2WcypAiOIEkBs\nH+dCTgalu8sRDoMh4ZIBBdgHfoZUycLqReQFLZZ4Sl4zSmzt5vQxQFhEKb9+ff/4\nrb1KAo6wifengxVfIsa2b5ljXzAqXs7JkPvmC6fa7X4ZZndRokaxYlu3cg8OV+uG\n/6YAHZilo8at0OpkkNdNFuhwuGlkBqrZKNUj/gSiYYc06gF/r/z6iWAjpXJRW1qq\n3CLZXdZFZ/VrqXeVjtOAu2A=\n";
        expectedResult = expectedResult.replace("=", "");
        expectedResult = expectedResult.replace('\n', '\t');
        expectedResult = expectedResult.replace('+', '-');
        expectedResult = expectedResult.replace('/', '_');
        String result = StringUtils.newStringUtf8((byte[])encoded);
        Assertions.assertEquals((Object)result, (Object)expectedResult, (String)"new Base64(64, \\t, true)");
    }

    @Test
    void testConstructors() {
        Base64 base64 = new Base64();
        base64 = new Base64(-1);
        base64 = new Base64(-1, new byte[0]);
        base64 = new Base64(64, new byte[0]);
        base64 = new Base64(64, new byte[]{36});
        Assertions.assertThrows(IllegalArgumentException.class, () -> new Base64(-1, new byte[]{65}), (String)"Should have rejected attempt to use 'A' as a line separator");
        Assertions.assertThrows(IllegalArgumentException.class, () -> new Base64(64, new byte[]{65}), (String)"Should have rejected attempt to use 'A' as a line separator");
        Assertions.assertThrows(IllegalArgumentException.class, () -> new Base64(64, new byte[]{61}), (String)"Should have rejected attempt to use '=' as a line separator");
        base64 = new Base64(64, new byte[]{36});
        Assertions.assertThrows(IllegalArgumentException.class, () -> new Base64(64, new byte[]{65, 36}), (String)"Should have rejected attempt to use 'A$' as a line separator");
        base64 = new Base64(64, new byte[]{32, 36, 10, 13, 9});
    }

    @Test
    void testCustomEncodingAlphabet() {
        byte[] encodeTable = new byte[]{46, 45, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 48, 49, 50, 51, 52, 53, 54, 55, 56, 57, 43, 47};
        Base64 b64 = new Base64();
        Base64 b64customEncoding = Base64.builder().setEncodeTable(encodeTable).get();
        String content = "! Hello World - this \u00a7$%";
        byte[] encodedBytes = b64.encode(StringUtils.getBytesUtf8((String)"! Hello World - this \u00a7$%"));
        String encodedContent = StringUtils.newStringUtf8((byte[])encodedBytes);
        byte[] encodedBytesCustom = b64customEncoding.encode(StringUtils.getBytesUtf8((String)"! Hello World - this \u00a7$%"));
        String encodedContentCustom = StringUtils.newStringUtf8((byte[])encodedBytesCustom);
        Assertions.assertTrue((encodedContent.contains("A") && encodedContent.contains("B") ? 1 : 0) != 0, (String)"testing precondition not met - ecodedContent should contain parts of modified table");
        Assertions.assertEquals((Object)encodedContent.replace('A', '.').replace('B', '-').replace("=", ""), (Object)encodedContentCustom);
        byte[] decode = b64customEncoding.decode(encodedBytesCustom);
        String decodeString = StringUtils.newStringUtf8((byte[])decode);
        Assertions.assertEquals((Object)"! Hello World - this \u00a7$%", (Object)decodeString);
    }

    @Test
    void testCustomEncodingAlphabet_illegal() {
        byte[] encodeTable = new byte[]{46, 45, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77};
        Assertions.assertThrows(IllegalArgumentException.class, () -> Base64.builder().setEncodeTable(encodeTable).get());
    }

    private void testDecodeEncode(String encodedText) {
        String decodedText = StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)encodedText));
        String encodedText2 = Base64.encodeBase64String((byte[])StringUtils.getBytesUtf8((String)decodedText));
        Assertions.assertEquals((Object)encodedText, (Object)encodedText2);
    }

    @Test
    void testDecodePadMarkerIndex2() {
        Assertions.assertEquals((Object)"A", (Object)new String(Base64.decodeBase64((byte[])"QQ==".getBytes(CHARSET_UTF8))));
    }

    @Test
    void testDecodePadMarkerIndex3() {
        Assertions.assertEquals((Object)"AA", (Object)new String(Base64.decodeBase64((byte[])"QUE=".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"AAA", (Object)new String(Base64.decodeBase64((byte[])"QUFB".getBytes(CHARSET_UTF8))));
    }

    @Test
    void testDecodePadOnly() {
        Assertions.assertEquals((int)0, (int)Base64.decodeBase64((byte[])"====".getBytes(CHARSET_UTF8)).length);
        Assertions.assertEquals((Object)"", (Object)new String(Base64.decodeBase64((byte[])"====".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((int)0, (int)Base64.decodeBase64((byte[])"===".getBytes(CHARSET_UTF8)).length);
        Assertions.assertEquals((int)0, (int)Base64.decodeBase64((byte[])"==".getBytes(CHARSET_UTF8)).length);
        Assertions.assertEquals((int)0, (int)Base64.decodeBase64((byte[])"=".getBytes(CHARSET_UTF8)).length);
        Assertions.assertEquals((int)0, (int)Base64.decodeBase64((byte[])"".getBytes(CHARSET_UTF8)).length);
    }

    @Test
    void testDecodePadOnlyChunked() {
        Assertions.assertEquals((int)0, (int)Base64.decodeBase64((byte[])"====\n".getBytes(CHARSET_UTF8)).length);
        Assertions.assertEquals((Object)"", (Object)new String(Base64.decodeBase64((byte[])"====\n".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((int)0, (int)Base64.decodeBase64((byte[])"===\n".getBytes(CHARSET_UTF8)).length);
        Assertions.assertEquals((int)0, (int)Base64.decodeBase64((byte[])"==\n".getBytes(CHARSET_UTF8)).length);
        Assertions.assertEquals((int)0, (int)Base64.decodeBase64((byte[])"=\n".getBytes(CHARSET_UTF8)).length);
        Assertions.assertEquals((int)0, (int)Base64.decodeBase64((byte[])"\n".getBytes(CHARSET_UTF8)).length);
    }

    @Test
    void testDecodeWithInnerPad() {
        String content = "SGVsbG8gV29ybGQ=SGVsbG8gV29ybGQ=";
        byte[] result = Base64.decodeBase64((String)"SGVsbG8gV29ybGQ=SGVsbG8gV29ybGQ=");
        byte[] shouldBe = StringUtils.getBytesUtf8((String)"Hello World");
        Assertions.assertArrayEquals((byte[])result, (byte[])shouldBe, (String)"decode should halt at pad (=)");
    }

    @Test
    void testDecodeWithWhitespace() throws Exception {
        String orig = "I am a late night coder.";
        byte[] encodedArray = Base64.encodeBase64((byte[])"I am a late night coder.".getBytes(CHARSET_UTF8));
        StringBuilder intermediate = new StringBuilder(new String(encodedArray));
        intermediate.insert(2, ' ');
        intermediate.insert(5, '\t');
        intermediate.insert(10, '\r');
        intermediate.insert(15, '\n');
        byte[] encodedWithWS = intermediate.toString().getBytes(CHARSET_UTF8);
        byte[] decodedWithWS = Base64.decodeBase64((byte[])encodedWithWS);
        String dest = new String(decodedWithWS);
        Assertions.assertEquals((Object)"I am a late night coder.", (Object)dest, (String)"Dest string doesn't equal the original");
    }

    @Test
    void testEmptyBase64() {
        byte[] empty = new byte[]{};
        byte[] result = Base64.encodeBase64((byte[])empty);
        Assertions.assertEquals((int)0, (int)result.length, (String)"empty base64 encode");
        Assertions.assertNull((Object)Base64.encodeBase64(null), (String)"empty base64 encode");
        result = new Base64().encode(empty, 0, 1);
        Assertions.assertEquals((int)0, (int)result.length, (String)"empty base64 encode");
        Assertions.assertNull((Object)new Base64().encode(null, 0, 1), (String)"empty base64 encode");
        empty = new byte[]{};
        result = Base64.decodeBase64((byte[])empty);
        Assertions.assertEquals((int)0, (int)result.length, (String)"empty base64 decode");
        Assertions.assertNull((Object)Base64.decodeBase64((byte[])null), (String)"empty base64 encode");
    }

    private void testEncodeDecode(String plainText) {
        String encodedText = Base64.encodeBase64String((byte[])StringUtils.getBytesUtf8((String)plainText));
        String decodedText = StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)encodedText));
        Assertions.assertEquals((Object)plainText, (Object)decodedText);
    }

    @Test
    void testEncodeDecodeRandom() {
        for (int i = 1; i < 5; ++i) {
            byte[] data = new byte[this.getRandom().nextInt(10000) + 1];
            this.getRandom().nextBytes(data);
            byte[] enc = Base64.encodeBase64((byte[])data);
            Assertions.assertTrue((boolean)Base64.isBase64((byte[])enc));
            byte[] data2 = Base64.decodeBase64((byte[])enc);
            Assertions.assertArrayEquals((byte[])data, (byte[])data2);
        }
    }

    @Test
    void testEncodeDecodeSmall() {
        for (int i = 0; i < 12; ++i) {
            byte[] data = new byte[i];
            this.getRandom().nextBytes(data);
            byte[] enc = Base64.encodeBase64((byte[])data);
            Assertions.assertTrue((boolean)Base64.isBase64((byte[])enc), (String)("\"" + new String(enc) + "\" is Base64 data."));
            byte[] data2 = Base64.decodeBase64((byte[])enc);
            Assertions.assertArrayEquals((byte[])data, (byte[])data2, (String)(this.toString(data) + " equals " + this.toString(data2)));
        }
    }

    @Test
    void testEncodeOverMaxSize() throws Exception {
        this.testEncodeOverMaxSize(-1);
        this.testEncodeOverMaxSize(0);
        this.testEncodeOverMaxSize(1);
        this.testEncodeOverMaxSize(2);
    }

    private void testEncodeOverMaxSize(int maxSize) {
        Assertions.assertThrows(IllegalArgumentException.class, () -> Base64.encodeBase64((byte[])BaseNTestData.DECODED, (boolean)true, (boolean)false, (int)maxSize));
    }

    @Test
    void testHugeLineSeparator() {
        int BaseNCodec_DEFAULT_BUFFER_SIZE = 8192;
        int Base64_BYTES_PER_ENCODED_BLOCK = 4;
        byte[] baLineSeparator = new byte[32765];
        Base64 b64 = new Base64(4, baLineSeparator);
        String strOriginal = "Hello World";
        String strDecoded = new String(b64.decode(b64.encode(StringUtils.getBytesUtf8((String)"Hello World"))));
        Assertions.assertEquals((Object)"Hello World", (Object)strDecoded, (String)"testDEFAULT_BUFFER_SIZE");
    }

    @Test
    void testIgnoringNonBase64InDecode() throws Exception {
        Assertions.assertEquals((Object)FOX_TEXT, (Object)new String(Base64.decodeBase64((byte[])FOX_BASE64.getBytes(CHARSET_UTF8))));
    }

    @ParameterizedTest
    @MethodSource(value={"testIsBase64"})
    void testIsArrayByteBase64(byte[] arrayOctet, boolean match) {
        Assertions.assertEquals((Object)match, (Object)Base64.isArrayByteBase64((byte[])arrayOctet));
    }

    @ParameterizedTest
    @MethodSource
    void testIsBase64(byte[] arrayOctet, boolean match) {
        Assertions.assertEquals((Object)match, (Object)Base64.isBase64((byte[])arrayOctet));
    }

    @Test
    void testIsStringBase64() {
        String nullString = null;
        String emptyString = "";
        String validString = "abc===defg\n\r123456\r789\r\rABC\n\nDEF==GHI\r\nJKL==============";
        String invalidString = "abc===defg\n\r123456\r789\r\rABC\n\nDEF==GHI\r\nJKL==============\u0000";
        Assertions.assertThrows(NullPointerException.class, () -> Base64.isBase64((String)nullString), (String)"Base64.isStringBase64() should not be null-safe.");
        Assertions.assertTrue((boolean)Base64.isBase64((String)""), (String)"Base64.isStringBase64(empty-string) is true");
        Assertions.assertTrue((boolean)Base64.isBase64((String)"abc===defg\n\r123456\r789\r\rABC\n\nDEF==GHI\r\nJKL=============="), (String)"Base64.isStringBase64(valid-string) is true");
        Assertions.assertFalse((boolean)Base64.isBase64((String)"abc===defg\n\r123456\r789\r\rABC\n\nDEF==GHI\r\nJKL==============\u0000"), (String)"Base64.isStringBase64(invalid-string) is false");
    }

    @Test
    void testIsUrlSafe() {
        Base64 base64Standard = new Base64(false);
        Base64 base64URLSafe = new Base64(true);
        Assertions.assertFalse((boolean)base64Standard.isUrlSafe(), (String)"Base64.isUrlSafe=false");
        Assertions.assertTrue((boolean)base64URLSafe.isUrlSafe(), (String)"Base64.isUrlSafe=true");
        byte[] whiteSpace = new byte[]{32, 10, 13, 9};
        Assertions.assertTrue((boolean)Base64.isBase64((byte[])whiteSpace), (String)"Base64.isBase64(whiteSpace)=true");
    }

    @Test
    void testKnownDecodings() {
        Assertions.assertEquals((Object)FOX_TEXT, (Object)new String(Base64.decodeBase64((byte[])"VGhlIHF1aWNrIGJyb3duIGZveCBqdW1wZWQgb3ZlciB0aGUgbGF6eSBkb2dzLg==".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"It was the best of times, it was the worst of times.", (Object)new String(Base64.decodeBase64((byte[])"SXQgd2FzIHRoZSBiZXN0IG9mIHRpbWVzLCBpdCB3YXMgdGhlIHdvcnN0IG9mIHRpbWVzLg==".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"http://jakarta.apache.org/commmons", (Object)new String(Base64.decodeBase64((byte[])"aHR0cDovL2pha2FydGEuYXBhY2hlLm9yZy9jb21tbW9ucw==".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQqRrSsTtUuVvWwXxYyZz", (Object)new String(Base64.decodeBase64((byte[])"QWFCYkNjRGRFZUZmR2dIaElpSmpLa0xsTW1Obk9vUHBRcVJyU3NUdFV1VnZXd1h4WXlaeg==".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 }", (Object)new String(Base64.decodeBase64((byte[])"eyAwLCAxLCAyLCAzLCA0LCA1LCA2LCA3LCA4LCA5IH0=".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"xyzzy!", (Object)new String(Base64.decodeBase64((byte[])"eHl6enkh".getBytes(CHARSET_UTF8))));
    }

    @Test
    void testKnownEncodings() {
        Assertions.assertEquals((Object)"VGhlIHF1aWNrIGJyb3duIGZveCBqdW1wZWQgb3ZlciB0aGUgbGF6eSBkb2dzLg==", (Object)new String(Base64.encodeBase64((byte[])FOX_TEXT.getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"YmxhaCBibGFoIGJsYWggYmxhaCBibGFoIGJsYWggYmxhaCBibGFoIGJsYWggYmxhaCBibGFoIGJs\r\nYWggYmxhaCBibGFoIGJsYWggYmxhaCBibGFoIGJsYWggYmxhaCBibGFoIGJsYWggYmxhaCBibGFo\r\nIGJsYWggYmxhaCBibGFoIGJsYWggYmxhaCBibGFoIGJsYWggYmxhaCBibGFoIGJsYWggYmxhaCBi\r\nbGFoIGJsYWg=\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])"blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah blah".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"SXQgd2FzIHRoZSBiZXN0IG9mIHRpbWVzLCBpdCB3YXMgdGhlIHdvcnN0IG9mIHRpbWVzLg==", (Object)new String(Base64.encodeBase64((byte[])"It was the best of times, it was the worst of times.".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"aHR0cDovL2pha2FydGEuYXBhY2hlLm9yZy9jb21tbW9ucw==", (Object)new String(Base64.encodeBase64((byte[])"http://jakarta.apache.org/commmons".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"QWFCYkNjRGRFZUZmR2dIaElpSmpLa0xsTW1Obk9vUHBRcVJyU3NUdFV1VnZXd1h4WXlaeg==", (Object)new String(Base64.encodeBase64((byte[])"AaBbCcDdEeFfGgHhIiJjKkLlMmNnOoPpQqRrSsTtUuVvWwXxYyZz".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"eyAwLCAxLCAyLCAzLCA0LCA1LCA2LCA3LCA4LCA5IH0=", (Object)new String(Base64.encodeBase64((byte[])"{ 0, 1, 2, 3, 4, 5, 6, 7, 8, 9 }".getBytes(CHARSET_UTF8))));
        Assertions.assertEquals((Object)"eHl6enkh", (Object)new String(Base64.encodeBase64((byte[])"xyzzy!".getBytes(CHARSET_UTF8))));
    }

    @Test
    void testNonBase64Test() throws Exception {
        byte[] bArray = new byte[]{37};
        Assertions.assertFalse((boolean)Base64.isBase64((byte[])bArray), (String)"Invalid Base64 array was incorrectly validated as an array of Base64 encoded data");
        try {
            Base64 b64 = new Base64();
            byte[] result = b64.decode(bArray);
            Assertions.assertEquals((int)0, (int)result.length, (String)"The result should be empty as the test encoded content did not contain any valid base 64 characters");
        }
        catch (Exception e) {
            Assertions.fail((String)("Exception '" + e.getClass().getName() + "' was thrown when trying to decode invalid base64 encoded data - RFC 2045 requires that all non base64 character be discarded, an exception should not have been thrown"));
        }
    }

    @Test
    void testObjectDecodeWithInvalidParameter() {
        Assertions.assertThrows(DecoderException.class, () -> new Base64().decode((Object)5), (String)"decode(Object) didn't throw an exception when passed an Integer object");
    }

    @Test
    void testObjectDecodeWithValidParameter() throws Exception {
        String original = "Hello World!";
        byte[] o = Base64.encodeBase64((byte[])"Hello World!".getBytes(CHARSET_UTF8));
        Base64 b64 = new Base64();
        Object oDecoded = b64.decode((Object)o);
        byte[] baDecoded = (byte[])oDecoded;
        String dest = new String(baDecoded);
        Assertions.assertEquals((Object)"Hello World!", (Object)dest, (String)"dest string does not equal original");
    }

    @Test
    void testObjectEncode() throws Exception {
        Base64 b64 = new Base64();
        Assertions.assertEquals((Object)"SGVsbG8gV29ybGQ=", (Object)new String(b64.encode("Hello World".getBytes(CHARSET_UTF8))));
    }

    @Test
    void testObjectEncodeWithInvalidParameter() {
        Assertions.assertThrows(EncoderException.class, () -> new Base64().encode((Object)"Yadayadayada"), (String)"encode(Object) didn't throw an exception when passed a String object");
    }

    @Test
    void testObjectEncodeWithValidParameter() throws Exception {
        String original = "Hello World!";
        byte[] origObj = "Hello World!".getBytes(CHARSET_UTF8);
        Base64 b64 = new Base64();
        Object oEncoded = b64.encode((Object)origObj);
        byte[] bArray = Base64.decodeBase64((byte[])((byte[])oEncoded));
        String dest = new String(bArray);
        Assertions.assertEquals((Object)"Hello World!", (Object)dest, (String)"dest string does not equal original");
    }

    @Test
    void testPairs() {
        Assertions.assertEquals((Object)"AAA=", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0})));
        for (int i = -128; i <= 127; ++i) {
            byte[] test = new byte[]{(byte)i, (byte)i};
            Assertions.assertArrayEquals((byte[])test, (byte[])Base64.decodeBase64((byte[])Base64.encodeBase64((byte[])test)));
        }
    }

    @Test
    void testRfc1421Section6Dot8ChunkSizeDefinition() {
        Assertions.assertEquals((int)64, (int)64);
    }

    @Test
    void testRfc2045Section2Dot1CrLfDefinition() {
        Assertions.assertArrayEquals((byte[])new byte[]{13, 10}, (byte[])BaseNCodec.CHUNK_SEPARATOR);
    }

    @Test
    void testRfc2045Section6Dot8ChunkSizeDefinition() {
        Assertions.assertEquals((int)76, (int)76);
    }

    @Test
    void testRfc4648Section10Decode() {
        Assertions.assertEquals((Object)"", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)"")));
        Assertions.assertEquals((Object)"f", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)"Zg==")));
        Assertions.assertEquals((Object)"fo", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)"Zm8=")));
        Assertions.assertEquals((Object)"foo", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)"Zm9v")));
        Assertions.assertEquals((Object)"foob", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)"Zm9vYg==")));
        Assertions.assertEquals((Object)"fooba", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)"Zm9vYmE=")));
        Assertions.assertEquals((Object)"foobar", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)"Zm9vYmFy")));
    }

    @ParameterizedTest
    @ValueSource(strings={"", "Zg==", "Zm8=", "Zm9v", "Zm9vYg==", "Zm9vYmE=", "Zm9vYmFy"})
    void testRfc4648Section10DecodeEncode(String input) {
        this.testDecodeEncode(input);
    }

    @Test
    void testRfc4648Section10DecodeWithCrLf() {
        String CRLF = StringUtils.newStringUsAscii((byte[])BaseNCodec.CHUNK_SEPARATOR);
        Assertions.assertEquals((Object)"", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)("" + CRLF))));
        Assertions.assertEquals((Object)"f", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)("Zg==" + CRLF))));
        Assertions.assertEquals((Object)"fo", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)("Zm8=" + CRLF))));
        Assertions.assertEquals((Object)"foo", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)("Zm9v" + CRLF))));
        Assertions.assertEquals((Object)"foob", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)("Zm9vYg==" + CRLF))));
        Assertions.assertEquals((Object)"fooba", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)("Zm9vYmE=" + CRLF))));
        Assertions.assertEquals((Object)"foobar", (Object)StringUtils.newStringUsAscii((byte[])Base64.decodeBase64((String)("Zm9vYmFy" + CRLF))));
    }

    @Test
    void testRfc4648Section10Encode() {
        Assertions.assertEquals((Object)"", (Object)Base64.encodeBase64String((byte[])StringUtils.getBytesUtf8((String)"")));
        Assertions.assertEquals((Object)"Zg==", (Object)Base64.encodeBase64String((byte[])StringUtils.getBytesUtf8((String)"f")));
        Assertions.assertEquals((Object)"Zm8=", (Object)Base64.encodeBase64String((byte[])StringUtils.getBytesUtf8((String)"fo")));
        Assertions.assertEquals((Object)"Zm9v", (Object)Base64.encodeBase64String((byte[])StringUtils.getBytesUtf8((String)"foo")));
        Assertions.assertEquals((Object)"Zm9vYg==", (Object)Base64.encodeBase64String((byte[])StringUtils.getBytesUtf8((String)"foob")));
        Assertions.assertEquals((Object)"Zm9vYmE=", (Object)Base64.encodeBase64String((byte[])StringUtils.getBytesUtf8((String)"fooba")));
        Assertions.assertEquals((Object)"Zm9vYmFy", (Object)Base64.encodeBase64String((byte[])StringUtils.getBytesUtf8((String)"foobar")));
    }

    @ParameterizedTest
    @ValueSource(strings={"", "f", "fo", "foo", "foob", "fooba", "foobar"})
    void testRfc4648Section10EncodeDecode(String input) {
        this.testEncodeDecode(input);
    }

    @Test
    void testSingletons() {
        Assertions.assertEquals((Object)"AA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0})));
        Assertions.assertEquals((Object)"AQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{1})));
        Assertions.assertEquals((Object)"Ag==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{2})));
        Assertions.assertEquals((Object)"Aw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{3})));
        Assertions.assertEquals((Object)"BA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{4})));
        Assertions.assertEquals((Object)"BQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{5})));
        Assertions.assertEquals((Object)"Bg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{6})));
        Assertions.assertEquals((Object)"Bw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{7})));
        Assertions.assertEquals((Object)"CA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{8})));
        Assertions.assertEquals((Object)"CQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{9})));
        Assertions.assertEquals((Object)"Cg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{10})));
        Assertions.assertEquals((Object)"Cw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{11})));
        Assertions.assertEquals((Object)"DA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{12})));
        Assertions.assertEquals((Object)"DQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{13})));
        Assertions.assertEquals((Object)"Dg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{14})));
        Assertions.assertEquals((Object)"Dw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{15})));
        Assertions.assertEquals((Object)"EA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{16})));
        Assertions.assertEquals((Object)"EQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{17})));
        Assertions.assertEquals((Object)"Eg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{18})));
        Assertions.assertEquals((Object)"Ew==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{19})));
        Assertions.assertEquals((Object)"FA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{20})));
        Assertions.assertEquals((Object)"FQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{21})));
        Assertions.assertEquals((Object)"Fg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{22})));
        Assertions.assertEquals((Object)"Fw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{23})));
        Assertions.assertEquals((Object)"GA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{24})));
        Assertions.assertEquals((Object)"GQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{25})));
        Assertions.assertEquals((Object)"Gg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{26})));
        Assertions.assertEquals((Object)"Gw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{27})));
        Assertions.assertEquals((Object)"HA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{28})));
        Assertions.assertEquals((Object)"HQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{29})));
        Assertions.assertEquals((Object)"Hg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{30})));
        Assertions.assertEquals((Object)"Hw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{31})));
        Assertions.assertEquals((Object)"IA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{32})));
        Assertions.assertEquals((Object)"IQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{33})));
        Assertions.assertEquals((Object)"Ig==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{34})));
        Assertions.assertEquals((Object)"Iw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{35})));
        Assertions.assertEquals((Object)"JA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{36})));
        Assertions.assertEquals((Object)"JQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{37})));
        Assertions.assertEquals((Object)"Jg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{38})));
        Assertions.assertEquals((Object)"Jw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{39})));
        Assertions.assertEquals((Object)"KA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{40})));
        Assertions.assertEquals((Object)"KQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{41})));
        Assertions.assertEquals((Object)"Kg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{42})));
        Assertions.assertEquals((Object)"Kw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{43})));
        Assertions.assertEquals((Object)"LA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{44})));
        Assertions.assertEquals((Object)"LQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{45})));
        Assertions.assertEquals((Object)"Lg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{46})));
        Assertions.assertEquals((Object)"Lw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{47})));
        Assertions.assertEquals((Object)"MA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{48})));
        Assertions.assertEquals((Object)"MQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{49})));
        Assertions.assertEquals((Object)"Mg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{50})));
        Assertions.assertEquals((Object)"Mw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{51})));
        Assertions.assertEquals((Object)"NA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{52})));
        Assertions.assertEquals((Object)"NQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{53})));
        Assertions.assertEquals((Object)"Ng==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{54})));
        Assertions.assertEquals((Object)"Nw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{55})));
        Assertions.assertEquals((Object)"OA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{56})));
        Assertions.assertEquals((Object)"OQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{57})));
        Assertions.assertEquals((Object)"Og==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{58})));
        Assertions.assertEquals((Object)"Ow==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{59})));
        Assertions.assertEquals((Object)"PA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{60})));
        Assertions.assertEquals((Object)"PQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{61})));
        Assertions.assertEquals((Object)"Pg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{62})));
        Assertions.assertEquals((Object)"Pw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{63})));
        Assertions.assertEquals((Object)"QA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{64})));
        Assertions.assertEquals((Object)"QQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{65})));
        Assertions.assertEquals((Object)"Qg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{66})));
        Assertions.assertEquals((Object)"Qw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{67})));
        Assertions.assertEquals((Object)"RA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{68})));
        Assertions.assertEquals((Object)"RQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{69})));
        Assertions.assertEquals((Object)"Rg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{70})));
        Assertions.assertEquals((Object)"Rw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{71})));
        Assertions.assertEquals((Object)"SA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{72})));
        Assertions.assertEquals((Object)"SQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{73})));
        Assertions.assertEquals((Object)"Sg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{74})));
        Assertions.assertEquals((Object)"Sw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{75})));
        Assertions.assertEquals((Object)"TA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{76})));
        Assertions.assertEquals((Object)"TQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{77})));
        Assertions.assertEquals((Object)"Tg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{78})));
        Assertions.assertEquals((Object)"Tw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{79})));
        Assertions.assertEquals((Object)"UA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{80})));
        Assertions.assertEquals((Object)"UQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{81})));
        Assertions.assertEquals((Object)"Ug==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{82})));
        Assertions.assertEquals((Object)"Uw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{83})));
        Assertions.assertEquals((Object)"VA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{84})));
        Assertions.assertEquals((Object)"VQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{85})));
        Assertions.assertEquals((Object)"Vg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{86})));
        Assertions.assertEquals((Object)"Vw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{87})));
        Assertions.assertEquals((Object)"WA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{88})));
        Assertions.assertEquals((Object)"WQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{89})));
        Assertions.assertEquals((Object)"Wg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{90})));
        Assertions.assertEquals((Object)"Ww==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{91})));
        Assertions.assertEquals((Object)"XA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{92})));
        Assertions.assertEquals((Object)"XQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{93})));
        Assertions.assertEquals((Object)"Xg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{94})));
        Assertions.assertEquals((Object)"Xw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{95})));
        Assertions.assertEquals((Object)"YA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{96})));
        Assertions.assertEquals((Object)"YQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{97})));
        Assertions.assertEquals((Object)"Yg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{98})));
        Assertions.assertEquals((Object)"Yw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{99})));
        Assertions.assertEquals((Object)"ZA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{100})));
        Assertions.assertEquals((Object)"ZQ==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{101})));
        Assertions.assertEquals((Object)"Zg==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{102})));
        Assertions.assertEquals((Object)"Zw==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{103})));
        Assertions.assertEquals((Object)"aA==", (Object)new String(Base64.encodeBase64((byte[])new byte[]{104})));
        for (int i = -128; i <= 127; ++i) {
            byte[] test = new byte[]{(byte)i};
            Assertions.assertArrayEquals((byte[])test, (byte[])Base64.decodeBase64((byte[])Base64.encodeBase64((byte[])test)));
        }
    }

    @Test
    void testSingletonsChunked() {
        Assertions.assertEquals((Object)"AA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0})));
        Assertions.assertEquals((Object)"AQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{1})));
        Assertions.assertEquals((Object)"Ag==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{2})));
        Assertions.assertEquals((Object)"Aw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{3})));
        Assertions.assertEquals((Object)"BA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{4})));
        Assertions.assertEquals((Object)"BQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{5})));
        Assertions.assertEquals((Object)"Bg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{6})));
        Assertions.assertEquals((Object)"Bw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{7})));
        Assertions.assertEquals((Object)"CA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{8})));
        Assertions.assertEquals((Object)"CQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{9})));
        Assertions.assertEquals((Object)"Cg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{10})));
        Assertions.assertEquals((Object)"Cw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{11})));
        Assertions.assertEquals((Object)"DA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{12})));
        Assertions.assertEquals((Object)"DQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{13})));
        Assertions.assertEquals((Object)"Dg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{14})));
        Assertions.assertEquals((Object)"Dw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{15})));
        Assertions.assertEquals((Object)"EA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{16})));
        Assertions.assertEquals((Object)"EQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{17})));
        Assertions.assertEquals((Object)"Eg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{18})));
        Assertions.assertEquals((Object)"Ew==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{19})));
        Assertions.assertEquals((Object)"FA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{20})));
        Assertions.assertEquals((Object)"FQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{21})));
        Assertions.assertEquals((Object)"Fg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{22})));
        Assertions.assertEquals((Object)"Fw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{23})));
        Assertions.assertEquals((Object)"GA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{24})));
        Assertions.assertEquals((Object)"GQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{25})));
        Assertions.assertEquals((Object)"Gg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{26})));
        Assertions.assertEquals((Object)"Gw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{27})));
        Assertions.assertEquals((Object)"HA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{28})));
        Assertions.assertEquals((Object)"HQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{29})));
        Assertions.assertEquals((Object)"Hg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{30})));
        Assertions.assertEquals((Object)"Hw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{31})));
        Assertions.assertEquals((Object)"IA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{32})));
        Assertions.assertEquals((Object)"IQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{33})));
        Assertions.assertEquals((Object)"Ig==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{34})));
        Assertions.assertEquals((Object)"Iw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{35})));
        Assertions.assertEquals((Object)"JA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{36})));
        Assertions.assertEquals((Object)"JQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{37})));
        Assertions.assertEquals((Object)"Jg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{38})));
        Assertions.assertEquals((Object)"Jw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{39})));
        Assertions.assertEquals((Object)"KA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{40})));
        Assertions.assertEquals((Object)"KQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{41})));
        Assertions.assertEquals((Object)"Kg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{42})));
        Assertions.assertEquals((Object)"Kw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{43})));
        Assertions.assertEquals((Object)"LA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{44})));
        Assertions.assertEquals((Object)"LQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{45})));
        Assertions.assertEquals((Object)"Lg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{46})));
        Assertions.assertEquals((Object)"Lw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{47})));
        Assertions.assertEquals((Object)"MA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{48})));
        Assertions.assertEquals((Object)"MQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{49})));
        Assertions.assertEquals((Object)"Mg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{50})));
        Assertions.assertEquals((Object)"Mw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{51})));
        Assertions.assertEquals((Object)"NA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{52})));
        Assertions.assertEquals((Object)"NQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{53})));
        Assertions.assertEquals((Object)"Ng==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{54})));
        Assertions.assertEquals((Object)"Nw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{55})));
        Assertions.assertEquals((Object)"OA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{56})));
        Assertions.assertEquals((Object)"OQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{57})));
        Assertions.assertEquals((Object)"Og==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{58})));
        Assertions.assertEquals((Object)"Ow==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{59})));
        Assertions.assertEquals((Object)"PA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{60})));
        Assertions.assertEquals((Object)"PQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{61})));
        Assertions.assertEquals((Object)"Pg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{62})));
        Assertions.assertEquals((Object)"Pw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{63})));
        Assertions.assertEquals((Object)"QA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{64})));
        Assertions.assertEquals((Object)"QQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{65})));
        Assertions.assertEquals((Object)"Qg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{66})));
        Assertions.assertEquals((Object)"Qw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{67})));
        Assertions.assertEquals((Object)"RA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{68})));
        Assertions.assertEquals((Object)"RQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{69})));
        Assertions.assertEquals((Object)"Rg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{70})));
        Assertions.assertEquals((Object)"Rw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{71})));
        Assertions.assertEquals((Object)"SA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{72})));
        Assertions.assertEquals((Object)"SQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{73})));
        Assertions.assertEquals((Object)"Sg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{74})));
        Assertions.assertEquals((Object)"Sw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{75})));
        Assertions.assertEquals((Object)"TA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{76})));
        Assertions.assertEquals((Object)"TQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{77})));
        Assertions.assertEquals((Object)"Tg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{78})));
        Assertions.assertEquals((Object)"Tw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{79})));
        Assertions.assertEquals((Object)"UA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{80})));
        Assertions.assertEquals((Object)"UQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{81})));
        Assertions.assertEquals((Object)"Ug==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{82})));
        Assertions.assertEquals((Object)"Uw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{83})));
        Assertions.assertEquals((Object)"VA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{84})));
        Assertions.assertEquals((Object)"VQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{85})));
        Assertions.assertEquals((Object)"Vg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{86})));
        Assertions.assertEquals((Object)"Vw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{87})));
        Assertions.assertEquals((Object)"WA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{88})));
        Assertions.assertEquals((Object)"WQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{89})));
        Assertions.assertEquals((Object)"Wg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{90})));
        Assertions.assertEquals((Object)"Ww==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{91})));
        Assertions.assertEquals((Object)"XA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{92})));
        Assertions.assertEquals((Object)"XQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{93})));
        Assertions.assertEquals((Object)"Xg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{94})));
        Assertions.assertEquals((Object)"Xw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{95})));
        Assertions.assertEquals((Object)"YA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{96})));
        Assertions.assertEquals((Object)"YQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{97})));
        Assertions.assertEquals((Object)"Yg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{98})));
        Assertions.assertEquals((Object)"Yw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{99})));
        Assertions.assertEquals((Object)"ZA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{100})));
        Assertions.assertEquals((Object)"ZQ==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{101})));
        Assertions.assertEquals((Object)"Zg==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{102})));
        Assertions.assertEquals((Object)"Zw==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{103})));
        Assertions.assertEquals((Object)"aA==\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{104})));
    }

    @Test
    void testStringToByteVariations() throws DecoderException {
        Base64 base64 = new Base64();
        String s1 = "SGVsbG8gV29ybGQ=\r\n";
        String s2 = "";
        String s3 = null;
        String s4a = "K/fMJwH+Q5e0nr7tWsxwkA==\r\n";
        String s4b = "K_fMJwH-Q5e0nr7tWsxwkA";
        byte[] b4 = Hex.decodeHex((String)"2bf7cc2701fe4397b49ebeed5acc7090");
        Assertions.assertEquals((Object)"Hello World", (Object)StringUtils.newStringUtf8((byte[])base64.decode("SGVsbG8gV29ybGQ=\r\n")), (String)"StringToByte Hello World");
        Assertions.assertEquals((Object)"Hello World", (Object)StringUtils.newStringUtf8((byte[])((byte[])base64.decode((Object)"SGVsbG8gV29ybGQ=\r\n"))), (String)"StringToByte Hello World");
        Assertions.assertEquals((Object)"Hello World", (Object)StringUtils.newStringUtf8((byte[])Base64.decodeBase64((String)"SGVsbG8gV29ybGQ=\r\n")), (String)"StringToByte static Hello World");
        Assertions.assertEquals((Object)"", (Object)StringUtils.newStringUtf8((byte[])base64.decode("")), (String)"StringToByte \"\"");
        Assertions.assertEquals((Object)"", (Object)StringUtils.newStringUtf8((byte[])Base64.decodeBase64((String)"")), (String)"StringToByte static \"\"");
        Assertions.assertNull((Object)StringUtils.newStringUtf8((byte[])base64.decode(s3)), (String)"StringToByte null");
        Assertions.assertNull((Object)StringUtils.newStringUtf8((byte[])Base64.decodeBase64(s3)), (String)"StringToByte static null");
        Assertions.assertArrayEquals((byte[])b4, (byte[])base64.decode("K_fMJwH-Q5e0nr7tWsxwkA"), (String)"StringToByte UUID");
        Assertions.assertArrayEquals((byte[])b4, (byte[])Base64.decodeBase64((String)"K/fMJwH+Q5e0nr7tWsxwkA==\r\n"), (String)"StringToByte static UUID");
        Assertions.assertArrayEquals((byte[])b4, (byte[])Base64.decodeBase64((String)"K_fMJwH-Q5e0nr7tWsxwkA"), (String)"StringToByte static-url-safe UUID");
    }

    @Test
    void testTriplets() {
        Assertions.assertEquals((Object)"AAAA", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 0})));
        Assertions.assertEquals((Object)"AAAB", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 1})));
        Assertions.assertEquals((Object)"AAAC", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 2})));
        Assertions.assertEquals((Object)"AAAD", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 3})));
        Assertions.assertEquals((Object)"AAAE", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 4})));
        Assertions.assertEquals((Object)"AAAF", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 5})));
        Assertions.assertEquals((Object)"AAAG", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 6})));
        Assertions.assertEquals((Object)"AAAH", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 7})));
        Assertions.assertEquals((Object)"AAAI", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 8})));
        Assertions.assertEquals((Object)"AAAJ", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 9})));
        Assertions.assertEquals((Object)"AAAK", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 10})));
        Assertions.assertEquals((Object)"AAAL", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 11})));
        Assertions.assertEquals((Object)"AAAM", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 12})));
        Assertions.assertEquals((Object)"AAAN", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 13})));
        Assertions.assertEquals((Object)"AAAO", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 14})));
        Assertions.assertEquals((Object)"AAAP", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 15})));
        Assertions.assertEquals((Object)"AAAQ", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 16})));
        Assertions.assertEquals((Object)"AAAR", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 17})));
        Assertions.assertEquals((Object)"AAAS", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 18})));
        Assertions.assertEquals((Object)"AAAT", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 19})));
        Assertions.assertEquals((Object)"AAAU", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 20})));
        Assertions.assertEquals((Object)"AAAV", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 21})));
        Assertions.assertEquals((Object)"AAAW", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 22})));
        Assertions.assertEquals((Object)"AAAX", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 23})));
        Assertions.assertEquals((Object)"AAAY", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 24})));
        Assertions.assertEquals((Object)"AAAZ", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 25})));
        Assertions.assertEquals((Object)"AAAa", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 26})));
        Assertions.assertEquals((Object)"AAAb", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 27})));
        Assertions.assertEquals((Object)"AAAc", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 28})));
        Assertions.assertEquals((Object)"AAAd", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 29})));
        Assertions.assertEquals((Object)"AAAe", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 30})));
        Assertions.assertEquals((Object)"AAAf", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 31})));
        Assertions.assertEquals((Object)"AAAg", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 32})));
        Assertions.assertEquals((Object)"AAAh", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 33})));
        Assertions.assertEquals((Object)"AAAi", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 34})));
        Assertions.assertEquals((Object)"AAAj", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 35})));
        Assertions.assertEquals((Object)"AAAk", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 36})));
        Assertions.assertEquals((Object)"AAAl", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 37})));
        Assertions.assertEquals((Object)"AAAm", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 38})));
        Assertions.assertEquals((Object)"AAAn", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 39})));
        Assertions.assertEquals((Object)"AAAo", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 40})));
        Assertions.assertEquals((Object)"AAAp", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 41})));
        Assertions.assertEquals((Object)"AAAq", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 42})));
        Assertions.assertEquals((Object)"AAAr", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 43})));
        Assertions.assertEquals((Object)"AAAs", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 44})));
        Assertions.assertEquals((Object)"AAAt", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 45})));
        Assertions.assertEquals((Object)"AAAu", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 46})));
        Assertions.assertEquals((Object)"AAAv", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 47})));
        Assertions.assertEquals((Object)"AAAw", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 48})));
        Assertions.assertEquals((Object)"AAAx", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 49})));
        Assertions.assertEquals((Object)"AAAy", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 50})));
        Assertions.assertEquals((Object)"AAAz", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 51})));
        Assertions.assertEquals((Object)"AAA0", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 52})));
        Assertions.assertEquals((Object)"AAA1", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 53})));
        Assertions.assertEquals((Object)"AAA2", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 54})));
        Assertions.assertEquals((Object)"AAA3", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 55})));
        Assertions.assertEquals((Object)"AAA4", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 56})));
        Assertions.assertEquals((Object)"AAA5", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 57})));
        Assertions.assertEquals((Object)"AAA6", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 58})));
        Assertions.assertEquals((Object)"AAA7", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 59})));
        Assertions.assertEquals((Object)"AAA8", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 60})));
        Assertions.assertEquals((Object)"AAA9", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 61})));
        Assertions.assertEquals((Object)"AAA+", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 62})));
        Assertions.assertEquals((Object)"AAA/", (Object)new String(Base64.encodeBase64((byte[])new byte[]{0, 0, 63})));
    }

    @Test
    void testTripletsChunked() {
        Assertions.assertEquals((Object)"AAAA\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 0})));
        Assertions.assertEquals((Object)"AAAB\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 1})));
        Assertions.assertEquals((Object)"AAAC\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 2})));
        Assertions.assertEquals((Object)"AAAD\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 3})));
        Assertions.assertEquals((Object)"AAAE\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 4})));
        Assertions.assertEquals((Object)"AAAF\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 5})));
        Assertions.assertEquals((Object)"AAAG\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 6})));
        Assertions.assertEquals((Object)"AAAH\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 7})));
        Assertions.assertEquals((Object)"AAAI\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 8})));
        Assertions.assertEquals((Object)"AAAJ\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 9})));
        Assertions.assertEquals((Object)"AAAK\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 10})));
        Assertions.assertEquals((Object)"AAAL\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 11})));
        Assertions.assertEquals((Object)"AAAM\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 12})));
        Assertions.assertEquals((Object)"AAAN\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 13})));
        Assertions.assertEquals((Object)"AAAO\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 14})));
        Assertions.assertEquals((Object)"AAAP\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 15})));
        Assertions.assertEquals((Object)"AAAQ\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 16})));
        Assertions.assertEquals((Object)"AAAR\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 17})));
        Assertions.assertEquals((Object)"AAAS\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 18})));
        Assertions.assertEquals((Object)"AAAT\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 19})));
        Assertions.assertEquals((Object)"AAAU\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 20})));
        Assertions.assertEquals((Object)"AAAV\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 21})));
        Assertions.assertEquals((Object)"AAAW\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 22})));
        Assertions.assertEquals((Object)"AAAX\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 23})));
        Assertions.assertEquals((Object)"AAAY\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 24})));
        Assertions.assertEquals((Object)"AAAZ\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 25})));
        Assertions.assertEquals((Object)"AAAa\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 26})));
        Assertions.assertEquals((Object)"AAAb\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 27})));
        Assertions.assertEquals((Object)"AAAc\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 28})));
        Assertions.assertEquals((Object)"AAAd\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 29})));
        Assertions.assertEquals((Object)"AAAe\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 30})));
        Assertions.assertEquals((Object)"AAAf\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 31})));
        Assertions.assertEquals((Object)"AAAg\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 32})));
        Assertions.assertEquals((Object)"AAAh\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 33})));
        Assertions.assertEquals((Object)"AAAi\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 34})));
        Assertions.assertEquals((Object)"AAAj\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 35})));
        Assertions.assertEquals((Object)"AAAk\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 36})));
        Assertions.assertEquals((Object)"AAAl\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 37})));
        Assertions.assertEquals((Object)"AAAm\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 38})));
        Assertions.assertEquals((Object)"AAAn\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 39})));
        Assertions.assertEquals((Object)"AAAo\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 40})));
        Assertions.assertEquals((Object)"AAAp\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 41})));
        Assertions.assertEquals((Object)"AAAq\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 42})));
        Assertions.assertEquals((Object)"AAAr\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 43})));
        Assertions.assertEquals((Object)"AAAs\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 44})));
        Assertions.assertEquals((Object)"AAAt\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 45})));
        Assertions.assertEquals((Object)"AAAu\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 46})));
        Assertions.assertEquals((Object)"AAAv\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 47})));
        Assertions.assertEquals((Object)"AAAw\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 48})));
        Assertions.assertEquals((Object)"AAAx\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 49})));
        Assertions.assertEquals((Object)"AAAy\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 50})));
        Assertions.assertEquals((Object)"AAAz\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 51})));
        Assertions.assertEquals((Object)"AAA0\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 52})));
        Assertions.assertEquals((Object)"AAA1\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 53})));
        Assertions.assertEquals((Object)"AAA2\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 54})));
        Assertions.assertEquals((Object)"AAA3\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 55})));
        Assertions.assertEquals((Object)"AAA4\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 56})));
        Assertions.assertEquals((Object)"AAA5\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 57})));
        Assertions.assertEquals((Object)"AAA6\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 58})));
        Assertions.assertEquals((Object)"AAA7\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 59})));
        Assertions.assertEquals((Object)"AAA8\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 60})));
        Assertions.assertEquals((Object)"AAA9\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 61})));
        Assertions.assertEquals((Object)"AAA+\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 62})));
        Assertions.assertEquals((Object)"AAA/\r\n", (Object)new String(Base64.encodeBase64Chunked((byte[])new byte[]{0, 0, 63})));
    }

    @Test
    void testUrlSafe() {
        Base64 codec = new Base64(true);
        for (int i = 0; i <= 150; ++i) {
            byte[][] randomData = BaseNTestData.randomData((BaseNCodec)codec, i);
            byte[] encoded = randomData[1];
            byte[] decoded = randomData[0];
            byte[] result = Base64.decodeBase64((byte[])encoded);
            Assertions.assertArrayEquals((byte[])decoded, (byte[])result, (String)("url-safe i=" + i));
            Assertions.assertFalse((boolean)BaseNTestData.bytesContain(encoded, (byte)61), (String)("url-safe i=" + i + " no '='"));
            Assertions.assertFalse((boolean)BaseNTestData.bytesContain(encoded, (byte)92), (String)("url-safe i=" + i + " no '\\'"));
            Assertions.assertFalse((boolean)BaseNTestData.bytesContain(encoded, (byte)43), (String)("url-safe i=" + i + " no '+'"));
        }
    }

    @Test
    void testUUID() throws DecoderException {
        byte[][] ids = new byte[][]{Hex.decodeHex((String)"94ed8d0319e4493399560fb67404d370"), Hex.decodeHex((String)"2bf7cc2701fe4397b49ebeed5acc7090"), Hex.decodeHex((String)"64be154b6ffa40258d1a01288e7c31ca"), Hex.decodeHex((String)"ff7f8fc01cdb471a8c8b5a9306183fe8")};
        byte[][] standard = new byte[][]{StringUtils.getBytesUtf8((String)"lO2NAxnkSTOZVg+2dATTcA=="), StringUtils.getBytesUtf8((String)"K/fMJwH+Q5e0nr7tWsxwkA=="), StringUtils.getBytesUtf8((String)"ZL4VS2/6QCWNGgEojnwxyg=="), StringUtils.getBytesUtf8((String)"/3+PwBzbRxqMi1qTBhg/6A==")};
        byte[][] urlSafe1 = new byte[][]{StringUtils.getBytesUtf8((String)"lO2NAxnkSTOZVg-2dATTcA=="), StringUtils.getBytesUtf8((String)"K_fMJwH-Q5e0nr7tWsxwkA=="), StringUtils.getBytesUtf8((String)"ZL4VS2_6QCWNGgEojnwxyg=="), StringUtils.getBytesUtf8((String)"_3-PwBzbRxqMi1qTBhg_6A==")};
        byte[][] urlSafe2 = new byte[][]{StringUtils.getBytesUtf8((String)"lO2NAxnkSTOZVg-2dATTcA="), StringUtils.getBytesUtf8((String)"K_fMJwH-Q5e0nr7tWsxwkA="), StringUtils.getBytesUtf8((String)"ZL4VS2_6QCWNGgEojnwxyg="), StringUtils.getBytesUtf8((String)"_3-PwBzbRxqMi1qTBhg_6A=")};
        byte[][] urlSafe3 = new byte[][]{StringUtils.getBytesUtf8((String)"lO2NAxnkSTOZVg-2dATTcA"), StringUtils.getBytesUtf8((String)"K_fMJwH-Q5e0nr7tWsxwkA"), StringUtils.getBytesUtf8((String)"ZL4VS2_6QCWNGgEojnwxyg"), StringUtils.getBytesUtf8((String)"_3-PwBzbRxqMi1qTBhg_6A")};
        for (int i = 0; i < 4; ++i) {
            byte[] encodedStandard = Base64.encodeBase64((byte[])ids[i]);
            byte[] encodedUrlSafe = Base64.encodeBase64URLSafe((byte[])ids[i]);
            byte[] decodedStandard = Base64.decodeBase64((byte[])standard[i]);
            byte[] decodedUrlSafe1 = Base64.decodeBase64((byte[])urlSafe1[i]);
            byte[] decodedUrlSafe2 = Base64.decodeBase64((byte[])urlSafe2[i]);
            byte[] decodedUrlSafe3 = Base64.decodeBase64((byte[])urlSafe3[i]);
            Assertions.assertArrayEquals((byte[])encodedStandard, (byte[])standard[i], (String)"standard encode uuid");
            Assertions.assertArrayEquals((byte[])encodedUrlSafe, (byte[])urlSafe3[i], (String)"url-safe encode uuid");
            Assertions.assertArrayEquals((byte[])decodedStandard, (byte[])ids[i], (String)"standard decode uuid");
            Assertions.assertArrayEquals((byte[])decodedUrlSafe1, (byte[])ids[i], (String)"url-safe1 decode uuid");
            Assertions.assertArrayEquals((byte[])decodedUrlSafe2, (byte[])ids[i], (String)"url-safe2 decode uuid");
            Assertions.assertArrayEquals((byte[])decodedUrlSafe3, (byte[])ids[i], (String)"url-safe3 decode uuid");
        }
    }

    private String toString(byte[] data) {
        return org.apache.commons.lang3.StringUtils.join((byte[])data, (char)',');
    }
}

