/* This file is part of the KDE project
   Copyright (C) 2004 Lucijan Busch <lucijan@kde.org>
   Copyright (C) 2004 Cedric Pasteur <cedric.pasteur@free.fr>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include <tqobjectlist.h>
#include <tqpainter.h>
#include <tqcursor.h>

#include <kdebug.h>

#include "kexireportform.h"

KexiReportForm::KexiReportForm(TQWidget *parent, const char *name/*, KexiDB::Connection *conn*/)
 : TQWidget(parent, name)
{
	//m_conn = conn;
	kexipluginsdbg << "KexiReportForm::KexiReportForm(): " << endl;
	setCursor(TQCursor(TQt::ArrowCursor)); //to avoid keeping Size cursor when moving from form's boundaries
	setBackgroundColor(white);
}

KexiReportForm::~KexiReportForm()
{
	kexipluginsdbg << "KexiReportForm::~KexiReportForm(): close" << endl;
}

//repaint all children widgets
static void repaintAll(TQWidget *w)
{
	TQObjectList *list = w->queryList("TQWidget");
	TQObjectListIt it(*list);
	for (TQObject *obj; (obj=it.current()); ++it ) {
		static_cast<TQWidget*>(obj)->repaint();
	}
	delete list;
}

void
KexiReportForm::drawRect(const TQRect& r, int type)
{
	TQValueList<TQRect> l;
	l.append(r);
	drawRects(l, type);
}

void
KexiReportForm::drawRects(const TQValueList<TQRect> &list, int type)
{
	TQPainter p;
	p.begin(this, true);
	bool unclipped = testWFlags( WPaintUnclipped );
	setWFlags( WPaintUnclipped );

	if (prev_rect.isValid()) {
		//redraw prev. selection's rectangle
		p.drawPixmap( TQPoint(prev_rect.x()-2, prev_rect.y()-2), buffer, TQRect(prev_rect.x()-2, prev_rect.y()-2, prev_rect.width()+4, prev_rect.height()+4));
	}
	p.setBrush(TQBrush::NoBrush);
	if(type == 1) // selection rect
		p.setPen(TQPen(white, 1, TQt::DotLine));
	else if(type == 2) // insert rect
		p.setPen(TQPen(white, 2));
	p.setRasterOp(XorROP);

	prev_rect = TQRect();
	TQValueList<TQRect>::ConstIterator endIt = list.constEnd();
	for(TQValueList<TQRect>::ConstIterator it = list.constBegin(); it != endIt; ++it) {
		p.drawRect(*it);
		prev_rect = prev_rect.unite(*it);
	}

	if (!unclipped)
		clearWFlags( WPaintUnclipped );
	p.end();
}

void
KexiReportForm::initBuffer()
{
	repaintAll(this);
	buffer.resize( width(), height() );
	buffer = TQPixmap::grabWindow( winId() );
	prev_rect = TQRect();
}

void
KexiReportForm::clearForm()
{
	TQPainter p;
	p.begin(this, true);
	bool unclipped = testWFlags( WPaintUnclipped );
	setWFlags( WPaintUnclipped );

	//redraw entire form surface
	p.drawPixmap( TQPoint(0,0), buffer, TQRect(0,0,buffer.width(), buffer.height()) );

	if (!unclipped)
		clearWFlags( WPaintUnclipped );
	p.end();

	repaintAll(this);
}

void
KexiReportForm::highlightWidgets(TQWidget *from, TQWidget *to)//, const TQPoint &point)
{
	TQPoint fromPoint, toPoint;
	if(from && from->parentWidget() && (from != this))
		fromPoint = from->parentWidget()->mapTo(this, from->pos());
	if(to && to->parentWidget() && (to != this))
		toPoint = to->parentWidget()->mapTo(this, to->pos());

	TQPainter p;
	p.begin(this, true);
	bool unclipped = testWFlags( WPaintUnclipped );
	setWFlags( WPaintUnclipped );

	if (prev_rect.isValid()) {
		//redraw prev. selection's rectangle
		p.drawPixmap( TQPoint(prev_rect.x(), prev_rect.y()), buffer, TQRect(prev_rect.x(), prev_rect.y(), prev_rect.width(), prev_rect.height()));
	}

	p.setPen( TQPen(TQt::red, 2) );

	if(to)
	{
		TQPixmap pix1 = TQPixmap::grabWidget(from);
		TQPixmap pix2 = TQPixmap::grabWidget(to);

		if((from != this) && (to != this))
			p.drawLine( from->parentWidget()->mapTo(this, from->geometry().center()), to->parentWidget()->mapTo(this, to->geometry().center()) );

		p.drawPixmap(fromPoint.x(), fromPoint.y(), pix1);
		p.drawPixmap(toPoint.x(), toPoint.y(), pix2);

		if(to == this)
			p.drawRoundRect(2, 2, width()-4, height()-4, 4, 4);
		else
			p.drawRoundRect(toPoint.x(), toPoint.y(), to->width(), to->height(), 5, 5);
	}

	if(from == this)
		p.drawRoundRect(2, 2, width()-4, height()-4, 4, 4);
	else
		p.drawRoundRect(fromPoint.x(),  fromPoint.y(), from->width(), from->height(), 5, 5);

	if((to == this) || (from == this))
		prev_rect = TQRect(0, 0, buffer.width(), buffer.height());
	else if(to)
	{
		prev_rect.setX( (fromPoint.x() < toPoint.x()) ? (fromPoint.x() - 5) : (toPoint.x() - 5) );
		prev_rect.setY( (fromPoint.y() < toPoint.y()) ? (fromPoint.y() - 5) : (toPoint.y() - 5) );
		prev_rect.setRight( (fromPoint.x() < toPoint.x()) ? (toPoint.x() + to->width() + 10) : (fromPoint.x() + from->width() + 10) );
		prev_rect.setBottom( (fromPoint.y() < toPoint.y()) ? (toPoint.y() + to->height() + 10) : (fromPoint.y() + from->height() + 10) ) ;
	}
	else
		prev_rect = TQRect(fromPoint.x()- 5,  fromPoint.y() -5, from->width() + 10, from->height() + 10);

	if (!unclipped)
		clearWFlags( WPaintUnclipped );
	p.end();
}

TQSize
KexiReportForm::sizeHint() const
{
	//todo: find better size (user configured?)
	return TQSize(400,300);
}

#include "kexireportform.moc"

