/*
 * Decompiled with CFR 0.152.
 */
package org.logstash.plugins.inputs.http.util;

import io.netty.handler.ssl.ClientAuth;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.UnrecoverableKeyException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.Set;
import javax.crypto.Cipher;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLException;
import javax.net.ssl.SSLServerSocketFactory;
import javax.net.ssl.TrustManagerFactory;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.logstash.plugins.inputs.http.util.SslBuilder;

public class SslSimpleBuilder
implements SslBuilder {
    private static final Logger logger = LogManager.getLogger(SslSimpleBuilder.class);
    public static final Set<String> SUPPORTED_CIPHERS = new HashSet<String>(Arrays.asList(((SSLServerSocketFactory)SSLServerSocketFactory.getDefault()).getSupportedCipherSuites()));
    private static final String[] DEFAULT_CIPHERS;
    private static final String[] DEFAULT_CIPHERS_LIMITED;
    private String[] protocols = new String[]{"TLSv1.2", "TLSv1.3"};
    private String[] ciphers = SslSimpleBuilder.getDefaultCiphers();
    private File sslKeyFile;
    private File sslCertificateFile;
    private String[] certificateAuthorities;
    private KeyStore keyStore;
    private char[] keyStorePassword;
    private KeyStore trustStore;
    private String passphrase;
    private SslClientVerifyMode verifyMode = SslClientVerifyMode.NONE;

    public static SslSimpleBuilder withPemCertificate(String sslCertificateFilePath, String sslKeyFilePath, String pass) {
        SslSimpleBuilder builder = new SslSimpleBuilder();
        builder.sslCertificateFile = new File(sslCertificateFilePath);
        if (!builder.sslCertificateFile.canRead()) {
            throw new IllegalArgumentException(String.format("Certificate file cannot be read. Please confirm the user running Logstash has permissions to read: %s", sslCertificateFilePath));
        }
        builder.sslKeyFile = new File(sslKeyFilePath);
        if (!builder.sslKeyFile.canRead()) {
            throw new IllegalArgumentException(String.format("Private key file cannot be read. Please confirm the user running Logstash has permissions to read: %s", sslKeyFilePath));
        }
        builder.passphrase = pass;
        return builder;
    }

    public static SslSimpleBuilder withKeyStore(String keyStoreType, String keyStoreFile, String keyStorePassword) throws Exception {
        SslSimpleBuilder builder = new SslSimpleBuilder();
        Path keyStorePath = Paths.get(Objects.requireNonNull(keyStoreFile, "Keystore path cannot be null"), new String[0]);
        if (!Files.isReadable(keyStorePath)) {
            throw new IllegalArgumentException(String.format("Keystore file cannot be read. Please confirm the user running Logstash has permissions to read: %s", keyStoreFile));
        }
        builder.keyStorePassword = SslSimpleBuilder.formatJksPassword(keyStorePassword);
        builder.keyStore = SslSimpleBuilder.readKeyStore(keyStorePath, SslSimpleBuilder.resolveKeyStoreType(keyStoreType, keyStorePath), builder.keyStorePassword);
        return builder;
    }

    private SslSimpleBuilder() {
    }

    public SslSimpleBuilder setProtocols(String[] protocols) {
        this.protocols = protocols;
        return this;
    }

    public SslSimpleBuilder setCipherSuites(String[] ciphersSuite) throws IllegalArgumentException {
        for (String cipher : ciphersSuite) {
            if (!SUPPORTED_CIPHERS.contains(cipher)) {
                if (!SslSimpleBuilder.isUnlimitedJCEAvailable()) {
                    logger.warn("JCE Unlimited Strength Jurisdiction Policy not installed");
                }
                throw new IllegalArgumentException("Cipher `" + cipher + "` is not available");
            }
            logger.debug("{} cipher is supported", (Object)cipher);
        }
        this.ciphers = ciphersSuite;
        return this;
    }

    public SslSimpleBuilder setClientAuthentication(SslClientVerifyMode verifyMode) {
        this.verifyMode = verifyMode;
        return this;
    }

    public SslSimpleBuilder setCertificateAuthorities(String[] certificateAuthorities) {
        if (certificateAuthorities == null || certificateAuthorities.length == 0) {
            throw new IllegalArgumentException("SSL certificate authorities is required");
        }
        this.certificateAuthorities = certificateAuthorities;
        return this;
    }

    public SslSimpleBuilder setTrustStore(String trustStoreType, String trustStoreFile, String trustStorePassword) throws Exception {
        Path trustStorePath = Paths.get(Objects.requireNonNull(trustStoreFile, "Trust Store path cannot be null"), new String[0]);
        if (!Files.isReadable(trustStorePath)) {
            throw new IllegalArgumentException(String.format("Trust Store file cannot be read. Please confirm the user running Logstash has permissions to read: %s", trustStoreFile));
        }
        this.trustStore = SslSimpleBuilder.readKeyStore(trustStorePath, SslSimpleBuilder.resolveKeyStoreType(trustStoreType, trustStorePath), SslSimpleBuilder.formatJksPassword(trustStorePassword));
        if (!this.hasTrustStoreEntry(this.trustStore) && this.isClientAuthenticationRequired()) {
            throw new IllegalArgumentException(String.format("The provided Trust Store file does not contains any trusted certificate entry: %s", trustStoreFile));
        }
        return this;
    }

    private boolean isClientAuthenticationEnabled(SslClientVerifyMode mode) {
        return mode == SslClientVerifyMode.OPTIONAL || mode == SslClientVerifyMode.REQUIRED;
    }

    public boolean isClientAuthenticationRequired() {
        return this.verifyMode == SslClientVerifyMode.REQUIRED;
    }

    @Override
    public SslContext build() throws Exception {
        if (this.trustStore != null && this.certificateAuthorities != null) {
            throw new IllegalStateException("Use either a bundle of Certificate Authorities or a Trust Store to configure client authentication");
        }
        if (logger.isDebugEnabled()) {
            logger.debug("Available ciphers: {}", SUPPORTED_CIPHERS);
            logger.debug("Ciphers: {}", (Object)Arrays.toString(this.ciphers));
        }
        SslContextBuilder builder = this.createSslContextBuilder().ciphers(Arrays.asList(this.ciphers)).protocols(this.protocols).clientAuth(this.verifyMode.toClientAuth());
        if (this.isClientAuthenticationEnabled(this.verifyMode)) {
            if (this.certificateAuthorities != null) {
                if (logger.isDebugEnabled()) {
                    logger.debug("Certificate Authorities: {}", (Object)Arrays.toString(this.certificateAuthorities));
                }
                builder.trustManager(this.loadCertificateCollection(this.certificateAuthorities));
            } else if (this.trustStore != null || this.keyStore != null) {
                builder.trustManager(this.createTrustManagerFactory());
            } else {
                throw new IllegalStateException("Either an SSL certificate or an SSL Trust Store is required when SSL is enabled");
            }
        }
        return SslSimpleBuilder.doBuild(builder);
    }

    private SslContextBuilder createSslContextBuilder() throws Exception {
        if (this.sslCertificateFile != null) {
            return SslContextBuilder.forServer((File)this.sslCertificateFile, (File)this.sslKeyFile, (String)this.passphrase);
        }
        if (this.keyStore != null) {
            return SslContextBuilder.forServer((KeyManagerFactory)this.createKeyManagerFactory());
        }
        throw new IllegalStateException("Either a KeyStore or an SSL certificate must be provided");
    }

    private KeyManagerFactory createKeyManagerFactory() throws NoSuchAlgorithmException, UnrecoverableKeyException, KeyStoreException {
        KeyManagerFactory kmf = this.getDefaultKeyManagerFactory();
        kmf.init(this.keyStore, this.keyStorePassword);
        return kmf;
    }

    KeyManagerFactory getDefaultKeyManagerFactory() throws NoSuchAlgorithmException {
        return KeyManagerFactory.getInstance(KeyManagerFactory.getDefaultAlgorithm());
    }

    private TrustManagerFactory createTrustManagerFactory() throws Exception {
        TrustManagerFactory tmf = this.getDefaultTrustManagerFactory();
        if (this.trustStore == null) {
            logger.info("SSL Trust Store not configured, using the provided Key Store instead.");
            if (logger.isDebugEnabled() && !this.hasTrustStoreEntry(this.keyStore)) {
                logger.debug("The provided SSL Key Store, used as Trust Store, has no trusted certificate entry.");
            }
            tmf.init(this.keyStore);
            return tmf;
        }
        tmf.init(this.trustStore);
        return tmf;
    }

    TrustManagerFactory getDefaultTrustManagerFactory() throws NoSuchAlgorithmException {
        return TrustManagerFactory.getInstance(TrustManagerFactory.getDefaultAlgorithm());
    }

    static SslContext doBuild(SslContextBuilder builder) throws Exception {
        try {
            return builder.build();
        }
        catch (SSLException e) {
            logger.debug("Failed to initialize SSL", (Throwable)e);
            if (("failed to initialize the server-side SSL context".equals(e.getMessage()) || "failed to initialize the client-side SSL context".equals(e.getMessage())) && e.getCause() instanceof Exception) {
                throw (Exception)e.getCause();
            }
            throw e;
        }
        catch (Exception e) {
            logger.debug("Failed to initialize SSL", (Throwable)e);
            throw e;
        }
    }

    private X509Certificate[] loadCertificateCollection(String[] certificates) throws IOException, CertificateException {
        logger.debug("Load certificates collection");
        CertificateFactory certificateFactory = CertificateFactory.getInstance("X.509");
        ArrayList collections = new ArrayList();
        for (String certificate : certificates) {
            logger.debug("Loading certificates from file {}", (Object)certificate);
            try (FileInputStream in = new FileInputStream(certificate);){
                List certificatesChains = (List)certificateFactory.generateCertificates(in);
                collections.addAll(certificatesChains);
            }
        }
        return collections.toArray(new X509Certificate[collections.size()]);
    }

    public static String[] getDefaultCiphers() {
        if (SslSimpleBuilder.isUnlimitedJCEAvailable()) {
            return DEFAULT_CIPHERS;
        }
        logger.warn("JCE Unlimited Strength Jurisdiction Policy not installed - max key length is 128 bits");
        return DEFAULT_CIPHERS_LIMITED;
    }

    public static boolean isUnlimitedJCEAvailable() {
        try {
            return Cipher.getMaxAllowedKeyLength("AES") > 128;
        }
        catch (NoSuchAlgorithmException e) {
            logger.warn("AES not available", (Throwable)e);
            return false;
        }
    }

    String[] getProtocols() {
        return this.protocols != null ? (String[])this.protocols.clone() : null;
    }

    String[] getCertificateAuthorities() {
        return this.certificateAuthorities != null ? (String[])this.certificateAuthorities.clone() : null;
    }

    String[] getCiphers() {
        return this.ciphers != null ? (String[])this.ciphers.clone() : null;
    }

    SslClientVerifyMode getVerifyMode() {
        return this.verifyMode;
    }

    static String resolveKeyStoreType(String type, Path path) {
        if (type == null || type.isEmpty()) {
            return SslSimpleBuilder.inferKeyStoreType(path);
        }
        return type;
    }

    private static String inferKeyStoreType(Path path) {
        String name;
        String string = name = path == null ? "" : path.getFileName().toString().toLowerCase(Locale.ROOT);
        if (name.endsWith(".p12") || name.endsWith(".pfx") || name.endsWith(".pkcs12")) {
            return "PKCS12";
        }
        return "jks";
    }

    private static char[] formatJksPassword(String password) {
        if (password == null) {
            return null;
        }
        return password.toCharArray();
    }

    private static KeyStore readKeyStore(Path path, String ksType, char[] password) throws GeneralSecurityException, IOException {
        KeyStore keyStore = KeyStore.getInstance(ksType);
        if (path != null) {
            try (InputStream in = Files.newInputStream(path, new OpenOption[0]);){
                keyStore.load(in, password);
            }
        }
        return keyStore;
    }

    private boolean hasTrustStoreEntry(KeyStore store) throws GeneralSecurityException {
        Enumeration<String> aliases = store.aliases();
        while (aliases.hasMoreElements()) {
            String alias = aliases.nextElement();
            if (!store.isCertificateEntry(alias)) continue;
            return true;
        }
        return false;
    }

    KeyStore getKeyStore() {
        return this.keyStore;
    }

    KeyStore getTrustStore() {
        return this.trustStore;
    }

    static {
        String[] defaultCipherCandidates = new String[]{"TLS_AES_256_GCM_SHA384", "TLS_AES_128_GCM_SHA256", "TLS_CHACHA20_POLY1305_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384", "TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384", "TLS_ECDHE_ECDSA_WITH_CHACHA20_POLY1305_SHA256", "TLS_ECDHE_RSA_WITH_CHACHA20_POLY1305_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256", "TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384", "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384", "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256"};
        DEFAULT_CIPHERS = (String[])Arrays.stream(defaultCipherCandidates).filter(SUPPORTED_CIPHERS::contains).toArray(String[]::new);
        DEFAULT_CIPHERS_LIMITED = new String[]{"TLS_ECDHE_ECDSA_WITH_AES_256_GCM_SHA384", "TLS_ECDHE_RSA_WITH_AES_256_GCM_SHA384", "TLS_ECDHE_ECDSA_WITH_AES_128_GCM_SHA256", "TLS_ECDHE_RSA_WITH_AES_128_GCM_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA384", "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA384", "TLS_ECDHE_RSA_WITH_AES_128_CBC_SHA256", "TLS_ECDHE_ECDSA_WITH_AES_128_CBC_SHA256"};
    }

    public static enum SslClientVerifyMode {
        NONE(ClientAuth.NONE),
        OPTIONAL(ClientAuth.OPTIONAL),
        REQUIRED(ClientAuth.REQUIRE);

        private final ClientAuth clientAuth;

        private SslClientVerifyMode(ClientAuth clientAuth) {
            this.clientAuth = clientAuth;
        }

        public ClientAuth toClientAuth() {
            return this.clientAuth;
        }
    }
}

