;;; -*- coding: iso-8859-1 -*-

;; Copyright (C) 2000 Free Software Foundation, Inc.
;; Copyright (C) 2010 Ben Wing.

;; Author: Yoshiki Hayashi  <yoshiki@xemacs.org>
;; Maintainer: Yoshiki Hayashi  <yoshiki@xemacs.org>
;; Created: 2000
;; Keywords: tests

;; This file is part of XEmacs.

;; XEmacs is free software: you can redistribute it and/or modify it
;; under the terms of the GNU General Public License as published by the
;; Free Software Foundation, either version 3 of the License, or (at your
;; option) any later version.

;; XEmacs is distributed in the hope that it will be useful, but WITHOUT
;; ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
;; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
;; for more details.

;; You should have received a copy of the GNU General Public License
;; along with XEmacs.  If not, see <http://www.gnu.org/licenses/>.

;;; Synched up with: Not in FSF.

;;; Commentary:

;; Test case-table related functionality.  See test-harness.el for
;; instructions on how to run these tests.

;; NOTE NOTE NOTE: See also:
;;
;; (1) regexp-tests.el, for case-related regexp searching.
;; (2) search-tests.el, for case-related non-regexp searching.
;; (3) lisp-tests.el, for case-related comparisons with `equalp'.

;; NOTE NOTE NOTE: There is some domain overlap among case-tests.el,
;; lisp-tests.el, regexp-tests.el, and search-tests.el.  The current rule
;; for what goes where is:
;;
;; (1) Anything regexp-related goes in regexp-tests.el, including searches.
;; (2) Non-regexp searches go in search-tests.el.  This includes case-folding
;;     searches in the situation where the test tests both folding and
;;     non-folding behavior.
;; (3) Anything else that involves case-testing but in an ancillary manner
;;     goes into whichever primary area it is involved in (e.g. searches for
;;     search-tests.el, Lisp primitives in lisp-tests.el).  But if it is
;;     primarily case-related and happens to involve other areas in an
;;     ancillary manner, it goes into case-tests.el.  This includes, for
;;     example, the Unicode case map torture tests.


(Assert (case-table-p (standard-case-table)))
;; Old case table test.
(Assert (case-table-p (list
		       (make-string 256 ?a)
		       nil nil nil)))
(Assert (case-table-p (list
		       (make-string 256 ?a)
		       (make-string 256 ?b)
		       nil nil)))
(Assert (case-table-p (list
		       (make-string 256 ?a)
		       (make-string 256 ?b)
		       (make-string 256 ?c)
		       nil)))
(Assert (case-table-p (list
		       (make-string 256 ?a)
		       (make-string 256 ?b)
		       (make-string 256 ?c)
		       (make-string 256 ?d))))
(Assert (not (case-table-p (list (make-string 256 ?a)
				 (make-string 256 ?b)
				 (make-string 256 ?c)
				 (make-string 254 ?d)))))
(Assert (not (case-table-p (list (make-string 256 ?a)))))

(Assert (case-table-p (set-case-table (current-case-table))))

(defvar string-0-through-32
  (let ((result (make-string 33 (int-to-char 0))))
    (dotimes (i 33)
      (aset result i (int-to-char i)))
    result)
  "String containing characters from code point 0 (NUL) through 32 (SPC).")

(defvar string-127-through-160
  (let ((result (make-string 34 (int-to-char 0))))
    (dotimes (i 34)
      (aset result i (int-to-char (+ 127 i))))
    result)
  "String containing characters from code point 127 (DEL) through 160
\(no-break-space).")

;; Case table sanity check.
(let* ((downcase1 "!\"#$%&'()*+,-./0123456789:;<=>?@abcdefghijklmnopqrstuvwxyz\
\[\\]^_`abcdefghijklmnopqrstuvwxyz{|}~")
       (upcase1   "!\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ\
\[\\]^_`ABCDEFGHIJKLMNOPQRSTUVWXYZ{|}~")
       (flipcase1 "!\"#$%&'()*+,-./0123456789:;<=>?@abcdefghijklmnopqrstuvwxyz\
\[\\]^_`ABCDEFGHIJKLMNOPQRSTUVWXYZ{|}~")
       (downcase2 "\
")
       (upcase2
	(format "%c\
%c%c"
		(or (decode-char 'ucs #x039c) ?\u00b5)
		(or (decode-char 'ucs #x1e9e) ?\u00df)
		(or (decode-char 'ucs #x0178) ?\u00ff)))
       (flipcase2
	(format "%c\
%c%c"
		(or (decode-char 'ucs #x039c) ?\u00b5)
		(or (decode-char 'ucs #x1e9e) ?\u00df)
		(or (decode-char 'ucs #x0178) ?\u00ff)))
       (downcase-string
	(concat string-0-through-32 downcase1 string-127-through-160
		downcase2))
       (upcase-string
	(concat string-0-through-32 flipcase1 string-127-through-160
		flipcase2))
       (table (standard-case-table)))
  (dotimes (i 256)
    (Assert (eq (get-case-table 'downcase (int-to-char i) table)
		(aref downcase-string i)))
    (Assert (eq (get-case-table 'upcase (int-to-char i) table)
		(aref upcase-string i))))

  (Check-Error-Message error "Char case must be downcase or upcase"
		       (get-case-table 'foo ?a (standard-case-table)))

  (Assert (string= (upcase downcase1) upcase1))

  (Assert (string= (upcase upcase1) upcase1))

  (Assert (string= (upcase downcase2) upcase2))
  
  (Assert (string= (upcase upcase2) upcase2))

  (Assert (string= (downcase downcase1) downcase1))

  (Assert (string= (downcase upcase1) downcase1))

  (Assert (string= (downcase downcase2) downcase2))

  (Assert (string= (downcase upcase2) downcase2))

  ;; Old case table format test.
  (with-temp-buffer
    (set-case-table
     (list
      (concat string-0-through-32 downcase1 string-127-through-160 downcase2)
      nil nil nil))
    (Assert
     (string=
      (upcase "!\"#$%&'()*+,-./0123456789:;<=>?@abcdefghijklmnopqrstuvwxyz")
      "!\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ"))
    (Assert
     (string=
      (downcase "!\"#$%&'()*+,-./0123456789:;<=>?@ABCDEFGHIJKLMNOPQRSTUVWXYZ")
      "!\"#$%&'()*+,-./0123456789:;<=>?@abcdefghijklmnopqrstuvwxyz")))
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; Torture test, using all the non-"full" mappings from the Unicode case
;; tables. (Full mappings are those that involve sequences of more than one
;; character, e.g. lowercase German  (esszet) -> uppercase sequence "SS").

(defun char-as-unicode-escape (ch)
  "Return a string corresponding to the Unicode read-expression of CH.
For example, if CH is ?\\u00F1, the return value will be the string
\"?\\\\u00F1\"."
  (let ((uni (char-to-unicode ch)))
    (if (< uni #x10000) (format "?\\u%04X" uni) (format "?\\U%08X" uni))))

;; NOTE NOTE NOTE! *DO NOT* apply nreverse to the following list, as we do in
;; uni-case-conv.el.  By keeping the order as-is, we ensure that some "obnoxious"
;; mappings get tested, i.e. those where upcase and downcase do not mutually
;; correspond.

(let* ((uni-mappings
	'(;  UPPER   LOWER
	  (#x0041 #x0061) ;; LATIN CAPITAL LETTER A
	  (#x0042 #x0062) ;; LATIN CAPITAL LETTER B
	  (#x0043 #x0063) ;; LATIN CAPITAL LETTER C
	  (#x0044 #x0064) ;; LATIN CAPITAL LETTER D
	  (#x0045 #x0065) ;; LATIN CAPITAL LETTER E
	  (#x0046 #x0066) ;; LATIN CAPITAL LETTER F
	  (#x0047 #x0067) ;; LATIN CAPITAL LETTER G
	  (#x0048 #x0068) ;; LATIN CAPITAL LETTER H
	  (#x0049 #x0069) ;; LATIN CAPITAL LETTER I
;;; WARNING: Unhandled Turkish mapping:
;;;        0049; T; 0131; # LATIN CAPITAL LETTER I
	  (#x004A #x006A) ;; LATIN CAPITAL LETTER J
	  (#x004B #x006B) ;; LATIN CAPITAL LETTER K
	  (#x004C #x006C) ;; LATIN CAPITAL LETTER L
	  (#x004D #x006D) ;; LATIN CAPITAL LETTER M
	  (#x004E #x006E) ;; LATIN CAPITAL LETTER N
	  (#x004F #x006F) ;; LATIN CAPITAL LETTER O
	  (#x0050 #x0070) ;; LATIN CAPITAL LETTER P
	  (#x0051 #x0071) ;; LATIN CAPITAL LETTER Q
	  (#x0052 #x0072) ;; LATIN CAPITAL LETTER R
	  (#x0053 #x0073) ;; LATIN CAPITAL LETTER S
	  (#x0054 #x0074) ;; LATIN CAPITAL LETTER T
	  (#x0055 #x0075) ;; LATIN CAPITAL LETTER U
	  (#x0056 #x0076) ;; LATIN CAPITAL LETTER V
	  (#x0057 #x0077) ;; LATIN CAPITAL LETTER W
	  (#x0058 #x0078) ;; LATIN CAPITAL LETTER X
	  (#x0059 #x0079) ;; LATIN CAPITAL LETTER Y
	  (#x005A #x007A) ;; LATIN CAPITAL LETTER Z
	  (#x00B5 #x03BC) ;; MICRO SIGN
	  (#x00C0 #x00E0) ;; LATIN CAPITAL LETTER A WITH GRAVE
	  (#x00C1 #x00E1) ;; LATIN CAPITAL LETTER A WITH ACUTE
	  (#x00C2 #x00E2) ;; LATIN CAPITAL LETTER A WITH CIRCUMFLEX
	  (#x00C3 #x00E3) ;; LATIN CAPITAL LETTER A WITH TILDE
	  (#x00C4 #x00E4) ;; LATIN CAPITAL LETTER A WITH DIAERESIS
	  (#x00C5 #x00E5) ;; LATIN CAPITAL LETTER A WITH RING ABOVE
	  (#x00C6 #x00E6) ;; LATIN CAPITAL LETTER AE
	  (#x00C7 #x00E7) ;; LATIN CAPITAL LETTER C WITH CEDILLA
	  (#x00C8 #x00E8) ;; LATIN CAPITAL LETTER E WITH GRAVE
	  (#x00C9 #x00E9) ;; LATIN CAPITAL LETTER E WITH ACUTE
	  (#x00CA #x00EA) ;; LATIN CAPITAL LETTER E WITH CIRCUMFLEX
	  (#x00CB #x00EB) ;; LATIN CAPITAL LETTER E WITH DIAERESIS
	  (#x00CC #x00EC) ;; LATIN CAPITAL LETTER I WITH GRAVE
	  (#x00CD #x00ED) ;; LATIN CAPITAL LETTER I WITH ACUTE
	  (#x00CE #x00EE) ;; LATIN CAPITAL LETTER I WITH CIRCUMFLEX
	  (#x00CF #x00EF) ;; LATIN CAPITAL LETTER I WITH DIAERESIS
	  (#x00D0 #x00F0) ;; LATIN CAPITAL LETTER ETH
	  (#x00D1 #x00F1) ;; LATIN CAPITAL LETTER N WITH TILDE
	  (#x00D2 #x00F2) ;; LATIN CAPITAL LETTER O WITH GRAVE
	  (#x00D3 #x00F3) ;; LATIN CAPITAL LETTER O WITH ACUTE
	  (#x00D4 #x00F4) ;; LATIN CAPITAL LETTER O WITH CIRCUMFLEX
	  (#x00D5 #x00F5) ;; LATIN CAPITAL LETTER O WITH TILDE
	  (#x00D6 #x00F6) ;; LATIN CAPITAL LETTER O WITH DIAERESIS
	  (#x00D8 #x00F8) ;; LATIN CAPITAL LETTER O WITH STROKE
	  (#x00D9 #x00F9) ;; LATIN CAPITAL LETTER U WITH GRAVE
	  (#x00DA #x00FA) ;; LATIN CAPITAL LETTER U WITH ACUTE
	  (#x00DB #x00FB) ;; LATIN CAPITAL LETTER U WITH CIRCUMFLEX
	  (#x00DC #x00FC) ;; LATIN CAPITAL LETTER U WITH DIAERESIS
	  (#x00DD #x00FD) ;; LATIN CAPITAL LETTER Y WITH ACUTE
	  (#x00DE #x00FE) ;; LATIN CAPITAL LETTER THORN
;;; WARNING: Unhandled full mapping:
;;;   00DF; F; 0073 0073; # LATIN SMALL LETTER SHARP S
	  (#x0100 #x0101) ;; LATIN CAPITAL LETTER A WITH MACRON
	  (#x0102 #x0103) ;; LATIN CAPITAL LETTER A WITH BREVE
	  (#x0104 #x0105) ;; LATIN CAPITAL LETTER A WITH OGONEK
	  (#x0106 #x0107) ;; LATIN CAPITAL LETTER C WITH ACUTE
	  (#x0108 #x0109) ;; LATIN CAPITAL LETTER C WITH CIRCUMFLEX
	  (#x010A #x010B) ;; LATIN CAPITAL LETTER C WITH DOT ABOVE
	  (#x010C #x010D) ;; LATIN CAPITAL LETTER C WITH CARON
	  (#x010E #x010F) ;; LATIN CAPITAL LETTER D WITH CARON
	  (#x0110 #x0111) ;; LATIN CAPITAL LETTER D WITH STROKE
	  (#x0112 #x0113) ;; LATIN CAPITAL LETTER E WITH MACRON
	  (#x0114 #x0115) ;; LATIN CAPITAL LETTER E WITH BREVE
	  (#x0116 #x0117) ;; LATIN CAPITAL LETTER E WITH DOT ABOVE
	  (#x0118 #x0119) ;; LATIN CAPITAL LETTER E WITH OGONEK
	  (#x011A #x011B) ;; LATIN CAPITAL LETTER E WITH CARON
	  (#x011C #x011D) ;; LATIN CAPITAL LETTER G WITH CIRCUMFLEX
	  (#x011E #x011F) ;; LATIN CAPITAL LETTER G WITH BREVE
	  (#x0120 #x0121) ;; LATIN CAPITAL LETTER G WITH DOT ABOVE
	  (#x0122 #x0123) ;; LATIN CAPITAL LETTER G WITH CEDILLA
	  (#x0124 #x0125) ;; LATIN CAPITAL LETTER H WITH CIRCUMFLEX
	  (#x0126 #x0127) ;; LATIN CAPITAL LETTER H WITH STROKE
	  (#x0128 #x0129) ;; LATIN CAPITAL LETTER I WITH TILDE
	  (#x012A #x012B) ;; LATIN CAPITAL LETTER I WITH MACRON
	  (#x012C #x012D) ;; LATIN CAPITAL LETTER I WITH BREVE
	  (#x012E #x012F) ;; LATIN CAPITAL LETTER I WITH OGONEK
;;; WARNING: Unhandled full mapping:
;;;   0130; F; 0069 0307; # LATIN CAPITAL LETTER I WITH DOT ABOVE
;;; WARNING: Unhandled Turkish mapping:
;;;        0130; T; 0069; # LATIN CAPITAL LETTER I WITH DOT ABOVE
	  (#x0132 #x0133) ;; LATIN CAPITAL LIGATURE IJ
	  (#x0134 #x0135) ;; LATIN CAPITAL LETTER J WITH CIRCUMFLEX
	  (#x0136 #x0137) ;; LATIN CAPITAL LETTER K WITH CEDILLA
	  (#x0139 #x013A) ;; LATIN CAPITAL LETTER L WITH ACUTE
	  (#x013B #x013C) ;; LATIN CAPITAL LETTER L WITH CEDILLA
	  (#x013D #x013E) ;; LATIN CAPITAL LETTER L WITH CARON
	  (#x013F #x0140) ;; LATIN CAPITAL LETTER L WITH MIDDLE DOT
	  (#x0141 #x0142) ;; LATIN CAPITAL LETTER L WITH STROKE
	  (#x0143 #x0144) ;; LATIN CAPITAL LETTER N WITH ACUTE
	  (#x0145 #x0146) ;; LATIN CAPITAL LETTER N WITH CEDILLA
	  (#x0147 #x0148) ;; LATIN CAPITAL LETTER N WITH CARON
;;; WARNING: Unhandled full mapping:
;;;   0149; F; 02BC 006E; # LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
	  (#x014A #x014B) ;; LATIN CAPITAL LETTER ENG
	  (#x014C #x014D) ;; LATIN CAPITAL LETTER O WITH MACRON
	  (#x014E #x014F) ;; LATIN CAPITAL LETTER O WITH BREVE
	  (#x0150 #x0151) ;; LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
	  (#x0152 #x0153) ;; LATIN CAPITAL LIGATURE OE
	  (#x0154 #x0155) ;; LATIN CAPITAL LETTER R WITH ACUTE
	  (#x0156 #x0157) ;; LATIN CAPITAL LETTER R WITH CEDILLA
	  (#x0158 #x0159) ;; LATIN CAPITAL LETTER R WITH CARON
	  (#x015A #x015B) ;; LATIN CAPITAL LETTER S WITH ACUTE
	  (#x015C #x015D) ;; LATIN CAPITAL LETTER S WITH CIRCUMFLEX
	  (#x015E #x015F) ;; LATIN CAPITAL LETTER S WITH CEDILLA
	  (#x0160 #x0161) ;; LATIN CAPITAL LETTER S WITH CARON
	  (#x0162 #x0163) ;; LATIN CAPITAL LETTER T WITH CEDILLA
	  (#x0164 #x0165) ;; LATIN CAPITAL LETTER T WITH CARON
	  (#x0166 #x0167) ;; LATIN CAPITAL LETTER T WITH STROKE
	  (#x0168 #x0169) ;; LATIN CAPITAL LETTER U WITH TILDE
	  (#x016A #x016B) ;; LATIN CAPITAL LETTER U WITH MACRON
	  (#x016C #x016D) ;; LATIN CAPITAL LETTER U WITH BREVE
	  (#x016E #x016F) ;; LATIN CAPITAL LETTER U WITH RING ABOVE
	  (#x0170 #x0171) ;; LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
	  (#x0172 #x0173) ;; LATIN CAPITAL LETTER U WITH OGONEK
	  (#x0174 #x0175) ;; LATIN CAPITAL LETTER W WITH CIRCUMFLEX
	  (#x0176 #x0177) ;; LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
	  (#x0178 #x00FF) ;; LATIN CAPITAL LETTER Y WITH DIAERESIS
	  (#x0179 #x017A) ;; LATIN CAPITAL LETTER Z WITH ACUTE
	  (#x017B #x017C) ;; LATIN CAPITAL LETTER Z WITH DOT ABOVE
	  (#x017D #x017E) ;; LATIN CAPITAL LETTER Z WITH CARON
	  (#x017F #x0073) ;; LATIN SMALL LETTER LONG S
	  (#x0181 #x0253) ;; LATIN CAPITAL LETTER B WITH HOOK
	  (#x0182 #x0183) ;; LATIN CAPITAL LETTER B WITH TOPBAR
	  (#x0184 #x0185) ;; LATIN CAPITAL LETTER TONE SIX
	  (#x0186 #x0254) ;; LATIN CAPITAL LETTER OPEN O
	  (#x0187 #x0188) ;; LATIN CAPITAL LETTER C WITH HOOK
	  (#x0189 #x0256) ;; LATIN CAPITAL LETTER AFRICAN D
	  (#x018A #x0257) ;; LATIN CAPITAL LETTER D WITH HOOK
	  (#x018B #x018C) ;; LATIN CAPITAL LETTER D WITH TOPBAR
	  (#x018E #x01DD) ;; LATIN CAPITAL LETTER REVERSED E
	  (#x018F #x0259) ;; LATIN CAPITAL LETTER SCHWA
	  (#x0190 #x025B) ;; LATIN CAPITAL LETTER OPEN E
	  (#x0191 #x0192) ;; LATIN CAPITAL LETTER F WITH HOOK
	  (#x0193 #x0260) ;; LATIN CAPITAL LETTER G WITH HOOK
	  (#x0194 #x0263) ;; LATIN CAPITAL LETTER GAMMA
	  (#x0196 #x0269) ;; LATIN CAPITAL LETTER IOTA
	  (#x0197 #x0268) ;; LATIN CAPITAL LETTER I WITH STROKE
	  (#x0198 #x0199) ;; LATIN CAPITAL LETTER K WITH HOOK
	  (#x019C #x026F) ;; LATIN CAPITAL LETTER TURNED M
	  (#x019D #x0272) ;; LATIN CAPITAL LETTER N WITH LEFT HOOK
	  (#x019F #x0275) ;; LATIN CAPITAL LETTER O WITH MIDDLE TILDE
	  (#x01A0 #x01A1) ;; LATIN CAPITAL LETTER O WITH HORN
	  (#x01A2 #x01A3) ;; LATIN CAPITAL LETTER OI
	  (#x01A4 #x01A5) ;; LATIN CAPITAL LETTER P WITH HOOK
	  (#x01A6 #x0280) ;; LATIN LETTER YR
	  (#x01A7 #x01A8) ;; LATIN CAPITAL LETTER TONE TWO
	  (#x01A9 #x0283) ;; LATIN CAPITAL LETTER ESH
	  (#x01AC #x01AD) ;; LATIN CAPITAL LETTER T WITH HOOK
	  (#x01AE #x0288) ;; LATIN CAPITAL LETTER T WITH RETROFLEX HOOK
	  (#x01AF #x01B0) ;; LATIN CAPITAL LETTER U WITH HORN
	  (#x01B1 #x028A) ;; LATIN CAPITAL LETTER UPSILON
	  (#x01B2 #x028B) ;; LATIN CAPITAL LETTER V WITH HOOK
	  (#x01B3 #x01B4) ;; LATIN CAPITAL LETTER Y WITH HOOK
	  (#x01B5 #x01B6) ;; LATIN CAPITAL LETTER Z WITH STROKE
	  (#x01B7 #x0292) ;; LATIN CAPITAL LETTER EZH
	  (#x01B8 #x01B9) ;; LATIN CAPITAL LETTER EZH REVERSED
	  (#x01BC #x01BD) ;; LATIN CAPITAL LETTER TONE FIVE
	  (#x01C4 #x01C6) ;; LATIN CAPITAL LETTER DZ WITH CARON
	  (#x01C5 #x01C6) ;; LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CARON
	  (#x01C7 #x01C9) ;; LATIN CAPITAL LETTER LJ
	  (#x01C8 #x01C9) ;; LATIN CAPITAL LETTER L WITH SMALL LETTER J
	  (#x01CA #x01CC) ;; LATIN CAPITAL LETTER NJ
	  (#x01CB #x01CC) ;; LATIN CAPITAL LETTER N WITH SMALL LETTER J
	  (#x01CD #x01CE) ;; LATIN CAPITAL LETTER A WITH CARON
	  (#x01CF #x01D0) ;; LATIN CAPITAL LETTER I WITH CARON
	  (#x01D1 #x01D2) ;; LATIN CAPITAL LETTER O WITH CARON
	  (#x01D3 #x01D4) ;; LATIN CAPITAL LETTER U WITH CARON
	  (#x01D5 #x01D6) ;; LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
	  (#x01D7 #x01D8) ;; LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
	  (#x01D9 #x01DA) ;; LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
	  (#x01DB #x01DC) ;; LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
	  (#x01DE #x01DF) ;; LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
	  (#x01E0 #x01E1) ;; LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
	  (#x01E2 #x01E3) ;; LATIN CAPITAL LETTER AE WITH MACRON
	  (#x01E4 #x01E5) ;; LATIN CAPITAL LETTER G WITH STROKE
	  (#x01E6 #x01E7) ;; LATIN CAPITAL LETTER G WITH CARON
	  (#x01E8 #x01E9) ;; LATIN CAPITAL LETTER K WITH CARON
	  (#x01EA #x01EB) ;; LATIN CAPITAL LETTER O WITH OGONEK
	  (#x01EC #x01ED) ;; LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
	  (#x01EE #x01EF) ;; LATIN CAPITAL LETTER EZH WITH CARON
;;; WARNING: Unhandled full mapping:
;;;   01F0; F; 006A 030C; # LATIN SMALL LETTER J WITH CARON
	  (#x01F1 #x01F3) ;; LATIN CAPITAL LETTER DZ
	  (#x01F2 #x01F3) ;; LATIN CAPITAL LETTER D WITH SMALL LETTER Z
	  (#x01F4 #x01F5) ;; LATIN CAPITAL LETTER G WITH ACUTE
	  (#x01F6 #x0195) ;; LATIN CAPITAL LETTER HWAIR
	  (#x01F7 #x01BF) ;; LATIN CAPITAL LETTER WYNN
	  (#x01F8 #x01F9) ;; LATIN CAPITAL LETTER N WITH GRAVE
	  (#x01FA #x01FB) ;; LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
	  (#x01FC #x01FD) ;; LATIN CAPITAL LETTER AE WITH ACUTE
	  (#x01FE #x01FF) ;; LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
	  (#x0200 #x0201) ;; LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
	  (#x0202 #x0203) ;; LATIN CAPITAL LETTER A WITH INVERTED BREVE
	  (#x0204 #x0205) ;; LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
	  (#x0206 #x0207) ;; LATIN CAPITAL LETTER E WITH INVERTED BREVE
	  (#x0208 #x0209) ;; LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
	  (#x020A #x020B) ;; LATIN CAPITAL LETTER I WITH INVERTED BREVE
	  (#x020C #x020D) ;; LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
	  (#x020E #x020F) ;; LATIN CAPITAL LETTER O WITH INVERTED BREVE
	  (#x0210 #x0211) ;; LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
	  (#x0212 #x0213) ;; LATIN CAPITAL LETTER R WITH INVERTED BREVE
	  (#x0214 #x0215) ;; LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
	  (#x0216 #x0217) ;; LATIN CAPITAL LETTER U WITH INVERTED BREVE
	  (#x0218 #x0219) ;; LATIN CAPITAL LETTER S WITH COMMA BELOW
	  (#x021A #x021B) ;; LATIN CAPITAL LETTER T WITH COMMA BELOW
	  (#x021C #x021D) ;; LATIN CAPITAL LETTER YOGH
	  (#x021E #x021F) ;; LATIN CAPITAL LETTER H WITH CARON
	  (#x0220 #x019E) ;; LATIN CAPITAL LETTER N WITH LONG RIGHT LEG
	  (#x0222 #x0223) ;; LATIN CAPITAL LETTER OU
	  (#x0224 #x0225) ;; LATIN CAPITAL LETTER Z WITH HOOK
	  (#x0226 #x0227) ;; LATIN CAPITAL LETTER A WITH DOT ABOVE
	  (#x0228 #x0229) ;; LATIN CAPITAL LETTER E WITH CEDILLA
	  (#x022A #x022B) ;; LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON
	  (#x022C #x022D) ;; LATIN CAPITAL LETTER O WITH TILDE AND MACRON
	  (#x022E #x022F) ;; LATIN CAPITAL LETTER O WITH DOT ABOVE
	  (#x0230 #x0231) ;; LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON
	  (#x0232 #x0233) ;; LATIN CAPITAL LETTER Y WITH MACRON
	  (#x023A #x2C65) ;; LATIN CAPITAL LETTER A WITH STROKE
	  (#x023B #x023C) ;; LATIN CAPITAL LETTER C WITH STROKE
	  (#x023D #x019A) ;; LATIN CAPITAL LETTER L WITH BAR
	  (#x023E #x2C66) ;; LATIN CAPITAL LETTER T WITH DIAGONAL STROKE
	  (#x0241 #x0242) ;; LATIN CAPITAL LETTER GLOTTAL STOP
	  (#x0243 #x0180) ;; LATIN CAPITAL LETTER B WITH STROKE
	  (#x0244 #x0289) ;; LATIN CAPITAL LETTER U BAR
	  (#x0245 #x028C) ;; LATIN CAPITAL LETTER TURNED V
	  (#x0246 #x0247) ;; LATIN CAPITAL LETTER E WITH STROKE
	  (#x0248 #x0249) ;; LATIN CAPITAL LETTER J WITH STROKE
	  (#x024A #x024B) ;; LATIN CAPITAL LETTER SMALL Q WITH HOOK TAIL
	  (#x024C #x024D) ;; LATIN CAPITAL LETTER R WITH STROKE
	  (#x024E #x024F) ;; LATIN CAPITAL LETTER Y WITH STROKE
	  (#x0345 #x03B9) ;; COMBINING GREEK YPOGEGRAMMENI
	  (#x0370 #x0371) ;; GREEK CAPITAL LETTER HETA
	  (#x0372 #x0373) ;; GREEK CAPITAL LETTER ARCHAIC SAMPI
	  (#x0376 #x0377) ;; GREEK CAPITAL LETTER PAMPHYLIAN DIGAMMA
	  (#x0386 #x03AC) ;; GREEK CAPITAL LETTER ALPHA WITH TONOS
	  (#x0388 #x03AD) ;; GREEK CAPITAL LETTER EPSILON WITH TONOS
	  (#x0389 #x03AE) ;; GREEK CAPITAL LETTER ETA WITH TONOS
	  (#x038A #x03AF) ;; GREEK CAPITAL LETTER IOTA WITH TONOS
	  (#x038C #x03CC) ;; GREEK CAPITAL LETTER OMICRON WITH TONOS
	  (#x038E #x03CD) ;; GREEK CAPITAL LETTER UPSILON WITH TONOS
	  (#x038F #x03CE) ;; GREEK CAPITAL LETTER OMEGA WITH TONOS
;;; WARNING: Unhandled full mapping:
;;;   0390; F; 03B9 0308 0301; # GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
	  (#x0391 #x03B1) ;; GREEK CAPITAL LETTER ALPHA
	  (#x0392 #x03B2) ;; GREEK CAPITAL LETTER BETA
	  (#x0393 #x03B3) ;; GREEK CAPITAL LETTER GAMMA
	  (#x0394 #x03B4) ;; GREEK CAPITAL LETTER DELTA
	  (#x0395 #x03B5) ;; GREEK CAPITAL LETTER EPSILON
	  (#x0396 #x03B6) ;; GREEK CAPITAL LETTER ZETA
	  (#x0397 #x03B7) ;; GREEK CAPITAL LETTER ETA
	  (#x0398 #x03B8) ;; GREEK CAPITAL LETTER THETA
	  (#x0399 #x03B9) ;; GREEK CAPITAL LETTER IOTA
	  (#x039A #x03BA) ;; GREEK CAPITAL LETTER KAPPA
	  (#x039B #x03BB) ;; GREEK CAPITAL LETTER LAMDA
	  (#x039C #x03BC) ;; GREEK CAPITAL LETTER MU
	  (#x039D #x03BD) ;; GREEK CAPITAL LETTER NU
	  (#x039E #x03BE) ;; GREEK CAPITAL LETTER XI
	  (#x039F #x03BF) ;; GREEK CAPITAL LETTER OMICRON
	  (#x03A0 #x03C0) ;; GREEK CAPITAL LETTER PI
	  (#x03A1 #x03C1) ;; GREEK CAPITAL LETTER RHO
	  (#x03A3 #x03C3) ;; GREEK CAPITAL LETTER SIGMA
	  (#x03A4 #x03C4) ;; GREEK CAPITAL LETTER TAU
	  (#x03A5 #x03C5) ;; GREEK CAPITAL LETTER UPSILON
	  (#x03A6 #x03C6) ;; GREEK CAPITAL LETTER PHI
	  (#x03A7 #x03C7) ;; GREEK CAPITAL LETTER CHI
	  (#x03A8 #x03C8) ;; GREEK CAPITAL LETTER PSI
	  (#x03A9 #x03C9) ;; GREEK CAPITAL LETTER OMEGA
	  (#x03AA #x03CA) ;; GREEK CAPITAL LETTER IOTA WITH DIALYTIKA
	  (#x03AB #x03CB) ;; GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
;;; WARNING: Unhandled full mapping:
;;;   03B0; F; 03C5 0308 0301; # GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS
	  (#x03C2 #x03C3) ;; GREEK SMALL LETTER FINAL SIGMA
	  (#x03CF #x03D7) ;; GREEK CAPITAL KAI SYMBOL
	  (#x03D0 #x03B2) ;; GREEK BETA SYMBOL
	  (#x03D1 #x03B8) ;; GREEK THETA SYMBOL
	  (#x03D5 #x03C6) ;; GREEK PHI SYMBOL
	  (#x03D6 #x03C0) ;; GREEK PI SYMBOL
	  (#x03D8 #x03D9) ;; GREEK LETTER ARCHAIC KOPPA
	  (#x03DA #x03DB) ;; GREEK LETTER STIGMA
	  (#x03DC #x03DD) ;; GREEK LETTER DIGAMMA
	  (#x03DE #x03DF) ;; GREEK LETTER KOPPA
	  (#x03E0 #x03E1) ;; GREEK LETTER SAMPI
	  (#x03E2 #x03E3) ;; COPTIC CAPITAL LETTER SHEI
	  (#x03E4 #x03E5) ;; COPTIC CAPITAL LETTER FEI
	  (#x03E6 #x03E7) ;; COPTIC CAPITAL LETTER KHEI
	  (#x03E8 #x03E9) ;; COPTIC CAPITAL LETTER HORI
	  (#x03EA #x03EB) ;; COPTIC CAPITAL LETTER GANGIA
	  (#x03EC #x03ED) ;; COPTIC CAPITAL LETTER SHIMA
	  (#x03EE #x03EF) ;; COPTIC CAPITAL LETTER DEI
	  (#x03F0 #x03BA) ;; GREEK KAPPA SYMBOL
	  (#x03F1 #x03C1) ;; GREEK RHO SYMBOL
	  (#x03F4 #x03B8) ;; GREEK CAPITAL THETA SYMBOL
	  (#x03F5 #x03B5) ;; GREEK LUNATE EPSILON SYMBOL
	  (#x03F7 #x03F8) ;; GREEK CAPITAL LETTER SHO
	  (#x03F9 #x03F2) ;; GREEK CAPITAL LUNATE SIGMA SYMBOL
	  (#x03FA #x03FB) ;; GREEK CAPITAL LETTER SAN
	  (#x03FD #x037B) ;; GREEK CAPITAL REVERSED LUNATE SIGMA SYMBOL
	  (#x03FE #x037C) ;; GREEK CAPITAL DOTTED LUNATE SIGMA SYMBOL
	  (#x03FF #x037D) ;; GREEK CAPITAL REVERSED DOTTED LUNATE SIGMA SYMBOL
	  (#x0400 #x0450) ;; CYRILLIC CAPITAL LETTER IE WITH GRAVE
	  (#x0401 #x0451) ;; CYRILLIC CAPITAL LETTER IO
	  (#x0402 #x0452) ;; CYRILLIC CAPITAL LETTER DJE
	  (#x0403 #x0453) ;; CYRILLIC CAPITAL LETTER GJE
	  (#x0404 #x0454) ;; CYRILLIC CAPITAL LETTER UKRAINIAN IE
	  (#x0405 #x0455) ;; CYRILLIC CAPITAL LETTER DZE
	  (#x0406 #x0456) ;; CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I
	  (#x0407 #x0457) ;; CYRILLIC CAPITAL LETTER YI
	  (#x0408 #x0458) ;; CYRILLIC CAPITAL LETTER JE
	  (#x0409 #x0459) ;; CYRILLIC CAPITAL LETTER LJE
	  (#x040A #x045A) ;; CYRILLIC CAPITAL LETTER NJE
	  (#x040B #x045B) ;; CYRILLIC CAPITAL LETTER TSHE
	  (#x040C #x045C) ;; CYRILLIC CAPITAL LETTER KJE
	  (#x040D #x045D) ;; CYRILLIC CAPITAL LETTER I WITH GRAVE
	  (#x040E #x045E) ;; CYRILLIC CAPITAL LETTER SHORT U
	  (#x040F #x045F) ;; CYRILLIC CAPITAL LETTER DZHE
	  (#x0410 #x0430) ;; CYRILLIC CAPITAL LETTER A
	  (#x0411 #x0431) ;; CYRILLIC CAPITAL LETTER BE
	  (#x0412 #x0432) ;; CYRILLIC CAPITAL LETTER VE
	  (#x0413 #x0433) ;; CYRILLIC CAPITAL LETTER GHE
	  (#x0414 #x0434) ;; CYRILLIC CAPITAL LETTER DE
	  (#x0415 #x0435) ;; CYRILLIC CAPITAL LETTER IE
	  (#x0416 #x0436) ;; CYRILLIC CAPITAL LETTER ZHE
	  (#x0417 #x0437) ;; CYRILLIC CAPITAL LETTER ZE
	  (#x0418 #x0438) ;; CYRILLIC CAPITAL LETTER I
	  (#x0419 #x0439) ;; CYRILLIC CAPITAL LETTER SHORT I
	  (#x041A #x043A) ;; CYRILLIC CAPITAL LETTER KA
	  (#x041B #x043B) ;; CYRILLIC CAPITAL LETTER EL
	  (#x041C #x043C) ;; CYRILLIC CAPITAL LETTER EM
	  (#x041D #x043D) ;; CYRILLIC CAPITAL LETTER EN
	  (#x041E #x043E) ;; CYRILLIC CAPITAL LETTER O
	  (#x041F #x043F) ;; CYRILLIC CAPITAL LETTER PE
	  (#x0420 #x0440) ;; CYRILLIC CAPITAL LETTER ER
	  (#x0421 #x0441) ;; CYRILLIC CAPITAL LETTER ES
	  (#x0422 #x0442) ;; CYRILLIC CAPITAL LETTER TE
	  (#x0423 #x0443) ;; CYRILLIC CAPITAL LETTER U
	  (#x0424 #x0444) ;; CYRILLIC CAPITAL LETTER EF
	  (#x0425 #x0445) ;; CYRILLIC CAPITAL LETTER HA
	  (#x0426 #x0446) ;; CYRILLIC CAPITAL LETTER TSE
	  (#x0427 #x0447) ;; CYRILLIC CAPITAL LETTER CHE
	  (#x0428 #x0448) ;; CYRILLIC CAPITAL LETTER SHA
	  (#x0429 #x0449) ;; CYRILLIC CAPITAL LETTER SHCHA
	  (#x042A #x044A) ;; CYRILLIC CAPITAL LETTER HARD SIGN
	  (#x042B #x044B) ;; CYRILLIC CAPITAL LETTER YERU
	  (#x042C #x044C) ;; CYRILLIC CAPITAL LETTER SOFT SIGN
	  (#x042D #x044D) ;; CYRILLIC CAPITAL LETTER E
	  (#x042E #x044E) ;; CYRILLIC CAPITAL LETTER YU
	  (#x042F #x044F) ;; CYRILLIC CAPITAL LETTER YA
	  (#x0460 #x0461) ;; CYRILLIC CAPITAL LETTER OMEGA
	  (#x0462 #x0463) ;; CYRILLIC CAPITAL LETTER YAT
	  (#x0464 #x0465) ;; CYRILLIC CAPITAL LETTER IOTIFIED E
	  (#x0466 #x0467) ;; CYRILLIC CAPITAL LETTER LITTLE YUS
	  (#x0468 #x0469) ;; CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS
	  (#x046A #x046B) ;; CYRILLIC CAPITAL LETTER BIG YUS
	  (#x046C #x046D) ;; CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS
	  (#x046E #x046F) ;; CYRILLIC CAPITAL LETTER KSI
	  (#x0470 #x0471) ;; CYRILLIC CAPITAL LETTER PSI
	  (#x0472 #x0473) ;; CYRILLIC CAPITAL LETTER FITA
	  (#x0474 #x0475) ;; CYRILLIC CAPITAL LETTER IZHITSA
	  (#x0476 #x0477) ;; CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
	  (#x0478 #x0479) ;; CYRILLIC CAPITAL LETTER UK
	  (#x047A #x047B) ;; CYRILLIC CAPITAL LETTER ROUND OMEGA
	  (#x047C #x047D) ;; CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
	  (#x047E #x047F) ;; CYRILLIC CAPITAL LETTER OT
	  (#x0480 #x0481) ;; CYRILLIC CAPITAL LETTER KOPPA
	  (#x048A #x048B) ;; CYRILLIC CAPITAL LETTER SHORT I WITH TAIL
	  (#x048C #x048D) ;; CYRILLIC CAPITAL LETTER SEMISOFT SIGN
	  (#x048E #x048F) ;; CYRILLIC CAPITAL LETTER ER WITH TICK
	  (#x0490 #x0491) ;; CYRILLIC CAPITAL LETTER GHE WITH UPTURN
	  (#x0492 #x0493) ;; CYRILLIC CAPITAL LETTER GHE WITH STROKE
	  (#x0494 #x0495) ;; CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
	  (#x0496 #x0497) ;; CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
	  (#x0498 #x0499) ;; CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
	  (#x049A #x049B) ;; CYRILLIC CAPITAL LETTER KA WITH DESCENDER
	  (#x049C #x049D) ;; CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE
	  (#x049E #x049F) ;; CYRILLIC CAPITAL LETTER KA WITH STROKE
	  (#x04A0 #x04A1) ;; CYRILLIC CAPITAL LETTER BASHKIR KA
	  (#x04A2 #x04A3) ;; CYRILLIC CAPITAL LETTER EN WITH DESCENDER
	  (#x04A4 #x04A5) ;; CYRILLIC CAPITAL LIGATURE EN GHE
	  (#x04A6 #x04A7) ;; CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
	  (#x04A8 #x04A9) ;; CYRILLIC CAPITAL LETTER ABKHASIAN HA
	  (#x04AA #x04AB) ;; CYRILLIC CAPITAL LETTER ES WITH DESCENDER
	  (#x04AC #x04AD) ;; CYRILLIC CAPITAL LETTER TE WITH DESCENDER
	  (#x04AE #x04AF) ;; CYRILLIC CAPITAL LETTER STRAIGHT U
	  (#x04B0 #x04B1) ;; CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE
	  (#x04B2 #x04B3) ;; CYRILLIC CAPITAL LETTER HA WITH DESCENDER
	  (#x04B4 #x04B5) ;; CYRILLIC CAPITAL LIGATURE TE TSE
	  (#x04B6 #x04B7) ;; CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
	  (#x04B8 #x04B9) ;; CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE
	  (#x04BA #x04BB) ;; CYRILLIC CAPITAL LETTER SHHA
	  (#x04BC #x04BD) ;; CYRILLIC CAPITAL LETTER ABKHASIAN CHE
	  (#x04BE #x04BF) ;; CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER
	  (#x04C0 #x04CF) ;; CYRILLIC LETTER PALOCHKA
	  (#x04C1 #x04C2) ;; CYRILLIC CAPITAL LETTER ZHE WITH BREVE
	  (#x04C3 #x04C4) ;; CYRILLIC CAPITAL LETTER KA WITH HOOK
	  (#x04C5 #x04C6) ;; CYRILLIC CAPITAL LETTER EL WITH TAIL
	  (#x04C7 #x04C8) ;; CYRILLIC CAPITAL LETTER EN WITH HOOK
	  (#x04C9 #x04CA) ;; CYRILLIC CAPITAL LETTER EN WITH TAIL
	  (#x04CB #x04CC) ;; CYRILLIC CAPITAL LETTER KHAKASSIAN CHE
	  (#x04CD #x04CE) ;; CYRILLIC CAPITAL LETTER EM WITH TAIL
	  (#x04D0 #x04D1) ;; CYRILLIC CAPITAL LETTER A WITH BREVE
	  (#x04D2 #x04D3) ;; CYRILLIC CAPITAL LETTER A WITH DIAERESIS
	  (#x04D4 #x04D5) ;; CYRILLIC CAPITAL LIGATURE A IE
	  (#x04D6 #x04D7) ;; CYRILLIC CAPITAL LETTER IE WITH BREVE
	  (#x04D8 #x04D9) ;; CYRILLIC CAPITAL LETTER SCHWA
	  (#x04DA #x04DB) ;; CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
	  (#x04DC #x04DD) ;; CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
	  (#x04DE #x04DF) ;; CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
	  (#x04E0 #x04E1) ;; CYRILLIC CAPITAL LETTER ABKHASIAN DZE
	  (#x04E2 #x04E3) ;; CYRILLIC CAPITAL LETTER I WITH MACRON
	  (#x04E4 #x04E5) ;; CYRILLIC CAPITAL LETTER I WITH DIAERESIS
	  (#x04E6 #x04E7) ;; CYRILLIC CAPITAL LETTER O WITH DIAERESIS
	  (#x04E8 #x04E9) ;; CYRILLIC CAPITAL LETTER BARRED O
	  (#x04EA #x04EB) ;; CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS
	  (#x04EC #x04ED) ;; CYRILLIC CAPITAL LETTER E WITH DIAERESIS
	  (#x04EE #x04EF) ;; CYRILLIC CAPITAL LETTER U WITH MACRON
	  (#x04F0 #x04F1) ;; CYRILLIC CAPITAL LETTER U WITH DIAERESIS
	  (#x04F2 #x04F3) ;; CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
	  (#x04F4 #x04F5) ;; CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
	  (#x04F6 #x04F7) ;; CYRILLIC CAPITAL LETTER GHE WITH DESCENDER
	  (#x04F8 #x04F9) ;; CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
	  (#x04FA #x04FB) ;; CYRILLIC CAPITAL LETTER GHE WITH STROKE AND HOOK
	  (#x04FC #x04FD) ;; CYRILLIC CAPITAL LETTER HA WITH HOOK
	  (#x04FE #x04FF) ;; CYRILLIC CAPITAL LETTER HA WITH STROKE
	  (#x0500 #x0501) ;; CYRILLIC CAPITAL LETTER KOMI DE
	  (#x0502 #x0503) ;; CYRILLIC CAPITAL LETTER KOMI DJE
	  (#x0504 #x0505) ;; CYRILLIC CAPITAL LETTER KOMI ZJE
	  (#x0506 #x0507) ;; CYRILLIC CAPITAL LETTER KOMI DZJE
	  (#x0508 #x0509) ;; CYRILLIC CAPITAL LETTER KOMI LJE
	  (#x050A #x050B) ;; CYRILLIC CAPITAL LETTER KOMI NJE
	  (#x050C #x050D) ;; CYRILLIC CAPITAL LETTER KOMI SJE
	  (#x050E #x050F) ;; CYRILLIC CAPITAL LETTER KOMI TJE
	  (#x0510 #x0511) ;; CYRILLIC CAPITAL LETTER REVERSED ZE
	  (#x0512 #x0513) ;; CYRILLIC CAPITAL LETTER EL WITH HOOK
	  (#x0514 #x0515) ;; CYRILLIC CAPITAL LETTER LHA
	  (#x0516 #x0517) ;; CYRILLIC CAPITAL LETTER RHA
	  (#x0518 #x0519) ;; CYRILLIC CAPITAL LETTER YAE
	  (#x051A #x051B) ;; CYRILLIC CAPITAL LETTER QA
	  (#x051C #x051D) ;; CYRILLIC CAPITAL LETTER WE
	  (#x051E #x051F) ;; CYRILLIC CAPITAL LETTER ALEUT KA
	  (#x0520 #x0521) ;; CYRILLIC CAPITAL LETTER EL WITH MIDDLE HOOK
	  (#x0522 #x0523) ;; CYRILLIC CAPITAL LETTER EN WITH MIDDLE HOOK
	  (#x0524 #x0525) ;; CYRILLIC CAPITAL LETTER PE WITH DESCENDER
	  (#x0531 #x0561) ;; ARMENIAN CAPITAL LETTER AYB
	  (#x0532 #x0562) ;; ARMENIAN CAPITAL LETTER BEN
	  (#x0533 #x0563) ;; ARMENIAN CAPITAL LETTER GIM
	  (#x0534 #x0564) ;; ARMENIAN CAPITAL LETTER DA
	  (#x0535 #x0565) ;; ARMENIAN CAPITAL LETTER ECH
	  (#x0536 #x0566) ;; ARMENIAN CAPITAL LETTER ZA
	  (#x0537 #x0567) ;; ARMENIAN CAPITAL LETTER EH
	  (#x0538 #x0568) ;; ARMENIAN CAPITAL LETTER ET
	  (#x0539 #x0569) ;; ARMENIAN CAPITAL LETTER TO
	  (#x053A #x056A) ;; ARMENIAN CAPITAL LETTER ZHE
	  (#x053B #x056B) ;; ARMENIAN CAPITAL LETTER INI
	  (#x053C #x056C) ;; ARMENIAN CAPITAL LETTER LIWN
	  (#x053D #x056D) ;; ARMENIAN CAPITAL LETTER XEH
	  (#x053E #x056E) ;; ARMENIAN CAPITAL LETTER CA
	  (#x053F #x056F) ;; ARMENIAN CAPITAL LETTER KEN
	  (#x0540 #x0570) ;; ARMENIAN CAPITAL LETTER HO
	  (#x0541 #x0571) ;; ARMENIAN CAPITAL LETTER JA
	  (#x0542 #x0572) ;; ARMENIAN CAPITAL LETTER GHAD
	  (#x0543 #x0573) ;; ARMENIAN CAPITAL LETTER CHEH
	  (#x0544 #x0574) ;; ARMENIAN CAPITAL LETTER MEN
	  (#x0545 #x0575) ;; ARMENIAN CAPITAL LETTER YI
	  (#x0546 #x0576) ;; ARMENIAN CAPITAL LETTER NOW
	  (#x0547 #x0577) ;; ARMENIAN CAPITAL LETTER SHA
	  (#x0548 #x0578) ;; ARMENIAN CAPITAL LETTER VO
	  (#x0549 #x0579) ;; ARMENIAN CAPITAL LETTER CHA
	  (#x054A #x057A) ;; ARMENIAN CAPITAL LETTER PEH
	  (#x054B #x057B) ;; ARMENIAN CAPITAL LETTER JHEH
	  (#x054C #x057C) ;; ARMENIAN CAPITAL LETTER RA
	  (#x054D #x057D) ;; ARMENIAN CAPITAL LETTER SEH
	  (#x054E #x057E) ;; ARMENIAN CAPITAL LETTER VEW
	  (#x054F #x057F) ;; ARMENIAN CAPITAL LETTER TIWN
	  (#x0550 #x0580) ;; ARMENIAN CAPITAL LETTER REH
	  (#x0551 #x0581) ;; ARMENIAN CAPITAL LETTER CO
	  (#x0552 #x0582) ;; ARMENIAN CAPITAL LETTER YIWN
	  (#x0553 #x0583) ;; ARMENIAN CAPITAL LETTER PIWR
	  (#x0554 #x0584) ;; ARMENIAN CAPITAL LETTER KEH
	  (#x0555 #x0585) ;; ARMENIAN CAPITAL LETTER OH
	  (#x0556 #x0586) ;; ARMENIAN CAPITAL LETTER FEH
;;; WARNING: Unhandled full mapping:
;;;   0587; F; 0565 0582; # ARMENIAN SMALL LIGATURE ECH YIWN
	  (#x10A0 #x2D00) ;; GEORGIAN CAPITAL LETTER AN
	  (#x10A1 #x2D01) ;; GEORGIAN CAPITAL LETTER BAN
	  (#x10A2 #x2D02) ;; GEORGIAN CAPITAL LETTER GAN
	  (#x10A3 #x2D03) ;; GEORGIAN CAPITAL LETTER DON
	  (#x10A4 #x2D04) ;; GEORGIAN CAPITAL LETTER EN
	  (#x10A5 #x2D05) ;; GEORGIAN CAPITAL LETTER VIN
	  (#x10A6 #x2D06) ;; GEORGIAN CAPITAL LETTER ZEN
	  (#x10A7 #x2D07) ;; GEORGIAN CAPITAL LETTER TAN
	  (#x10A8 #x2D08) ;; GEORGIAN CAPITAL LETTER IN
	  (#x10A9 #x2D09) ;; GEORGIAN CAPITAL LETTER KAN
	  (#x10AA #x2D0A) ;; GEORGIAN CAPITAL LETTER LAS
	  (#x10AB #x2D0B) ;; GEORGIAN CAPITAL LETTER MAN
	  (#x10AC #x2D0C) ;; GEORGIAN CAPITAL LETTER NAR
	  (#x10AD #x2D0D) ;; GEORGIAN CAPITAL LETTER ON
	  (#x10AE #x2D0E) ;; GEORGIAN CAPITAL LETTER PAR
	  (#x10AF #x2D0F) ;; GEORGIAN CAPITAL LETTER ZHAR
	  (#x10B0 #x2D10) ;; GEORGIAN CAPITAL LETTER RAE
	  (#x10B1 #x2D11) ;; GEORGIAN CAPITAL LETTER SAN
	  (#x10B2 #x2D12) ;; GEORGIAN CAPITAL LETTER TAR
	  (#x10B3 #x2D13) ;; GEORGIAN CAPITAL LETTER UN
	  (#x10B4 #x2D14) ;; GEORGIAN CAPITAL LETTER PHAR
	  (#x10B5 #x2D15) ;; GEORGIAN CAPITAL LETTER KHAR
	  (#x10B6 #x2D16) ;; GEORGIAN CAPITAL LETTER GHAN
	  (#x10B7 #x2D17) ;; GEORGIAN CAPITAL LETTER QAR
	  (#x10B8 #x2D18) ;; GEORGIAN CAPITAL LETTER SHIN
	  (#x10B9 #x2D19) ;; GEORGIAN CAPITAL LETTER CHIN
	  (#x10BA #x2D1A) ;; GEORGIAN CAPITAL LETTER CAN
	  (#x10BB #x2D1B) ;; GEORGIAN CAPITAL LETTER JIL
	  (#x10BC #x2D1C) ;; GEORGIAN CAPITAL LETTER CIL
	  (#x10BD #x2D1D) ;; GEORGIAN CAPITAL LETTER CHAR
	  (#x10BE #x2D1E) ;; GEORGIAN CAPITAL LETTER XAN
	  (#x10BF #x2D1F) ;; GEORGIAN CAPITAL LETTER JHAN
	  (#x10C0 #x2D20) ;; GEORGIAN CAPITAL LETTER HAE
	  (#x10C1 #x2D21) ;; GEORGIAN CAPITAL LETTER HE
	  (#x10C2 #x2D22) ;; GEORGIAN CAPITAL LETTER HIE
	  (#x10C3 #x2D23) ;; GEORGIAN CAPITAL LETTER WE
	  (#x10C4 #x2D24) ;; GEORGIAN CAPITAL LETTER HAR
	  (#x10C5 #x2D25) ;; GEORGIAN CAPITAL LETTER HOE
	  (#x1E00 #x1E01) ;; LATIN CAPITAL LETTER A WITH RING BELOW
	  (#x1E02 #x1E03) ;; LATIN CAPITAL LETTER B WITH DOT ABOVE
	  (#x1E04 #x1E05) ;; LATIN CAPITAL LETTER B WITH DOT BELOW
	  (#x1E06 #x1E07) ;; LATIN CAPITAL LETTER B WITH LINE BELOW
	  (#x1E08 #x1E09) ;; LATIN CAPITAL LETTER C WITH CEDILLA AND ACUTE
	  (#x1E0A #x1E0B) ;; LATIN CAPITAL LETTER D WITH DOT ABOVE
	  (#x1E0C #x1E0D) ;; LATIN CAPITAL LETTER D WITH DOT BELOW
	  (#x1E0E #x1E0F) ;; LATIN CAPITAL LETTER D WITH LINE BELOW
	  (#x1E10 #x1E11) ;; LATIN CAPITAL LETTER D WITH CEDILLA
	  (#x1E12 #x1E13) ;; LATIN CAPITAL LETTER D WITH CIRCUMFLEX BELOW
	  (#x1E14 #x1E15) ;; LATIN CAPITAL LETTER E WITH MACRON AND GRAVE
	  (#x1E16 #x1E17) ;; LATIN CAPITAL LETTER E WITH MACRON AND ACUTE
	  (#x1E18 #x1E19) ;; LATIN CAPITAL LETTER E WITH CIRCUMFLEX BELOW
	  (#x1E1A #x1E1B) ;; LATIN CAPITAL LETTER E WITH TILDE BELOW
	  (#x1E1C #x1E1D) ;; LATIN CAPITAL LETTER E WITH CEDILLA AND BREVE
	  (#x1E1E #x1E1F) ;; LATIN CAPITAL LETTER F WITH DOT ABOVE
	  (#x1E20 #x1E21) ;; LATIN CAPITAL LETTER G WITH MACRON
	  (#x1E22 #x1E23) ;; LATIN CAPITAL LETTER H WITH DOT ABOVE
	  (#x1E24 #x1E25) ;; LATIN CAPITAL LETTER H WITH DOT BELOW
	  (#x1E26 #x1E27) ;; LATIN CAPITAL LETTER H WITH DIAERESIS
	  (#x1E28 #x1E29) ;; LATIN CAPITAL LETTER H WITH CEDILLA
	  (#x1E2A #x1E2B) ;; LATIN CAPITAL LETTER H WITH BREVE BELOW
	  (#x1E2C #x1E2D) ;; LATIN CAPITAL LETTER I WITH TILDE BELOW
	  (#x1E2E #x1E2F) ;; LATIN CAPITAL LETTER I WITH DIAERESIS AND ACUTE
	  (#x1E30 #x1E31) ;; LATIN CAPITAL LETTER K WITH ACUTE
	  (#x1E32 #x1E33) ;; LATIN CAPITAL LETTER K WITH DOT BELOW
	  (#x1E34 #x1E35) ;; LATIN CAPITAL LETTER K WITH LINE BELOW
	  (#x1E36 #x1E37) ;; LATIN CAPITAL LETTER L WITH DOT BELOW
	  (#x1E38 #x1E39) ;; LATIN CAPITAL LETTER L WITH DOT BELOW AND MACRON
	  (#x1E3A #x1E3B) ;; LATIN CAPITAL LETTER L WITH LINE BELOW
	  (#x1E3C #x1E3D) ;; LATIN CAPITAL LETTER L WITH CIRCUMFLEX BELOW
	  (#x1E3E #x1E3F) ;; LATIN CAPITAL LETTER M WITH ACUTE
	  (#x1E40 #x1E41) ;; LATIN CAPITAL LETTER M WITH DOT ABOVE
	  (#x1E42 #x1E43) ;; LATIN CAPITAL LETTER M WITH DOT BELOW
	  (#x1E44 #x1E45) ;; LATIN CAPITAL LETTER N WITH DOT ABOVE
	  (#x1E46 #x1E47) ;; LATIN CAPITAL LETTER N WITH DOT BELOW
	  (#x1E48 #x1E49) ;; LATIN CAPITAL LETTER N WITH LINE BELOW
	  (#x1E4A #x1E4B) ;; LATIN CAPITAL LETTER N WITH CIRCUMFLEX BELOW
	  (#x1E4C #x1E4D) ;; LATIN CAPITAL LETTER O WITH TILDE AND ACUTE
	  (#x1E4E #x1E4F) ;; LATIN CAPITAL LETTER O WITH TILDE AND DIAERESIS
	  (#x1E50 #x1E51) ;; LATIN CAPITAL LETTER O WITH MACRON AND GRAVE
	  (#x1E52 #x1E53) ;; LATIN CAPITAL LETTER O WITH MACRON AND ACUTE
	  (#x1E54 #x1E55) ;; LATIN CAPITAL LETTER P WITH ACUTE
	  (#x1E56 #x1E57) ;; LATIN CAPITAL LETTER P WITH DOT ABOVE
	  (#x1E58 #x1E59) ;; LATIN CAPITAL LETTER R WITH DOT ABOVE
	  (#x1E5A #x1E5B) ;; LATIN CAPITAL LETTER R WITH DOT BELOW
	  (#x1E5C #x1E5D) ;; LATIN CAPITAL LETTER R WITH DOT BELOW AND MACRON
	  (#x1E5E #x1E5F) ;; LATIN CAPITAL LETTER R WITH LINE BELOW
	  (#x1E60 #x1E61) ;; LATIN CAPITAL LETTER S WITH DOT ABOVE
	  (#x1E62 #x1E63) ;; LATIN CAPITAL LETTER S WITH DOT BELOW
	  (#x1E64 #x1E65) ;; LATIN CAPITAL LETTER S WITH ACUTE AND DOT ABOVE
	  (#x1E66 #x1E67) ;; LATIN CAPITAL LETTER S WITH CARON AND DOT ABOVE
	  (#x1E68 #x1E69) ;; LATIN CAPITAL LETTER S WITH DOT BELOW AND DOT ABOVE
	  (#x1E6A #x1E6B) ;; LATIN CAPITAL LETTER T WITH DOT ABOVE
	  (#x1E6C #x1E6D) ;; LATIN CAPITAL LETTER T WITH DOT BELOW
	  (#x1E6E #x1E6F) ;; LATIN CAPITAL LETTER T WITH LINE BELOW
	  (#x1E70 #x1E71) ;; LATIN CAPITAL LETTER T WITH CIRCUMFLEX BELOW
	  (#x1E72 #x1E73) ;; LATIN CAPITAL LETTER U WITH DIAERESIS BELOW
	  (#x1E74 #x1E75) ;; LATIN CAPITAL LETTER U WITH TILDE BELOW
	  (#x1E76 #x1E77) ;; LATIN CAPITAL LETTER U WITH CIRCUMFLEX BELOW
	  (#x1E78 #x1E79) ;; LATIN CAPITAL LETTER U WITH TILDE AND ACUTE
	  (#x1E7A #x1E7B) ;; LATIN CAPITAL LETTER U WITH MACRON AND DIAERESIS
	  (#x1E7C #x1E7D) ;; LATIN CAPITAL LETTER V WITH TILDE
	  (#x1E7E #x1E7F) ;; LATIN CAPITAL LETTER V WITH DOT BELOW
	  (#x1E80 #x1E81) ;; LATIN CAPITAL LETTER W WITH GRAVE
	  (#x1E82 #x1E83) ;; LATIN CAPITAL LETTER W WITH ACUTE
	  (#x1E84 #x1E85) ;; LATIN CAPITAL LETTER W WITH DIAERESIS
	  (#x1E86 #x1E87) ;; LATIN CAPITAL LETTER W WITH DOT ABOVE
	  (#x1E88 #x1E89) ;; LATIN CAPITAL LETTER W WITH DOT BELOW
	  (#x1E8A #x1E8B) ;; LATIN CAPITAL LETTER X WITH DOT ABOVE
	  (#x1E8C #x1E8D) ;; LATIN CAPITAL LETTER X WITH DIAERESIS
	  (#x1E8E #x1E8F) ;; LATIN CAPITAL LETTER Y WITH DOT ABOVE
	  (#x1E90 #x1E91) ;; LATIN CAPITAL LETTER Z WITH CIRCUMFLEX
	  (#x1E92 #x1E93) ;; LATIN CAPITAL LETTER Z WITH DOT BELOW
	  (#x1E94 #x1E95) ;; LATIN CAPITAL LETTER Z WITH LINE BELOW
;;; WARNING: Unhandled full mapping:
;;;   1E96; F; 0068 0331; # LATIN SMALL LETTER H WITH LINE BELOW
;;; WARNING: Unhandled full mapping:
;;;   1E97; F; 0074 0308; # LATIN SMALL LETTER T WITH DIAERESIS
;;; WARNING: Unhandled full mapping:
;;;   1E98; F; 0077 030A; # LATIN SMALL LETTER W WITH RING ABOVE
;;; WARNING: Unhandled full mapping:
;;;   1E99; F; 0079 030A; # LATIN SMALL LETTER Y WITH RING ABOVE
;;; WARNING: Unhandled full mapping:
;;;   1E9A; F; 0061 02BE; # LATIN SMALL LETTER A WITH RIGHT HALF RING
	  (#x1E9B #x1E61) ;; LATIN SMALL LETTER LONG S WITH DOT ABOVE
;;; WARNING: Unhandled full mapping:
;;;   1E9E; F; 0073 0073; # LATIN CAPITAL LETTER SHARP S
	  (#x1E9E #x00DF) ;; LATIN CAPITAL LETTER SHARP S
	  (#x1EA0 #x1EA1) ;; LATIN CAPITAL LETTER A WITH DOT BELOW
	  (#x1EA2 #x1EA3) ;; LATIN CAPITAL LETTER A WITH HOOK ABOVE
	  (#x1EA4 #x1EA5) ;; LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND ACUTE
	  (#x1EA6 #x1EA7) ;; LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND GRAVE
	  (#x1EA8 #x1EA9) ;; LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND HOOK ABOVE
	  (#x1EAA #x1EAB) ;; LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND TILDE
	  (#x1EAC #x1EAD) ;; LATIN CAPITAL LETTER A WITH CIRCUMFLEX AND DOT BELOW
	  (#x1EAE #x1EAF) ;; LATIN CAPITAL LETTER A WITH BREVE AND ACUTE
	  (#x1EB0 #x1EB1) ;; LATIN CAPITAL LETTER A WITH BREVE AND GRAVE
	  (#x1EB2 #x1EB3) ;; LATIN CAPITAL LETTER A WITH BREVE AND HOOK ABOVE
	  (#x1EB4 #x1EB5) ;; LATIN CAPITAL LETTER A WITH BREVE AND TILDE
	  (#x1EB6 #x1EB7) ;; LATIN CAPITAL LETTER A WITH BREVE AND DOT BELOW
	  (#x1EB8 #x1EB9) ;; LATIN CAPITAL LETTER E WITH DOT BELOW
	  (#x1EBA #x1EBB) ;; LATIN CAPITAL LETTER E WITH HOOK ABOVE
	  (#x1EBC #x1EBD) ;; LATIN CAPITAL LETTER E WITH TILDE
	  (#x1EBE #x1EBF) ;; LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND ACUTE
	  (#x1EC0 #x1EC1) ;; LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND GRAVE
	  (#x1EC2 #x1EC3) ;; LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND HOOK ABOVE
	  (#x1EC4 #x1EC5) ;; LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND TILDE
	  (#x1EC6 #x1EC7) ;; LATIN CAPITAL LETTER E WITH CIRCUMFLEX AND DOT BELOW
	  (#x1EC8 #x1EC9) ;; LATIN CAPITAL LETTER I WITH HOOK ABOVE
	  (#x1ECA #x1ECB) ;; LATIN CAPITAL LETTER I WITH DOT BELOW
	  (#x1ECC #x1ECD) ;; LATIN CAPITAL LETTER O WITH DOT BELOW
	  (#x1ECE #x1ECF) ;; LATIN CAPITAL LETTER O WITH HOOK ABOVE
	  (#x1ED0 #x1ED1) ;; LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND ACUTE
	  (#x1ED2 #x1ED3) ;; LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND GRAVE
	  (#x1ED4 #x1ED5) ;; LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND HOOK ABOVE
	  (#x1ED6 #x1ED7) ;; LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND TILDE
	  (#x1ED8 #x1ED9) ;; LATIN CAPITAL LETTER O WITH CIRCUMFLEX AND DOT BELOW
	  (#x1EDA #x1EDB) ;; LATIN CAPITAL LETTER O WITH HORN AND ACUTE
	  (#x1EDC #x1EDD) ;; LATIN CAPITAL LETTER O WITH HORN AND GRAVE
	  (#x1EDE #x1EDF) ;; LATIN CAPITAL LETTER O WITH HORN AND HOOK ABOVE
	  (#x1EE0 #x1EE1) ;; LATIN CAPITAL LETTER O WITH HORN AND TILDE
	  (#x1EE2 #x1EE3) ;; LATIN CAPITAL LETTER O WITH HORN AND DOT BELOW
	  (#x1EE4 #x1EE5) ;; LATIN CAPITAL LETTER U WITH DOT BELOW
	  (#x1EE6 #x1EE7) ;; LATIN CAPITAL LETTER U WITH HOOK ABOVE
	  (#x1EE8 #x1EE9) ;; LATIN CAPITAL LETTER U WITH HORN AND ACUTE
	  (#x1EEA #x1EEB) ;; LATIN CAPITAL LETTER U WITH HORN AND GRAVE
	  (#x1EEC #x1EED) ;; LATIN CAPITAL LETTER U WITH HORN AND HOOK ABOVE
	  (#x1EEE #x1EEF) ;; LATIN CAPITAL LETTER U WITH HORN AND TILDE
	  (#x1EF0 #x1EF1) ;; LATIN CAPITAL LETTER U WITH HORN AND DOT BELOW
	  (#x1EF2 #x1EF3) ;; LATIN CAPITAL LETTER Y WITH GRAVE
	  (#x1EF4 #x1EF5) ;; LATIN CAPITAL LETTER Y WITH DOT BELOW
	  (#x1EF6 #x1EF7) ;; LATIN CAPITAL LETTER Y WITH HOOK ABOVE
	  (#x1EF8 #x1EF9) ;; LATIN CAPITAL LETTER Y WITH TILDE
	  (#x1EFA #x1EFB) ;; LATIN CAPITAL LETTER MIDDLE-WELSH LL
	  (#x1EFC #x1EFD) ;; LATIN CAPITAL LETTER MIDDLE-WELSH V
	  (#x1EFE #x1EFF) ;; LATIN CAPITAL LETTER Y WITH LOOP
	  (#x1F08 #x1F00) ;; GREEK CAPITAL LETTER ALPHA WITH PSILI
	  (#x1F09 #x1F01) ;; GREEK CAPITAL LETTER ALPHA WITH DASIA
	  (#x1F0A #x1F02) ;; GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA
	  (#x1F0B #x1F03) ;; GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA
	  (#x1F0C #x1F04) ;; GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA
	  (#x1F0D #x1F05) ;; GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA
	  (#x1F0E #x1F06) ;; GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI
	  (#x1F0F #x1F07) ;; GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI
	  (#x1F18 #x1F10) ;; GREEK CAPITAL LETTER EPSILON WITH PSILI
	  (#x1F19 #x1F11) ;; GREEK CAPITAL LETTER EPSILON WITH DASIA
	  (#x1F1A #x1F12) ;; GREEK CAPITAL LETTER EPSILON WITH PSILI AND VARIA
	  (#x1F1B #x1F13) ;; GREEK CAPITAL LETTER EPSILON WITH DASIA AND VARIA
	  (#x1F1C #x1F14) ;; GREEK CAPITAL LETTER EPSILON WITH PSILI AND OXIA
	  (#x1F1D #x1F15) ;; GREEK CAPITAL LETTER EPSILON WITH DASIA AND OXIA
	  (#x1F28 #x1F20) ;; GREEK CAPITAL LETTER ETA WITH PSILI
	  (#x1F29 #x1F21) ;; GREEK CAPITAL LETTER ETA WITH DASIA
	  (#x1F2A #x1F22) ;; GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA
	  (#x1F2B #x1F23) ;; GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA
	  (#x1F2C #x1F24) ;; GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA
	  (#x1F2D #x1F25) ;; GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA
	  (#x1F2E #x1F26) ;; GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI
	  (#x1F2F #x1F27) ;; GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI
	  (#x1F38 #x1F30) ;; GREEK CAPITAL LETTER IOTA WITH PSILI
	  (#x1F39 #x1F31) ;; GREEK CAPITAL LETTER IOTA WITH DASIA
	  (#x1F3A #x1F32) ;; GREEK CAPITAL LETTER IOTA WITH PSILI AND VARIA
	  (#x1F3B #x1F33) ;; GREEK CAPITAL LETTER IOTA WITH DASIA AND VARIA
	  (#x1F3C #x1F34) ;; GREEK CAPITAL LETTER IOTA WITH PSILI AND OXIA
	  (#x1F3D #x1F35) ;; GREEK CAPITAL LETTER IOTA WITH DASIA AND OXIA
	  (#x1F3E #x1F36) ;; GREEK CAPITAL LETTER IOTA WITH PSILI AND PERISPOMENI
	  (#x1F3F #x1F37) ;; GREEK CAPITAL LETTER IOTA WITH DASIA AND PERISPOMENI
	  (#x1F48 #x1F40) ;; GREEK CAPITAL LETTER OMICRON WITH PSILI
	  (#x1F49 #x1F41) ;; GREEK CAPITAL LETTER OMICRON WITH DASIA
	  (#x1F4A #x1F42) ;; GREEK CAPITAL LETTER OMICRON WITH PSILI AND VARIA
	  (#x1F4B #x1F43) ;; GREEK CAPITAL LETTER OMICRON WITH DASIA AND VARIA
	  (#x1F4C #x1F44) ;; GREEK CAPITAL LETTER OMICRON WITH PSILI AND OXIA
	  (#x1F4D #x1F45) ;; GREEK CAPITAL LETTER OMICRON WITH DASIA AND OXIA
;;; WARNING: Unhandled full mapping:
;;;   1F50; F; 03C5 0313; # GREEK SMALL LETTER UPSILON WITH PSILI
;;; WARNING: Unhandled full mapping:
;;;   1F52; F; 03C5 0313 0300; # GREEK SMALL LETTER UPSILON WITH PSILI AND VARIA
;;; WARNING: Unhandled full mapping:
;;;   1F54; F; 03C5 0313 0301; # GREEK SMALL LETTER UPSILON WITH PSILI AND OXIA
;;; WARNING: Unhandled full mapping:
;;;   1F56; F; 03C5 0313 0342; # GREEK SMALL LETTER UPSILON WITH PSILI AND PERISPOMENI
	  (#x1F59 #x1F51) ;; GREEK CAPITAL LETTER UPSILON WITH DASIA
	  (#x1F5B #x1F53) ;; GREEK CAPITAL LETTER UPSILON WITH DASIA AND VARIA
	  (#x1F5D #x1F55) ;; GREEK CAPITAL LETTER UPSILON WITH DASIA AND OXIA
	  (#x1F5F #x1F57) ;; GREEK CAPITAL LETTER UPSILON WITH DASIA AND PERISPOMENI
	  (#x1F68 #x1F60) ;; GREEK CAPITAL LETTER OMEGA WITH PSILI
	  (#x1F69 #x1F61) ;; GREEK CAPITAL LETTER OMEGA WITH DASIA
	  (#x1F6A #x1F62) ;; GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA
	  (#x1F6B #x1F63) ;; GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA
	  (#x1F6C #x1F64) ;; GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA
	  (#x1F6D #x1F65) ;; GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA
	  (#x1F6E #x1F66) ;; GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI
	  (#x1F6F #x1F67) ;; GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI
;;; WARNING: Unhandled full mapping:
;;;   1F80; F; 1F00 03B9; # GREEK SMALL LETTER ALPHA WITH PSILI AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F81; F; 1F01 03B9; # GREEK SMALL LETTER ALPHA WITH DASIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F82; F; 1F02 03B9; # GREEK SMALL LETTER ALPHA WITH PSILI AND VARIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F83; F; 1F03 03B9; # GREEK SMALL LETTER ALPHA WITH DASIA AND VARIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F84; F; 1F04 03B9; # GREEK SMALL LETTER ALPHA WITH PSILI AND OXIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F85; F; 1F05 03B9; # GREEK SMALL LETTER ALPHA WITH DASIA AND OXIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F86; F; 1F06 03B9; # GREEK SMALL LETTER ALPHA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F87; F; 1F07 03B9; # GREEK SMALL LETTER ALPHA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F88; F; 1F00 03B9; # GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI
	  (#x1F88 #x1F80) ;; GREEK CAPITAL LETTER ALPHA WITH PSILI AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F89; F; 1F01 03B9; # GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI
	  (#x1F89 #x1F81) ;; GREEK CAPITAL LETTER ALPHA WITH DASIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F8A; F; 1F02 03B9; # GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI
	  (#x1F8A #x1F82) ;; GREEK CAPITAL LETTER ALPHA WITH PSILI AND VARIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F8B; F; 1F03 03B9; # GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI
	  (#x1F8B #x1F83) ;; GREEK CAPITAL LETTER ALPHA WITH DASIA AND VARIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F8C; F; 1F04 03B9; # GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI
	  (#x1F8C #x1F84) ;; GREEK CAPITAL LETTER ALPHA WITH PSILI AND OXIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F8D; F; 1F05 03B9; # GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI
	  (#x1F8D #x1F85) ;; GREEK CAPITAL LETTER ALPHA WITH DASIA AND OXIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F8E; F; 1F06 03B9; # GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
	  (#x1F8E #x1F86) ;; GREEK CAPITAL LETTER ALPHA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F8F; F; 1F07 03B9; # GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
	  (#x1F8F #x1F87) ;; GREEK CAPITAL LETTER ALPHA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F90; F; 1F20 03B9; # GREEK SMALL LETTER ETA WITH PSILI AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F91; F; 1F21 03B9; # GREEK SMALL LETTER ETA WITH DASIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F92; F; 1F22 03B9; # GREEK SMALL LETTER ETA WITH PSILI AND VARIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F93; F; 1F23 03B9; # GREEK SMALL LETTER ETA WITH DASIA AND VARIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F94; F; 1F24 03B9; # GREEK SMALL LETTER ETA WITH PSILI AND OXIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F95; F; 1F25 03B9; # GREEK SMALL LETTER ETA WITH DASIA AND OXIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F96; F; 1F26 03B9; # GREEK SMALL LETTER ETA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F97; F; 1F27 03B9; # GREEK SMALL LETTER ETA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F98; F; 1F20 03B9; # GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI
	  (#x1F98 #x1F90) ;; GREEK CAPITAL LETTER ETA WITH PSILI AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F99; F; 1F21 03B9; # GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI
	  (#x1F99 #x1F91) ;; GREEK CAPITAL LETTER ETA WITH DASIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F9A; F; 1F22 03B9; # GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI
	  (#x1F9A #x1F92) ;; GREEK CAPITAL LETTER ETA WITH PSILI AND VARIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F9B; F; 1F23 03B9; # GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI
	  (#x1F9B #x1F93) ;; GREEK CAPITAL LETTER ETA WITH DASIA AND VARIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F9C; F; 1F24 03B9; # GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI
	  (#x1F9C #x1F94) ;; GREEK CAPITAL LETTER ETA WITH PSILI AND OXIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F9D; F; 1F25 03B9; # GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI
	  (#x1F9D #x1F95) ;; GREEK CAPITAL LETTER ETA WITH DASIA AND OXIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F9E; F; 1F26 03B9; # GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
	  (#x1F9E #x1F96) ;; GREEK CAPITAL LETTER ETA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1F9F; F; 1F27 03B9; # GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
	  (#x1F9F #x1F97) ;; GREEK CAPITAL LETTER ETA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FA0; F; 1F60 03B9; # GREEK SMALL LETTER OMEGA WITH PSILI AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FA1; F; 1F61 03B9; # GREEK SMALL LETTER OMEGA WITH DASIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FA2; F; 1F62 03B9; # GREEK SMALL LETTER OMEGA WITH PSILI AND VARIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FA3; F; 1F63 03B9; # GREEK SMALL LETTER OMEGA WITH DASIA AND VARIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FA4; F; 1F64 03B9; # GREEK SMALL LETTER OMEGA WITH PSILI AND OXIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FA5; F; 1F65 03B9; # GREEK SMALL LETTER OMEGA WITH DASIA AND OXIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FA6; F; 1F66 03B9; # GREEK SMALL LETTER OMEGA WITH PSILI AND PERISPOMENI AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FA7; F; 1F67 03B9; # GREEK SMALL LETTER OMEGA WITH DASIA AND PERISPOMENI AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FA8; F; 1F60 03B9; # GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI
	  (#x1FA8 #x1FA0) ;; GREEK CAPITAL LETTER OMEGA WITH PSILI AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FA9; F; 1F61 03B9; # GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI
	  (#x1FA9 #x1FA1) ;; GREEK CAPITAL LETTER OMEGA WITH DASIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FAA; F; 1F62 03B9; # GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI
	  (#x1FAA #x1FA2) ;; GREEK CAPITAL LETTER OMEGA WITH PSILI AND VARIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FAB; F; 1F63 03B9; # GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI
	  (#x1FAB #x1FA3) ;; GREEK CAPITAL LETTER OMEGA WITH DASIA AND VARIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FAC; F; 1F64 03B9; # GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI
	  (#x1FAC #x1FA4) ;; GREEK CAPITAL LETTER OMEGA WITH PSILI AND OXIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FAD; F; 1F65 03B9; # GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI
	  (#x1FAD #x1FA5) ;; GREEK CAPITAL LETTER OMEGA WITH DASIA AND OXIA AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FAE; F; 1F66 03B9; # GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
	  (#x1FAE #x1FA6) ;; GREEK CAPITAL LETTER OMEGA WITH PSILI AND PERISPOMENI AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FAF; F; 1F67 03B9; # GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
	  (#x1FAF #x1FA7) ;; GREEK CAPITAL LETTER OMEGA WITH DASIA AND PERISPOMENI AND PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FB2; F; 1F70 03B9; # GREEK SMALL LETTER ALPHA WITH VARIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FB3; F; 03B1 03B9; # GREEK SMALL LETTER ALPHA WITH YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FB4; F; 03AC 03B9; # GREEK SMALL LETTER ALPHA WITH OXIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FB6; F; 03B1 0342; # GREEK SMALL LETTER ALPHA WITH PERISPOMENI
;;; WARNING: Unhandled full mapping:
;;;   1FB7; F; 03B1 0342 03B9; # GREEK SMALL LETTER ALPHA WITH PERISPOMENI AND YPOGEGRAMMENI
	  (#x1FB8 #x1FB0) ;; GREEK CAPITAL LETTER ALPHA WITH VRACHY
	  (#x1FB9 #x1FB1) ;; GREEK CAPITAL LETTER ALPHA WITH MACRON
	  (#x1FBA #x1F70) ;; GREEK CAPITAL LETTER ALPHA WITH VARIA
	  (#x1FBB #x1F71) ;; GREEK CAPITAL LETTER ALPHA WITH OXIA
;;; WARNING: Unhandled full mapping:
;;;   1FBC; F; 03B1 03B9; # GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
	  (#x1FBC #x1FB3) ;; GREEK CAPITAL LETTER ALPHA WITH PROSGEGRAMMENI
	  (#x1FBE #x03B9) ;; GREEK PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FC2; F; 1F74 03B9; # GREEK SMALL LETTER ETA WITH VARIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FC3; F; 03B7 03B9; # GREEK SMALL LETTER ETA WITH YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FC4; F; 03AE 03B9; # GREEK SMALL LETTER ETA WITH OXIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FC6; F; 03B7 0342; # GREEK SMALL LETTER ETA WITH PERISPOMENI
;;; WARNING: Unhandled full mapping:
;;;   1FC7; F; 03B7 0342 03B9; # GREEK SMALL LETTER ETA WITH PERISPOMENI AND YPOGEGRAMMENI
	  (#x1FC8 #x1F72) ;; GREEK CAPITAL LETTER EPSILON WITH VARIA
	  (#x1FC9 #x1F73) ;; GREEK CAPITAL LETTER EPSILON WITH OXIA
	  (#x1FCA #x1F74) ;; GREEK CAPITAL LETTER ETA WITH VARIA
	  (#x1FCB #x1F75) ;; GREEK CAPITAL LETTER ETA WITH OXIA
;;; WARNING: Unhandled full mapping:
;;;   1FCC; F; 03B7 03B9; # GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
	  (#x1FCC #x1FC3) ;; GREEK CAPITAL LETTER ETA WITH PROSGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FD2; F; 03B9 0308 0300; # GREEK SMALL LETTER IOTA WITH DIALYTIKA AND VARIA
;;; WARNING: Unhandled full mapping:
;;;   1FD3; F; 03B9 0308 0301; # GREEK SMALL LETTER IOTA WITH DIALYTIKA AND OXIA
;;; WARNING: Unhandled full mapping:
;;;   1FD6; F; 03B9 0342; # GREEK SMALL LETTER IOTA WITH PERISPOMENI
;;; WARNING: Unhandled full mapping:
;;;   1FD7; F; 03B9 0308 0342; # GREEK SMALL LETTER IOTA WITH DIALYTIKA AND PERISPOMENI
	  (#x1FD8 #x1FD0) ;; GREEK CAPITAL LETTER IOTA WITH VRACHY
	  (#x1FD9 #x1FD1) ;; GREEK CAPITAL LETTER IOTA WITH MACRON
	  (#x1FDA #x1F76) ;; GREEK CAPITAL LETTER IOTA WITH VARIA
	  (#x1FDB #x1F77) ;; GREEK CAPITAL LETTER IOTA WITH OXIA
;;; WARNING: Unhandled full mapping:
;;;   1FE2; F; 03C5 0308 0300; # GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND VARIA
;;; WARNING: Unhandled full mapping:
;;;   1FE3; F; 03C5 0308 0301; # GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND OXIA
;;; WARNING: Unhandled full mapping:
;;;   1FE4; F; 03C1 0313; # GREEK SMALL LETTER RHO WITH PSILI
;;; WARNING: Unhandled full mapping:
;;;   1FE6; F; 03C5 0342; # GREEK SMALL LETTER UPSILON WITH PERISPOMENI
;;; WARNING: Unhandled full mapping:
;;;   1FE7; F; 03C5 0308 0342; # GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND PERISPOMENI
	  (#x1FE8 #x1FE0) ;; GREEK CAPITAL LETTER UPSILON WITH VRACHY
	  (#x1FE9 #x1FE1) ;; GREEK CAPITAL LETTER UPSILON WITH MACRON
	  (#x1FEA #x1F7A) ;; GREEK CAPITAL LETTER UPSILON WITH VARIA
	  (#x1FEB #x1F7B) ;; GREEK CAPITAL LETTER UPSILON WITH OXIA
	  (#x1FEC #x1FE5) ;; GREEK CAPITAL LETTER RHO WITH DASIA
;;; WARNING: Unhandled full mapping:
;;;   1FF2; F; 1F7C 03B9; # GREEK SMALL LETTER OMEGA WITH VARIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FF3; F; 03C9 03B9; # GREEK SMALL LETTER OMEGA WITH YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FF4; F; 03CE 03B9; # GREEK SMALL LETTER OMEGA WITH OXIA AND YPOGEGRAMMENI
;;; WARNING: Unhandled full mapping:
;;;   1FF6; F; 03C9 0342; # GREEK SMALL LETTER OMEGA WITH PERISPOMENI
;;; WARNING: Unhandled full mapping:
;;;   1FF7; F; 03C9 0342 03B9; # GREEK SMALL LETTER OMEGA WITH PERISPOMENI AND YPOGEGRAMMENI
	  (#x1FF8 #x1F78) ;; GREEK CAPITAL LETTER OMICRON WITH VARIA
	  (#x1FF9 #x1F79) ;; GREEK CAPITAL LETTER OMICRON WITH OXIA
	  (#x1FFA #x1F7C) ;; GREEK CAPITAL LETTER OMEGA WITH VARIA
	  (#x1FFB #x1F7D) ;; GREEK CAPITAL LETTER OMEGA WITH OXIA
;;; WARNING: Unhandled full mapping:
;;;   1FFC; F; 03C9 03B9; # GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
	  (#x1FFC #x1FF3) ;; GREEK CAPITAL LETTER OMEGA WITH PROSGEGRAMMENI
	  (#x2126 #x03C9) ;; OHM SIGN
	  (#x212A #x006B) ;; KELVIN SIGN
	  (#x212B #x00E5) ;; ANGSTROM SIGN
	  (#x2132 #x214E) ;; TURNED CAPITAL F
	  (#x2160 #x2170) ;; ROMAN NUMERAL ONE
	  (#x2161 #x2171) ;; ROMAN NUMERAL TWO
	  (#x2162 #x2172) ;; ROMAN NUMERAL THREE
	  (#x2163 #x2173) ;; ROMAN NUMERAL FOUR
	  (#x2164 #x2174) ;; ROMAN NUMERAL FIVE
	  (#x2165 #x2175) ;; ROMAN NUMERAL SIX
	  (#x2166 #x2176) ;; ROMAN NUMERAL SEVEN
	  (#x2167 #x2177) ;; ROMAN NUMERAL EIGHT
	  (#x2168 #x2178) ;; ROMAN NUMERAL NINE
	  (#x2169 #x2179) ;; ROMAN NUMERAL TEN
	  (#x216A #x217A) ;; ROMAN NUMERAL ELEVEN
	  (#x216B #x217B) ;; ROMAN NUMERAL TWELVE
	  (#x216C #x217C) ;; ROMAN NUMERAL FIFTY
	  (#x216D #x217D) ;; ROMAN NUMERAL ONE HUNDRED
	  (#x216E #x217E) ;; ROMAN NUMERAL FIVE HUNDRED
	  (#x216F #x217F) ;; ROMAN NUMERAL ONE THOUSAND
	  (#x2183 #x2184) ;; ROMAN NUMERAL REVERSED ONE HUNDRED
	  (#x24B6 #x24D0) ;; CIRCLED LATIN CAPITAL LETTER A
	  (#x24B7 #x24D1) ;; CIRCLED LATIN CAPITAL LETTER B
	  (#x24B8 #x24D2) ;; CIRCLED LATIN CAPITAL LETTER C
	  (#x24B9 #x24D3) ;; CIRCLED LATIN CAPITAL LETTER D
	  (#x24BA #x24D4) ;; CIRCLED LATIN CAPITAL LETTER E
	  (#x24BB #x24D5) ;; CIRCLED LATIN CAPITAL LETTER F
	  (#x24BC #x24D6) ;; CIRCLED LATIN CAPITAL LETTER G
	  (#x24BD #x24D7) ;; CIRCLED LATIN CAPITAL LETTER H
	  (#x24BE #x24D8) ;; CIRCLED LATIN CAPITAL LETTER I
	  (#x24BF #x24D9) ;; CIRCLED LATIN CAPITAL LETTER J
	  (#x24C0 #x24DA) ;; CIRCLED LATIN CAPITAL LETTER K
	  (#x24C1 #x24DB) ;; CIRCLED LATIN CAPITAL LETTER L
	  (#x24C2 #x24DC) ;; CIRCLED LATIN CAPITAL LETTER M
	  (#x24C3 #x24DD) ;; CIRCLED LATIN CAPITAL LETTER N
	  (#x24C4 #x24DE) ;; CIRCLED LATIN CAPITAL LETTER O
	  (#x24C5 #x24DF) ;; CIRCLED LATIN CAPITAL LETTER P
	  (#x24C6 #x24E0) ;; CIRCLED LATIN CAPITAL LETTER Q
	  (#x24C7 #x24E1) ;; CIRCLED LATIN CAPITAL LETTER R
	  (#x24C8 #x24E2) ;; CIRCLED LATIN CAPITAL LETTER S
	  (#x24C9 #x24E3) ;; CIRCLED LATIN CAPITAL LETTER T
	  (#x24CA #x24E4) ;; CIRCLED LATIN CAPITAL LETTER U
	  (#x24CB #x24E5) ;; CIRCLED LATIN CAPITAL LETTER V
	  (#x24CC #x24E6) ;; CIRCLED LATIN CAPITAL LETTER W
	  (#x24CD #x24E7) ;; CIRCLED LATIN CAPITAL LETTER X
	  (#x24CE #x24E8) ;; CIRCLED LATIN CAPITAL LETTER Y
	  (#x24CF #x24E9) ;; CIRCLED LATIN CAPITAL LETTER Z
	  (#x2C00 #x2C30) ;; GLAGOLITIC CAPITAL LETTER AZU
	  (#x2C01 #x2C31) ;; GLAGOLITIC CAPITAL LETTER BUKY
	  (#x2C02 #x2C32) ;; GLAGOLITIC CAPITAL LETTER VEDE
	  (#x2C03 #x2C33) ;; GLAGOLITIC CAPITAL LETTER GLAGOLI
	  (#x2C04 #x2C34) ;; GLAGOLITIC CAPITAL LETTER DOBRO
	  (#x2C05 #x2C35) ;; GLAGOLITIC CAPITAL LETTER YESTU
	  (#x2C06 #x2C36) ;; GLAGOLITIC CAPITAL LETTER ZHIVETE
	  (#x2C07 #x2C37) ;; GLAGOLITIC CAPITAL LETTER DZELO
	  (#x2C08 #x2C38) ;; GLAGOLITIC CAPITAL LETTER ZEMLJA
	  (#x2C09 #x2C39) ;; GLAGOLITIC CAPITAL LETTER IZHE
	  (#x2C0A #x2C3A) ;; GLAGOLITIC CAPITAL LETTER INITIAL IZHE
	  (#x2C0B #x2C3B) ;; GLAGOLITIC CAPITAL LETTER I
	  (#x2C0C #x2C3C) ;; GLAGOLITIC CAPITAL LETTER DJERVI
	  (#x2C0D #x2C3D) ;; GLAGOLITIC CAPITAL LETTER KAKO
	  (#x2C0E #x2C3E) ;; GLAGOLITIC CAPITAL LETTER LJUDIJE
	  (#x2C0F #x2C3F) ;; GLAGOLITIC CAPITAL LETTER MYSLITE
	  (#x2C10 #x2C40) ;; GLAGOLITIC CAPITAL LETTER NASHI
	  (#x2C11 #x2C41) ;; GLAGOLITIC CAPITAL LETTER ONU
	  (#x2C12 #x2C42) ;; GLAGOLITIC CAPITAL LETTER POKOJI
	  (#x2C13 #x2C43) ;; GLAGOLITIC CAPITAL LETTER RITSI
	  (#x2C14 #x2C44) ;; GLAGOLITIC CAPITAL LETTER SLOVO
	  (#x2C15 #x2C45) ;; GLAGOLITIC CAPITAL LETTER TVRIDO
	  (#x2C16 #x2C46) ;; GLAGOLITIC CAPITAL LETTER UKU
	  (#x2C17 #x2C47) ;; GLAGOLITIC CAPITAL LETTER FRITU
	  (#x2C18 #x2C48) ;; GLAGOLITIC CAPITAL LETTER HERU
	  (#x2C19 #x2C49) ;; GLAGOLITIC CAPITAL LETTER OTU
	  (#x2C1A #x2C4A) ;; GLAGOLITIC CAPITAL LETTER PE
	  (#x2C1B #x2C4B) ;; GLAGOLITIC CAPITAL LETTER SHTA
	  (#x2C1C #x2C4C) ;; GLAGOLITIC CAPITAL LETTER TSI
	  (#x2C1D #x2C4D) ;; GLAGOLITIC CAPITAL LETTER CHRIVI
	  (#x2C1E #x2C4E) ;; GLAGOLITIC CAPITAL LETTER SHA
	  (#x2C1F #x2C4F) ;; GLAGOLITIC CAPITAL LETTER YERU
	  (#x2C20 #x2C50) ;; GLAGOLITIC CAPITAL LETTER YERI
	  (#x2C21 #x2C51) ;; GLAGOLITIC CAPITAL LETTER YATI
	  (#x2C22 #x2C52) ;; GLAGOLITIC CAPITAL LETTER SPIDERY HA
	  (#x2C23 #x2C53) ;; GLAGOLITIC CAPITAL LETTER YU
	  (#x2C24 #x2C54) ;; GLAGOLITIC CAPITAL LETTER SMALL YUS
	  (#x2C25 #x2C55) ;; GLAGOLITIC CAPITAL LETTER SMALL YUS WITH TAIL
	  (#x2C26 #x2C56) ;; GLAGOLITIC CAPITAL LETTER YO
	  (#x2C27 #x2C57) ;; GLAGOLITIC CAPITAL LETTER IOTATED SMALL YUS
	  (#x2C28 #x2C58) ;; GLAGOLITIC CAPITAL LETTER BIG YUS
	  (#x2C29 #x2C59) ;; GLAGOLITIC CAPITAL LETTER IOTATED BIG YUS
	  (#x2C2A #x2C5A) ;; GLAGOLITIC CAPITAL LETTER FITA
	  (#x2C2B #x2C5B) ;; GLAGOLITIC CAPITAL LETTER IZHITSA
	  (#x2C2C #x2C5C) ;; GLAGOLITIC CAPITAL LETTER SHTAPIC
	  (#x2C2D #x2C5D) ;; GLAGOLITIC CAPITAL LETTER TROKUTASTI A
	  (#x2C2E #x2C5E) ;; GLAGOLITIC CAPITAL LETTER LATINATE MYSLITE
	  (#x2C60 #x2C61) ;; LATIN CAPITAL LETTER L WITH DOUBLE BAR
	  (#x2C62 #x026B) ;; LATIN CAPITAL LETTER L WITH MIDDLE TILDE
	  (#x2C63 #x1D7D) ;; LATIN CAPITAL LETTER P WITH STROKE
	  (#x2C64 #x027D) ;; LATIN CAPITAL LETTER R WITH TAIL
	  (#x2C67 #x2C68) ;; LATIN CAPITAL LETTER H WITH DESCENDER
	  (#x2C69 #x2C6A) ;; LATIN CAPITAL LETTER K WITH DESCENDER
	  (#x2C6B #x2C6C) ;; LATIN CAPITAL LETTER Z WITH DESCENDER
	  (#x2C6D #x0251) ;; LATIN CAPITAL LETTER ALPHA
	  (#x2C6E #x0271) ;; LATIN CAPITAL LETTER M WITH HOOK
	  (#x2C6F #x0250) ;; LATIN CAPITAL LETTER TURNED A
	  (#x2C70 #x0252) ;; LATIN CAPITAL LETTER TURNED ALPHA
	  (#x2C72 #x2C73) ;; LATIN CAPITAL LETTER W WITH HOOK
	  (#x2C75 #x2C76) ;; LATIN CAPITAL LETTER HALF H
	  (#x2C7E #x023F) ;; LATIN CAPITAL LETTER S WITH SWASH TAIL
	  (#x2C7F #x0240) ;; LATIN CAPITAL LETTER Z WITH SWASH TAIL
	  (#x2C80 #x2C81) ;; COPTIC CAPITAL LETTER ALFA
	  (#x2C82 #x2C83) ;; COPTIC CAPITAL LETTER VIDA
	  (#x2C84 #x2C85) ;; COPTIC CAPITAL LETTER GAMMA
	  (#x2C86 #x2C87) ;; COPTIC CAPITAL LETTER DALDA
	  (#x2C88 #x2C89) ;; COPTIC CAPITAL LETTER EIE
	  (#x2C8A #x2C8B) ;; COPTIC CAPITAL LETTER SOU
	  (#x2C8C #x2C8D) ;; COPTIC CAPITAL LETTER ZATA
	  (#x2C8E #x2C8F) ;; COPTIC CAPITAL LETTER HATE
	  (#x2C90 #x2C91) ;; COPTIC CAPITAL LETTER THETHE
	  (#x2C92 #x2C93) ;; COPTIC CAPITAL LETTER IAUDA
	  (#x2C94 #x2C95) ;; COPTIC CAPITAL LETTER KAPA
	  (#x2C96 #x2C97) ;; COPTIC CAPITAL LETTER LAULA
	  (#x2C98 #x2C99) ;; COPTIC CAPITAL LETTER MI
	  (#x2C9A #x2C9B) ;; COPTIC CAPITAL LETTER NI
	  (#x2C9C #x2C9D) ;; COPTIC CAPITAL LETTER KSI
	  (#x2C9E #x2C9F) ;; COPTIC CAPITAL LETTER O
	  (#x2CA0 #x2CA1) ;; COPTIC CAPITAL LETTER PI
	  (#x2CA2 #x2CA3) ;; COPTIC CAPITAL LETTER RO
	  (#x2CA4 #x2CA5) ;; COPTIC CAPITAL LETTER SIMA
	  (#x2CA6 #x2CA7) ;; COPTIC CAPITAL LETTER TAU
	  (#x2CA8 #x2CA9) ;; COPTIC CAPITAL LETTER UA
	  (#x2CAA #x2CAB) ;; COPTIC CAPITAL LETTER FI
	  (#x2CAC #x2CAD) ;; COPTIC CAPITAL LETTER KHI
	  (#x2CAE #x2CAF) ;; COPTIC CAPITAL LETTER PSI
	  (#x2CB0 #x2CB1) ;; COPTIC CAPITAL LETTER OOU
	  (#x2CB2 #x2CB3) ;; COPTIC CAPITAL LETTER DIALECT-P ALEF
	  (#x2CB4 #x2CB5) ;; COPTIC CAPITAL LETTER OLD COPTIC AIN
	  (#x2CB6 #x2CB7) ;; COPTIC CAPITAL LETTER CRYPTOGRAMMIC EIE
	  (#x2CB8 #x2CB9) ;; COPTIC CAPITAL LETTER DIALECT-P KAPA
	  (#x2CBA #x2CBB) ;; COPTIC CAPITAL LETTER DIALECT-P NI
	  (#x2CBC #x2CBD) ;; COPTIC CAPITAL LETTER CRYPTOGRAMMIC NI
	  (#x2CBE #x2CBF) ;; COPTIC CAPITAL LETTER OLD COPTIC OOU
	  (#x2CC0 #x2CC1) ;; COPTIC CAPITAL LETTER SAMPI
	  (#x2CC2 #x2CC3) ;; COPTIC CAPITAL LETTER CROSSED SHEI
	  (#x2CC4 #x2CC5) ;; COPTIC CAPITAL LETTER OLD COPTIC SHEI
	  (#x2CC6 #x2CC7) ;; COPTIC CAPITAL LETTER OLD COPTIC ESH
	  (#x2CC8 #x2CC9) ;; COPTIC CAPITAL LETTER AKHMIMIC KHEI
	  (#x2CCA #x2CCB) ;; COPTIC CAPITAL LETTER DIALECT-P HORI
	  (#x2CCC #x2CCD) ;; COPTIC CAPITAL LETTER OLD COPTIC HORI
	  (#x2CCE #x2CCF) ;; COPTIC CAPITAL LETTER OLD COPTIC HA
	  (#x2CD0 #x2CD1) ;; COPTIC CAPITAL LETTER L-SHAPED HA
	  (#x2CD2 #x2CD3) ;; COPTIC CAPITAL LETTER OLD COPTIC HEI
	  (#x2CD4 #x2CD5) ;; COPTIC CAPITAL LETTER OLD COPTIC HAT
	  (#x2CD6 #x2CD7) ;; COPTIC CAPITAL LETTER OLD COPTIC GANGIA
	  (#x2CD8 #x2CD9) ;; COPTIC CAPITAL LETTER OLD COPTIC DJA
	  (#x2CDA #x2CDB) ;; COPTIC CAPITAL LETTER OLD COPTIC SHIMA
	  (#x2CDC #x2CDD) ;; COPTIC CAPITAL LETTER OLD NUBIAN SHIMA
	  (#x2CDE #x2CDF) ;; COPTIC CAPITAL LETTER OLD NUBIAN NGI
	  (#x2CE0 #x2CE1) ;; COPTIC CAPITAL LETTER OLD NUBIAN NYI
	  (#x2CE2 #x2CE3) ;; COPTIC CAPITAL LETTER OLD NUBIAN WAU
	  (#x2CEB #x2CEC) ;; COPTIC CAPITAL LETTER CRYPTOGRAMMIC SHEI
	  (#x2CED #x2CEE) ;; COPTIC CAPITAL LETTER CRYPTOGRAMMIC GANGIA
	  (#xA640 #xA641) ;; CYRILLIC CAPITAL LETTER ZEMLYA
	  (#xA642 #xA643) ;; CYRILLIC CAPITAL LETTER DZELO
	  (#xA644 #xA645) ;; CYRILLIC CAPITAL LETTER REVERSED DZE
	  (#xA646 #xA647) ;; CYRILLIC CAPITAL LETTER IOTA
	  (#xA648 #xA649) ;; CYRILLIC CAPITAL LETTER DJERV
	  (#xA64A #xA64B) ;; CYRILLIC CAPITAL LETTER MONOGRAPH UK
	  (#xA64C #xA64D) ;; CYRILLIC CAPITAL LETTER BROAD OMEGA
	  (#xA64E #xA64F) ;; CYRILLIC CAPITAL LETTER NEUTRAL YER
	  (#xA650 #xA651) ;; CYRILLIC CAPITAL LETTER YERU WITH BACK YER
	  (#xA652 #xA653) ;; CYRILLIC CAPITAL LETTER IOTIFIED YAT
	  (#xA654 #xA655) ;; CYRILLIC CAPITAL LETTER REVERSED YU
	  (#xA656 #xA657) ;; CYRILLIC CAPITAL LETTER IOTIFIED A
	  (#xA658 #xA659) ;; CYRILLIC CAPITAL LETTER CLOSED LITTLE YUS
	  (#xA65A #xA65B) ;; CYRILLIC CAPITAL LETTER BLENDED YUS
	  (#xA65C #xA65D) ;; CYRILLIC CAPITAL LETTER IOTIFIED CLOSED LITTLE YUS
	  (#xA65E #xA65F) ;; CYRILLIC CAPITAL LETTER YN
	  (#xA662 #xA663) ;; CYRILLIC CAPITAL LETTER SOFT DE
	  (#xA664 #xA665) ;; CYRILLIC CAPITAL LETTER SOFT EL
	  (#xA666 #xA667) ;; CYRILLIC CAPITAL LETTER SOFT EM
	  (#xA668 #xA669) ;; CYRILLIC CAPITAL LETTER MONOCULAR O
	  (#xA66A #xA66B) ;; CYRILLIC CAPITAL LETTER BINOCULAR O
	  (#xA66C #xA66D) ;; CYRILLIC CAPITAL LETTER DOUBLE MONOCULAR O
	  (#xA680 #xA681) ;; CYRILLIC CAPITAL LETTER DWE
	  (#xA682 #xA683) ;; CYRILLIC CAPITAL LETTER DZWE
	  (#xA684 #xA685) ;; CYRILLIC CAPITAL LETTER ZHWE
	  (#xA686 #xA687) ;; CYRILLIC CAPITAL LETTER CCHE
	  (#xA688 #xA689) ;; CYRILLIC CAPITAL LETTER DZZE
	  (#xA68A #xA68B) ;; CYRILLIC CAPITAL LETTER TE WITH MIDDLE HOOK
	  (#xA68C #xA68D) ;; CYRILLIC CAPITAL LETTER TWE
	  (#xA68E #xA68F) ;; CYRILLIC CAPITAL LETTER TSWE
	  (#xA690 #xA691) ;; CYRILLIC CAPITAL LETTER TSSE
	  (#xA692 #xA693) ;; CYRILLIC CAPITAL LETTER TCHE
	  (#xA694 #xA695) ;; CYRILLIC CAPITAL LETTER HWE
	  (#xA696 #xA697) ;; CYRILLIC CAPITAL LETTER SHWE
	  (#xA722 #xA723) ;; LATIN CAPITAL LETTER EGYPTOLOGICAL ALEF
	  (#xA724 #xA725) ;; LATIN CAPITAL LETTER EGYPTOLOGICAL AIN
	  (#xA726 #xA727) ;; LATIN CAPITAL LETTER HENG
	  (#xA728 #xA729) ;; LATIN CAPITAL LETTER TZ
	  (#xA72A #xA72B) ;; LATIN CAPITAL LETTER TRESILLO
	  (#xA72C #xA72D) ;; LATIN CAPITAL LETTER CUATRILLO
	  (#xA72E #xA72F) ;; LATIN CAPITAL LETTER CUATRILLO WITH COMMA
	  (#xA732 #xA733) ;; LATIN CAPITAL LETTER AA
	  (#xA734 #xA735) ;; LATIN CAPITAL LETTER AO
	  (#xA736 #xA737) ;; LATIN CAPITAL LETTER AU
	  (#xA738 #xA739) ;; LATIN CAPITAL LETTER AV
	  (#xA73A #xA73B) ;; LATIN CAPITAL LETTER AV WITH HORIZONTAL BAR
	  (#xA73C #xA73D) ;; LATIN CAPITAL LETTER AY
	  (#xA73E #xA73F) ;; LATIN CAPITAL LETTER REVERSED C WITH DOT
	  (#xA740 #xA741) ;; LATIN CAPITAL LETTER K WITH STROKE
	  (#xA742 #xA743) ;; LATIN CAPITAL LETTER K WITH DIAGONAL STROKE
	  (#xA744 #xA745) ;; LATIN CAPITAL LETTER K WITH STROKE AND DIAGONAL STROKE
	  (#xA746 #xA747) ;; LATIN CAPITAL LETTER BROKEN L
	  (#xA748 #xA749) ;; LATIN CAPITAL LETTER L WITH HIGH STROKE
	  (#xA74A #xA74B) ;; LATIN CAPITAL LETTER O WITH LONG STROKE OVERLAY
	  (#xA74C #xA74D) ;; LATIN CAPITAL LETTER O WITH LOOP
	  (#xA74E #xA74F) ;; LATIN CAPITAL LETTER OO
	  (#xA750 #xA751) ;; LATIN CAPITAL LETTER P WITH STROKE THROUGH DESCENDER
	  (#xA752 #xA753) ;; LATIN CAPITAL LETTER P WITH FLOURISH
	  (#xA754 #xA755) ;; LATIN CAPITAL LETTER P WITH SQUIRREL TAIL
	  (#xA756 #xA757) ;; LATIN CAPITAL LETTER Q WITH STROKE THROUGH DESCENDER
	  (#xA758 #xA759) ;; LATIN CAPITAL LETTER Q WITH DIAGONAL STROKE
	  (#xA75A #xA75B) ;; LATIN CAPITAL LETTER R ROTUNDA
	  (#xA75C #xA75D) ;; LATIN CAPITAL LETTER RUM ROTUNDA
	  (#xA75E #xA75F) ;; LATIN CAPITAL LETTER V WITH DIAGONAL STROKE
	  (#xA760 #xA761) ;; LATIN CAPITAL LETTER VY
	  (#xA762 #xA763) ;; LATIN CAPITAL LETTER VISIGOTHIC Z
	  (#xA764 #xA765) ;; LATIN CAPITAL LETTER THORN WITH STROKE
	  (#xA766 #xA767) ;; LATIN CAPITAL LETTER THORN WITH STROKE THROUGH DESCENDER
	  (#xA768 #xA769) ;; LATIN CAPITAL LETTER VEND
	  (#xA76A #xA76B) ;; LATIN CAPITAL LETTER ET
	  (#xA76C #xA76D) ;; LATIN CAPITAL LETTER IS
	  (#xA76E #xA76F) ;; LATIN CAPITAL LETTER CON
	  (#xA779 #xA77A) ;; LATIN CAPITAL LETTER INSULAR D
	  (#xA77B #xA77C) ;; LATIN CAPITAL LETTER INSULAR F
	  (#xA77D #x1D79) ;; LATIN CAPITAL LETTER INSULAR G
	  (#xA77E #xA77F) ;; LATIN CAPITAL LETTER TURNED INSULAR G
	  (#xA780 #xA781) ;; LATIN CAPITAL LETTER TURNED L
	  (#xA782 #xA783) ;; LATIN CAPITAL LETTER INSULAR R
	  (#xA784 #xA785) ;; LATIN CAPITAL LETTER INSULAR S
	  (#xA786 #xA787) ;; LATIN CAPITAL LETTER INSULAR T
	  (#xA78B #xA78C) ;; LATIN CAPITAL LETTER SALTILLO
;;; WARNING: Unhandled full mapping:
;;;   FB00; F; 0066 0066; # LATIN SMALL LIGATURE FF
;;; WARNING: Unhandled full mapping:
;;;   FB01; F; 0066 0069; # LATIN SMALL LIGATURE FI
;;; WARNING: Unhandled full mapping:
;;;   FB02; F; 0066 006C; # LATIN SMALL LIGATURE FL
;;; WARNING: Unhandled full mapping:
;;;   FB03; F; 0066 0066 0069; # LATIN SMALL LIGATURE FFI
;;; WARNING: Unhandled full mapping:
;;;   FB04; F; 0066 0066 006C; # LATIN SMALL LIGATURE FFL
;;; WARNING: Unhandled full mapping:
;;;   FB05; F; 0073 0074; # LATIN SMALL LIGATURE LONG S T
;;; WARNING: Unhandled full mapping:
;;;   FB06; F; 0073 0074; # LATIN SMALL LIGATURE ST
;;; WARNING: Unhandled full mapping:
;;;   FB13; F; 0574 0576; # ARMENIAN SMALL LIGATURE MEN NOW
;;; WARNING: Unhandled full mapping:
;;;   FB14; F; 0574 0565; # ARMENIAN SMALL LIGATURE MEN ECH
;;; WARNING: Unhandled full mapping:
;;;   FB15; F; 0574 056B; # ARMENIAN SMALL LIGATURE MEN INI
;;; WARNING: Unhandled full mapping:
;;;   FB16; F; 057E 0576; # ARMENIAN SMALL LIGATURE VEW NOW
;;; WARNING: Unhandled full mapping:
;;;   FB17; F; 0574 056D; # ARMENIAN SMALL LIGATURE MEN XEH
	  (#xFF21 #xFF41) ;; FULLWIDTH LATIN CAPITAL LETTER A
	  (#xFF22 #xFF42) ;; FULLWIDTH LATIN CAPITAL LETTER B
	  (#xFF23 #xFF43) ;; FULLWIDTH LATIN CAPITAL LETTER C
	  (#xFF24 #xFF44) ;; FULLWIDTH LATIN CAPITAL LETTER D
	  (#xFF25 #xFF45) ;; FULLWIDTH LATIN CAPITAL LETTER E
	  (#xFF26 #xFF46) ;; FULLWIDTH LATIN CAPITAL LETTER F
	  (#xFF27 #xFF47) ;; FULLWIDTH LATIN CAPITAL LETTER G
	  (#xFF28 #xFF48) ;; FULLWIDTH LATIN CAPITAL LETTER H
	  (#xFF29 #xFF49) ;; FULLWIDTH LATIN CAPITAL LETTER I
	  (#xFF2A #xFF4A) ;; FULLWIDTH LATIN CAPITAL LETTER J
	  (#xFF2B #xFF4B) ;; FULLWIDTH LATIN CAPITAL LETTER K
	  (#xFF2C #xFF4C) ;; FULLWIDTH LATIN CAPITAL LETTER L
	  (#xFF2D #xFF4D) ;; FULLWIDTH LATIN CAPITAL LETTER M
	  (#xFF2E #xFF4E) ;; FULLWIDTH LATIN CAPITAL LETTER N
	  (#xFF2F #xFF4F) ;; FULLWIDTH LATIN CAPITAL LETTER O
	  (#xFF30 #xFF50) ;; FULLWIDTH LATIN CAPITAL LETTER P
	  (#xFF31 #xFF51) ;; FULLWIDTH LATIN CAPITAL LETTER Q
	  (#xFF32 #xFF52) ;; FULLWIDTH LATIN CAPITAL LETTER R
	  (#xFF33 #xFF53) ;; FULLWIDTH LATIN CAPITAL LETTER S
	  (#xFF34 #xFF54) ;; FULLWIDTH LATIN CAPITAL LETTER T
	  (#xFF35 #xFF55) ;; FULLWIDTH LATIN CAPITAL LETTER U
	  (#xFF36 #xFF56) ;; FULLWIDTH LATIN CAPITAL LETTER V
	  (#xFF37 #xFF57) ;; FULLWIDTH LATIN CAPITAL LETTER W
	  (#xFF38 #xFF58) ;; FULLWIDTH LATIN CAPITAL LETTER X
	  (#xFF39 #xFF59) ;; FULLWIDTH LATIN CAPITAL LETTER Y
	  (#xFF3A #xFF5A) ;; FULLWIDTH LATIN CAPITAL LETTER Z
	  (#x10400 #x10428) ;; DESERET CAPITAL LETTER LONG I
	  (#x10401 #x10429) ;; DESERET CAPITAL LETTER LONG E
	  (#x10402 #x1042A) ;; DESERET CAPITAL LETTER LONG A
	  (#x10403 #x1042B) ;; DESERET CAPITAL LETTER LONG AH
	  (#x10404 #x1042C) ;; DESERET CAPITAL LETTER LONG O
	  (#x10405 #x1042D) ;; DESERET CAPITAL LETTER LONG OO
	  (#x10406 #x1042E) ;; DESERET CAPITAL LETTER SHORT I
	  (#x10407 #x1042F) ;; DESERET CAPITAL LETTER SHORT E
	  (#x10408 #x10430) ;; DESERET CAPITAL LETTER SHORT A
	  (#x10409 #x10431) ;; DESERET CAPITAL LETTER SHORT AH
	  (#x1040A #x10432) ;; DESERET CAPITAL LETTER SHORT O
	  (#x1040B #x10433) ;; DESERET CAPITAL LETTER SHORT OO
	  (#x1040C #x10434) ;; DESERET CAPITAL LETTER AY
	  (#x1040D #x10435) ;; DESERET CAPITAL LETTER OW
	  (#x1040E #x10436) ;; DESERET CAPITAL LETTER WU
	  (#x1040F #x10437) ;; DESERET CAPITAL LETTER YEE
	  (#x10410 #x10438) ;; DESERET CAPITAL LETTER H
	  (#x10411 #x10439) ;; DESERET CAPITAL LETTER PEE
	  (#x10412 #x1043A) ;; DESERET CAPITAL LETTER BEE
	  (#x10413 #x1043B) ;; DESERET CAPITAL LETTER TEE
	  (#x10414 #x1043C) ;; DESERET CAPITAL LETTER DEE
	  (#x10415 #x1043D) ;; DESERET CAPITAL LETTER CHEE
	  (#x10416 #x1043E) ;; DESERET CAPITAL LETTER JEE
	  (#x10417 #x1043F) ;; DESERET CAPITAL LETTER KAY
	  (#x10418 #x10440) ;; DESERET CAPITAL LETTER GAY
	  (#x10419 #x10441) ;; DESERET CAPITAL LETTER EF
	  (#x1041A #x10442) ;; DESERET CAPITAL LETTER VEE
	  (#x1041B #x10443) ;; DESERET CAPITAL LETTER ETH
	  (#x1041C #x10444) ;; DESERET CAPITAL LETTER THEE
	  (#x1041D #x10445) ;; DESERET CAPITAL LETTER ES
	  (#x1041E #x10446) ;; DESERET CAPITAL LETTER ZEE
	  (#x1041F #x10447) ;; DESERET CAPITAL LETTER ESH
	  (#x10420 #x10448) ;; DESERET CAPITAL LETTER ZHEE
	  (#x10421 #x10449) ;; DESERET CAPITAL LETTER ER
	  (#x10422 #x1044A) ;; DESERET CAPITAL LETTER EL
	  (#x10423 #x1044B) ;; DESERET CAPITAL LETTER EM
	  (#x10424 #x1044C) ;; DESERET CAPITAL LETTER EN
	  (#x10425 #x1044D) ;; DESERET CAPITAL LETTER ENG
	  (#x10426 #x1044E) ;; DESERET CAPITAL LETTER OI
	  (#x10427 #x1044F) ;; DESERET CAPITAL LETTER EW
	  ))
       ;; a table to track mappings that overlap with some other mapping
       (multi-hash (make-hash-table))
       (uni-casetab
	(loop
	  with case-table = (make-case-table)
	  for (uc lc) in uni-mappings 
	  if (and (setq uc (decode-char 'ucs uc))
		  (setq lc (decode-char 'ucs lc)))
	  ;; see if there are existing mappings for either char of the new
	  ;; mapping pair.
	  do
	  (let* ((curucval (get-case-table 'downcase uc case-table))
		 (curlcval (get-case-table 'upcase lc case-table))
		 (curucval (and (not (eq curucval uc)) curucval))
		 (curlcval (and (not (eq curlcval lc)) curlcval))
		 )
	    ;; if so, flag both the existing and new mapping pair as having
	    ;; an overlapping mapping. 
	    (when (or curucval curlcval)
	      (loop for ch in (list curucval curlcval uc lc) do
		(puthash ch t multi-hash)))

	    ;; finally, make the new mapping.
	    (put-case-table-pair uc lc case-table))
	  finally return case-table)))
  (labels ((ismulti (uc lc)
             (or (gethash uc multi-hash) (gethash lc multi-hash))))
    (let (
	  ;; All lowercase
	  (lowermulti (with-output-to-string
			(loop for (uc lc) in uni-mappings do
			  (if (decode-char 'ucs lc)
			      (write-char (decode-char 'ucs lc))))))
	  ;; All uppercase
	  (uppermulti (with-output-to-string
			(loop for (uc lc) in uni-mappings do
			  (if (decode-char 'ucs uc)
			      (write-char (decode-char 'ucs uc))))))
	  ;; For each pair, lower followed by upper
	  (loweruppermulti (with-output-to-string
			     (loop for (uc lc) in uni-mappings
			       if (and (setq uc (decode-char 'ucs uc))
				       (setq lc (decode-char 'ucs lc)))
			       do (write-char lc) (write-char uc)))) 
	  ;; For each pair, upper followed by lower
	  (upperlowermulti (with-output-to-string
			     (loop for (uc lc) in uni-mappings
			       if (and (setq uc (decode-char 'ucs uc))
				       (setq lc (decode-char 'ucs lc)))
			       do (write-char uc) (write-char lc))))
	  ;; All lowercase, no complex mappings
	  (lower (with-output-to-string
		   (loop for (uc lc) in uni-mappings
		     if (and (setq uc (decode-char 'ucs uc))
			     (setq lc (decode-char 'ucs lc)))
		     do (unless (ismulti uc lc) (princ lc)))))
	  ;; All uppercase, no complex mappings
	  (upper (with-output-to-string
		   (loop for (uc lc) in uni-mappings
		     if (and (setq uc (decode-char 'ucs uc))
			     (setq lc (decode-char 'ucs lc)))
		     do (unless (ismulti uc lc) (princ uc)))))
	  ;; For each pair, lower followed by upper, no complex mappings
	  (lowerupper (with-output-to-string
			(loop for (uc lc) in uni-mappings
			  if (and (setq uc (decode-char 'ucs uc))
				  (setq lc (decode-char 'ucs lc)))
			  do (unless (ismulti uc lc) (princ lc) (princ uc)))))
	  ;; For each pair, upper followed by lower, no complex mappings
	  (upperlower (with-output-to-string
			(loop for (uc lc) in uni-mappings
			  if (and (setq uc (decode-char 'ucs uc))
				  (setq lc (decode-char 'ucs lc)))
			  do (unless (ismulti uc lc) (princ uc) (princ lc))))))
      (with-case-table
	uni-casetab
	;; Comparison with `equalp' uses a canonical mapping internally and
	;; so should be able to handle multi-mappings.  Just comparing
	;; using downcase and upcase, however, won't necessarily work in
	;; the presence of such mappings -- that's what the internal canon
	;; and eqv tables are for.
	(Assert (equalp lowermulti uppermulti))
	(Assert (equalp loweruppermulti upperlowermulti))
	(Assert (equal lower (downcase upper)))
	(Assert (equal upper (upcase lower)))
	(Assert (equal (downcase lower) (downcase (downcase lower))))
	(Assert (equal (upcase lowerupper) (upcase upperlower)))
	(Assert (equal (downcase lowerupper) (downcase upperlower)))
	;; Individually -- we include multi-mappings since we're using
	;; `equalp'.
	(loop
	  for (uc lc) in uni-mappings
	  if (and (setq uc (decode-char 'ucs uc))
		  (setq lc (decode-char 'ucs lc)))
	  do (progn (Assert (equalp uc lc))
		    (Assert (equalp (string uc) (string lc))))))

      ;; Here we include multi-mappings -- searching should be able to
      ;; handle it.
      (with-temp-buffer
	(set-case-table uni-casetab)
	(loop for (str1 str2) in `((,lowermulti ,uppermulti)
				   (,loweruppermulti ,upperlowermulti)
				   (,uppermulti ,lowermulti)
				   (,upperlowermulti ,loweruppermulti))
	  do
	  (erase-buffer)
	  (Assert (= (point-min) 1))
	  (Assert (= (point) 1))
	  (insert str1)
	  (let ((point (point))
		(case-fold-search t))
	    (Assert (= (length str1) (1- point)))
	    (goto-char (point-min))
	    (Assert (eql (search-forward str2 nil t) point))))
	(loop for (uc lc) in uni-mappings
	      if (and (setq uc (decode-char 'ucs uc))
		      (setq lc (decode-char 'ucs lc)))
	      do
	      (loop for (ch1 ch2) in `((,uc ,lc)
				       (,lc ,uc))
		    do
		    (erase-buffer)
		    (insert ?0)
		    (insert ch1)
		    (insert ?1)
		    (goto-char (point-min))
		    (Assert (eql (search-forward (char-to-string ch2) nil t) 3)
			    (format
			     "Case-folded searching doesn't equate %s and %s"
			     (char-as-unicode-escape ch1)
			     (char-as-unicode-escape ch2)))))))))

;; end of case-tests.el
