import { ISignal } from '@lumino/signaling';
import { Kernel, KernelMessage } from '../kernel';
import { ServerConnection } from '..';
import * as Session from './session';
/**
 * Session object for accessing the session REST api. The session
 * should be used to start kernels and then shut them down -- for
 * all other kernel operations, the kernel object should be used.
 */
export declare class SessionConnection implements Session.ISessionConnection {
    /**
     * Construct a new session.
     */
    constructor(options: Session.ISessionConnection.IOptions);
    /**
     * A signal emitted when the session is disposed.
     */
    get disposed(): ISignal<this, void>;
    /**
     * A signal emitted when the kernel changes.
     */
    get kernelChanged(): ISignal<this, Session.ISessionConnection.IKernelChangedArgs>;
    /**
     * A signal proxied from the connection about the kernel status.
     */
    get statusChanged(): ISignal<this, Kernel.Status>;
    /**
     * A signal proxied from the kernel about the connection status.
     */
    get connectionStatusChanged(): ISignal<this, Kernel.ConnectionStatus>;
    /**
     * A signal proxied from the kernel pending input.
     */
    get pendingInput(): ISignal<this, boolean>;
    /**
     * A signal proxied from the kernel about iopub kernel messages.
     */
    get iopubMessage(): ISignal<this, KernelMessage.IIOPubMessage>;
    /**
     * A signal proxied from the kernel for an unhandled kernel message.
     */
    get unhandledMessage(): ISignal<this, KernelMessage.IMessage>;
    /**
     * A signal proxied from the kernel emitted for any kernel message.
     *
     * #### Notes
     * The behavior is undefined if the message is modified during message
     * handling. As such, it should be treated as read-only.
     */
    get anyMessage(): ISignal<this, Kernel.IAnyMessageArgs>;
    /**
     * A signal emitted when a session property changes.
     */
    get propertyChanged(): ISignal<this, 'path' | 'name' | 'type'>;
    /**
     * Get the session id.
     */
    get id(): string;
    /**
     * Get the session kernel connection object.
     *
     * #### Notes
     * This is a read-only property, and can be altered by [changeKernel].
     */
    get kernel(): Kernel.IKernelConnection | null;
    /**
     * Get the session path.
     */
    get path(): string;
    /**
     * Get the session type.
     */
    get type(): string;
    /**
     * Get the session name.
     */
    get name(): string;
    /**
     * Get the model associated with the session.
     */
    get model(): Session.IModel;
    /**
     * The server settings of the session.
     */
    readonly serverSettings: ServerConnection.ISettings;
    /**
     * Test whether the session has been disposed.
     */
    get isDisposed(): boolean;
    /**
     * Update the session based on a session model from the server.
     *
     * #### Notes
     * This only updates this session connection instance. Use `setPath`,
     * `setName`, `setType`, and `changeKernel` to change the session values on
     * the server.
     */
    update(model: Session.IModel): void;
    /**
     * Dispose of the resources held by the session.
     */
    dispose(): void;
    /**
     * Change the session path.
     *
     * @param path - The new session path.
     *
     * @returns A promise that resolves when the session has renamed.
     *
     * #### Notes
     * This uses the Jupyter REST API, and the response is validated.
     * The promise is fulfilled on a valid response and rejected otherwise.
     */
    setPath(path: string): Promise<void>;
    /**
     * Change the session name.
     */
    setName(name: string): Promise<void>;
    /**
     * Change the session type.
     */
    setType(type: string): Promise<void>;
    /**
     * Change the kernel.
     *
     * @params options - The name or id of the new kernel.
     *
     * #### Notes
     * This shuts down the existing kernel and creates a new kernel,
     * keeping the existing session ID and session path.
     */
    changeKernel(options: Partial<Kernel.IModel>): Promise<Kernel.IKernelConnection | null>;
    /**
     * Kill the kernel and shutdown the session.
     *
     * @returns - The promise fulfilled on a valid response from the server.
     *
     * #### Notes
     * Uses the [Jupyter Notebook API](http://petstore.swagger.io/?url=https://raw.githubusercontent.com/jupyter/notebook/master/notebook/services/api/api.yaml#!/sessions), and validates the response.
     * Disposes of the session and emits a [sessionDied] signal on success.
     */
    shutdown(): Promise<void>;
    /**
     * Create a new kernel connection and connect to its signals.
     *
     * #### Notes
     * This method is not meant to be subclassed.
     */
    protected setupKernel(model: Kernel.IModel | null): void;
    /**
     * Handle to changes in the Kernel status.
     */
    protected onKernelStatus(sender: Kernel.IKernelConnection, state: Kernel.Status): void;
    /**
     * Handle to changes in the Kernel status.
     */
    protected onKernelConnectionStatus(sender: Kernel.IKernelConnection, state: Kernel.ConnectionStatus): void;
    /**
     * Handle a change in the pendingInput.
     */
    protected onPendingInput(sender: Kernel.IKernelConnection, state: boolean): void;
    /**
     * Handle iopub kernel messages.
     */
    protected onIOPubMessage(sender: Kernel.IKernelConnection, msg: KernelMessage.IIOPubMessage): void;
    /**
     * Handle unhandled kernel messages.
     */
    protected onUnhandledMessage(sender: Kernel.IKernelConnection, msg: KernelMessage.IMessage): void;
    /**
     * Handle any kernel messages.
     */
    protected onAnyMessage(sender: Kernel.IKernelConnection, args: Kernel.IAnyMessageArgs): void;
    /**
     * Send a PATCH to the server, updating the session path or the kernel.
     */
    private _patch;
    /**
     * Handle a change to the model.
     */
    private _handleModelChange;
    private _id;
    private _path;
    private _name;
    private _type;
    private _username;
    private _clientId;
    private _kernel;
    private _isDisposed;
    private _disposed;
    private _kernelChanged;
    private _statusChanged;
    private _connectionStatusChanged;
    private _pendingInput;
    private _iopubMessage;
    private _unhandledMessage;
    private _anyMessage;
    private _propertyChanged;
    private _connectToKernel;
    private _kernelConnectionOptions;
}
