"use strict";
// Copyright (c) Jupyter Development Team.
// Distributed under the terms of the Modified BSD License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServiceManager = void 0;
const signaling_1 = require("@lumino/signaling");
const builder_1 = require("./builder");
const nbconvert_1 = require("./nbconvert");
const contents_1 = require("./contents");
const event_1 = require("./event");
const kernel_1 = require("./kernel");
const kernelspec_1 = require("./kernelspec");
const session_1 = require("./session");
const setting_1 = require("./setting");
const terminal_1 = require("./terminal");
const serverconnection_1 = require("./serverconnection");
const user_1 = require("./user");
const workspace_1 = require("./workspace");
/**
 * A Jupyter services manager.
 */
class ServiceManager {
    /**
     * Construct a new services provider.
     */
    constructor(options = {}) {
        var _a, _b;
        this._isDisposed = false;
        this._connectionFailure = new signaling_1.Signal(this);
        this._isReady = false;
        const defaultDrive = options.defaultDrive;
        const serverSettings = (_a = options.serverSettings) !== null && _a !== void 0 ? _a : serverconnection_1.ServerConnection.makeSettings();
        const standby = (_b = options.standby) !== null && _b !== void 0 ? _b : 'when-hidden';
        const normalized = { defaultDrive, serverSettings, standby };
        const kernelManager = new kernel_1.KernelManager(normalized);
        this.serverSettings = serverSettings;
        this.contents = new contents_1.ContentsManager(normalized);
        this.events = new event_1.EventManager(normalized);
        this.sessions = new session_1.SessionManager(Object.assign(Object.assign({}, normalized), { kernelManager: kernelManager }));
        this.settings = new setting_1.SettingManager(normalized);
        this.terminals = new terminal_1.TerminalManager(normalized);
        this.builder = new builder_1.BuildManager(normalized);
        this.workspaces = new workspace_1.WorkspaceManager(normalized);
        this.nbconvert = new nbconvert_1.NbConvertManager(normalized);
        this.kernelspecs = new kernelspec_1.KernelSpecManager(normalized);
        this.user = new user_1.UserManager(normalized);
        // Proxy all connection failures from the individual service managers.
        this.kernelspecs.connectionFailure.connect(this._onConnectionFailure, this);
        this.sessions.connectionFailure.connect(this._onConnectionFailure, this);
        this.terminals.connectionFailure.connect(this._onConnectionFailure, this);
        // Define promises that need to be resolved before service manager is ready.
        const readyList = [this.sessions.ready, this.kernelspecs.ready];
        if (this.terminals.isAvailable()) {
            readyList.push(this.terminals.ready);
        }
        this._readyPromise = Promise.all(readyList).then(() => {
            this._isReady = true;
        });
    }
    /**
     * A signal emitted when there is a connection failure with the kernel.
     */
    get connectionFailure() {
        return this._connectionFailure;
    }
    /**
     * Test whether the service manager is disposed.
     */
    get isDisposed() {
        return this._isDisposed;
    }
    /**
     * Dispose of the resources used by the manager.
     */
    dispose() {
        if (this.isDisposed) {
            return;
        }
        this._isDisposed = true;
        signaling_1.Signal.clearData(this);
        this.contents.dispose();
        this.events.dispose();
        this.sessions.dispose();
        this.terminals.dispose();
    }
    /**
     * Test whether the manager is ready.
     */
    get isReady() {
        return this._isReady;
    }
    /**
     * A promise that fulfills when the manager is ready.
     */
    get ready() {
        return this._readyPromise;
    }
    _onConnectionFailure(sender, err) {
        this._connectionFailure.emit(err);
    }
}
exports.ServiceManager = ServiceManager;
//# sourceMappingURL=manager.js.map