/*
Copyright 2024 The Kubernetes Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package testing

import (
	"context"
	"testing"

	"k8s.io/apimachinery/pkg/runtime"
	"k8s.io/apimachinery/pkg/util/sets"
	"k8s.io/apimachinery/pkg/util/validation/field"
)

type VersionValidationRunner func(t *testing.T, gv string, versionValidationErrors field.ErrorList)

// RunValidationForEachVersion runs f as a subtest of t for each version of the given unversioned object.
// Each subtest is named by GroupVersionKind.  Each call to f is provided the field.ErrorList results
// of converting the unversioned object to a version and validating it.
//
// Only autogenerated validation is run. To test both handwritten and autogenerated validation:
//
//		RunValidationForEachVersion(t, testCase.pod, func(t *testing.T, versionValidationErrors field.ErrorList) {
//		  errs := ValidatePod(testCase.obj) // hand written validation
//	      errs = append(errs, versionValidationErrors...) // generated declarative validation
//		  // Validate that the errors are what was expected for this test case.
//		})
func RunValidationForEachVersion(t *testing.T, scheme *runtime.Scheme, options sets.Set[string], unversioned runtime.Object, fn VersionValidationRunner) {
	runValidation(t, scheme, options, unversioned, fn)
}

// RunUpdateValidationForEachVersion is like RunValidationForEachVersion but for update validation.
func RunUpdateValidationForEachVersion(t *testing.T, scheme *runtime.Scheme, options sets.Set[string], unversioned, unversionedOld runtime.Object, fn VersionValidationRunner) {
	runUpdateValidation(t, scheme, options, unversioned, unversionedOld, fn)
}

// RunStatusValidationForEachVersion is like RunUpdateValidationForEachVersion but for status validation.
func RunStatusValidationForEachVersion(t *testing.T, scheme *runtime.Scheme, options sets.Set[string], unversioned, unversionedOld runtime.Object, fn VersionValidationRunner) {
	runUpdateValidation(t, scheme, options, unversioned, unversionedOld, fn, "status")
}

func runValidation(t *testing.T, scheme *runtime.Scheme, options sets.Set[string], unversioned runtime.Object, fn VersionValidationRunner, subresources ...string) {
	unversionedGVKs, _, err := scheme.ObjectKinds(unversioned)
	if err != nil {
		t.Fatal(err)
	}
	for _, unversionedGVK := range unversionedGVKs {
		gvs := scheme.VersionsForGroupKind(unversionedGVK.GroupKind())
		for _, gv := range gvs {
			gvk := gv.WithKind(unversionedGVK.Kind)
			t.Run(gvk.String(), func(t *testing.T) {
				if gvk.Version != runtime.APIVersionInternal { // skip internal
					versioned, err := scheme.New(gvk)
					if err != nil {
						t.Fatal(err)
					}
					err = scheme.Convert(unversioned, versioned, nil)
					if err != nil {
						t.Fatal(err)
					}
					fn(t, gv.String(), scheme.Validate(context.Background(), options, versioned, subresources...))
				}
			})
		}
	}
}

func runUpdateValidation(t *testing.T, scheme *runtime.Scheme, options sets.Set[string], unversionedNew, unversionedOld runtime.Object, fn VersionValidationRunner, subresources ...string) {
	unversionedGVKs, _, err := scheme.ObjectKinds(unversionedNew)
	if err != nil {
		t.Fatal(err)
	}
	for _, unversionedGVK := range unversionedGVKs {
		gvs := scheme.VersionsForGroupKind(unversionedGVK.GroupKind())
		for _, gv := range gvs {
			gvk := gv.WithKind(unversionedGVK.Kind)
			t.Run(gvk.String(), func(t *testing.T) {
				if gvk.Version != runtime.APIVersionInternal { // skip internal
					versionedNew, err := scheme.New(gvk)
					if err != nil {
						t.Fatal(err)
					}
					err = scheme.Convert(unversionedNew, versionedNew, nil)
					if err != nil {
						t.Fatal(err)
					}

					var versionedOld runtime.Object
					if unversionedOld != nil {
						versionedOld, err = scheme.New(gvk)
						if err != nil {
							t.Fatal(err)
						}

						err = scheme.Convert(unversionedOld, versionedOld, nil)
						if err != nil {
							t.Fatal(err)
						}
					}

					fn(t, gv.String(), scheme.ValidateUpdate(context.Background(), options, versionedNew, versionedOld, subresources...))
				}
			})
		}
	}
}
