// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief B -> Xc ell - nu_ell
  class BELLE_2007_I732595 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BELLE_2007_I732595);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // projections
      declare(UnstableParticles(Cuts::abspid==511 ||
				Cuts::abspid==521),"UFS");
      // histograms
      for(unsigned int ix=0;ix<2;++ix) {
	book(_p[ix],1,1,1+2*ix);
      }
    }

    void findDecayProducts(Particle parent, Particles & em, Particles & ep,
			   Particles & nue, Particles & nueBar, bool & charm) {
      for(const Particle & p : parent.children()) {
	if(PID::isCharmHadron(p.pid())) {
	  charm=true;
	}
	else if(p.pid() == PID::EMINUS || p.pid()==PID::MUON) {
	  em.push_back(p);
	}
	else if(p.pid() == PID::EPLUS || p.pid()==PID::ANTIMUON) {
	  ep.push_back(p);
	}
	else if(p.pid() == PID::NU_E  || p.pid()==PID::NU_MU) {
	  nue.push_back(p);
	}
	else if(p.pid() == PID::NU_EBAR || p.pid()==PID::NU_MUBAR) {
	  nueBar.push_back(p);
	}
	else if(PID::isBottomHadron(p.pid())) {
	  findDecayProducts(p,em,ep,nue,nueBar,charm);
	}
	else if(!PID::isHadron(p.pid())) {
	  findDecayProducts(p,em,ep,nue,nueBar,charm);
	}
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      // find and loop over B mesons
      for (const Particle& p : apply<UnstableParticles>(event, "UFS").particles()) {
	if(p.children().empty() ||
	   (p.children().size()==1 && p.children()[1].abspid()==p.abspid())) continue;
       	// find decay products
       	bool charm = false;
       	Particles em,ep,nue,nueBar;
       	findDecayProducts(p,em,ep,nue,nueBar,charm);
       	if(!charm) continue;
	FourMomentum pl,pnu;
	if(em.size()==1 && nueBar.size()==1 && em[0].pid()+1==-nueBar[0].pid()) {
	  pl  = em[0].momentum();
	  pnu = nueBar[0].momentum();
	}
	else if(ep.size()==1 && nue.size()==1 && nue[0].pid()==-ep[0].pid()+1) {
	  pl  = ep[0].momentum();
	  pnu = nue[0].momentum();
	}
	else
	  continue;
	// boost to rest frame
	LorentzTransform boost = LorentzTransform::mkFrameTransformFromBeta(p.momentum().betaVec());
	double mX2 = (p.momentum()-pl-pnu).mass2();
       	pl = boost.transform(pl);
       	double Estar = pl.E();
	for(unsigned int ix=0;ix<2;++ix) {
       	  for(const auto & bin : _p[ix]->bins())
       	    if(bin.xMin()<Estar) _p[ix]->fill(bin.xMid(),mX2);
	  mX2 *=mX2;
       	}
      }
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      // compute <(mx2-<mx2>)^2> = <mx4>-<mx2>^2
      Scatter2DPtr tmp;
      book(tmp,1,1,2);
      for(unsigned int ix=0;ix<_p[0]->numBins();++ix) {
	double x  = 0.5*(_p[0]->xEdges()[ix+1]+_p[0]->xEdges()[ix]);
	double dx = 0.5*(_p[0]->xEdges()[ix+1]-_p[0]->xEdges()[ix]);
	double value = _p[1]->bins()[ix].mean()-sqr(_p[0]->bins()[ix].mean());
	double error = value*sqrt(sqr(_p[1]->bins()[ix].relErr())+4.*sqr(_p[0]->bins()[ix].relErr()));
	tmp->addPoint(x,value,make_pair(dx,dx),make_pair(error,error));
      }
    }

    /// @}


    /// @name Histograms
    /// @{
    Profile1DPtr _p[2];
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BELLE_2007_I732595);

}
