const name = /[a-zA-Z_]\w*/;
const exponent = /[eE][+-]?[0-9]+/;
const positive_exponent = /[eE]+?[0-9]+/;
const binary_exponent = /[pP][+-]?[0-9]+/;
const nonzero_decimal_digits = /0|([1-9][0-9_]*)/

let cur = 1;
const PREC = {
  ASSIGN: cur++,
  LOR: cur++,
  LXOR: cur++,
  LAND: cur++,
  EQUALITY: cur++,
  COMPARISON: cur++,
  BOR: cur++,
  BXOR: cur++,
  BAND: cur++,
  SHIFT: cur++,
  ADD: cur++,
  MULTIPLY: cur++,
  CAST: cur++,
  UNARY: cur++,
  POSTFIX: cur++,
};

module.exports = grammar ({
  name: 'hare',

  extras:  $ => [
    /\s/,
    $.comment,
  ],

  word: $ => $.keyword,

  rules: {
    sub_unit: $ => seq(
      optional($.imports),
      optional($.declarations),
    ),

    keyword: $ => token(choice(
      'abort', 'align', 'alloc', 'append', 'as', 'assert', 'bool', 'break',
      'case', 'const', 'continue', 'def', 'defer', 'delete', 'done', 'else',
      'enum', 'export', 'f32', 'f64', 'false', 'fn', 'for', 'free', 'i16',
      'i32', 'i64', 'i8', 'if', 'insert', 'int', 'is', 'len', 'let', 'match',
      'null', 'nullable', 'offset', 'return', 'rune', 'size', 'static', 'str',
      'struct', 'switch', 'true', 'type', 'u16', 'u32', 'u64', 'u8', 'uint',
      'uintptr', 'union', 'use', 'vaarg', 'vaend', 'valist', 'vastart', 'void',
      'opaque', 'yield', '_', 'never',
    )),

    name: $ => /[a-zA-Z_]\w*/,

    identifier: $ => token(seq(name, repeat(seq('::', name)))),

    comment: $ => /\/\/[^\n]*/,

    // Types (§6.5)
    type: $ => seq(optional('const'), optional('!'), $._storage_class),

    _storage_class: $ => choice(
      $._primitive_type,
      $.struct_union_type,
      $.tuple_type,
      $.tagged_union_type,
      $.slice_array_type,
      $.function_type,
      $.alias_type,
      $.unwrapped_alias,
      $._string_type,
    ),

    _primitive_type: $ => choice(
      $._integer_type,
      $._floating_type,
      $.pointer_type,
      'rune',
      'bool',
      'valist',
      'void',
      'opaque',
      'done',
      'never',
    ),

    _integer_type: $ => choice(
      'i8', 'i16', 'i32', 'i64', 'u8', 'u16', 'u32', 'u64', 'int', 'uint',
      'size', 'uintptr',
    ),

    _floating_type: $ => choice(
      'f32', 'f64',
    ),

    pointer_type: $ => choice(
      seq('*', $.type),
      seq('nullable', '*', $.type),
    ),

    struct_union_type: $ => choice(
      seq('struct', optional('@packed'), '{', $.struct_fields, '}'),
      seq('union', '{', $.struct_union_fields, '}'),
    ),

    struct_union_fields: $ => seq(
      $.struct_union_field,
      repeat(seq(',', $.struct_union_field)),
      optional(','),
    ),

    struct_union_field: $ => choice(
      seq($.name, ':', $.type),
      $.struct_union_type,
      $.identifier,
    ),

    struct_fields: $ => seq(
      $.struct_field,
      repeat(seq(',', $.struct_field)),
      optional(','),
    ),

    struct_field: $ => seq(optional($.offset_specifier), $.struct_union_field),

    offset_specifier: $ => seq('@offset', '(', $.expression, ')'),

    tuple_type: $ =>
      seq('(', $.type, repeat1(seq(',', $.type)), optional(','), ')'),

    tagged_union_type: $ =>
      seq('(', $.type, repeat1(seq('|', $.type)), optional('|'), ')'),

    slice_array_type: $ => choice(
      seq('[',']', $.type),
      seq('[', $.expression, ']', $.type),
      seq('[', '*', ']', $.type),
      seq('[', '_', ']', $.type),
    ),

    _string_type: $ => 'str',

    function_type: $ => seq('fn', $._prototype),

    _prototype: $ => seq(
      '(',
       optional(seq($.parameter, repeat(seq(',', $.parameter)))),
       optional(','),
       optional('...'),
      ')',
      $.type
    ),

    parameter: $ => seq(
      optional(seq($.name, ':')),
      $.type,
      optional(seq('=', $.expression)),
    ),

    alias_type: $ => $.identifier,

    unwrapped_alias: $ => seq('...', $.identifier),

    // Expressions (§6.6)
    literal: $ => choice(
      $.integer_literal,
      $.floating_literal,
      $.rune_literal,
      $.string_literal,
      $.array_literal,
      $.struct_literal,
      $.tuple_literal,
      'true',
      'false',
      'null',
      'void',
    ),

    floating_literal: $ => token(choice(
      seq(/0x[0-9a-fA-F]+(\.[0-9a-fA-F]+)?/, binary_exponent, optional(choice('f32', 'f64'))),
      seq(nonzero_decimal_digits, /\.[0-9]+/, optional(exponent)),
      seq(nonzero_decimal_digits, optional(exponent), choice('f32', 'f64')),
    )),

    integer_literal: $ => token(seq(
      choice(
        /0x[0-9A-Fa-f_]+/,
        /0o[0-8_]+/,
        /0b[01_]+/,
        seq(nonzero_decimal_digits, optional(positive_exponent)),
      ),
      optional(choice(
        'i', 'u', 'z', 'i8', 'i16', 'i32', 'i64', 'u8', 'u16', 'u32', 'u64',
      )),
    )),

    rune_literal: $ => seq(
      "'",
      choice($.escape_sequence, token.immediate(prec(1, /[^\\']/))),
      token.immediate("'"),
    ),

    escape_sequence: $ => token.immediate(choice(
      /\\[0abfnrtv\\'"]/,
      /\\x[0-9A-Fa-f]{2}/,
      /\\u[0-9A-Fa-f]{4}/,
      /\\U[0-9A-Fa-f]{8}/,
    )),

    string_literal: $ => choice(
      /`[^`]+`/,
      seq(
        '"',
        repeat(choice($.escape_sequence, token.immediate(prec(1, /[^\\"]+/)))),
        token.immediate('"'),
      )
    ),

    array_literal: $ => seq(
      '[',
      optional(seq(
        $.expression,
        repeat(seq(',', $.expression)),
        optional('...'),
        optional(','),
      )),
      ']'
    ),

    struct_literal: $ => seq(
      choice('struct', $.identifier),
      '{',
      optional(seq(
        $.field_value,
        repeat(seq(',', $.field_value)),
        optional(',')
      )),
      '}',
    ),

    field_value: $ => choice(
      seq($.name, '=', $.expression),
      seq($.name, ':', $.type, '=', $.expression),
      $.struct_literal,
      '...',
    ),

    tuple_literal: $ => seq(
      '(',
      $.expression,
      repeat1(seq(',', $.expression)),
      optional(','),
      ')'
    ),

    _plain_expression: $ => choice(
      $.identifier,
      $.literal,
    ),

    nested_expression: $ => choice(
      $._plain_expression,
      seq('(', $.expression, ')'),
    ),

    allocation_expression: $ => choice(
      seq('alloc', '(', $.expression, ')'),
      seq('alloc', '(', $.expression, '...', ')'),
      seq('alloc', '(', $.expression, ',', $.expression, ')'),
      seq('free', '(', $.expression, ')'),
    ),
    
    assertion_expression: $ => seq(
      optional('static'),
      choice(
        seq('assert', '(', $.expression, ')'),
        seq('assert', '(', $.expression, ',', $.string_literal, ')'),
        seq('abort',  '(', optional($.string_literal), ')'),
      ),
    ),

    call_expression: $ => seq($.postfix_expression, '(', optional($.argument_list), ')'),

    argument_list: $ => choice(
      seq($.expression, optional(',')),
      seq($.expression, '...', optional(',')),
      seq($.expression, ',', $.argument_list),
    ),

    measurement_expression: $ => choice(
      $.align_expression,
      $.size_expression,
      $.length_expression,
      $.offset_expression,
    ),

    align_expression: $ => seq('align', '(', $.type, ')'),

    size_expression: $ => seq('size', '(', $.type, ')'),

    length_expression: $ => seq('len', '(', $.expression, ')'),

    offset_expression: $ => seq('offset', '(', $.field_access_expression, ')'),

    field_access_expression: $ => choice(
      seq($.postfix_expression, field('selector', seq('.', $.name))),
      seq($.postfix_expression, field('selector', seq('.', $.integer_literal)))
    ),

    indexing_expression: $ => seq($.postfix_expression, '[', $.expression, ']'),

    slicing_expression: $ => seq(
      $.postfix_expression, '[', optional($.expression),
        '..',  optional($.expression), ']',
    ),

    slice_mutation_expression: $ =>
      choice($.append_expression, $.insert_expression, $.delete_expression),

    append_expression: $ => seq(
      optional('static'),
      choice(
        seq('append', '(', $.expression, ',', $.expression, ')'),
        seq('append', '(', $.expression, ',', $.expression, '...', ')'),
        seq('append', '(', $.expression, ',', $.expression, ',', $.expression, ')'),
      ),
    ),

    insert_expression: $ => seq(
      optional('static'), 'insert', '(',
      $.indexing_expression, ',', $.expression, choice(optional('...'), seq(',', $.expression)), ')',
    ),

    delete_expression: $ => seq(
      optional('static'),
      choice(
        seq('delete', '(', $.slicing_expression, ')'),
        seq('delete', '(', $.indexing_expression, ')'),
      ),
    ),

    error_propagation: $ => seq($.postfix_expression, choice('?', '!')),

    postfix_expression: $ => choice(
      $.nested_expression,
      $.call_expression,
      $.field_access_expression,
      $.indexing_expression,
      $.slicing_expression,
      $.error_propagation,
    ),

    variadic_expression: $ => choice(
      seq('vastart', '(', ')'),
      seq('vaarg', '(', $.expression, ')'),
      seq('vaend', '(', $.expression, ')'),
    ),

    builtin_expression: $ => choice(
      $.allocation_expression,
      $.assertion_expression,
      $.measurement_expression,
      $.slice_mutation_expression,
      $.variadic_expression,
    ),

    unary_expression: $ => prec.left(PREC.UNARY,
      seq(choice('+', '-', '~', '!', '*', '&'), $.expression)),

    cast_expression: $ => prec.left(PREC.CAST, choice(
      field('type_cast', seq($.expression, ':', $.type)),
      field('as_cast', seq($.expression, 'as', $.nullable_type)),
      field('is_cast', seq($.expression, 'is', $.nullable_type)),
    )),

    nullable_type: $ => choice($.type, 'null'),

    multiplicative_expression: $ => prec.left(PREC.MULTIPLY,
      seq($.expression, choice('*', '/', '%'), $.expression)),

    additive_expression: $ => prec.left(PREC.ADD,
      seq($.expression, choice('+', '-'), $.expression)),

    shift_expression: $ => prec.left(PREC.SHIFT,
      seq($.expression, choice('<<', '>>'), $.expression)),

    and_expression: $ => prec.left(PREC.BAND,
      seq($.expression, '&', $.expression)),

    exclusive_or_expression: $ => prec.left(PREC.BXOR,
      seq($.expression, '^', $.expression)),

    inclusive_or_expression: $ => prec.left(PREC.BOR,
      seq($.expression, '|', $.expression)),

    comparison_expression: $ => prec.left(PREC.COMPARISON,
      seq($.expression, choice('<', '>', '<=', '>='), $.expression)),

    equality_expression: $ => prec.left(PREC.EQUALITY,
      seq($.expression, choice('==', '!='), $.expression)),

    logical_and_expression: $ => prec.left(PREC.LAND,
      seq($.expression, '&&', $.expression)),

    logical_xor_expression: $ => prec.left(PREC.LXOR,
      seq($.expression, '^^', $.expression)),

    logical_or_expression: $ => prec.left(PREC.LOR,
      seq($.expression, '||', $.expression)),

    if_expression: $ => prec.right(choice(
      seq('if', $.conditional_branch),
      seq('if', $.conditional_branch, 'else', $.expression),
    )),

    conditional_branch: $ => seq('(' , $.expression, ')',  $.expression),

    for_loop: $ => seq(
      'for', optional($.label), '(', $.for_predicate, ')', $.expression,
    ),

    for_predicate: $ => choice(
      $.iterable_binding,
      $.expression,
      seq(field('binding', $.binding_list), ';', field('afterthought', $.expression)),
      seq(field('condition', $.expression), ';', field('afterthought', $.expression)),
      seq(field('binding', $.binding_list), ';', field('condition', $.expression), ';', field('afterthought', $.expression)),
    ),

    iterable_binding: $ => seq(
      choice('const', 'let'),
      $.binding_name,
      optional(seq(':', $.type)),
      choice('..', '&..', '=>'),
      $.expression,
    ),

    switch_expression: $ => seq(
      'switch', optional($.label), '(', $.expression, ')', '{', $.switch_cases, '}',
    ),

    switch_cases: $ => repeat1($.switch_case),

    switch_case: $ => choice(
     seq('case', $.case_options, '=>', $.expression_list),
     seq('case', '=>', $.expression_list),
    ),

    case_options: $ => choice(
      seq($.expression, optional(',')),
      seq($.expression, ',',  $.case_options),
    ),

    match_expression: $ =>
      seq('match', optional($.label), '(', $.expression, ')', '{', $.match_cases, '}'),

    match_cases: $ => repeat1($.match_case),

    match_case: $ => choice(
      seq('case', 'let', $.name, ':', $.type, '=>', $.expression_list),
      seq('case', $.nullable_type, '=>', $.expression_list),
      seq('case', '=>', $.expression_list),
    ),

    assignment: $ => prec.right(PREC.ASSIGN, choice(
      seq($.expression, $.assignment_op, $.expression),
      seq('(', $.tuple_binding_names, ')', '=', $.expression),
    )),

    assignment_op: $ => choice(
      '=', '+=', '-=', '*=', '/=', '%=', '<<=', '>>=',
      '&=', '|=', '^=', '&&=', '||=', '^^=',
    ),

    binding_list: $ => choice(
      seq(optional('static'), 'let', $.bindings),
      seq(optional('static'), 'const', $.bindings),
      seq('def', $.binding),
    ),

    bindings: $ => choice(
      seq($.binding, optional(',')),
      seq($.binding, ',', $.bindings),
    ),

    binding: $ => choice(
      seq($.binding_name, '=', $.expression),
      seq($.binding_name, ':', $.type, '=', $.expression),
    ),

    binding_name: $ => choice($.name, seq('(', $.tuple_binding_names, ')')),

    tuple_binding_names: $ => choice(
      seq($.tuple_binding_name, ',', $.tuple_binding_name),
      seq($.tuple_binding_name, ',', $.tuple_binding_names),
    ),

    tuple_binding_name: $ => choice($.name, '_'),

    defer_expression: $ => seq('defer', $.expression),

    expression_list: $ => choice(
      seq($.expression, ';', optional($.expression_list)),
      seq($.binding_list, ';', optional($.expression_list)),
      seq($.defer_expression, ';', optional($.expression_list)),
    ),

    compound_expression: $ => seq(optional($.label), '{', $.expression_list, '}'),

    label: $ => seq(':', $.name),

    control_expression: $ => prec.right(0, choice(
      seq('break', optional($.label)),
      seq('continue', optional($.label)),
      seq('return', optional($.expression)),
      seq('yield', optional($.expression)),
      seq('yield', $.label, optional(seq(', ', $.expression))),
    )),

    expression: $ => choice(
      $.assignment,
      $.logical_or_expression,
      $.logical_xor_expression,
      $.logical_and_expression,
      $.equality_expression,
      $.comparison_expression,
      $.inclusive_or_expression,
      $.exclusive_or_expression,
      $.and_expression,
      $.shift_expression,
      $.additive_expression,
      $.multiplicative_expression,
      $.cast_expression,
      $.unary_expression,
      $.postfix_expression,
      $.builtin_expression,
      $.compound_expression,
      $.match_expression,
      $.switch_expression,
      $.if_expression,
      $.for_loop,
      $.control_expression,
    ),

    // Declarations (§6.11)
    declarations: $ => repeat1(seq(optional('export'), $.declaration, ';')),

    declaration: $ => choice(
      $.global_declaration,
      $.constant_declaration,
      $.type_declaration,
      $.function_declaration,
    ),

    global_declaration: $ => choice(
      seq('let', $.global_bindings),
      seq('const', $.global_bindings),
    ),

    global_bindings: $ => choice(
      seq($.global_binding, optional(',')),
      seq($.global_binding, ',', $.global_bindings),
    ),

    global_binding: $ => seq(
      optional($.decl_attr),
      optional('@threadlocal'),
      $.identifier,
      choice(
        seq('=', $.expression),
        seq(':', $.type, optional(seq('=', $.expression))),
      ),
    ),

    decl_attr: $ => seq(
      '@symbol', '(', $.string_literal, ')',
    ),

    constant_declaration: $ => seq('def', $.constant_bindings),

    constant_bindings: $ => choice(
      seq($.constant_binding, optional(',')),
      seq($.constant_binding, ',', $.constant_bindings),
    ),

    constant_binding: $ => seq(
      $.identifier,
      optional(seq(':', $.type)),
      '=',
      $.expression,
    ),

    type_declaration: $ => seq('type', $.type_bindings),

    type_bindings: $ => choice(
      seq($.type_binding, optional(',')),
      seq($.type_binding, ',', $.type_bindings),
    ),

    type_binding: $ => seq($.identifier, '=', choice($.type, $.enum_type)),

    enum_type: $ => seq('enum', optional($.enum_storage), '{', $.enum_values, '}'),

    enum_values: $ => choice(
      seq($.enum_value, optional(',')),
      seq($.enum_value, ',', $.enum_values),
    ),

    enum_value: $ => choice(
      $.name,
      seq($.name, '=', $.expression),
    ),

    enum_storage: $ => choice($._integer_type, 'rune'),

    function_declaration: $ =>
      seq(optional($.fndec_attrs), 'fn',
        field('name', $.identifier), field('type', $._prototype),
        optional(seq('=', field('body', $.expression))),
      ),

    fndec_attrs: $ => repeat1($.fndec_attr),

    fndec_attr: $ => choice(
      '@fini',
      '@init',
      '@test',
      $.decl_attr,
    ),

    // Units (§6.12)
    imports: $ => repeat1($.use_directive),

    use_directive: $ => seq(
      'use',
      choice(
	$.identifier,
      	seq($.name, '=', $.identifier),
	seq($.identifier, '::', '*'),
	seq($.identifier, '::', '{', $.member_list, '}'),
      ),
      ';',
    ),

    member_list: $ => choice(
      seq($.member, optional(',')),
      seq($.member, ',', $.member_list),
    ),

    member: $ => choice(
      $.name,
      seq($.name, '=', $.name),
    ),
  },
})
