searchData={"items":[{"type":"module","title":"diameter","doc":"Main API of the diameter application.\n\nThis module provides the interface with which a user can implement a Diameter\nnode that sends and receives messages using the Diameter protocol as defined in\nRFC 6733.\n\nBasic usage consists of creating a representation of a locally implemented\nDiameter node and its capabilities with `start_service/2`, adding transport\ncapability using `add_transport/2` and sending Diameter requests and receiving\nDiameter answers with `call/4`. Incoming Diameter requests are communicated as\ncallbacks to a `m:diameter_app` callback modules as specified in the service\nconfiguration.\n\nBeware the difference between _diameter_ (not capitalized) and _Diameter_\n(capitalized). The former refers to the Erlang application named diameter whose\nmain api is defined here, the latter to Diameter protocol in the sense of\nRFC 6733.\n\nThe diameter application must be started before calling most functions in this\nmodule.","ref":"diameter.html"},{"type":"module","title":"DATA TYPES - diameter","doc":"- **`Address()`**\n\n- **`DiameterIdentity()`**\n\n- **`Grouped()`**\n\n- **`OctetString()`**\n\n- **`Time()`**\n\n- **`Unsigned32()`**\n\n- **`UTF8String()`** - Types corresponding to RFC 6733 AVP Data Formats. Defined\n  in [diameter_dict(4)](diameter_dict.md#DATA_TYPES).\n\n- **`elapsed_time()`** - Elapsed time since a given time.\n\n  [](){: #application_alias }\n\n- **`application_alias() = term()`** - Name identifying a Diameter application\n  in service configuration. Passed to `call/4` when sending requests defined by\n  the application.\n\n  [](){: #application_module }\n\n- **`application_module() = Mod | [Mod | ExtraArgs] | #diameter_callback{}`**\n\n  ```text\n  Mod = atom()\n  ExtraArgs = list()\n  ```\n\n  Module implementing the callback interface defined in `m:diameter_app`, along\n  with any extra arguments to be appended to those documented. Note that extra\n  arguments specific to an outgoing request can be specified to `call/4`, in\n  which case those are appended to any module-specific extra arguments.\n\n  Specifying a `#diameter_callback{}` record allows individual functions to be\n  configured in place of the usual `m:diameter_app` callbacks. See\n  `diameter_callback.erl` for details.\n\n  [](){: #application_opt }\n\n- **`application_opt()`** - Options defining a Diameter application. Has one of\n  the following types.\n\n  - **`{alias, `[`application_alias()`](`m:diameter#application_alias`)`}`** -\n    Unique identifier for the application in the scope of the service. Defaults\n    to the value of the `dictionary` option.\n\n  - **`{dictionary, atom()}`** - Name of an encode/decode module for the\n    Diameter messages defined by the application. These modules are generated\n    from files whose format is documented in\n    [diameter_dict(4)](diameter_dict.md).\n\n  - **`{module, `[`application_module()`](`m:diameter#application_module`)`}`** -\n    Callback module in which messages of the Diameter application are handled.\n    See `m:diameter_app` for the required interface and semantics.\n\n  - **`{state, term()}`** - Initial callback state. The prevailing state is\n    passed to some `m:diameter_app` callbacks, which can then return a new\n    state. Defaults to the value of the `alias` option.\n\n  - **`{call_mutates_state, true|false}`** - Whether or not the\n    [pick_peer/4](`c:diameter_app:pick_peer/4`) application callback can modify\n    the application state. Defaults to `false`.\n\n    > #### Warning {: .warning }\n    >\n    > [pick_peer/4](`c:diameter_app:pick_peer/4`) callbacks are serialized when\n    > this option is `true`, which is a potential performance bottleneck. A\n    > simple Diameter client may suffer no ill effects from using mutable state\n    > but a server or agent that responds to incoming request should probably\n    > avoid it.\n\n  - **`{answer_errors, callback|report|discard}`** - Manner in which incoming\n    answer messages containing decode errors are handled.\n\n    If `callback` then errors result in a\n    [handle_answer/4](`c:diameter_app:handle_answer/4`) callback in the same\n    fashion as for [handle_request/3](`c:diameter_app:handle_request/3`), with\n    errors communicated in the `errors` field of the `#diameter_packet{}` passed\n    to the callback. If `report` then an answer containing errors is discarded\n    without a callback and a warning report is written to the log. If `discard`\n    then an answer containing errors is silently discarded without a callback.\n    In both the `report` and `discard` cases the return value for the `call/4`\n    invocation in question is as if a callback had taken place and returned\n    `{error, failure}`.\n\n    Defaults to `discard`.\n\n  - **`{request_errors, answer_3xxx|answer|callback}`** - Manner in which\n    incoming requests are handled when an error other than 3007\n    (DIAMETER_APPLICATION_UNSUPPORTED, which cannot be associated with an\n    application callback module), is detected.\n\n    If `answer_3xxx` then requests are answered without a\n    [handle_request/3](`c:diameter_app:handle_request/3`) callback taking place.\n    If `answer` then even 5xxx errors are answered without a callback unless the\n    connection in question has configured the RFC 3588 common dictionary as\n    noted below. If `callback` then a\n    [handle_request/3](`c:diameter_app:handle_request/3`) callback always takes\n    place and its return value determines the answer sent to the peer, if any.\n\n    Defaults to `answer_3xxx`.\n\n    > #### Note {: .info }\n    >\n    > Answers sent by diameter set the E-bit in the Diameter Header. Since RFC\n    > 3588 allows only 3xxx result codes in an `answer-message`, `answer` has\n    > the same semantics as `answer_3xxx` when the transport in question has\n    > been configured with `diameter_gen_base_rfc3588` as its common dictionary.\n    > Since RFC 6733 allows both 3xxx and 5xxx result codes in an\n    > `answer-message`, a transport with `diameter_gen_base_rfc6733` as its\n    > common dictionary does distinguish between `answer_3xxx` and `answer`.\n\n  [](){: #call_opt }\n\n- **`call_opt()`** - Options available to `call/4` when sending an outgoing\n  Diameter request. Has one of the following types.\n\n  - **`{extra, list()}`** - Extra arguments to append to callbacks to the\n    callback module in question. These are appended to any extra arguments\n    configured on the callback itself. Multiple options append to the argument\n    list.\n\n  - **`{filter, `[`peer_filter()`](`m:diameter#peer_filter`)`}`** - Filter to\n    apply to the list of available peers before passing it to the\n    [pick_peer/4](`c:diameter_app:pick_peer/4`) callback for the application in\n    question. Multiple options are equivalent a single `all` filter on the\n    corresponding list of filters. Defaults to `none`.\n\n  - **`{peer, `[`diameter_app:peer_ref()`](`t:diameter_app:peer_ref/0`)`}`** -\n    Peer to which the request in question can be sent, preempting the selection\n    of peers having advertised support for the Diameter application in question.\n    Multiple options can be specified, and their order is respected in the\n    candidate lists passed to a subsequent\n    [pick_peer/4](`c:diameter_app:pick_peer/4`) callback.\n\n  - **`{timeout, `[`Unsigned32()`](diameter_dict.md#DATA_TYPES)`}`** - Number of\n    milliseconds after which the request should timeout. Defaults to 5000.\n\n  - **`detach`** - Cause `call/4` to return `ok` as soon as the request in\n    question has been encoded, instead of waiting for and returning the result\n    from a subsequent [handle_answer/4](`c:diameter_app:handle_answer/4`) or\n    [handle_error/4](`c:diameter_app:handle_error/4`) callback.\n\n  An invalid option will cause `call/4` to fail.\n\n  [](){: #capability }\n\n- **`capability()`** - AVP values sent in outgoing CER or CEA messages during\n  capabilities exchange. Can be configured both on a service and a transport,\n  values on the latter taking precedence. Has one of the following types.\n\n  - **`{'Origin-Host', `[`DiameterIdentity()`](diameter_dict.md#DATA_TYPES)`}`**\n\n  - **`{'Origin-Realm', `[`DiameterIdentity()`](diameter_dict.md#DATA_TYPES)`}`**\n\n  - **`{'Host-IP-Address', [`[`Address()`](diameter_dict.md#DATA_TYPES)`]}`** -\n    An address list is available to the start function of a\n    [transport module](`m:diameter_transport`), which can return a new list for\n    use in the subsequent CER or CEA. Host-IP-Address need not be specified if\n    the transport module in question communicates an address list as described\n    in `m:diameter_transport`\n\n  - **`{'Vendor-Id', `[`Unsigned32()`](diameter_dict.md#DATA_TYPES)`}`**\n\n  - **`{'Product-Name', `[`UTF8String()`](diameter_dict.md#DATA_TYPES)`}`**\n\n  - **`{'Origin-State-Id', `[`Unsigned32()`](diameter_dict.md#DATA_TYPES)`}`** -\n    Origin-State-Id is optional but, if configured, will be included in outgoing\n    CER/CEA and DWR/DWA messages. Setting a value of `0` (zero) is equivalent to\n    not setting a value, as documented in RFC 6733. The function\n    `origin_state_id/0` can be used as to retrieve a value that is computed when\n    the diameter application is started.\n\n  - **`{'Supported-Vendor-Id', [`[`Unsigned32()`](diameter_dict.md#DATA_TYPES)`]}`**\n\n  - **`{'Auth-Application-Id', [`[`Unsigned32()`](diameter_dict.md#DATA_TYPES)`]}`**\n\n  - **`{'Inband-Security-Id', [`[`Unsigned32()`](diameter_dict.md#DATA_TYPES)`]}`** -\n    Inband-Security-Id defaults to the empty list, which is equivalent to a list\n    containing only 0 (NO_INBAND_SECURITY). If 1 (TLS) is specified then TLS is\n    selected if the CER/CEA received from the peer offers it.\n\n  - **`{'Acct-Application-Id', [`[`Unsigned32()`](diameter_dict.md#DATA_TYPES)`]}`**\n\n  - **`{'Vendor-Specific-Application-Id', [`[`Grouped()`](diameter_dict.md#DATA_TYPES)`]}`**\n\n  - **`{'Firmware-Revision', `[`Unsigned32()`](diameter_dict.md#DATA_TYPES)`}`**\n\n  Note that each tuple communicates one or more AVP values. It is an error to\n  specify duplicate tuples.\n\n  [](){: #eval }\n\n- **`eval() = {M,F,A} | fun() | [eval() | A]`** - An expression that can be\n  evaluated as a function in the following sense.\n\n  ```erlang\n  eval([{M,F,A} | T]) ->\n      apply(M, F, T ++ A);\n  eval([[F|A] | T]) ->\n      eval([F | T ++ A]);\n  eval([F|A]) ->\n      apply(F, A);\n  eval(F) ->\n      eval([F]).\n  ```\n\n  Applying an [`eval()`](`m:diameter#eval`) `E` to an argument list `A` is meant\n  in the sense of `eval([E|A])`.\n\n  > #### Warning {: .warning }\n  >\n  > Beware of using fun expressions of the form `fun Name/Arity` in situations\n  > in which the fun is not short-lived and code is to be upgraded at runtime\n  > since any processes retaining such a fun will have a reference to old code.\n  > In particular, such a value is typically inappropriate in configuration\n  > passed to `start_service/2` or `add_transport/2`.\n\n  [](){: #peer_filter }\n\n- **`peer_filter() = term()`** - Filter passed to `call/4` in order to select\n  candidate peers for a [pick_peer/4](`c:diameter_app:pick_peer/4`) callback.\n  Has one of the following types.\n\n  - **`none`** - Matches any peer. This is a convenience that provides a filter\n    equivalent to no filter.\n\n  - **`host`** - Matches only those peers whose Origin-Host has the same value\n    as Destination-Host in the outgoing request in question, or any peer if the\n    request does not contain a Destination-Host AVP.\n\n  - **`realm`** - Matches only those peers whose Origin-Realm has the same value\n    as Destination-Realm in the outgoing request in question, or any peer if the\n    request does not contain a Destination-Realm AVP.\n\n  - **`{host, any|`[`DiameterIdentity()`](diameter_dict.md#DATA_TYPES)`}`** -\n    Matches only those peers whose Origin-Host has the specified value, or all\n    peers if the atom `any`.\n\n  - **`{realm, any|`[`DiameterIdentity()`](diameter_dict.md#DATA_TYPES)`}`** -\n    Matches only those peers whose Origin-Realm has the specified value, or all\n    peers if the atom `any`.\n\n  - **`{eval, `[`eval()`](`m:diameter#eval`)`}`** - Matches only those peers for\n    which the specified [`eval()`](`m:diameter#eval`) returns `true` when\n    applied to the connection's `diameter_caps` record. Any other return value\n    or exception is equivalent to `false`.\n\n  - **`{neg, `[`peer_filter()`](`m:diameter#peer_filter`)`}`** - Matches only\n    those peers not matched by the specified filter.\n\n  - **`{all, [`[`peer_filter()`](`m:diameter#peer_filter`)`]}`** - Matches only\n    those peers matched by each filter in the specified list.\n\n  - **`{any, [`[`peer_filter()`](`m:diameter#peer_filter`)`]}`** - Matches only\n    those peers matched by at least one filter in the specified list. The\n    resulting list will be in match order, peers matching the first filter of\n    the list sorting before those matched by the second, and so on.\n\n  - **`{first, [`[`peer_filter()`](`m:diameter#peer_filter`)`]}`** - Like `any`,\n    but stops at the first filter for which there are matches, which can be much\n    more efficient when there are many peers. For example, the following filter\n    causes only peers best matching both the host and realm filters to be\n    presented.\n\n    ```text\n    {first, [{all, [host, realm]}, realm]}\n    ```\n\n  An invalid filter is equivalent to `{any,[]}`, a filter that matches no peer.\n\n  > #### Note {: .info }\n  >\n  > The `host` and `realm` filters cause the Destination-Host and\n  > Destination-Realm AVPs to be extracted from the outgoing request, assuming\n  > it to be a record- or list-valued\n  > [`diameter_codec:message()`](`m:diameter_codec#message`), and assuming at\n  > most one of each AVP. If this is not the case then the\n  > `{host|realm, `[`DiameterIdentity()`](diameter_dict.md#DATA_TYPES)`}`\n  > filters must be used to achieve the desired result. An empty\n  > [`DiameterIdentity()`](diameter_dict.md#DATA_TYPES) (which should not be\n  > typical) matches all hosts/realms for the purposes of filtering.\n\n  > #### Warning {: .warning }\n  >\n  > A `host` filter is not typically desirable when setting Destination-Host\n  > since it will remove peer agents from the candidates list.\n\n  [](){: #service_event }\n\n- **`service_event() = #diameter_event{service = `[`service_name()`](`m:diameter#service_name`)`, info = `[`service_event_info()`](`m:diameter#service_event_info`)`}`** -\n  An event message sent to processes that have subscribed to these using\n  `subscribe/1`.\n\n  [](){: #service_event_info }\n\n- **`service_event_info() = term()`** - The `info` field of a\n  [service_event()](`m:diameter#service_event`) record. Can have one of the\n  following types.\n\n  - **`start`**\n\n  - **`stop`** - The service is being started or stopped. No event precedes a\n    `start` event. No event follows a `stop` event, and this event implies the\n    termination of all transport processes.\n\n  - **`{up, Ref, Peer, Config, Pkt}`**\n\n  - **`{up, Ref, Peer, Config}`**\n\n  - **`{down, Ref, Peer, Config}`**\n\n    ```c\n    Ref    = transport_ref()\n    Peer   = diameter_app:peer()\n    Config = {connect|listen, [transport_opt()]}\n    Pkt    = #diameter_packet{}\n    ```\n\n    The RFC 3539 watchdog state machine has transitioned into (`up`) or out of\n    (`down`) the OKAY state. If a `#diameter_packet{}` is present in an `up`\n    event then there has been a capabilities exchange on a newly established\n    transport connection and the record contains the received CER or CEA.\n\n    Note that a single `up` or `down` event for a given peer corresponds to\n    multiple [peer_up/3](`c:diameter_app:peer_up/3`) or\n    [peer_down/3](`c:diameter_app:peer_down/3`) callbacks, one for each of the\n    Diameter applications negotiated during capabilities exchange. That is, the\n    event communicates connectivity with the peer as a whole while the callbacks\n    communicate connectivity with respect to individual Diameter applications.\n\n  - **`{reconnect, Ref, Opts}`**\n\n    ```text\n    Ref  = transport_ref()\n    Opts = [transport_opt()]\n    ```\n\n    A connecting transport is attempting to establish/reestablish a transport\n    connection with a peer following [connect_timer](`m:diameter#connect_timer`)\n    or [watchdog_timer](`m:diameter#watchdog_timer`) expiry.\n\n  - **`{closed, Ref, Reason, Config}`**\n\n    ```text\n    Ref = transport_ref()\n    Config = {connect|listen, [transport_opt()]}\n    ```\n\n    Capabilities exchange has failed. `Reason` can have one of the following\n    types.\n\n    - **`{'CER', Result, Caps, Pkt}`**\n\n      ```text\n      Result = ResultCode | {capabilities_cb, CB, ResultCode|discard}\n      Caps = #diameter_caps{}\n      Pkt  = #diameter_packet{}\n      ResultCode = integer()\n      CB = eval()\n      ```\n\n      An incoming CER has been answered with the indicated result code, or\n      discarded. `Caps` contains pairs of values, for the local node and remote\n      peer respectively. `Pkt` contains the CER in question. In the case of\n      rejection by a capabilities callback, the tuple contains the rejecting\n      callback.\n\n    - **`{'CER', Caps, {ResultCode, Pkt}}`**\n\n      ```text\n      ResultCode = integer()\n      Caps = #diameter_caps{}\n      Pkt  = #diameter_packet{}\n      ```\n\n      An incoming CER contained errors and has been answered with the indicated\n      result code. `Caps` contains values for the local node only. `Pkt`\n      contains the CER in question.\n\n    - **`{'CER', timeout}`** - An expected CER was not received within\n      [capx_timeout](`m:diameter#capx_timeout`) of connection establishment.\n\n    - **`{'CEA', Result, Caps, Pkt}`**\n\n      ```erlang\n      Result = ResultCode | atom() | {capabilities_cb, CB, ResultCode|discard}\n      Caps = #diameter_caps{}\n      Pkt  = #diameter_packet{}\n      ResultCode = integer()\n      ```\n\n      An incoming CEA has been rejected for the indicated reason. An\n      integer-valued `Result` indicates the result code sent by the peer. `Caps`\n      contains pairs of values for the local node and remote peer. `Pkt`\n      contains the CEA in question. In the case of rejection by a capabilities\n      callback, the tuple contains the rejecting callback.\n\n    - **`{'CEA', Caps, Pkt}`**\n\n      ```text\n      Caps = #diameter_caps{}\n      Pkt  = #diameter_packet{}\n      ```\n\n      An incoming CEA contained errors and has been rejected. `Caps` contains\n      only values for the local node. `Pkt` contains the CEA in question.\n\n    - **`{'CEA', timeout}`** - An expected CEA was not received within\n      [capx_timeout](`m:diameter#capx_timeout`) of connection establishment.\n\n  - **`{watchdog, Ref, PeerRef, {From, To}, Config}`**\n\n    ```c\n    Ref = transport_ref()\n    PeerRef = diameter_app:peer_ref()\n    From, To = initial | okay | suspect | down | reopen\n    Config = {connect|listen, [transport_opt()]}\n    ```\n\n    An RFC 3539 watchdog state machine has changed state.\n\n  - **`t:any/0`** - For forward compatibility, a subscriber should be prepared\n    to receive info fields of forms other than the above.\n\n  [](){: #service_name }\n\n- **`service_name() = term()`** - Name of a service as passed to\n  `start_service/2` and with which the service is identified. There can be at\n  most one service with a given name on a given node. Note that\n  `erlang:make_ref/0` can be used to generate a service name that is somewhat\n  unique.\n\n  [](){: #service_opt }\n\n- **`service_opt()`** - Option passed to `start_service/2`. Can be any\n  [`capability()`](`m:diameter#capability`) as well as the following.\n\n  - **`{application, [`[`application_opt()`](`m:diameter#application_opt`)`]}`** -\n    A Diameter application supported by the service.\n\n    A service must configure one tuple for each Diameter application it intends\n    to support. For an outgoing request, the relevant\n    [`application_alias()`](`m:diameter#application_alias`) is passed to\n    `call/4`, while for an incoming request the application identifier in the\n    message header determines the application, the identifier being specified in\n    the application's [dictionary](diameter_dict.md) file.\n\n    > #### Warning {: .warning }\n    >\n    > The capabilities advertised by a node must match its configured\n    > applications. In particular, `application` configuration must be matched\n    > by corresponding [capability()](`m:diameter#capability`) configuration, of\n    > \\*-Application-Id AVPs in particular.\n\n  - **`{decode_format, record | list | map | none}`{: #decode_format }** - The\n    format of decoded messages and grouped AVPs in the `msg` field of\n    diameter_packet records and `value` field of diameter_avp records\n    respectively. If `record` then a record whose definition is generated from\n    the dictionary file in question. If `list` or `map` then a `[Name | Avps]`\n    pair where `Avps` is a list of AVP name/values pairs or a map keyed on AVP\n    names respectively. If `none` then the atom-value message name, or\n    `undefined` for a Grouped AVP. See also\n    [diameter_codec:message()](`m:diameter_codec#message`).\n\n    Defaults to `record`.\n\n    > #### Note {: .info }\n    >\n    > AVPs are decoded into a list of diameter_avp records in `avps` field of\n    > diameter_packet records independently of `decode_format`.\n\n  - **`{restrict_connections, false | node | nodes | [node()] | eval()}`** - The\n    degree to which the service allows multiple transport connections to the\n    same peer, as identified by its Origin-Host at capabilities exchange.\n\n    If `[node()]` then a connection is rejected if another already exists on any\n    of the specified nodes. Types `false`, `node`, `nodes` and\n    [eval()](`m:diameter#eval`) are equivalent to `[]`, `[node()]`,\n    `[node()|nodes()]` and the evaluated value respectively, evaluation of each\n    expression taking place whenever a new connection is to be established. Note\n    that `false` allows an unlimited number of connections to be established\n    with the same peer.\n\n    Multiple connections are independent and governed by their own peer and\n    watchdog state machines.\n\n    Defaults to `nodes`.\n\n  - **`{sequence, {H,N} | `[`eval()`](`m:diameter#eval`)`}`** - A constant value\n    `H` for the topmost `32-N` bits of of 32-bit End-to-End and Hop-by-Hop\n    Identifiers generated by the service, either explicitly or as a return value\n    of a function to be evaluated at `start_service/2`. In particular, an\n    identifier `Id` is mapped to a new identifier as follows.\n\n    ```text\n    (H bsl N) bor (Id band ((1 bsl N) - 1))\n    ```\n\n    Note that RFC 6733 requires that End-to-End Identifiers remain unique for a\n    period of at least 4 minutes and that this and the call rate places a lower\n    bound on appropriate values of `N`: at a rate of `R` requests per second, an\n    `N`\\-bit counter traverses all of its values in `(1 bsl N) div (R*60)`\n    minutes, so the bound is `4*R*60 =< 1 bsl N`.\n\n    `N` must lie in the range `0..32` and `H` must be a non-negative integer\n    less than `1 bsl (32-N)`.\n\n    Defaults to `{0,32}`.\n\n    > #### Warning {: .warning }\n    >\n    > Multiple Erlang nodes implementing the same Diameter node should be\n    > configured with different sequence masks to ensure that each node uses a\n    > unique range of End-to-End and Hop-by-Hop Identifiers for outgoing\n    > requests.\n\n  - **`{share_peers, boolean() | [node()] | eval()}`** - Nodes to which peer\n    connections established on the local Erlang node are communicated. Shared\n    peers become available in the remote candidates list passed to\n    [pick_peer/4](`c:diameter_app:pick_peer/4`) callbacks on remote nodes whose\n    services are configured to use them: see `use_shared_peers` below.\n\n    If `false` then peers are not shared. If `[node()]` then peers are shared\n    with the specified list of nodes. If `eval()` then peers are shared with the\n    nodes returned by the specified function, evaluated whenever a peer\n    connection becomes available or a remote service requests information about\n    local connections. The value `true` is equivalent to `fun ``erlang:nodes/0`.\n    The value `node/0` in a list is ignored, so a collection of services can all\n    be configured to share with the same list of nodes.\n\n    Defaults to `false`.\n\n    > #### Note {: .info }\n    >\n    > Peers are only shared with services of the same name for the purpose of\n    > sending outgoing requests. Since the value of the\n    > [application_opt()](`m:diameter#application_opt`) `alias`, passed to\n    > `call/4`, is the handle for identifying a peer as a suitable candidate,\n    > services that share peers must use the same aliases to identify their\n    > supported applications. They should typically also configure identical\n    > [capabilities()](`m:diameter#capabilities`), since by sharing peer\n    > connections they are distributing the implementation of a single Diameter\n    > node across multiple Erlang nodes.\n\n  - **`{strict_arities, boolean() | encode | decode}`{: #strict_arities }** -\n    Whether or not to require that the number of AVPs in a message or grouped\n    AVP agree with those specified in the dictionary in question when passing\n    messages to `m:diameter_app` callbacks. If `true` then mismatches in an\n    outgoing messages cause message encoding to fail, while mismatches in an\n    incoming message are reported as 5005/5009 errors in the errors field of the\n    diameter_packet record passed to\n    [handle_request/3](`c:diameter_app:handle_request/3`) or\n    [handle_answer/4](`c:diameter_app:handle_answer/4`) callbacks. If `false`\n    then neither error is enforced/detected. If `encode` or `decode` then errors\n    are only enforced/detected on outgoing or incoming messages respectively.\n\n    Defaults to `true`.\n\n    > #### Note {: .info }\n    >\n    > Disabling arity checks affects the form of messages at encode/decode. In\n    > particular, decoded AVPs are represented as lists of values, regardless of\n    > the AVP's arity (ie. expected number in the message/AVP grammar in\n    > question), and values are expected to be supplied as lists at encode. This\n    > differs from the historic decode behaviour of representing AVPs of arity 1\n    > as bare values, not wrapped in a list.\n\n  - **`{string_decode, boolean()}`{: #string_decode }** - Whether or not to\n    decode AVPs of type [OctetString()](diameter_dict.md#DATA_TYPES) and its\n    derived types [DiameterIdentity()](diameter_dict.md#DATA_TYPES),\n    [DiameterURI()](diameter_dict.md#DATA_TYPES),\n    [IPFilterRule()](diameter_dict.md#DATA_TYPES),\n    [QoSFilterRule()](diameter_dict.md#DATA_TYPES), and\n    [UTF8String()](diameter_dict.md#DATA_TYPES). If `true` then AVPs of these\n    types are decoded to string(). If `false` then values are retained as\n    binary().\n\n    Defaults to `true`.\n\n    > #### Warning {: .warning }\n    >\n    > This option should be set to `false` since a sufficiently malicious peer\n    > can otherwise cause large amounts of memory to be consumed when decoded\n    > Diameter messages are passed between processes. The default value is for\n    > backwards compatibility.\n\n  - **`{traffic_counters, boolean()}`{: #traffic_counters }** - Whether or not\n    to count application-specific messages; those for which `m:diameter_app`\n    callbacks take place. If false then only messages handled by diameter itself\n    are counted: CER/CEA, DWR/DWA, DPR/DPA.\n\n    Defaults to `true`.\n\n    > #### Note {: .info }\n    >\n    > Disabling counters is a performance improvement, but means that the\n    > omitted counters are not returned by `service_info/2`.\n\n  - **`{use_shared_peers, boolean() | [node()] | eval()}`** - Nodes from which\n    communicated peers are made available in the remote candidates list of\n    [pick_peer/4](`c:diameter_app:pick_peer/4`) callbacks.\n\n    If `false` then remote peers are not used. If `[node()]` then only peers\n    from the specified list of nodes are used. If `eval()` then only peers\n    returned by the specified function are used, evaluated whenever a remote\n    service communicates information about an available peer connection. The\n    value `true` is equivalent to `fun ``erlang:nodes/0`. The value `node/0` in\n    a list is ignored.\n\n    Defaults to `false`.\n\n    > #### Note {: .info }\n    >\n    > A service that does not use shared peers will always pass the empty list\n    > as the second argument of [pick_peer/4](`c:diameter_app:pick_peer/4`)\n    > callbacks.\n\n    > #### Warning {: .warning }\n    >\n    > Sending a request over a peer connection on a remote node is less\n    > efficient than sending it over a local connection. It may be preferable to\n    > make use of the [service_opt()](`m:diameter#service_opt`)\n    > `restrict_connections` and maintain a dedicated connection on each node\n    > from which requests are sent.\n\n  - **[`transport_opt()`](`m:diameter#transport_opt`)** - Any transport option\n    except `applications`, `capabilities`, `transport_config`, and\n    `transport_module`. Used as defaults for transport configuration, values\n    passed to `add_transport/2` overriding values configured on the service.\n\n  [](){: #transport_opt }\n\n- **`transport_opt()`** - Option passed to `add_transport/2`. Has one of the\n  following types.\n\n  - **`{applications, [`[`application_alias()`](`m:diameter#application_alias`)`]}`{:\n    #applications }** - Diameter applications to which the transport should be\n    restricted. Defaults to all applications configured on the service in\n    question. Applications not configured on the service in question are\n    ignored.\n\n    > #### Warning {: .warning }\n    >\n    > The capabilities advertised by a node must match its configured\n    > applications. In particular, setting `applications` on a transport\n    > typically implies having to set matching \\*-Application-Id AVPs in a\n    > [capabilities()](`m:diameter#capabilities`) tuple.\n\n  - **`{avp_dictionaries, [module()]}`{: #avp_dictionaries }** - A list of\n    alternate dictionary modules with which to encode/decode AVPs that are not\n    defined by the dictionary of the application in question. At decode, such\n    AVPs are represented as diameter_avp records in the `'AVP'` field of a\n    decoded message or Grouped AVP, the first alternate that succeeds in\n    decoding the AVP setting the record's value field. At encode, values in an\n    `'AVP'` list can be passed as AVP name/value 2-tuples, and it is an encode\n    error for no alternate to define the AVP of such a tuple.\n\n    Defaults to the empty list.\n\n    > #### Note {: .info }\n    >\n    > The motivation for alternate dictionaries is RFC 7683, Diameter Overload\n    > Indication Conveyance (DOIC), which defines AVPs to be piggybacked onto\n    > existing application messages rather than defining an application of its\n    > own. The DOIC dictionary is provided by the diameter application, as\n    > module `diameter_gen_doic_rfc7683`, but alternate dictionaries can be used\n    > to encode/decode any set of AVPs not known to an application dictionary.\n\n  - **`{capabilities, [`[`capability()`](`m:diameter#capability`)`]}`{:\n    #capabilities }** - AVPs used to construct outgoing CER/CEA messages. Values\n    take precedence over any specified on the service in question.\n\n    Specifying a capability as a transport option may be particularly\n    appropriate for Inband-Security-Id, in case TLS is desired over TCP as\n    implemented by `m:diameter_tcp`.\n\n  - **`{capabilities_cb, `[`eval()`](`m:diameter#eval`)`}`{: #capabilities_cb\n    }** - Callback invoked upon reception of CER/CEA during capabilities\n    exchange in order to ask whether or not the connection should be accepted.\n    Applied to the [`transport_ref()`](`m:diameter#transport_ref`) and\n    `#diameter_caps{}` record of the connection.\n\n    The return value can have one of the following types.\n\n    - **`ok`** - Accept the connection.\n\n    - **`t:integer/0`** - Causes an incoming CER to be answered with the\n      specified Result-Code.\n\n    - **`discard`** - Causes an incoming CER to be discarded without CEA being\n      sent.\n\n    - **`unknown`** - Equivalent to returning `3010`, DIAMETER_UNKNOWN_PEER.\n\n    Returning anything but `ok` or a 2xxx series result code causes the\n    transport connection to be broken. Multiple\n    [capabilities_cb](`m:diameter#capabilities_cb`) options can be specified, in\n    which case the corresponding callbacks are applied until either all return\n    `ok` or one does not.\n\n  - **`{capx_timeout, `[`Unsigned32()`](diameter_dict.md#DATA_TYPES)`}`{:\n    #capx_timeout }** - Number of milliseconds after which a transport process\n    having an established transport connection will be terminated if the\n    expected capabilities exchange message (CER or CEA) is not received from the\n    peer. For a connecting transport, the timing of connection attempts is\n    governed by [connect_timer](`m:diameter#connect_timer`) or\n    [watchdog_timer](`m:diameter#watchdog_timer`) expiry. For a listening\n    transport, the peer determines the timing.\n\n    Defaults to 10000.\n\n  - **`{connect_timer, Tc}`{: #connect_timer }**\n\n    ```text\n    Tc = Unsigned32()\n    ```\n\n    For a connecting transport, the RFC 6733 Tc timer, in milliseconds. This\n    timer determines the frequency with which a transport attempts to establish\n    an initial connection with its peer following transport configuration. Once\n    an initial connection has been established,\n    [watchdog_timer](`m:diameter#watchdog_timer`) determines the frequency of\n    reconnection attempts, as required by RFC 3539.\n\n    For a listening transport, the timer specifies the time after which a\n    previously connected peer will be forgotten: a connection after this time is\n    regarded as an initial connection rather than reestablishment, causing the\n    RFC 3539 state machine to pass to state OKAY rather than REOPEN. Note that\n    these semantics are not governed by the RFC and that a listening transport's\n    [connect_timer](`m:diameter#connect_timer`) should be greater than its\n    peer's Tw plus jitter.\n\n    Defaults to 30000 for a connecting transport and 60000 for a listening\n    transport.\n\n  - **`{disconnect_cb, `[`eval()`](`m:diameter#eval`)`}`{: #disconnect_cb }** -\n    Callback invoked prior to terminating the transport process of a transport\n    connection having watchdog state `OKAY`. Applied to\n    `application|service|transport` and the\n    [`transport_ref()`](`m:diameter#transport_ref`) and\n    [`diameter_app:peer()`](`t:diameter_app:peer/0`) in question: `application`\n    indicates that the diameter application is being stopped, `service` that the\n    service in question is being stopped by `stop_service/1`, and `transport`\n    that the transport in question is being removed by `remove_transport/2`.\n\n    The return value can have one of the following types.\n\n    - **`{dpr, [option()]}`** - Send Disconnect-Peer-Request to the peer, the\n      transport process being terminated following reception of\n      Disconnect-Peer-Answer or timeout. An `option()` can be one of the\n      following.\n\n      - **`{cause, 0|rebooting|1|busy|2|goaway}`** - Disconnect-Cause to send,\n        `REBOOTING`, `BUSY` and `DO_NOT_WANT_TO_TALK_TO_YOU` respectively.\n        Defaults to `rebooting` for `Reason=service|application` and `goaway`\n        for `Reason=transport`.\n\n      - **`{timeout, `[`Unsigned32()`](diameter_dict.md#DATA_TYPES)`}`** -\n        Number of milliseconds after which the transport process is terminated\n        if DPA has not been received. Defaults to the value of\n        [dpa_timeout](`m:diameter#dpa_timeout`).\n\n    - **`dpr`** - Equivalent to `{dpr, []}`.\n\n    - **`close`** - Terminate the transport process without\n      Disconnect-Peer-Request being sent to the peer.\n\n    - **`ignore`** - Equivalent to not having configured the callback.\n\n    Multiple [disconnect_cb](`m:diameter#disconnect_cb`) options can be\n    specified, in which case the corresponding callbacks are applied until one\n    of them returns a value other than `ignore`. All callbacks returning\n    `ignore` is equivalent to not having configured them.\n\n    Defaults to a single callback returning `dpr`.\n\n  - **`{dpa_timeout, `[`Unsigned32()`](diameter_dict.md#DATA_TYPES)`}`{:\n    #dpa_timeout }** - Number of milliseconds after which a transport connection\n    is terminated following an outgoing DPR if DPA is not received.\n\n    Defaults to 1000.\n\n  - **`{dpr_timeout, `[`Unsigned32()`](diameter_dict.md#DATA_TYPES)`}`{:\n    #dpr_timeout }** - Number of milliseconds after which a transport connection\n    is terminated following an incoming DPR if the peer does not close the\n    connection.\n\n    Defaults to 5000.\n\n  - **`{incoming_maxlen, 0..16777215}`{: #incoming_maxlen }** - Bound on the\n    expected size of incoming Diameter messages. Messages larger than the\n    specified number of bytes are discarded.\n\n    Defaults to `16777215`, the maximum value of the 24-bit Message Length field\n    in a Diameter Header.\n\n  - **`{length_errors, exit|handle|discard}`{: #length_errors }** - How to deal\n    with errors in the Message Length field of the Diameter Header in an\n    incoming message. An error in this context is that the length is not at\n    least 20 bytes (the length of a Header), is not a multiple of 4 (a valid\n    length) or is not the length of the message in question, as received over\n    the transport interface documented in `m:diameter_transport`.\n\n    If `exit` then the transport process in question exits. If `handle` then the\n    message is processed as usual, a resulting\n    [handle_request/3](`c:diameter_app:handle_request/3`) or\n    [handle_answer/4](`c:diameter_app:handle_answer/4`) callback (if one takes\n    place) indicating the `5015` error (DIAMETER_INVALID_MESSAGE_LENGTH). If\n    `discard` then the message in question is silently discarded.\n\n    Defaults to `exit`.\n\n    > #### Note {: .info }\n    >\n    > The default value reflects the fact that a transport module for a\n    > stream-oriented transport like TCP may not be able to recover from a\n    > message length error since such a transport must use the Message Length\n    > header to divide the incoming byte stream into individual Diameter\n    > messages. An invalid length leaves it with no reliable way to rediscover\n    > message boundaries, which may result in the failure of subsequent\n    > messages. See `m:diameter_tcp` for the behaviour of that module.\n\n  - **`{pool_size, pos_integer()}`** - Number of transport processes to start.\n    For a listening transport, determines the size of the pool of accepting\n    transport processes, a larger number being desirable for processing multiple\n    concurrent peer connection attempts. For a connecting transport, determines\n    the number of connections to the peer in question that will be attempted to\n    be establshed: the [service_opt()](`m:diameter#service_opt`):\n    `restrict_connections` should also be configured on the service in question\n    to allow multiple connections to the same peer.\n\n  - **`{spawn_opt, [term()] | {M,F,A}}`{: #spawn_opt }** - An options list\n    passed to `erlang:spawn_opt/2` to spawn a handler process for an incoming\n    Diameter request on the local node, or an MFA that returns the pid of a\n    handler process.\n\n    Options `monitor` and `link` are ignored in the list-valued case. An MFA is\n    applied with an additional term prepended to its argument list, and should\n    return either the pid of the handler process that invokes\n    `diameter_traffic:request/1` on the argument in order to process the\n    request, or the atom `discard`. The handler process need not be local, and\n    diameter need not be started on the remote node, but diameter and relevant\n    application callbacks must be on the code path.\n\n    Defaults to the empty list.\n\n  - **`{strict_capx, boolean()]}`{: #strict_capx }** - Whether or not to enforce\n    the RFC 6733 requirement that any message before capabilities exchange\n    should close the peer connection. If false then unexpected messages are\n    discarded.\n\n    Defaults to true. Changing this results in non-standard behaviour, but can\n    be useful in case peers are known to be behave badly.\n\n  - **`{strict_mbit, boolean()}`{: #strict_mbit }** - Whether or not to regard\n    an AVP setting the M-bit as erroneous when the command grammar in question\n    does not explicitly allow the AVP. If `true` then such AVPs are regarded as\n    5001 errors, DIAMETER_AVP_UNSUPPORTED. If `false` then the M-bit is ignored\n    and policing it becomes the receiver's responsibility.\n\n    Defaults to `true`.\n\n    > #### Warning {: .warning }\n    >\n    > RFC 6733 is unclear about the semantics of the M-bit. One the one hand,\n    > the CCF specification in section 3.2 documents AVP in a command grammar as\n    > meaning _any_ arbitrary AVP; on the other hand, 1.3.4 states that AVPs\n    > setting the M-bit cannot be added to an existing command: the modified\n    > command must instead be placed in a new Diameter application.\n    >\n    > The reason for the latter is presumably interoperability: allowing\n    > arbitrary AVPs setting the M-bit in a command makes its interpretation\n    > implementation-dependent, since there's no guarantee that all\n    > implementations will understand the same set of arbitrary AVPs in the\n    > context of a given command. However, interpreting `AVP` in a command\n    > grammar as any AVP, regardless of M-bit, renders 1.3.4 meaningless, since\n    > the receiver can simply ignore any AVP it thinks isn't relevant,\n    > regardless of the sender's intent.\n    >\n    > Beware of confusing mandatory in the sense of the M-bit with mandatory in\n    > the sense of the command grammar. The former is a semantic requirement:\n    > that the receiver understand the semantics of the AVP in the context in\n    > question. The latter is a syntactic requirement: whether or not the AVP\n    > must occur in the message in question.\n\n  - **`{transport_config, term()}`{: #transport_config }**\n\n  - **`{transport_config, term(), `[`Unsigned32()`](diameter_dict.md#DATA_TYPES)` | infinity}`** -\n    Term passed as the third argument to the\n    [start/3](`c:diameter_transport:start/3`) function of the relevant\n    [transport module](`m:diameter_transport`) in order to start a transport\n    process. Defaults to the empty list.\n\n    The 3-tuple form additionally specifies an interval, in milliseconds, after\n    which a started transport process should be terminated if it has not yet\n    established a connection. For example, the following options on a connecting\n    transport request a connection with one peer over SCTP or another (typically\n    the same) over TCP.\n\n    ```erlang\n    {transport_module, diameter_sctp}\n    {transport_config, SctpOpts, 5000}\n    {transport_module, diameter_tcp}\n    {transport_config, TcpOpts}\n    ```\n\n    To listen on both SCTP and TCP, define one transport for each.\n\n  - **`{transport_module, atom()}`{: #transport_module }** - Module implementing\n    a transport process as defined in `m:diameter_transport`. Defaults to\n    `diameter_tcp`.\n\n    Multiple `transport_module` and\n    [transport_config](`m:diameter#transport_config`) options are allowed. The\n    order of these is significant in this case (and only in this case), a\n    `transport_module` being paired with the first\n    [transport_config](`m:diameter#transport_config`) following it in the\n    options list, or the default value for trailing modules. Transport starts\n    will be attempted with each of the modules in order until one establishes a\n    connection within the corresponding timeout (see below) or all fail.\n\n  - **`{watchdog_config, [{okay|suspect, non_neg_integer()}]}`{:\n    #watchdog_config }** - Configuration that alters the behaviour of the\n    watchdog state machine. On key `okay`, the non-negative number of answered\n    DWR messages before transitioning from REOPEN to OKAY. On key `suspect`, the\n    number of watchdog timeouts before transitioning from OKAY to SUSPECT when\n    DWR is unanswered, or 0 to not make the transition.\n\n    Defaults to `[{okay, 3}, {suspect, 1}]`. Not specifying a key is equivalent\n    to specifying the default value for that key.\n\n    > #### Warning {: .warning }\n    >\n    > The default value is as required by RFC 3539: changing it results in\n    > non-standard behaviour that should only be used to simulate misbehaving\n    > nodes during test.\n\n  - **`{watchdog_timer, TwInit}`{: #watchdog_timer }**\n\n    ```text\n    TwInit = Unsigned32()\n           | {M,F,A}\n    ```\n\n    The RFC 3539 watchdog timer. An integer value is interpreted as the RFC's\n    TwInit in milliseconds, a jitter of ± 2 seconds being added at each rearming\n    of the timer to compute the RFC's Tw. An MFA is expected to return the RFC's\n    Tw directly, with jitter applied, allowing the jitter calculation to be\n    performed by the callback.\n\n    An integer value must be at least 6000 as required by RFC 3539. Defaults\n    to 30000.\n\n  Unrecognized options are silently ignored but are returned unmodified by\n  `service_info/2` and can be referred to in predicate functions passed to\n  `remove_transport/2`.\n\n- **`transport_ref() = reference()`{: #transport_ref }** - Reference returned by\n  `add_transport/2` that identifies the configuration.","ref":"diameter.html#module-data-types"},{"type":"module","title":"SEE ALSO - diameter","doc":"`m:diameter_app`, `m:diameter_transport`, [diameter_dict(4)](diameter_dict.md)","ref":"diameter.html#module-see-also"},{"type":"type","title":"diameter.Address/0","doc":"","ref":"diameter.html#t:Address/0"},{"type":"function","title":"diameter.add_transport/2","doc":"Add transport capability to a service.\n\nThe service will start transport processes as required in order to establish a\nconnection with the peer, either by connecting to the peer (`connect`) or by\naccepting incoming connection requests (`listen`). A connecting transport\nestablishes transport connections with at most one peer, an listening transport\npotentially with many.\n\nThe diameter application takes responsibility for exchanging CER/CEA with the\npeer. Upon successful completion of capabilities exchange the service calls each\nrelevant application module's [peer_up/3](`c:diameter_app:peer_up/3`) callback\nafter which the caller can exchange Diameter messages with the peer over the\ntransport. In addition to CER/CEA, the service takes responsibility for the\nhandling of DWR/DWA and required by RFC 3539, as well as for DPR/DPA.\n\nThe returned reference uniquely identifies the transport within the scope of the\nservice. Note that the function returns before a transport connection has been\nestablished.\n\n> #### Note {: .info }\n>\n> It is not an error to add a transport to a service that has not yet been\n> configured: a service can be started after configuring its transports.","ref":"diameter.html#add_transport/2"},{"type":"type","title":"diameter.app_alias/0","doc":"","ref":"diameter.html#t:app_alias/0"},{"type":"type","title":"diameter.app_module/0","doc":"","ref":"diameter.html#t:app_module/0"},{"type":"type","title":"diameter.application_opt/0","doc":"","ref":"diameter.html#t:application_opt/0"},{"type":"function","title":"diameter.call/4","doc":"Send a Diameter request message.\n\n`App` specifies the Diameter application in which the request is defined and\ncallbacks to the corresponding callback module will follow as described below\nand in `m:diameter_app`. Unless the `detach` option is specified, the call\nreturns either when an answer message is received from the peer or an error\noccurs. In the answer case, the return value is as returned by a\n[handle_answer/4](`c:diameter_app:handle_answer/4`) callback. In the error case,\nwhether or not the error is returned directly by diameter or from a\n[handle_error/4](`c:diameter_app:handle_error/4`) callback depends on whether or\nnot the outgoing request is successfully encoded for transmission to the peer,\nthe cases being documented below.\n\nIf there are no suitable peers, or if\n[pick_peer/4](`c:diameter_app:pick_peer/4`) rejects them by returning `false`,\nthen `{error,no_connection}` is returned. Otherwise\n[pick_peer/4](`c:diameter_app:pick_peer/4`) is followed by a\n[prepare_request/3](`c:diameter_app:prepare_request/3`) callback, the message is\nencoded and then sent.\n\nThere are several error cases which may prevent an answer from being received\nand passed to a [handle_answer/4](`c:diameter_app:handle_answer/4`) callback:\n\n- If the initial encode of the outgoing request fails, then the request process\n  fails and `{error,encode}` is returned.\n- If the request is successfully encoded and sent but the answer times out then\n  a [handle_error/4](`c:diameter_app:handle_error/4`) callback takes place with\n  `Reason = timeout`.\n- If the request is successfully encoded and sent but the service in question is\n  stopped before an answer is received then a\n  [handle_error/4](`c:diameter_app:handle_error/4`) callback takes place with\n  `Reason = cancel`.\n- If the transport connection with the peer goes down after the request has been\n  sent but before an answer has been received then an attempt is made to resend\n  the request to an alternate peer. If no such peer is available, or if the\n  subsequent [pick_peer/4](`c:diameter_app:pick_peer/4`) callback rejects the\n  candidates, then a [handle_error/4](`c:diameter_app:handle_error/4`) callback\n  takes place with `Reason = failover`. If a peer is selected then a\n  [prepare_retransmit/3](`c:diameter_app:prepare_retransmit/3`) callback takes\n  place, after which the semantics are the same as following an initial\n  [prepare_request/3](`c:diameter_app:prepare_request/3`) callback.\n- If an encode error takes place during retransmission then the request process\n  fails and `{error,failure}` is returned.\n- If an application callback made in processing the request fails (pick_peer,\n  prepare_request, prepare_retransmit, handle_answer or handle_error) then\n  either `{error,encode}` or `{error,failure}` is returned depending on whether\n  or not there has been an attempt to send the request over the transport.\n\nNote that `{error,encode}` is the only return value which guarantees that the\nrequest has _not_ been sent over the transport connection.","ref":"diameter.html#call/4"},{"type":"type","title":"diameter.call_opt/0","doc":"","ref":"diameter.html#t:call_opt/0"},{"type":"type","title":"diameter.capability/0","doc":"","ref":"diameter.html#t:capability/0"},{"type":"type","title":"diameter.common_opt/0","doc":"","ref":"diameter.html#t:common_opt/0"},{"type":"type","title":"diameter.DiameterIdentity/0","doc":"","ref":"diameter.html#t:DiameterIdentity/0"},{"type":"type","title":"diameter.DiameterURI/0","doc":"","ref":"diameter.html#t:DiameterURI/0"},{"type":"type","title":"diameter.decode_format/0","doc":"","ref":"diameter.html#t:decode_format/0"},{"type":"type","title":"diameter.Enumerated/0","doc":"","ref":"diameter.html#t:Enumerated/0"},{"type":"type","title":"diameter.elapsed_time/0","doc":"","ref":"diameter.html#t:elapsed_time/0"},{"type":"type","title":"diameter.eval/0","doc":"","ref":"diameter.html#t:eval/0"},{"type":"type","title":"diameter.evaluable/0","doc":"","ref":"diameter.html#t:evaluable/0"},{"type":"type","title":"diameter.Float32/0","doc":"","ref":"diameter.html#t:Float32/0"},{"type":"type","title":"diameter.Float64/0","doc":"","ref":"diameter.html#t:Float64/0"},{"type":"type","title":"diameter.Grouped/0","doc":"","ref":"diameter.html#t:Grouped/0"},{"type":"type","title":"diameter.Integer32/0","doc":"","ref":"diameter.html#t:Integer32/0"},{"type":"type","title":"diameter.Integer64/0","doc":"","ref":"diameter.html#t:Integer64/0"},{"type":"type","title":"diameter.IPFilterRule/0","doc":"","ref":"diameter.html#t:IPFilterRule/0"},{"type":"type","title":"diameter.message_length/0","doc":"","ref":"diameter.html#t:message_length/0"},{"type":"type","title":"diameter.OctetString/0","doc":"","ref":"diameter.html#t:OctetString/0"},{"type":"function","title":"diameter.origin_state_id/0","doc":"Return a reasonable value for use as Origin-State-Id in outgoing messages.\n\nThe value returned is the number of seconds since 19680120T031408Z, the first\nvalue that can be encoded as a Diameter [`Time()`](diameter_dict.md#DATA_TYPES),\nat the time the diameter application was started.","ref":"diameter.html#origin_state_id/0"},{"type":"type","title":"diameter.peer_filter/0","doc":"","ref":"diameter.html#t:peer_filter/0"},{"type":"opaque","title":"diameter.peer_ref/0","doc":"","ref":"diameter.html#t:peer_ref/0"},{"type":"type","title":"diameter.QoSFilterRule/0","doc":"","ref":"diameter.html#t:QoSFilterRule/0"},{"type":"type","title":"diameter.remotes/0","doc":"","ref":"diameter.html#t:remotes/0"},{"type":"function","title":"diameter.remove_transport/2","doc":"Remove previously added transports.\n\n`Pred` determines which transports to remove. An arity-3-valued `Pred` removes\nall transports for which `Pred(Ref, Type, Opts)` returns `true`, where `Type`\nand `Opts` are as passed to `add_transport/2` and `Ref` is as returned by it.\nThe remaining forms are equivalent to an arity-3 fun as follows.\n\n```erlang\nPred = fun(transport_ref(), list()):  fun(Ref, _, Opts) -> Pred(Ref, Opts) end\nPred = fun(list()):                   fun(_, _, Opts) -> Pred(Opts) end\nPred = transport_ref():               fun(Ref, _, _)  -> Pred == Ref end\nPred = list():                        fun(_, _, Opts) -> [] == Pred -- Opts end\nPred = true:                          fun(_, _, _) -> true end\nPred = false:                         fun(_, _, _) -> false end\nPred = {M,F,A}:  fun(Ref, Type, Opts) -> apply(M, F, [Ref, Type, Opts | A]) end\n```\n\nRemoving a transport causes the corresponding transport processes to be\nterminated. Whether or not a DPR message is sent to a peer is controlled by\nvalue of [disconnect_cb](`m:diameter#disconnect_cb`) configured on the\ntransport.","ref":"diameter.html#remove_transport/2"},{"type":"type","title":"diameter.restriction/0","doc":"","ref":"diameter.html#t:restriction/0"},{"type":"type","title":"diameter.sequence/0","doc":"","ref":"diameter.html#t:sequence/0"},{"type":"function","title":"diameter.service_info/2","doc":"Return information about a started service. Requesting info for an unknown\nservice causes `undefined` to be returned. Requesting a list of items causes a\ntagged list to be returned.\n\n`Item` can be one of the following.\n\n- **`'Origin-Host'`**\n\n- **`'Origin-Realm'`**\n\n- **`'Vendor-Id'`**\n\n- **`'Product-Name'`**\n\n- **`'Origin-State-Id'`**\n\n- **`'Host-IP-Address'`**\n\n- **`'Supported-Vendor'`**\n\n- **`'Auth-Application-Id'`**\n\n- **`'Inband-Security-Id'`**\n\n- **`'Acct-Application-Id'`**\n\n- **`'Vendor-Specific-Application-Id'`**\n\n- **`'Firmware-Revision'`** - Return a capability value as configured with\n  `start_service/2`.\n\n- **`applications`** - Return the list of applications as configured with\n  `start_service/2`.\n\n- **`capabilities`** - Return a tagged list of all capabilities values as\n  configured with `start_service/2`.\n\n- **`transport`** - Return a list containing one entry for each of the service's\n  transport as configured with `add_transport/2`. Each entry is a tagged list\n  containing both configuration and information about established peer\n  connections. An example return value with for a client service with\n  Origin-Host \"client.example.com\" configured with a single transport connected\n  to \"server.example.com\" might look as follows.\n\n  ```erlang\n  [[{ref,#Ref<0.0.0.93>},\n    {type,connect},\n    {options,[{transport_module,diameter_tcp},\n              {transport_config,[{ip,{127,0,0,1}},\n                                 {raddr,{127,0,0,1}},\n                                 {rport,3868},\n                                 {reuseaddr,true}]}]},\n    {watchdog,{<0.66.0>,-576460736368485571,okay}},\n    {peer,{<0.67.0>,-576460736357885808}},\n    {apps,[{0,common}]},\n    {caps,[{origin_host,{\"client.example.com\",\"server.example.com\"}},\n           {origin_realm,{\"example.com\",\"example.com\"}},\n           {host_ip_address,{[{127,0,0,1}],[{127,0,0,1}]}},\n           {vendor_id,{0,193}},\n           {product_name,{\"Client\",\"Server\"}},\n           {origin_state_id,{[],[]}},\n           {supported_vendor_id,{[],[]}},\n           {auth_application_id,{[0],[0]}},\n           {inband_security_id,{[],[0]}},\n           {acct_application_id,{[],[]}},\n           {vendor_specific_application_id,{[],[]}},\n           {firmware_revision,{[],[]}},\n           {avp,{[],[]}}]},\n    {port,[{owner,<0.69.0>},\n           {module,diameter_tcp},\n           {socket,{{127,0,0,1},48758}},\n           {peer,{{127,0,0,1},3868}},\n           {statistics,[{recv_oct,656},\n                        {recv_cnt,6},\n                        {recv_max,148},\n                        {recv_avg,109},\n                        {recv_dvi,19},\n                        {send_oct,836},\n                        {send_cnt,6},\n                        {send_max,184},\n                        {send_avg,139},\n                        {send_pend,0}]}]},\n    {statistics,[{{{0,258,0},recv},3},\n                 {{{0,258,1},send},3},\n                 {{{0,258,0},recv,{'Result-Code',2001}},3},\n                 {{{0,257,0},recv},1},\n                 {{{0,257,1},send},1},\n                 {{{0,257,0},recv,{'Result-Code',2001}},1},\n                 {{{0,280,1},recv},2},\n                 {{{0,280,0},send},2},\n                 {{{0,280,0},send,{'Result-Code',2001}},2}]}]]\n  ```\n\n  Here `ref` is a [`transport_ref()`](`m:diameter#transport_ref`) and `options`\n  the corresponding [`transport_opt()`](`m:diameter#transport_opt`) list passed\n  to `add_transport/2`. The `watchdog` entry shows the state of a connection's\n  RFC 3539 watchdog state machine. The `peer` entry identifies the\n  [`diameter_app:peer_ref()`](`t:diameter_app:peer_ref/0`) for which there will\n  have been [peer_up/3](`c:diameter_app:peer_up/3`) callbacks for the Diameter\n  applications identified by the `apps` entry, `common` being the\n  [`application_alias()`](`m:diameter#application_alias`). The `caps` entry\n  identifies the capabilities sent by the local node and received from the peer\n  during capabilities exchange. The `port` entry displays socket-level\n  information about the transport connection. The `statistics` entry presents\n  Diameter-level counters, an entry like `{{{0,280,1},recv},2}` saying that the\n  client has received 2 DWR messages:\n  `{0,280,1} = {Application_Id, Command_Code, R_Flag}`.\n\n  Note that `watchdog`, `peer`, `apps`, `caps` and `port` entries depend on\n  connectivity with the peer and may not be present. Note also that the\n  `statistics` entry presents values accumulated during the lifetime of the\n  transport configuration.\n\n  A listening transport presents its information slightly differently since\n  there may be multiple accepted connections for the same\n  [`transport_ref()`](`m:diameter#transport_ref`). The `transport` info returned\n  by a server with a single client connection might look as follows.\n\n  ```erlang\n  [[{ref,#Ref<0.0.0.61>},\n    {type,listen},\n    {options,[{transport_module,diameter_tcp},\n              {transport_config,[{reuseaddr,true},\n                                 {ip,{127,0,0,1}},\n                                 {port,3868}]}]},\n    {accept,[[{watchdog,{<0.56.0>,-576460739249514012,okay}},\n              {peer,{<0.58.0>,-576460638229179167}},\n              {apps,[{0,common}]},\n              {caps,[{origin_host,{\"server.example.com\",\"client.example.com\"}},\n                     {origin_realm,{\"example.com\",\"example.com\"}},\n                     {host_ip_address,{[{127,0,0,1}],[{127,0,0,1}]}},\n                     {vendor_id,{193,0}},\n                     {product_name,{\"Server\",\"Client\"}},\n                     {origin_state_id,{[],[]}},\n                     {supported_vendor_id,{[],[]}},\n                     {auth_application_id,{[0],[0]}},\n                     {inband_security_id,{[],[]}},\n                     {acct_application_id,{[],[]}},\n                     {vendor_specific_application_id,{[],[]}},\n                     {firmware_revision,{[],[]}},\n                     {avp,{[],[]}}]},\n              {port,[{owner,<0.62.0>},\n                     {module,diameter_tcp},\n                     {socket,{{127,0,0,1},3868}},\n                     {peer,{{127,0,0,1},48758}},\n                     {statistics,[{recv_oct,1576},\n                                  {recv_cnt,16},\n                                  {recv_max,184},\n                                  {recv_avg,98},\n                                  {recv_dvi,26},\n                                  {send_oct,1396},\n                                  {send_cnt,16},\n                                  {send_max,148},\n                                  {send_avg,87},\n                                  {send_pend,0}]}]}],\n             [{watchdog,{<0.72.0>,-576460638229717546,initial}}]]},\n    {statistics,[{{{0,280,0},recv},7},\n                 {{{0,280,1},send},7},\n                 {{{0,280,0},recv,{'Result-Code',2001}},7},\n                 {{{0,258,1},recv},3},\n                 {{{0,258,0},send},3},\n                 {{{0,258,0},send,{'Result-Code',2001}},3},\n                 {{{0,280,1},recv},5},\n                 {{{0,280,0},send},5},\n                 {{{0,280,0},send,{'Result-Code',2001}},5},\n                 {{{0,257,1},recv},1},\n                 {{{0,257,0},send},1},\n                 {{{0,257,0},send,{'Result-Code',2001}},1}]}]]\n  ```\n\n  The information presented here is as in the `connect` case except that the\n  client connections are grouped under an `accept` tuple.\n\n  Whether or not the [transport_opt()](`m:diameter#transport_opt`) `pool_size`\n  has been configured affects the format of the listing in the case of a\n  connecting transport, since a value greater than 1 implies multiple transport\n  processes for the same [`transport_ref()`](`m:diameter#transport_ref`), as in\n  the listening case. The format in this case is similar to the listening case,\n  with a `pool` tuple in place of an `accept` tuple.\n\n- **`connections`** - Return a list containing one entry for every established\n  transport connection whose watchdog state machine is not in the `down` state.\n  This is a flat view of `transport` info which lists only active connections\n  and for which Diameter-level statistics are accumulated only for the lifetime\n  of the transport connection. A return value for the server above might look as\n  follows.\n\n  ```erlang\n  [[{ref,#Ref<0.0.0.61>},\n    {type,accept},\n    {options,[{transport_module,diameter_tcp},\n              {transport_config,[{reuseaddr,true},\n                                 {ip,{127,0,0,1}},\n                                 {port,3868}]}]},\n    {watchdog,{<0.56.0>,-576460739249514012,okay}},\n    {peer,{<0.58.0>,-576460638229179167}},\n    {apps,[{0,common}]},\n    {caps,[{origin_host,{\"server.example.com\",\"client.example.com\"}},\n           {origin_realm,{\"example.com\",\"example.com\"}},\n           {host_ip_address,{[{127,0,0,1}],[{127,0,0,1}]}},\n           {vendor_id,{193,0}},\n           {product_name,{\"Server\",\"Client\"}},\n           {origin_state_id,{[],[]}},\n           {supported_vendor_id,{[],[]}},\n           {auth_application_id,{[0],[0]}},\n           {inband_security_id,{[],[]}},\n           {acct_application_id,{[],[]}},\n           {vendor_specific_application_id,{[],[]}},\n           {firmware_revision,{[],[]}},\n           {avp,{[],[]}}]},\n    {port,[{owner,<0.62.0>},\n           {module,diameter_tcp},\n           {socket,{{127,0,0,1},3868}},\n           {peer,{{127,0,0,1},48758}},\n           {statistics,[{recv_oct,10124},\n                        {recv_cnt,132},\n                        {recv_max,184},\n                        {recv_avg,76},\n                        {recv_dvi,9},\n                        {send_oct,10016},\n                        {send_cnt,132},\n                        {send_max,148},\n                        {send_avg,75},\n                        {send_pend,0}]}]},\n    {statistics,[{{{0,280,0},recv},62},\n                 {{{0,280,1},send},62},\n                 {{{0,280,0},recv,{'Result-Code',2001}},62},\n                 {{{0,258,1},recv},3},\n                 {{{0,258,0},send},3},\n                 {{{0,258,0},send,{'Result-Code',2001}},3},\n                 {{{0,280,1},recv},66},\n                 {{{0,280,0},send},66},\n                 {{{0,280,0},send,{'Result-Code',2001}},66},\n                 {{{0,257,1},recv},1},\n                 {{{0,257,0},send},1},\n                 {{{0,257,0},send,{'Result-Code',2001}},1}]}]]\n  ```\n\n  Note that there may be multiple entries with the same `ref`, in contrast to\n  `transport` info.\n\n- **`statistics`** - Return a `{{Counter, Ref}, non_neg_integer()}` list of\n  counter values. `Ref` can be either a\n  [`transport_ref()`](`m:diameter#transport_ref`) or a\n  [`diameter_app:peer_ref()`](`t:diameter_app:peer_ref/0`). Entries for the latter\n  are folded into corresponding entries for the former as peer connections go\n  down. Entries for both are removed at `remove_transport/2`. The Diameter-level\n  statistics returned by `transport` and `connections` info are based upon these\n  entries.\n\n- **[`diameter_app:peer_ref()`](`t:diameter_app:peer_ref/0`)** - Return transport\n  configuration associated with a single peer, as passed to `add_transport/2`.\n  The returned list is empty if the peer is unknown. Otherwise it contains the\n  `ref`, `type` and `options` tuples as in `transport` and `connections` info\n  above. For example:\n\n  ```erlang\n  [{ref,#Ref<0.0.0.61>},\n   {type,accept},\n   {options,[{transport_module,diameter_tcp},\n             {transport_config,[{reuseaddr,true},\n                                {ip,{127,0,0,1}},\n                                {port,3868}]}]}]\n  ```","ref":"diameter.html#service_info/2"},{"type":"type","title":"diameter.service_name/0","doc":"","ref":"diameter.html#t:service_name/0"},{"type":"type","title":"diameter.service_opt/0","doc":"","ref":"diameter.html#t:service_opt/0"},{"type":"function","title":"diameter.services/0","doc":"Return the list of started services.","ref":"diameter.html#services/0"},{"type":"function","title":"diameter.session_id/1","doc":"Return a value for a Session-Id AVP.\n\nThe value has the form required by section 8.8 of RFC 6733. Ident should be the\nOrigin-Host of the peer from which the message containing the returned value\nwill be sent.","ref":"diameter.html#session_id/1"},{"type":"function","title":"diameter.start/0","doc":"Start the diameter application.\n\nThe diameter application must be started before starting a service. In a\nproduction system this is typically accomplished by a boot file, not by calling\n`start/0` explicitly.","ref":"diameter.html#start/0"},{"type":"function","title":"diameter.start_service/2","doc":"Start a diameter service.\n\nA service defines a locally-implemented Diameter node, specifying the\ncapabilities to be advertised during capabilities exchange. Transports are added\nto a service using `add_transport/2`.\n\n> #### Note {: .info }\n>\n> A transport can both override its service's capabilities and restrict its\n> supported Diameter applications so \"service = Diameter node as identified by\n> Origin-Host\" is not necessarily the case.","ref":"diameter.html#start_service/2"},{"type":"function","title":"diameter.stop/0","doc":"Stop the diameter application.","ref":"diameter.html#stop/0"},{"type":"function","title":"diameter.stop_service/1","doc":"Stop a diameter service.\n\nStopping a service causes all associated transport connections to be broken. A\nDPR message will be sent as in the case of `remove_transport/2`.\n\n> #### Note {: .info }\n>\n> Stopping a service does not remove any associated transports:\n> `remove_transport/2` must be called to remove transport configuration.","ref":"diameter.html#stop_service/1"},{"type":"type","title":"diameter.strict_arities/0","doc":"","ref":"diameter.html#t:strict_arities/0"},{"type":"function","title":"diameter.subscribe/1","doc":"Subscribe to [`service_event()`](`m:diameter#service_event`) messages from a\nservice.\n\nIt is not an error to subscribe to events from a service that does not yet\nexist. Doing so before adding transports is required to guarantee the reception\nof all transport-related events.","ref":"diameter.html#subscribe/1"},{"type":"type","title":"diameter.Time/0","doc":"","ref":"diameter.html#t:Time/0"},{"type":"type","title":"diameter.transport_opt/0","doc":"","ref":"diameter.html#t:transport_opt/0"},{"type":"type","title":"diameter.transport_pred/0","doc":"","ref":"diameter.html#t:transport_pred/0"},{"type":"type","title":"diameter.transport_ref/0","doc":"","ref":"diameter.html#t:transport_ref/0"},{"type":"type","title":"diameter.Unsigned32/0","doc":"","ref":"diameter.html#t:Unsigned32/0"},{"type":"type","title":"diameter.Unsigned64/0","doc":"","ref":"diameter.html#t:Unsigned64/0"},{"type":"type","title":"diameter.UTF8String/0","doc":"","ref":"diameter.html#t:UTF8String/0"},{"type":"function","title":"diameter.unsubscribe/1","doc":"Unsubscribe to event messages from a service.","ref":"diameter.html#unsubscribe/1"},{"type":"function","title":"diameter.which_connections/0","doc":"Return a list of _all_ connections, grouped by the service they\nare associated with.","ref":"diameter.html#which_connections/0"},{"type":"function","title":"diameter.which_connections/1","doc":"Return a list of connections associated with the service 'SvcName'.","ref":"diameter.html#which_connections/1"},{"type":"function","title":"diameter.which_transports/0","doc":"Return a list of _all_ transports.","ref":"diameter.html#which_transports/0"},{"type":"function","title":"diameter.which_transports/1","doc":"Return a list of transports associated with the service 'SvcName'.","ref":"diameter.html#which_transports/1"},{"type":"function","title":"diameter.which_watchdogs/0","doc":"Return a list of _all_ watchdogs.","ref":"diameter.html#which_watchdogs/0"},{"type":"function","title":"diameter.which_watchdogs/1","doc":"Return a list of watchdogs associated with the service 'SvcName'.","ref":"diameter.html#which_watchdogs/1"},{"type":"behaviour","title":"diameter_app","doc":"Callback module of a Diameter application.\n\nA diameter service as started by `diameter:start_service/2` configures one of\nmore Diameter applications, each of whose configuration specifies a callback\nthat handles messages specific to the application. The messages and AVPs of the\napplication are defined in a dictionary file whose format is documented in\n[diameter_dict(4)](diameter_dict.md) while the callback module is documented\nhere. The callback module implements the Diameter application-specific\nfunctionality of a service.\n\nA callback module must export all of the functions documented below. The\nfunctions themselves are of three distinct flavours:\n\n- `c:peer_up/3` and `c:peer_down/3` signal the attainment or loss of\n  connectivity with a Diameter peer.\n- `c:pick_peer/4`, `c:prepare_request/3`, `c:prepare_retransmit/3`,\n  `c:handle_answer/4` and `c:handle_error/4` are (or may be) called as a\n  consequence of a call to `diameter:call/4` to send an outgoing Diameter\n  request message.\n- `c:handle_request/3` is called in response to an incoming Diameter request\n  message.\n\nThe arities for the the callback functions here assume no extra arguments. All\nfunctions will also be passed any extra arguments configured with the callback\nmodule itself when calling `diameter:start_service/2` and, for the call-specific\ncallbacks, any extra arguments passed to `diameter:call/4`.","ref":"diameter_app.html"},{"type":"type","title":"diameter_app.capabilities/0","doc":"A record containing the identities of the local Diameter node and the remote Diameter\npeer having an established transport connection, as well as the capabilities\nas determined by capabilities exchange. Each field of the record is a 2-tuple\nconsisting of values for the (local) host and (remote) peer. Optional or\npossibly multiple values are encoded as lists of values, mandatory values as\nthe bare value.","ref":"diameter_app.html#t:capabilities/0"},{"type":"callback","title":"diameter_app.handle_answer/4","doc":"Invoked when an answer message is received from a peer. The return value is\nreturned from `diameter:call/4` unless the `detach` option was specified.\n\nThe decoded answer record and undecoded binary are in the `msg` and `bin` fields\nof the argument [packet()](`t:packet/0`) respectively. `Request` is\nthe outgoing request message as was returned from `c:prepare_request/3` or\n`c:prepare_retransmit/3`.\n\nFor any given call to `diameter:call/4` there is at most one `c:handle_answer/4`\ncallback: any duplicate answer (due to retransmission or otherwise) is\ndiscarded. Similarly, only one of `c:handle_answer/4` or `c:handle_error/4` is\ncalled.\n\nBy default, an incoming answer message that cannot be successfully decoded\ncauses the request process to fail, causing `diameter:call/4` to return\n`{error, failure}` unless the `detach` option was specified. In particular,\nthere is no `c:handle_error/4` callback in this case. The\n[diameter:application_opt()](`m:diameter#application_opt`) `answer_errors` can\nbe set to change this behaviour.","ref":"diameter_app.html#c:handle_answer/4"},{"type":"callback","title":"diameter_app.handle_error/4","doc":"Invoked when an error occurs before an answer message is received in response to\nan outgoing request. The return value is returned from `diameter:call/4` unless\nthe `detach` option was specified.\n\nReason `timeout` indicates that an answer message has not been received within\nthe time specified with the corresponding\n[diameter:call_opt()](`m:diameter#call_opt`). Reason `failover` indicates that\nthe transport connection to the peer to which the request has been sent has\nbecome unavailable and that not alternate peer was not selected.","ref":"diameter_app.html#c:handle_error/4"},{"type":"callback","title":"diameter_app.handle_request/3","doc":"Invoked when a request message is received from a peer. The application in which\nthe callback takes place (that is, the callback module as configured with\n`diameter:start_service/2`) is determined by the Application Identifier in the\nheader of the incoming request message, the selected module being the one whose\ncorresponding dictionary declares itself as defining either the application in\nquestion or the Relay application.\n\nThe argument [packet()](`t:packet/0`) has the following signature.\n\n```erlang\n#diameter_packet{header = #diameter_header{},\n                 avps   = [#diameter_avp{}],\n                 msg    = record() | undefined,\n                 errors = [Unsigned32() | {Unsigned32(), #diameter_avp{}}],\n                 bin    = binary(),\n                 transport_data = term()}\n```\n\nThe `msg` field will be `undefined` in case the request has been received in the\nrelay application. Otherwise it contains the record representing the request as\noutlined in [diameter_dict(4)](diameter_dict.md#MESSAGE_RECORDS).\n\nThe `errors` field specifies any results codes identifying errors found while\ndecoding the request. This is used to set Result-Code and/or Failed-AVP in a\nreturned answer unless the callback returns a `#diameter_packet{}` whose\n`errors` field is set to either a non-empty list of its own, in which case this\nlist is used instead, or the atom `false` to disable any setting of Result-Code\nand Failed-AVP. Note that the errors detected by diameter are of the 3xxx and\n5xxx series, Protocol Errors and Permanent Failures respectively. The `errors`\nlist is empty if the request has been received in the relay application.\n\nThe `transport_data` field contains an arbitrary term passed into diameter from\nthe transport module in question, or the atom `undefined` if the transport\nspecified no data. The term is preserved if a\n[message()](`t:message/0`) is returned but must be set explicitly in\na returned [packet()](`t:packet/0`).\n\nThe semantics of each of the possible return values are as follows.\n\n- **`{reply, `[`packet()`](`t:packet/0`)`|`[`message()`](`t:message/0`)`}`** -\n  Send the specified answer message to the peer. In the case of a\n  [packet()](`t:packet/0`), the message to be sent must be set in the\n  `msg` field and the `header` field can be set to a `#diameter_header{}` to\n  specify values that should be preserved in the outgoing answer, appropriate\n  values otherwise being set by diameter.\n\n- **`{answer_message, 3000..3999|5000..5999}`** - Send an answer message to the\n  peer containing the specified Result-Code. Equivalent to\n\n  ```text\n  {reply, ['answer-message' | Avps]\n  ```\n\n  where `Avps` sets the Origin-Host, Origin-Realm, the specified Result-Code and\n  (if the request contained one) Session-Id AVPs, and possibly Failed-AVP as\n  described below.\n\n  Returning a value other than 3xxx or 5xxx will cause the request process in\n  question to fail, as will returning a 5xxx value if the peer connection in\n  question has been configured with the RFC 3588 common dictionary\n  `diameter_gen_base_rfc3588`. (Since RFC 3588 only allows 3xxx values in an\n  answer-message.)\n\n  When returning 5xxx, Failed-AVP will be populated with the AVP of the first\n  matching Result-Code/AVP pair in the `errors` field of the argument\n  [packet()](`t:packet/0`), if found. If this is not appropriate then\n  an answer-message should be constructed explicitly and returned in a `reply`\n  tuple instead.\n\n- **`{relay, Opts}`** - Relay a request to another peer in the role of a\n  Diameter relay agent. If a routing loop is detected then the request is\n  answered with 3005 (DIAMETER_LOOP_DETECTED). Otherwise a Route-Record AVP\n  (containing the sending peer's Origin-Host) is added to the request and\n  `c:pick_peer/4` and subsequent callbacks take place just as if\n  `diameter:call/4` had been called explicitly. The End-to-End Identifier of the\n  incoming request is preserved in the header of the relayed request.\n\n  The returned `Opts` should not specify `detach`. A subsequent\n  `c:handle_answer/4` callback for the relayed request must return its first\n  argument, the [packet()](`t:packet/0`) containing the answer\n  message. Note that the `extra` option can be specified to supply arguments\n  that can distinguish the relay case from others if so desired. Any other\n  return value (for example, from a `c:handle_error/4` callback) causes the\n  request to be answered with 3002 (DIAMETER_UNABLE_TO_DELIVER).\n\n- **`discard`** - Discard the request. No answer message is sent to the peer.\n\n- **`{eval, Action, PostF}`** - Handle the request as if `Action` has been\n  returned and then evaluate `PostF` in the request process. The return value is\n  ignored.\n\n- **`{eval_packet, Action, PostF}`** - Like `eval` but evaluate `PostF` on any\n  encoded `#diameter_packet{}` prior to transmission, the `bin` field containing\n  the encoded binary. The return value is ignored.\n\n- **`{protocol_error, 3000..3999}`** - Equivalent to\n  `{answer_message, 3000..3999}`.\n\n> #### Note {: .info }\n>\n> Requests containing errors may be answered by diameter, without a callback\n> taking place, depending on the value of the\n> [diameter:application_opt()](`m:diameter#application_opt`) `request_errors`.","ref":"diameter_app.html#c:handle_request/3"},{"type":"type","title":"diameter_app.message/0","doc":"The representation of a Diameter message as passed to `diameter:call/4` or\nreturned from a `c:handle_request/3` callback.","ref":"diameter_app.html#t:message/0"},{"type":"type","title":"diameter_app.packet/0","doc":"A container for incoming and outgoing Diameter messages that's passed through\nencode/decode and transport. Fields should not be set in return values except\nas documented.","ref":"diameter_app.html#t:packet/0"},{"type":"type","title":"diameter_app.peer/0","doc":"A tuple representing a Diameter peer connection.","ref":"diameter_app.html#t:peer/0"},{"type":"callback","title":"diameter_app.peer_down/3","doc":"Invoked to signal that a peer connection on the local Erlang node is no longer\navailable following a previous call to `c:peer_up/3`. In particular, that the\nRFC 3539 watchdog state machine for the connection has left state `OKAY` and the\npeer will no longer be a candidate in `c:pick_peer/4` callbacks.","ref":"diameter_app.html#c:peer_down/3"},{"type":"type","title":"diameter_app.peer_ref/0","doc":"A term identifying a transport connection with a Diameter peer.","ref":"diameter_app.html#t:peer_ref/0"},{"type":"callback","title":"diameter_app.peer_up/3","doc":"Invoked to signal the availability of a peer connection on the local Erlang\nnode. In particular, capabilities exchange with the peer has indicated support\nfor the application in question, the RFC 3539 watchdog state machine for the\nconnection has reached state `OKAY` and Diameter messages can be both sent and\nreceived.\n\n> #### Note {: .info }\n>\n> A watchdog state machine can reach state `OKAY` from state `SUSPECT` without a\n> new capabilities exchange taking place. A new transport connection (and\n> capabilities exchange) results in a new peer_ref().\n\n> #### Note {: .info }\n>\n> There is no requirement that a callback return before incoming requests are\n> received: `c:handle_request/3` callbacks must be handled independently of\n> `c:peer_up/3` and `c:peer_down/3`.","ref":"diameter_app.html#c:peer_up/3"},{"type":"callback","title":"diameter_app.pick_peer/4","doc":"Invoked as a consequence of a call to `diameter:call/4` to select a destination\npeer for an outgoing request. The return value indicates the selected peer.\n\nThe candidate lists contain only those peers that have advertised support for\nthe Diameter application in question during capabilities exchange, that have not\nbe excluded by a `filter` option in the call to `diameter:call/4` and whose\nwatchdog state machine is in the `OKAY` state. The order of the elements is\nunspecified except that any peers whose Origin-Host and Origin-Realm matches\nthat of the outgoing request (in the sense of a `{filter, {all, [host, realm]}}`\noption to `diameter:call/4`) will be placed at the head of the list.\n`LocalCandidates` contains peers whose transport process resides on the local\nErlang node while `RemoteCandidates` contains peers that have been communicated\nfrom other nodes by services of the same name.\n\nA callback that returns a peer() will be followed by a `c:prepare_request/3`\ncallback and, if the latter indicates that the request should be sent, by either\n`c:handle_answer/4` or `c:handle_error/4` depending on whether or not an answer\nmessage is received from the peer. If the transport becomes unavailable after\n`c:prepare_request/3` then a new `c:pick_peer/4` callback may take place to\nfailover to an alternate peer, after which `c:prepare_retransmit/3` takes the\nplace of `c:prepare_request/3` in resending the request. There is no guarantee\nthat a `c:pick_peer/4` callback to select an alternate peer will be followed by\nany additional callbacks since a retransmission to an alternate peer is\nabandoned if an answer is received from a previously selected peer.\n\nThe return values `false` and `{false, State}` (that is, `NewState = State`) are\nequivalent, as are `{ok, Peer}` and `{Peer, State}`.\n\n> #### Note {: .info }\n>\n> The [diameter:service_opt()](`m:diameter#service_opt`) `use_shared_peers`\n> determines whether or not a service uses peers shared from other nodes. If not\n> then `RemoteCandidates` is the empty list.\n\n> #### Warning {: .warning }\n>\n> The return value `{Peer, NewState}` is only allowed if the Diameter\n> application in question was configured with the\n> [diameter:application_opt()](`m:diameter#application_opt`)\n> `{call_mutates_state, true}`. Otherwise, the `State` argument is always the\n> initial value as configured on the application, not any subsequent value\n> returned by a `c:peer_up/3` or `c:peer_down/3` callback.","ref":"diameter_app.html#c:pick_peer/4"},{"type":"callback","title":"diameter_app.prepare_request/3","doc":"Invoked to return a request for encoding and transport. Allows the sender to use\nthe selected peer's capabilities to modify the outgoing request. Many\nimplementations may simply want to return `{send, Packet}`\n\nA returned [packet()](`t:packet/0`) should set the request to be\nencoded in its `msg` field and can set the `transport_data` field in order to\npass information to the transport process. Extra arguments passed to\n`diameter:call/4` can be used to communicate transport (or any other) data to\nthe callback.\n\nA returned [packet()](`t:packet/0`) can set the `header` field to a\n`#diameter_header{}` to specify values that should be preserved in the outgoing\nrequest, values otherwise being those in the header record contained in\n`Packet`. A returned `length`, `cmd_code` or `application_id` is ignored.\n\nA returned `PostF` will be evaluated on any encoded `#diameter_packet{}` prior\nto transmission, the `bin` field containing the encoded binary. The return value\nis ignored.\n\nReturning `{discard, Reason}` causes the request to be aborted and the\n`diameter:call/4` for which the callback has taken place to return\n`{error, Reason}`. Returning `discard` is equivalent to returning\n`{discard, discarded}`.","ref":"diameter_app.html#c:prepare_request/3"},{"type":"callback","title":"diameter_app.prepare_retransmit/3","doc":"Invoked to return a request for encoding and retransmission. Has the same role\nas `c:prepare_request/3` in the case that a peer connection is lost an an\nalternate peer selected but the argument [packet()](`t:packet/0`) is\nas returned by the initial `c:prepare_request/3`.\n\nReturning `{discard, Reason}` causes the request to be aborted and a\n`c:handle_error/4` callback to take place with `Reason` as initial argument.\nReturning `discard` is equivalent to returning `{discard, discarded}`.","ref":"diameter_app.html#c:prepare_retransmit/3"},{"type":"type","title":"diameter_app.state/0","doc":"The state maintained by the application\ncallback functions `c:peer_up/3`, `c:peer_down/3` and (optionally)\n`c:pick_peer/4`. The initial state is configured in the call to\n`diameter:start_service/2` that configures the application on a service.\nCallback functions returning a state are evaluated in a common\nservice-specific process while those not returning state are evaluated in a\nrequest-specific process.","ref":"diameter_app.html#t:state/0"},{"type":"module","title":"diameter_codec","doc":"Decode and encode of Diameter messages.\n\nIncoming Diameter messages are decoded from binary() before being communicated\nto `m:diameter_app` callbacks. Similarly, outgoing Diameter messages are encoded\ninto binary() before being passed to the appropriate `m:diameter_transport`\nmodule for transmission. The functions documented here implement the default\nencode/decode.\n\n> #### Warning {: .warning }\n>\n> The diameter user does not need to call functions here explicitly when sending\n> and receiving messages using `diameter:call/4` and the callback interface\n> documented in `m:diameter_app`: diameter itself provides encode/decode as a\n> consequence of configuration passed to `diameter:start_service/2`, and the\n> results may differ from those returned by the functions documented here,\n> depending on configuration.\n\nThe [header()](`m:diameter_codec#header`) and\n[packet()](`m:diameter_codec#packet`) records below are defined in diameter.hrl,\nwhich can be included as follows.\n\n```erlang\n-include_lib(\"diameter/include/diameter.hrl\").\n```\n\nApplication-specific records are defined in the hrl files resulting from\ndictionary file compilation.","ref":"diameter_codec.html"},{"type":"module","title":"DATA TYPES - diameter_codec","doc":"- **`uint8()  = 0..255`{: #integers }**\n\n- **`uint24() = 0..16777215`**\n\n- **`uint32() = 0..4294967295`** - 8-bit, 24-bit and 32-bit integers occurring\n  in Diameter and AVP headers.\n\n- **`avp() = #diameter_avp{}`{: #avp }** - The application-neutral\n  representation of an AVP. Primarily intended for use by relay applications\n  that need to handle arbitrary Diameter applications. A service implementing a\n  specific Diameter application (for which it configures a dictionary) can\n  manipulate values of type [message()](`m:diameter_codec#message`) instead.\n\n  Fields have the following types.\n\n  - **`code = uint32()`**\n\n  - **`is_mandatory = boolean()`**\n\n  - **`need_encryption = boolean()`**\n\n  - **`vendor_id = uint32() | undefined`** - Values in the AVP header,\n    corresponding to AVP Code, the M flag, P flags and Vendor-ID respectively. A\n    Vendor-ID other than `undefined` implies a set V flag.\n\n  - **`data = iolist()`** - The data bytes of the AVP.\n\n  - **`name = atom()`** - The name of the AVP as defined in the dictionary file\n    in question, or `undefined` if the AVP is unknown to the dictionary file in\n    question.\n\n  - **`value = term()`** - The decoded value of an AVP. Will be `undefined` on\n    decode if the data bytes could not be decoded, the AVP is unknown, or if the\n    [decode format](`m:diameter#decode_format`) is `none`. The type of a decoded\n    value is as document in [diameter_dict(4)](diameter_dict.md#DATA_TYPES).\n\n  - **`type = atom()`** - The type of the AVP as specified in the dictionary\n    file in question (or one it inherits). Possible types are `undefined` and\n    the Diameter types: `OctetString`, `Integer32`, `Integer64`, `Unsigned32`,\n    `Unsigned64`, `Float32`, `Float64`, `Grouped`, `Enumerated`, `Address`,\n    `Time`, `UTF8String`, `DiameterIdentity`, `DiameterURI`, `IPFilterRule` and\n    `QoSFilterRule`.\n\n- **`dictionary() = module()`{: #dictionary }** - The name of a generated\n  dictionary module as generated by [diameterc(1)](diameterc_cmd.md) or\n  `diameter_make:codec/2`. The interface provided by a dictionary module is an\n  implementation detail that may change.\n\n- **`header() = #diameter_header{}`{: #header }** - The record representation of\n  the Diameter header. Values in a [packet()](`m:diameter_codec#packet`)\n  returned by `decode/2` are as extracted from the incoming message. Values set\n  in an [packet()](`m:diameter_codec#packet`) passed to `encode/2` are preserved\n  in the encoded binary(), with the exception of `length`, `cmd_code` and\n  `application_id`, all of which are determined by the\n  [dictionary()](`m:diameter_codec#dictionary`) in question.\n\n  > #### Note {: .info }\n  >\n  > It is not necessary to set header fields explicitly in outgoing messages as\n  > diameter itself will set appropriate values. Setting inappropriate values\n  > can be useful for test purposes.\n\n  Fields have the following types.\n\n  - **`version = uint8()`**\n\n  - **`length = uint24()`**\n\n  - **`cmd_code = uint24()`**\n\n  - **`application_id = uint32()`**\n\n  - **`hop_by_hop_id = uint32()`**\n\n  - **`end_to_end_id = uint32()`** - Values of the Version, Message Length,\n    Command-Code, Application-ID, Hop-by-Hop Identifier and End-to-End\n    Identifier fields of the Diameter header.\n\n  - **`is_request = boolean()`**\n\n  - **`is_proxiable = boolean()`**\n\n  - **`is_error = boolean()`**\n\n  - **`is_retransmitted = boolean()`** - Values corresponding to the R(equest),\n    P(roxiable), E(rror) and T(Potentially re-transmitted message) flags of the\n    Diameter header.\n\n- **`message() = record() | maybe_improper_list()`{: #message }** - The\n  representation of a Diameter message as passed to `diameter:call/4` or\n  returned from a [handle_request/3](`c:diameter_app:handle_request/3`)\n  callback. The record representation is as outlined in\n  [diameter_dict(4)](diameter_dict.md#MESSAGE_RECORDS): a message as defined in\n  a dictionary file is encoded as a record with one field for each component\n  AVP. Equivalently, a message can also be encoded as a list whose head is the\n  atom-valued message name (as specified in the relevant dictionary file) and\n  whose tail is either a list of AVP name/values pairs or a map with values\n  keyed on AVP names. The format at decode is determined by\n  [diameter:service_opt()](`m:diameter#service_opt`)\n  [decode_format](`m:diameter#decode_format`). Any of the formats is accepted at\n  encode.\n\n  Another list-valued representation allows a message to be specified as a list\n  whose head is a [header()](`m:diameter_codec#header`) and whose tail is an\n  [avp()](`m:diameter_codec#avp`) list. This representation is used by diameter\n  itself when relaying requests as directed by the return value of a\n  [handle_request/3](`c:diameter_app:handle_request/3`) callback. It differs\n  from the other two in that it bypasses the checks for messages that do not\n  agree with their definitions in the dictionary in question: messages are sent\n  exactly as specified.\n\n- **`packet() = #diameter_packet{}`{: #packet }** - A container for incoming and\n  outgoing Diameter messages. Fields have the following types.\n\n  - **`header = `[`header()`](`m:diameter_codec#header`)` | undefined`** - The\n    Diameter header of the message. Can be (and typically should be) `undefined`\n    for an outgoing message in a non-relay application, in which case diameter\n    provides appropriate values.\n\n  - **`avps = [`[`avp()`](`m:diameter_codec#avp`)`] | undefined`** - The AVPs of\n    the message. Ignored for an outgoing message if the `msg` field is set to a\n    value other than `undefined`.\n\n  - **`msg = `[`message()`](`m:diameter_codec#message`)` | undefined`** - The\n    incoming/outgoing message. For an incoming message, a term corresponding to\n    the configured [decode format](`m:diameter#decode_format`) if the message\n    can be decoded in a non-relay application, `undefined` otherwise. For an\n    outgoing message, setting a\n    `[`[`header()`](`m:diameter_codec#header`)`|`[`avp()`](`m:diameter_codec#avp`)`]`\n    list is equivalent to setting the `header` and `avps` fields to the\n    corresponding values.\n\n    > #### Warning {: .warning }\n    >\n    > A value in the `msg` field does _not_ imply an absence of decode errors.\n    > The `errors` field should also be examined.\n\n  - **`bin = binary()`** - The incoming message prior to encode or the outgoing\n    message after encode.\n\n  - **`errors = [5000..5999 | {5000..5999, avp()}]`** - Errors detected at\n    decode of an incoming message, as identified by a corresponding 5xxx series\n    Result-Code (Permanent Failures). For an incoming request, these should be\n    used to formulate an appropriate answer as documented for the\n    [handle_request/3](`c:diameter_app:handle_request/3`) callback in\n    `m:diameter_app`. For an incoming answer, the\n    [diameter:application_opt()](`m:diameter#application_opt`) `answer_errors`\n    determines the behaviour.\n\n  - **`transport_data = term()`** - An arbitrary term of meaning only to the\n    transport process in question, as documented in `m:diameter_transport`.","ref":"diameter_codec.html#module-data-types"},{"type":"module","title":"SEE ALSO - diameter_codec","doc":"[diameterc(1)](diameterc_cmd.md), `m:diameter_app`,\n[diameter_dict(4)](diameter_dict.md), `m:diameter_make`","ref":"diameter_codec.html#module-see-also"},{"type":"function","title":"diameter_codec.decode/2","doc":"Decode a Diameter message.","ref":"diameter_codec.html#decode/2"},{"type":"type","title":"diameter_codec.dictionary/0","doc":"The name of a generated [dictionary module](`m:diameter_codec#dictionary`)\nas generated by [diameterc(1)](diameterc_cmd.md) or `diameter_make:codec/2`.\nThe interface provided by a dictionary module is an\nimplementation detail that may change.","ref":"diameter_codec.html#t:dictionary/0"},{"type":"function","title":"diameter_codec.encode/2","doc":"Encode a Diameter message.","ref":"diameter_codec.html#encode/2"},{"type":"type","title":"diameter_codec.message/0","doc":"The representation of a Diameter [message](`m:diameter_codec#message`)\nas passed to `diameter:call/4` or returned from a\n[handle_request/3](`c:diameter_app:handle_request/3`) callback.","ref":"diameter_codec.html#t:message/0"},{"type":"type","title":"diameter_codec.packet/0","doc":"A [container](`m:diameter_codec#packet`) for incoming and outgoing\nDiameter messages.","ref":"diameter_codec.html#t:packet/0"},{"type":"type","title":"diameter_codec.record/0","doc":"","ref":"diameter_codec.html#t:record/0"},{"type":"module","title":"diameter_make","doc":"Diameter dictionary compilation.\n\nThe function `codec/2` is used to compile a diameter\n[dictionary file](diameter_dict.md) into Erlang source. The resulting source\nimplements the interface diameter requires to encode and decode the dictionary's\nmessages and AVPs.\n\nThe utility [diameterc(1)](diameterc_cmd.md) provides an alternate compilation\ninterface.","ref":"diameter_make.html"},{"type":"module","title":"BUGS - diameter_make","doc":"Unrecognized options are silently ignored.","ref":"diameter_make.html#module-bugs"},{"type":"module","title":"SEE ALSO - diameter_make","doc":"[diameterc(1)](diameterc_cmd.md), [diameter_dict(4)](diameter_dict.md)","ref":"diameter_make.html#module-see-also"},{"type":"function","title":"diameter_make.codec/2","doc":"Compile a single dictionary file.\n\nThe input `File` can be either a path or a literal dictionary, the occurrence\nof newline (ascii NL) or carriage return (ascii CR) identifying the latter.\n`Opt` determines the format of the results and whether they are written to\nfile or returned, and can have the following types.\n\n- **`parse | forms | erl | hrl`** - Specifies an output format. Whether the\n  output is returned or written to file depends on whether or not option\n  `return` is specified. When written to file, the resulting file(s) will have\n  extensions `.D`, `.F`, `.erl`, and `.hrl` respectively, basenames defaulting\n  to `dictionary` if the input dictionary is literal and does not specify\n  [`@name`](diameter_dict.md#name). When returned, results are in the order of\n  the corresponding format options. Format options default to `erl` and `hrl`\n  (in this order) if unspecified.\n\n  The `parse` format is an internal representation that can be passed to\n  `flatten/1` and `format/1`, while the `forms` format can be passed to\n  `compile:forms/2`. The `erl` and `hrl` formats are returned as iolists.\n\n- **`{include, string()}`** - Prepend the specified directory to the code path.\n  Use to point at beam files compiled from inherited dictionaries,\n  [`@inherits`](diameter_dict.md#inherits) in a dictionary file creating a beam\n  dependency, not an erl/hrl dependency.\n\n  Multiple `include` options can be specified.\n\n- **`{outdir, string()}`** - Write generated source to the specified directory.\n  Defaults to the current working directory. Has no effect if option `return` is\n  specified.\n\n- **`return`** - Return results in a `{ok, [Out]}` tuple instead of writing to\n  file and returning `ok`.\n\n- **`{name|prefix, string()}`** - Transform the input dictionary before\n  compilation, setting [`@name`](diameter_dict.md#name) or\n  [`@prefix`](diameter_dict.md#prefix) to the specified string.\n\n- **`{inherits, string()}`** - Transform the input dictionary before\n  compilation, appending [`@inherits`](diameter_dict.md#inherits) of the\n  specified string.\n\n  Two forms have special meaning:\n\n  ```text\n  {inherits, \"-\"}\n  {inherits, \"Prev/Mod\"}\n  ```\n\n  The first has the effect of clearing any previous inherits, the second of\n  replacing a previous inherits of `Prev` to one of `Mod`. This allows the\n  semantics of the input dictionary to be changed without modifying the file\n  itself.\n\n  Multiple `inherits` options can be specified.\n\nNote that a dictionary's [`@name`](diameter_dict.md#name), together with the\n`outdir` option, determine the output paths when the `return` option is not\nspecified. The [`@name`](diameter_dict.md#name) of a literal input dictionary\ndefaults to `dictionary`.\n\nA returned error reason can be converted into a readable string using\n`format_error/1`.","ref":"diameter_make.html#codec/2"},{"type":"type","title":"diameter_make.dict/0","doc":"","ref":"diameter_make.html#t:dict/0"},{"type":"function","title":"diameter_make.flatten/1","doc":"Reconstitute a parsed dictionary, as returned by `codec/2`, without using\n[`@inherits`](diameter_dict.md#inherits). That is, construct an equivalent\ndictionary in which all AVP's are definined in the dictionary itself. The return\nvalue is also a parsed dictionary.","ref":"diameter_make.html#flatten/1"},{"type":"function","title":"diameter_make.format/1","doc":"Turns a parsed dictionary, as returned by `codec/2`, back into the dictionary\nformat.","ref":"diameter_make.html#format/1"},{"type":"function","title":"diameter_make.format_error/1","doc":"Turn an error reason returned by `codec/2` into a readable string.","ref":"diameter_make.html#format_error/1"},{"type":"type","title":"diameter_make.opt/0","doc":"","ref":"diameter_make.html#t:opt/0"},{"type":"type","title":"diameter_make.parsed/0","doc":"","ref":"diameter_make.html#t:parsed/0"},{"type":"module","title":"diameter_sctp","doc":"Diameter transport over SCTP.\n\nThis module implements diameter transport over SCTP using `m:gen_sctp`. It can\nbe specified as the value of a transport_module option to\n`diameter:add_transport/2` and implements the behaviour documented in\n`m:diameter_transport`.\n\n[](){: #start }","ref":"diameter_sctp.html"},{"type":"module","title":"SEE ALSO - diameter_sctp","doc":"`m:diameter`, `m:diameter_transport`, `m:gen_sctp`, `m:inet`","ref":"diameter_sctp.html#module-see-also"},{"type":"type","title":"diameter_sctp.connect_option/0","doc":"","ref":"diameter_sctp.html#t:connect_option/0"},{"type":"type","title":"diameter_sctp.listen_option/0","doc":"","ref":"diameter_sctp.html#t:listen_option/0"},{"type":"type","title":"diameter_sctp.match/0","doc":"","ref":"diameter_sctp.html#t:match/0"},{"type":"type","title":"diameter_sctp.option/0","doc":"","ref":"diameter_sctp.html#t:option/0"},{"type":"function","title":"diameter_sctp.start/3","doc":"The start function required by `m:diameter_transport`.\n\nOptions `raddr` and `rport` specify the remote address and port for a connecting\ntransport and not valid for a listening transport: the former is required while\nlatter defaults to 3868 if unspecified. Multiple `raddr` options can be\nspecified, in which case the connecting transport in question attempts each in\nsequence until an association is established.\n\nOption `accept` specifies remote addresses for a listening transport and is not\nvalid for a connecting transport. If specified, a remote address that does not\nmatch one of the specified addresses causes the association to be aborted.\nMultiple `accept` options can be specified. A string-valued `Match` that does\nnot parse as an address is interpreted as a regular expression.\n\nOption `unordered` specifies whether or not to use unordered delivery, integer\n`N` being equivalent to `N =  #### Warning {: .warning }\n>\n> An small receive buffer may result in a peer having to resend incoming\n> messages: set the `m:inet` option `recbuf` to increase the buffer size.\n>\n> An small send buffer may result in outgoing messages being discarded: set the\n> `m:inet` option `sndbuf` to increase the buffer size.","ref":"diameter_sctp.html#start/3"},{"type":"module","title":"diameter_service","doc":"","ref":"diameter_service.html"},{"type":"type","title":"diameter_service.wd_state/0","doc":"State of the watchdog","ref":"diameter_service.html#t:wd_state/0"},{"type":"module","title":"diameter_tcp","doc":"Diameter transport over TCP.\n\nThis module implements diameter transport over TCP using `m:gen_tcp`. It can be\nspecified as the value of a `transport_module` option to\n`diameter:add_transport/2` and implements the behaviour documented in\n`m:diameter_transport`. TLS security is supported, either as an upgrade\nfollowing capabilities exchange or at connection establishment.\n\nNote that the ssl application is required for TLS and must be started before\nconfiguring TLS capability on diameter transports.\n\n[](){: #start }","ref":"diameter_tcp.html"},{"type":"module","title":"SEE ALSO - diameter_tcp","doc":"`m:diameter`, `m:diameter_transport`, `m:gen_tcp`, `m:inet`, `m:ssl`","ref":"diameter_tcp.html#module-see-also"},{"type":"type","title":"diameter_tcp.connect_option/0","doc":"","ref":"diameter_tcp.html#t:connect_option/0"},{"type":"type","title":"diameter_tcp.listen_option/0","doc":"","ref":"diameter_tcp.html#t:listen_option/0"},{"type":"type","title":"diameter_tcp.match/0","doc":"","ref":"diameter_tcp.html#t:match/0"},{"type":"type","title":"diameter_tcp.option/0","doc":"","ref":"diameter_tcp.html#t:option/0"},{"type":"function","title":"diameter_tcp.start/3","doc":"start({Type, Ref}, Svc, [Opt]) -> {ok, Pid} | {ok, Pid, [LAddr]} | {error,\nReason}\n\nThe start function required by `m:diameter_transport`.\n\nOptions `raddr` and `rport` specify the remote address and port for a connecting\ntransport and are not valid for a listening transport.\n\nOption `accept` specifies remote addresses for a listening transport and is not\nvalid for a connecting transport. If specified, a remote address that does not\nmatch one of the specified addresses causes the connection to be aborted.\nMultiple `accept` options can be specified. A string-valued `Match` that does\nnot parse as an address is interpreted as a regular expression.\n\nOption `ssl_options` must be specified for a transport that should support TLS:\na value of `true` results in a TLS handshake immediately upon connection\nestablishment while `t:list/0` specifies options to be passed to `ssl:connect/2`\nor `ssl:handshake/2` after capabilities exchange if TLS is negotiated.\n\nOption `fragment_timer` specifies the timeout, in milliseconds, of a timer used\nto flush messages from the incoming byte stream even if the number of bytes\nindicated in the Message Length field of its Diameter Header have not yet been\naccumulated: such a message is received over the transport interface after two\nsuccessive timeouts without the reception of additional bytes. Defaults to 1000.\n\n[](){: #sender }\n\nOption `sender` specifies whether or not to use a dedicated process for sending\noutgoing messages, which avoids the possibility of send blocking reception.\nDefaults to `false`. If set to `true` then a `message_cb` that avoids the\npossibility of messages being queued in the sender process without bound should\nbe configured.\n\nOption `message_cb` specifies a callback that is invoked on incoming and\noutgoing messages, that can be used to implement flow control. It is applied to\ntwo arguments: an atom indicating the reason for the callback (`send`, `recv`,\nor `ack` after a completed send), and the message in question (binary() on\n`recv`, binary() or diameter*packet record on `send` or `ack`, or `false` on\n`ack` when an incoming request has been discarded). It should return a list of\nactions and a new callback as tail; eg. `[fun cb/3, State]`. Valid actions are\nthe atoms `send` or `recv`, to cause a following message-valued action to be\nsent/received, a message to send/receive (binary() or diameter_packet record),\nor a boolean() to enable/disable reading on the socket. More than one\n`send`/`recv`/message sequence can be returned from the same callback, and an\ninitial `send`/`recv` can be omitted if the same as the value passed as the\ncallback's first argument. Reading is initially enabled, and returning `false`\ndoes not imply there cannot be subsequent `recv` callbacks since messages may\nalready have been read. An empty tail is equivalent to the prevailing callback.\nDefaults to a callback equivalent to `fun(ack, *) -> []; (\\_, Msg) -> [Msg]\nend`.\n\nRemaining options are any accepted by `ssl:connect/3` or `gen_tcp:connect/3` for\na connecting transport, or `ssl:listen/2` or `gen_tcp:listen/2` for a listening\ntransport, depending on whether or not `{ssl_options, true}` has been specified.\nOptions `binary`, `packet` and `active` cannot be specified. Also, option `port`\ncan be specified for a listening transport to specify the local listening port,\nthe default being the standardized 3868. Note that the option `ip` specifies the\nlocal address.\n\nAn `ssl_options` list must be specified if and only if the transport in question\nhas set `Inband-Security-Id` to 1 (`TLS`), as specified to either\n`diameter:start_service/2` or `diameter:add_transport/2`, so that the transport\nprocess will receive notification of whether or not to commence with a TLS\nhandshake following capabilities exchange. Failing to specify an options list on\na TLS-capable transport for which TLS is negotiated will cause TLS handshake to\nfail. Failing to specify TLS capability when `ssl_options` has been specified\nwill cause the transport process to wait for a notification that will not be\nforthcoming, which will eventually cause the RFC 3539 watchdog to take down the\nconnection.\n\nThe first element of a non-empty `Host-IP-Address` list in `Svc` provides the\nlocal IP address if an `ip` option is not specified. The local address is either\nreturned from`start/3` or passed in a `connected` message over the transport\ninterface.","ref":"diameter_tcp.html#start/3"},{"type":"behaviour","title":"diameter_transport","doc":"Diameter transport interface.\n\nA module specified as a `transport_module` to `diameter:add_transport/2` must\nimplement the interface documented here. The interface consists of a function\nwith which diameter starts a transport process and a message interface with\nwhich the transport process communicates with the process that starts it (aka\nits parent).","ref":"diameter_transport.html"},{"type":"behaviour","title":"DATA TYPES - diameter_transport","doc":"- **`message() = binary() | `[`diameter_codec:packet()`](`m:diameter_codec#packet`)**{: #message } -\n  A Diameter message as passed over the transport interface.\n\n  For an inbound message from a transport process, a\n  [diameter_codec:packet()](`m:diameter_codec#packet`) must contain the received\n  message in its `bin` field. In the case of an inbound request, any value set\n  in the `transport_data` field will passed back to the transport module in the\n  corresponding answer message, unless the sender supplies another value.\n\n  For an outbound message to a transport process, a\n  [diameter_codec:packet()](`m:diameter_codec#packet`) has a value other than\n  `undefined` in its `transport_data` field and has the binary() to send in its\n  `bin` field.\n\n[](){: #MESSAGES }","ref":"diameter_transport.html#module-data-types"},{"type":"behaviour","title":"MESSAGES - diameter_transport","doc":"All messages sent over the transport interface are of the form\n`{diameter, term()}`.\n\nA transport process can expect messages of the following types from its parent.\n\n- **`{diameter, {send, `[`message()`](`m:diameter_transport#message`)` | false}}`** -\n  An outbound Diameter message. The atom `false` can only be received when\n  request acknowledgements have been requests: see the `ack` message below.\n\n- **`{diameter, {close, Pid}}`** - A request to terminate the transport process\n  after having received DPA in response to DPR. The transport process should\n  exit. `Pid` is the pid() of the parent process.\n\n- **`{diameter, {tls, Ref, Type, Bool}}`** - Indication of whether or not\n  capabilities exchange has selected inband security using TLS. `Ref` is a\n  reference() that must be included in the `{diameter, {tls, Ref}}` reply\n  message to the transport's parent process (see below). `Type` is either\n  `connect` or `accept` depending on whether the process has been started for a\n  connecting or listening transport respectively. `Bool` is a boolean()\n  indicating whether or not the transport connection should be upgraded to TLS.\n\n  If TLS is requested (`Bool=true`) then a connecting process should initiate a\n  TLS handshake with the peer and an accepting process should prepare to accept\n  a handshake. A successful handshake should be followed by a\n  `{diameter, {tls, Ref}}` message to the parent process. A failed handshake\n  should cause the process to exit.\n\n  This message is only sent to a transport process over whose\n  `Inband-Security-Id` configuration has indicated support for TLS.\n\nA transport process should send messages of the following types to its parent.\n\n- **`{diameter, {self(), connected}}`** - Inform the parent that the transport\n  process with `Type=accept` has established a connection with the peer. Not\n  sent if the transport process has `Type=connect`.\n\n- **`{diameter, {self(), connected, Remote}}`**\n\n- **`{diameter, {self(), connected, Remote, [LocalAddr]}}`** - Inform the parent\n  that the transport process with `Type=connect` has established a connection\n  with a peer. Not sent if the transport process has `Type=accept`.  `Remote` is\n  an arbitrary term that uniquely identifies the remote endpoint to which the\n  transport has connected. A `LocalAddr` list has the same semantics as one\n  returned from `c:start/3`.\n\n- **`{diameter, ack}`** - Request acknowledgements of unanswered requests. A\n  transport process should send this once before passing incoming Diameter\n  messages into diameter. As a result, every Diameter request passed into\n  diameter with a `recv` message (below) will be answered with a `send` message\n  (above), either a [message()](`m:diameter_transport#message`) for the\n  transport process to send or the atom `false` if the request has been\n  discarded or otherwise not answered.\n\n  This is to allow a transport process to keep count of the number of incoming\n  request messages that have not yet been answered or discarded, to allow it to\n  regulate the amount of incoming traffic. Both diameter_tcp and diameter_sctp\n  request acknowledgements when a `message_cb` is configured, turning send/recv\n  message into callbacks that can be used to regulate traffic.\n\n- **`{diameter, {recv, `[`message()`](`m:diameter_transport#message`)`}}`** - An\n  inbound Diameter message.\n\n- **`{diameter, {tls, Ref}}`** - Acknowledgment of a successful TLS handshake.\n  `Ref` is the reference() received in the `{diameter, {tls, Ref, Type, Bool}}`\n  message in response to which the reply is sent. A transport must exit if a\n  handshake is not successful.","ref":"diameter_transport.html#module-messages"},{"type":"behaviour","title":"SEE ALSO - diameter_transport","doc":"`m:diameter_tcp`, `m:diameter_sctp`","ref":"diameter_transport.html#module-see-also"},{"type":"callback","title":"diameter_transport.start/3","doc":"Start a transport process. Called by diameter as a consequence of a call to\n`diameter:add_transport/2` in order to establish or accept a transport\nconnection respectively. A transport process maintains a connection with a\nsingle remote peer.\n\n`Type` indicates whether the transport process in question is being started for\na connecting (`Type=connect`) or listening (`Type=accept`) transport. In the\nlatter case, transport processes are started as required to accept connections\nfrom multiple peers.\n\nRef is the value that was returned from the call to `diameter:add_transport/2`\nthat has lead to starting of a transport process.\n\n`Svc` contains capabilities passed to `diameter:start_service/2` and\n`diameter:add_transport/2`, values passed to the latter overriding those passed\nto the former.\n\n`Config` is as passed in `transport_config` tuple in the\n[diameter:transport_opt()](`m:diameter#transport_opt`) list passed to\n`diameter:add_transport/2`.\n\nThe start function should use the `Host-IP-Address` list in `Svc` and/or\n`Config` to select and return an appropriate list of local IP addresses. In the\nconnecting case, the local address list can instead be communicated in a\n`connected` message (see [MESSAGES](`m:diameter_transport#MESSAGES`) below)\nfollowing connection establishment. In either case, the local address list is\nused to populate `Host-IP-Address` AVPs in outgoing capabilities exchange\nmessages if `Host-IP-Address` is unspecified.\n\nA transport process must implement the message interface documented below. It\nshould retain the pid of its parent, monitor the parent and terminate if it\ndies. It should not link to the parent. It should exit if its transport\nconnection with its peer is lost.","ref":"diameter_transport.html#c:start/3"},{"type":"extras","title":"Release Notes","doc":"<!--\n%CopyrightBegin%\n\nSPDX-License-Identifier: Apache-2.0\n\nCopyright Ericsson AB 2023-2025. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Release Notes\n\nReleases are listed in reverse chronological order, most recent first.","ref":"notes.html"},{"type":"extras","title":"diameter 2.5.1 - Release Notes","doc":"","ref":"notes.html#diameter-2-5-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- With this change message_cb callback will be called with updated state for processing 'ack' after 'send'.\n\n  Own Id: OTP-19753 Aux Id: [PR-9815]\n\n[PR-9815]: https://github.com/erlang/otp/pull/9815","ref":"notes.html#fixed-bugs-and-malfunctions"},{"type":"extras","title":"diameter 2.5 - Release Notes","doc":"","ref":"notes.html#diameter-2-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- With this change diameter will not crash when decoding a DiameterURI without port number.\n\n  Own Id: OTP-19620 Aux Id: [PR-9321]\n\n[PR-9321]: https://github.com/erlang/otp/pull/9321","ref":"notes.html#fixed-bugs-and-malfunctions-1"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- [EEP-69: Nominal Types](https://www.erlang.org/eeps/eep-0069) has been implemented. As a side effect, nominal types can encode opaque types. We changed all opaque-handling logic and improved opaque warnings in Dialyzer.\n  \n  All existing Erlang type systems are structural: two types are seen as equivalent if their structures are the same. Type comparisons are based on the structures of the types, not on how the user explicitly defines them. For example, in the following example, `meter()` and `foot()` are equivalent. The two types can be used interchangeably. Neither of them differ from the basic type `integer()`.\n  \n  ````\n  -type meter() :: integer().\n  -type foot() :: integer().\n  ````\n  \n  Nominal typing is an alternative type system, where two types are equivalent if and only if they are declared with the same type name. The EEP proposes one new syntax -nominal for declaring nominal types. Under nominal typing, `meter()` and `foot()` are no longer compatible. Whenever a function expects type `meter()`, passing in type `foot()` would result in a Dialyzer error.\n  \n  ````\n  -nominal meter() :: integer().\n  -nominal foot() :: integer().\n  ````\n  \n  More nominal type-checking rules can be found in the EEP. It is worth noting that most work for adding nominal types and type-checking is in `erl_types.erl`. The rest are changes that removed the previous opaque type-checking, and added an improved version of it using nominal type-checking with reworked warnings.\n  \n  Backwards compatibility for opaque type-checking is not preserved by this PR. Previous opaque warnings can appear with slightly different wordings. A new kind of opaque warning `opaque_union` is added, together with a Dialyzer option `no_opaque_union` to turn this kind of warnings off.\n\n  Own Id: OTP-19364 Aux Id: [PR-9079]\n\n- The license and copyright header has changed format to include an `SPDX-License-Identifier`. At the same time, most files have been updated to follow a uniform standard for license headers.\n\n  Own Id: OTP-19575 Aux Id: [PR-9670]\n\n- With this change diameter will not use slave terminology\n\n  Own Id: OTP-19621 Aux Id: [PR-9786]\n\n[PR-9079]: https://github.com/erlang/otp/pull/9079\n[PR-9670]: https://github.com/erlang/otp/pull/9670\n[PR-9786]: https://github.com/erlang/otp/pull/9786","ref":"notes.html#improvements-and-new-features"},{"type":"extras","title":"diameter 2.4.1 - Release Notes","doc":"","ref":"notes.html#diameter-2-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Function specs for the main API module has been updated.\n\n  Own Id: OTP-19126 Aux Id: [#8399]\n\n- Man pages are now available for `erl`, `erlc`, `dialyzer`, and all other programs that are included in Erlang/OTP.\n\n  Own Id: OTP-19201 Aux Id: [PR-8740]\n\n- `diameter:stop_service/1` has been made more synchronous.\n\n  Own Id: OTP-19206 Aux Id: ERIERL-1102\n\n[#8399]: https://github.com/erlang/otp/issues/8399\n[PR-8740]: https://github.com/erlang/otp/pull/8740","ref":"notes.html#fixed-bugs-and-malfunctions-2"},{"type":"extras","title":"diameter 2.4 - Release Notes","doc":"","ref":"notes.html#diameter-2-4"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- `-callback` attributes have been added to `m:diameter_app` and `m:diameter_transport`.\n\n  Own Id: OTP-18783 Aux Id: [PR-7699]\n\n- The documentation has been migrated to use Markdown and ExDoc.\n\n  Own Id: OTP-18955 Aux Id: [PR-8026]\n\n- Pick peer can now also handle request of type `#diameter_packet{}`.\n\n  Own Id: OTP-19090 Aux Id: [PR-8399]\n\n[PR-7699]: https://github.com/erlang/otp/pull/7699\n[PR-8026]: https://github.com/erlang/otp/pull/8026\n[PR-8399]: https://github.com/erlang/otp/pull/8399","ref":"notes.html#improvements-and-new-features-1"},{"type":"extras","title":"diameter 2.3.2.2 - Release Notes","doc":"","ref":"notes.html#diameter-2-3-2-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"* Stop service has been made more synchronous.\n\n  Own Id: OTP-19206 Aux Id: ERIERL-1102","ref":"notes.html#fixed-bugs-and-malfunctions-3"},{"type":"extras","title":"diameter 2.3.2.1 - Release Notes","doc":"","ref":"notes.html#diameter-2-3-2-1"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"* Pick peer can now also handle request of type #diameter_packet\\{\\}.\n\n  Own Id: OTP-19090 Aux Id: PR-8399","ref":"notes.html#improvements-and-new-features-2"},{"type":"extras","title":"diameter 2.3.2 - Release Notes","doc":"","ref":"notes.html#diameter-2-3-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"* Reduce the impact of calling service_info by not counting the binaries (on the heap) info, This is done by introducing an option, bins_info, which controls this.\n\n  Own Id: OTP-19040 Aux Id: ERIERL-1060","ref":"notes.html#fixed-bugs-and-malfunctions-4"},{"type":"extras","title":"diameter 2.3.1 - Release Notes","doc":"","ref":"notes.html#diameter-2-3-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Replaced unintentional Erlang Public License 1.1 headers in some files with\n  the intended Apache License 2.0 header.\n\n  Own Id: OTP-18815 Aux Id: PR-7780","ref":"notes.html#fixed-bugs-and-malfunctions-5"},{"type":"extras","title":"diameter 2.3 - Release Notes","doc":"","ref":"notes.html#diameter-2-3"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Replace size/1 with either tuple_size/1 or byte_size/1\n\n  The [`size/1`](`size/1`) BIF is not optimized by the JIT, and its use can\n  result in worse types for Dialyzer.\n\n  When one knows that the value being tested must be a tuple,\n  [`tuple_size/1`](`tuple_size/1`) should always be preferred.\n\n  When one knows that the value being tested must be a binary,\n  [`byte_size/1`](`byte_size/1`) should be preferred. However,\n  [`byte_size/1`](`byte_size/1`) also accepts a bitstring (rounding up size to a\n  whole number of bytes), so one must make sure that the call to `byte_size/` is\n  preceded by a call to [`is_binary/1`](`is_binary/1`) to ensure that bitstrings\n  are rejected. Note that the compiler removes redundant calls to\n  [`is_binary/1`](`is_binary/1`), so if one is not sure whether previous code\n  had made sure that the argument is a binary, it does not harm to add an\n  [`is_binary/1`](`is_binary/1`) test immediately before the call to\n  [`byte_size/1`](`byte_size/1`).\n\n  Own Id: OTP-18405 Aux Id:\n  GH-6672,PR-6702,PR-6768,PR-6700,PR-6769,PR-6812,PR-6814\n\n- Deprecates `dbg:stop_clear/0` because it is simply a function alias to\n  `dbg:stop/0`\n\n  Own Id: OTP-18478 Aux Id: GH-6903\n\n- The implementation has been fixed to use `proc_lib:init_fail/2,3` where\n  appropriate, instead of `proc_lib:init_ack/1,2`.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-18490 Aux Id: OTP-18471, GH-6339, PR-6843","ref":"notes.html#improvements-and-new-features-3"},{"type":"extras","title":"diameter 2.2.7.2 - Release Notes","doc":"","ref":"notes.html#diameter-2-2-7-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"* \\`diameter:stop_service/1\\` has been made more synchronous.\n\n  Own Id: OTP-19206 Aux Id: ERIERL-1102","ref":"notes.html#fixed-bugs-and-malfunctions-6"},{"type":"extras","title":"diameter 2.2.7.1 - Release Notes","doc":"","ref":"notes.html#diameter-2-2-7-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"* Reduce the impact of calling service_info by not counting the binaries (on the heap) info, This is done by introducing an option, bins_info, which controls this.\n\n  Own Id: OTP-19040 Aux Id: ERIERL-1060","ref":"notes.html#fixed-bugs-and-malfunctions-7"},{"type":"extras","title":"diameter 2.2.7 - Release Notes","doc":"","ref":"notes.html#diameter-2-2-7"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- There is a new configure option, `--enable-deterministic-build`, which will\n  apply the `deterministic` compiler option when building Erlang/OTP. The\n  `deterministic` option has been improved to eliminate more sources of\n  non-determinism in several applications.\n\n  Own Id: OTP-18165 Aux Id: PR-5965","ref":"notes.html#improvements-and-new-features-4"},{"type":"extras","title":"diameter 2.2.6 - Release Notes","doc":"","ref":"notes.html#diameter-2-2-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix decode of non-IP address types; that is, of values of the derived AVP data\n  format Address whose first two octets specify an address family other than 1\n  (IP) or 2 (IP6). Such values have never been decoded, and were treated as\n  decode errors. They're now decoded to a 2-tuple of the integer() address\n  family and binary() remaining octets, with no family-specific decode. The\n  2-tuple distinguishes the decode from the 4-tuple and 8-tuple IP address\n  decodes. 2-tuples are also now encoded.\n\n  Note that even currently unassigned address families are decoded: only the\n  reserved values, 0 and 65535, are treated as errors.\n\n  Own Id: OTP-17976 Aux Id: GH-5463","ref":"notes.html#fixed-bugs-and-malfunctions-8"},{"type":"extras","title":"diameter 2.2.5 - Release Notes","doc":"","ref":"notes.html#diameter-2-2-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- The compilation time is no longer recorded in BEAM files. There remained\n  several undocumented functions that attempted to retrieve compilation times.\n  Those have now been removed.\n\n  Own Id: OTP-17962","ref":"notes.html#fixed-bugs-and-malfunctions-9"},{"type":"extras","title":"diameter 2.2.4 - Release Notes","doc":"","ref":"notes.html#diameter-2-2-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- The unordered option was ignored on a client diameter_sctp transport, causing\n  all delivery to be ordered.\n\n  The association id was not specified to gen_sctp when requesting unordered\n  delivery, causing the setting to be applied to the whole endpoint.\n\n  Thanks to Bengt Kleberg and Andreas Schultz.\n\n  Own Id: OTP-17366 Aux Id: GH-4775","ref":"notes.html#fixed-bugs-and-malfunctions-10"},{"type":"extras","title":"diameter 2.2.3 - Release Notes","doc":"","ref":"notes.html#diameter-2-2-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Add the 'first' tuple to type diameter:peer_filter/0. The filter was added in\n  OTP-17.5.6.8 and OTP-18.3, but neither release updated the type specification.\n\n  Own Id: OTP-16548 Aux Id: ERL-1191","ref":"notes.html#fixed-bugs-and-malfunctions-11"},{"type":"extras","title":"diameter 2.2.2 - Release Notes","doc":"","ref":"notes.html#diameter-2-2-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- The possibility of choosing a handler process for an incoming Diameter request\n  with a configured MFA was documented in OTP 20.0, but counters (with\n  \\{traffic_counters, true\\}) were not incremented when this process was on a\n  remote node. Counters are now incremented on the node that configures the\n  transport in question.\n\n  Introduced in OTP 21.3.\n\n  Own Id: OTP-16457\n\n- Transport options differing from those passed to diameter:add_transport/2 were\n  used in several situations: when starting a transport process after\n  connect_timer expiry after an initial connection attempt has failed, when\n  starting a transport process after a connection has been accepted, when\n  sending events, when returning options in diameter:service_info/2, and\n  possibly more. In particular, the following configuration options to\n  diameter:add_transport/2 were dropped: avp_dictionaries, incoming_maxlen,\n  spawn_opt, strict_mbit.\n\n  Moreover, any service options mistakenly passed to diameter:add_transport/2\n  were interpreted as such, instead of being ignored as the documentation\n  states, with the consequence that outgoing and incoming requests saw different\n  values of some options, some were always taken from transport options, and\n  others from service options.\n\n  diameter:add_transport/2 must be called in new code for the fix to have\n  effect.\n\n  Introduced in OTP 20.1.\n\n  Own Id: OTP-16459","ref":"notes.html#fixed-bugs-and-malfunctions-12"},{"type":"extras","title":"diameter 2.2.1 - Release Notes","doc":"","ref":"notes.html#diameter-2-2-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix inadvertently broad monitor that resulted in gen_server cast messages to\n  hidden nodes from module diameter_dist.\n\n  Own Id: OTP-15768","ref":"notes.html#fixed-bugs-and-malfunctions-13"},{"type":"extras","title":"diameter 2.2 - Release Notes","doc":"","ref":"notes.html#diameter-2-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix failure of incoming answer message with faulty Experimental-Result-Code.\n  Failure to decode the AVP resulted in an uncaught exception, with no no\n  handle_answer/error callback as a consequence.\n\n  Own Id: OTP-15569 Aux Id: ERIERL-302","ref":"notes.html#fixed-bugs-and-malfunctions-14"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Add spawn_opt MFA configuration to allow a callback to spawn a handler process\n  for an incoming Diameter request on an an arbitrary node. Module diameter_dist\n  provides a route_session/2 that can be used to distribute requests based on\n  Session-Id, although this module is currently only documented in the module\n  itself and may change.\n\n  Own Id: OTP-15398","ref":"notes.html#improvements-and-new-features-5"},{"type":"extras","title":"diameter 2.1.6 - Release Notes","doc":"","ref":"notes.html#diameter-2-1-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix function_clause when sending an outgoing request after DPA has been sent\n  in response to an incoming DPR. The caused the diameter_peer_fsm gen_server\n  associated with the peer connection to fail, which could then result in the\n  transport connection being reset before the peer closed it upon reception of\n  DPA.\n\n  Own Id: OTP-15198 Aux Id: ERIERL-213","ref":"notes.html#fixed-bugs-and-malfunctions-15"},{"type":"extras","title":"diameter 2.1.5 - Release Notes","doc":"","ref":"notes.html#diameter-2-1-5"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix documentation typos.\n\n  Own Id: OTP-15045","ref":"notes.html#fixed-bugs-and-malfunctions-16"},{"type":"extras","title":"diameter 2.1.4.1 - Release Notes","doc":"","ref":"notes.html#diameter-2-1-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix failure of incoming answer message with faulty Experimental-Result-Code.\n  Failure to decode the AVP resulted in an uncaught exception, with no no\n  handle_answer/error callback as a consequence.\n\n  Own Id: OTP-15569 Aux Id: ERIERL-302","ref":"notes.html#fixed-bugs-and-malfunctions-17"},{"type":"extras","title":"diameter 2.1.4 - Release Notes","doc":"","ref":"notes.html#diameter-2-1-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix close of diameter_tcp/sctp listening socket at\n  diameter:remove_transport/2, that was broken in diameter 2.1. A reconfigured\n  transport could not listen on the same endpoint as a result.\n\n  Own Id: OTP-14839\n\n- Fix handling of SUSPECT connections at service termination. A connection with\n  this watchdog state caused diameter_service:terminate/2 to fail.\n\n  Own Id: OTP-14947 Aux Id: ERIERL-124","ref":"notes.html#fixed-bugs-and-malfunctions-18"},{"type":"extras","title":"diameter 2.1.3 - Release Notes","doc":"","ref":"notes.html#diameter-2-1-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix documentation typo: peer_up/3 was written where peer_down/3 was intended.\n\n  Own Id: OTP-14805","ref":"notes.html#fixed-bugs-and-malfunctions-19"},{"type":"extras","title":"diameter 2.1.2 - Release Notes","doc":"","ref":"notes.html#diameter-2-1-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- A fault introduced in diameter 2.1 could cause decode errors to be ignored in\n  AVPs following the header of a Grouped AVP.\n\n  Own Id: OTP-14684 Aux Id: ERIERL-85","ref":"notes.html#fixed-bugs-and-malfunctions-20"},{"type":"extras","title":"diameter 2.1.1 - Release Notes","doc":"","ref":"notes.html#diameter-2-1-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- An inadvertently removed monitor in diameter 2.1 caused the ets table\n  diameter_reg to leak entries, and caused service restart and more to fail.\n\n  Own Id: OTP-14668 Aux Id: ERIERL-83","ref":"notes.html#fixed-bugs-and-malfunctions-21"},{"type":"extras","title":"diameter 2.1 - Release Notes","doc":"","ref":"notes.html#diameter-2-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix handling of Proxy-Info in answer messages setting the E-bit.\n\n  RFC 6733 requires that Proxy-Info AVPs in an incoming request be echoed in an\n  outgoing answer. This was not done in answers formulated by diameter; for\n  example, as a result of a handle_request callback having returned an\n  'answer-message' or protocol_error tuple.\n\n  Own Id: OTP-9869\n\n- React to nodeup/nodedown when sharing peer connections.\n\n  Service configuration share_peers and use_shared_peers did not respond to the\n  coming and going of remote nodes.\n\n  Own Id: OTP-14011\n\n- Fix inappropriate message callbacks.\n\n  An incoming CER or DPR was regarded as discarded, resulting in a corresponding\n  message callback (if configured) in diameter_tcp/sctp.\n\n  Own Id: OTP-14486\n\n- Fix handling of 5009 errors (DIAMETER_AVP_OCCURS_TOO_MANY TIMES).\n\n  RFC 6733 says that the first AVP that exceeds the bound should be reported,\n  but the suggestions in the errors field of a diameter_packet record counted\n  AVPs from the rear of the message, not the front. Additionally, diameter 2.0\n  in OTP 20.0 broke the counting by accepting one more AVP than the message\n  grammar in question allowed.\n\n  Own Id: OTP-14512\n\n- Match case insensitively in diameter_tcp/sctp accept tuple.\n\n  Matching of remote addresses when accepting connections in a listening\n  transport was case-sensitive, causing the semantics to change as a consequence\n  of (kernel) OTP-13006.\n\n  Own Id: OTP-14535 Aux Id: OTP-13006\n\n- Fix backwards incompatibility of remote send when sharing transports.\n\n  The sending of requests over a transport connection on a remote node running\n  an older version of diameter was broken by diameter 2.0 in OTP 20.0.\n\n  Own Id: OTP-14552\n\n- Fix diameter_packet.avps decode of Grouped AVP errors in Failed-AVP.\n\n  Decode didn't produce a list of diameter_avp records, so information about\n  faulty component AVPs was lost.\n\n  Own Id: OTP-14607","ref":"notes.html#fixed-bugs-and-malfunctions-22"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Let unordered delivery be configured in diameter_sctp.\n\n  With option \\{unordered, boolean() | pos_integer()\\}, with false the default,\n  and N equivalent to OS =< N, where OS is the number of outbound streams\n  negotiated on the association in question. If configured, unordered sending\n  commences upon reception of a second message, outgoing messages being sent on\n  stream 0 before this.\n\n  The default false is for backwards compatibility, but false or 1 should be set\n  to follow RFC 6733's recommendation on the use of unordered sending to avoid\n  head-of-line blocking. There is typically no meaningful order to preserve,\n  since the order in which outgoing messages are received by a transport process\n  isn't known to the sender.\n\n  Own Id: OTP-10889\n\n- Complete/simplify Standards Compliance in User's Guide.\n\n  Own Id: OTP-10927\n\n- Add service option decode_format.\n\n  To allow incoming messages to be decoded into maps or lists instead of\n  records. Messages can be presented in any of the formats for encode.\n\n  Decode performance has also been improved.\n\n  Own Id: OTP-14511 Aux Id: OTP-14343\n\n- Add service option traffic_counters.\n\n  To let message-related counters be disabled, which can be a performance\n  improvement in some usecases.\n\n  Own Id: OTP-14521\n\n- Allow loopback/any as local addresses in diameter_tcp/sctp.\n\n  The atoms were implied by documentation, but not handled in code.\n\n  Own Id: OTP-14544\n\n- Add transport option strict_capx.\n\n  To allow the RFC 6733 requirement that a transport connection be closed if a\n  message is received before capabilities exchange to be relaxed.\n\n  Own Id: OTP-14546\n\n- Be consistent with service/transport configuration.\n\n  For options for which it's meaningful, defaults values for transport options\n  can now be configured on a service. This was previously the case only for an\n  arbitrary subset of options.\n\n  Own Id: OTP-14555\n\n- Add service/transport option avp_dictionaries.\n\n  To provide better support for AVPs that are not defined in the application\n  dictionary: configuring additional dictionaries in an avp_dictionaries tuple\n  allows their AVPs to be encoded/decoded in much the same fashion as\n  application AVPs.\n\n  The motivation is RFC 7683 Diameter Overload, Indicator Conveyance (DOIC),\n  that defines AVPs intended to be piggybacked onto arbitrary messages. A DOIC\n  dictionary has been included in the installation, in module\n  diameter_gen_doic_rfc7683.\n\n  Own Id: OTP-14588\n\n- Decode application AVPs in answers setting the E-bit.\n\n  AVPs defined in the application of the message being sent were previously not\n  decoded, only those in the common application that defines the answer-message\n  grammar.\n\n  Own Id: OTP-14596","ref":"notes.html#improvements-and-new-features-6"},{"type":"extras","title":"diameter 2.0 - Release Notes","doc":"","ref":"notes.html#diameter-2-0"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Let candidate peers be passed to diameter:call/4\n\n  With call option peer, to allow a request to be sent to a peer that hasn't\n  advertised support for the application in question.\n\n  RFC 6733 2.4 requires a node to send the application identifiers of all\n  locally supported applications at capabilities exchange, but not all nodes\n  respect this for the common application, and diameter itself will send\n  D\\[WP]\\[RA] without the common application having been explicitly advertised.\n  Regarding the common application as implicit renders Result-Code 5010\n  (DIAMETER_NO_COMMON_APPLICATION) meaningless however, so allow any request to\n  be sent as long as there is a configured dictionary to support it.\n\n  Own Id: OTP-14338\n\n- Improve performance of message encode/decode and related handling.\n\n  Dictionaries using @custom_types or @codecs will need to adapt the\n  corresponding functions to accept an additional argument that is now passed\n  through encode/decode, which was required to remove various process\n  dictionary-based workarounds that have been used to solve problems in the\n  past.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-14343\n\n- Add transport options to avoid deadlock and allow for load regulation.\n\n  Both diameter_tcp and diameter_sctp now accept two new configuration options:\n  sender and message_cb. The former causes outgoing sends to take place in a\n  dedicated process, to avoid the possibility of deadlock when both the\n  transport process and its peer block in send. The latter allows a callback to\n  control the reading of messages on the socket, to allow for backpressure\n  towards peers when the rate of incoming traffic is greater than can otherwise\n  be handled.\n\n  Neither of these options are yet documented, but are unlikely to change unless\n  problems are discovered. The sender option is not the default since it should\n  probably always be used in combination with message_cb, to prevent incoming\n  requests from being read at a higher rate than a peer allows outgoing answers\n  to be sent.\n\n  Own Id: OTP-14455 Aux Id: ERL-332","ref":"notes.html#improvements-and-new-features-7"},{"type":"extras","title":"diameter 1.12.2 - Release Notes","doc":"","ref":"notes.html#diameter-1-12-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- An improvement in the handling of peer failover in diameter 1.12.1 adversely\n  affected performance when sending requests. Further, the inefficient use of a\n  public table to route incoming answers has been removed.\n\n  Own Id: OTP-14206\n\n- Fixed xml issues in old release notes\n\n  Own Id: OTP-14269","ref":"notes.html#fixed-bugs-and-malfunctions-23"},{"type":"extras","title":"diameter 1.12.1 - Release Notes","doc":"","ref":"notes.html#diameter-1-12-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Close diameter_tcp/sctp listening sockets at diameter:stop_service/1.\n\n  Broken by OTP-13611.\n\n  Own Id: OTP-13787 Aux Id: OTP-13611\n\n- Update build scripts to not make assumptions about where env, cp and perl are\n  located.\n\n  Own Id: OTP-13800","ref":"notes.html#fixed-bugs-and-malfunctions-24"},{"type":"extras","title":"diameter 1.12 - Release Notes","doc":"","ref":"notes.html#diameter-1-12"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Ensure listening socket is closed at transport removal.\n\n  Transport removal did not immediately close a `diameter_tcp/sctp` listening\n  socket, and a subsequent peer connection caused it to remain open.\n\n  Own Id: OTP-13611","ref":"notes.html#fixed-bugs-and-malfunctions-25"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Add `diameter:peer_info/1`.\n\n  That retrieves information in the style of `diameter:service_info/2`, but for\n  a single peer connection.\n\n  Own Id: OTP-13508","ref":"notes.html#improvements-and-new-features-8"},{"type":"extras","title":"diameter 1.11.2 - Release Notes","doc":"","ref":"notes.html#diameter-1-11-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Make peer handling more efficient.\n\n  Inefficient lookup and manipulation of peer lists could result in poor\n  performance when many outgoing requests were sent simultaneously, or when many\n  peers connected simultaneously. Filtering peer lists on realm/host is now also\n  more efficient in many cases.\n\n  Own Id: OTP-13164\n\n- Fix handling of shared peer connections in watchdog state SUSPECT.\n\n  A peer connection shared from a remote node was regarded as being up for the\n  lifetime of the connection, ignoring watchdog transitions into state SUSPECT.\n\n  Own Id: OTP-13342","ref":"notes.html#fixed-bugs-and-malfunctions-26"},{"type":"extras","title":"diameter 1.11.1 - Release Notes","doc":"","ref":"notes.html#diameter-1-11-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix request table leaks\n\n  The End-to-End and Hop-by-Hop identifiers of outgoing Diameter requests are\n  stored in a table in order for the caller to be located when the corresponding\n  answer message is received. Entries were orphaned if the handler was\n  terminated by an exit signal as a consequence of actions taken by callback\n  functions, or if callbacks modified identifiers in retransmission cases.\n\n  Own Id: OTP-13137","ref":"notes.html#fixed-bugs-and-malfunctions-27"},{"type":"extras","title":"diameter 1.11 - Release Notes","doc":"","ref":"notes.html#diameter-1-11"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix relay encode of nested, Grouped AVPs.\n\n  A fault in OTP-12475 caused encode to fail if the first AVP in a Grouped AVP\n  was itself Grouped.\n\n  Own Id: OTP-12879 Aux Id: OTP-12475\n\n- Match acceptable peer addresses case insensitively.\n\n  Regular expressions passed in an 'accept' tuple to diameter_tcp or\n  diameter_sctp inappropriately matched case.\n\n  Own Id: OTP-12902\n\n- Fix diameter_watchdog function clause.\n\n  OTP-12912 introduced an error with accepting transports setting\n  `{restrict_connections, false}`, causing processes to fail when peer\n  connections were terminated.\n\n  Own Id: OTP-12969","ref":"notes.html#fixed-bugs-and-malfunctions-28"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Don't report 5005 (DIAMETER_AVP_MISSING) errors unnecessarily.\n\n  An AVP whose decode failed was reported as missing, despite having been\n  reported with another error as a consequence of the failure.\n\n  Own Id: OTP-12871\n\n- Improve decode performance.\n\n  The time required to decode a message increased quadratically with the number\n  of AVPs in the worst case, leading to extremely long execution times.\n\n  Own Id: OTP-12891\n\n- Improve watchdog and statistics performance.\n\n  Inefficient use of timers contributed to poor performance at high load, as did\n  ordering of the table statistics are written to.\n\n  Own Id: OTP-12912\n\n- Add service_opt() strict_mbit.\n\n  There are differing opinions on whether or not reception of an arbitrary AVP\n  setting the M-bit is an error. The default interpretation is strict: if a\n  command grammar doesn't explicitly allow an AVP setting the M-bit then\n  reception of such an AVP is regarded as an error. Setting\n  `{strict_mbit, false}` disables this check.\n\n  Own Id: OTP-12947","ref":"notes.html#improvements-and-new-features-9"},{"type":"extras","title":"diameter 1.10 - Release Notes","doc":"","ref":"notes.html#diameter-1-10"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix decode of Grouped AVPs containing errors.\n\n  RFC 6733 says this of Failed-AVP in 7.5:\n\n  - \\_\\_\\_\\_ -\n    `In the case where the offending AVP is embedded within a Grouped AVP, the Failed-AVP MAY contain the grouped AVP, which in turn contains the single offending AVP. The same method MAY be employed if the grouped AVP itself is embedded in yet another grouped AVP and so on. In this case, the Failed-AVP MAY contain the grouped AVP hierarchy up to the single offending AVP. This enables the recipient to detect the location of the offending AVP when embedded in a group.`\n\n  It says this of DIAMETER_INVALID_AVP_LENGTH in 7.1.5:\n\n  - \\_\\_\\_\\_ -\n    `The request contained an AVP with an invalid length. A Diameter message indicating this error MUST include the offending AVPs within a Failed-AVP AVP. In cases where the erroneous AVP length value exceeds the message length or is less than the minimum AVP header length, it is sufficient to include the offending AVP header and a zero filled payload of the minimum required length for the payloads data type. If the AVP is a Grouped AVP, the Grouped AVP header with an empty payload would be sufficient to indicate the offending AVP. In the case where the offending AVP header cannot be fully decoded when the AVP length is less than the minimum AVP header length, it is sufficient to include an offending AVP header that is formulated by padding the incomplete AVP header with zero up to the minimum AVP header length.`\n\n  The AVPs placed in the errors field of a diameter_packet record are intended\n  to be appropriate for inclusion in a Failed-AVP, but neither of the above\n  paragraphs has been followed in the Grouped case: the entire faulty AVP\n  (non-faulty components and all) has been included. This made it difficult to\n  identify the actual faulty AVP in all but simple cases.\n\n  The decode is now adapted to the RFC, and implements the suggested single\n  faulty AVP, nested in as many Grouped containers as required.\n\n  Own Id: OTP-12721\n\n- Fix SCTP problems on Solaris.\n\n  The allocation of association ids in Solaris was in conflict with an\n  assumption made in diameter_sctp, resulting in failures when accepting\n  multiple peer connections.\n\n  Own Id: OTP-12768\n\n- Fix start order of alternate transports.\n\n  A transport configured with diameter:add_transport/2 can be passed multiple\n  transport_module/transport_config tuples in order to specify alternate\n  configuration, modules being attempted in order until one succeeds. This is\n  primarily for the connecting case; for example, to allow a transport to be\n  configured to first attempt connection over SCTP, and then TCP in case SCTP\n  fails. Multiple module tuples can be paired with a single config tuple, but in\n  this case the start order was reversed relative to the order in which the\n  modules were specified.\n\n  Own Id: OTP-12851","ref":"notes.html#fixed-bugs-and-malfunctions-29"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Change license text from Erlang Public License to Apache Public License v2.\n\n  Own Id: OTP-12845","ref":"notes.html#improvements-and-new-features-10"},{"type":"extras","title":"diameter 1.9.2 - Release Notes","doc":"","ref":"notes.html#diameter-1-9-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix broken relay counters.\n\n  OTP-12654 in OTP 17.5.3 broke counters in the case of answer messages received\n  in the relay application. Counters were accumulated as unknown messages or\n  no_result_code instead of as relayed messages on the intended Result-Code and\n  'Experimental-Result' tuples.\n\n  Own Id: OTP-12741\n\n- Fix diameter_sctp listener race.\n\n  An oversight in OTP-12428 made it possible to start a transport process that\n  could not establish associations.\n\n  Own Id: OTP-12744","ref":"notes.html#fixed-bugs-and-malfunctions-30"},{"type":"extras","title":"diameter 1.9.1 - Release Notes","doc":"","ref":"notes.html#diameter-1-9-1"},{"type":"extras","title":"Known Bugs and Problems - Release Notes","doc":"- Don't leave extra bit in decoded AVP data.\n\n  OTP-12074 in OTP 17.3 missed one case: a length error on a trailing AVP\n  unknown to the dictionary in question.\n\n  Own Id: OTP-12642\n\n- Don't confuse Result-Code and Experimental-Result.\n\n  The errors field of a decoded diameter_packet record was populated with a\n  Result-Code AVP when an Experimental-Result containing a 3xxx Result-Code was\n  received in an answer not setting the E-bit. The correct AVP is now extracted\n  from the incoming message.\n\n  Own Id: OTP-12654\n\n- Don't count on unknown Application Id.\n\n  OTP-11721 in OTP 17.1 missed the case of an Application Id not agreeing with\n  that of the dictionary in question, causing counters to be accumulated on keys\n  containing the unknown id.\n\n  Own Id: OTP-12701","ref":"notes.html#known-bugs-and-problems"},{"type":"extras","title":"diameter 1.9 - Release Notes","doc":"","ref":"notes.html#diameter-1-9"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Don't discard outgoing answers unnecessarily.\n\n  Answers missing a Result-Code AVP or setting an E-bit inappropriately were\n  discarded even if encode was successful.\n\n  Own Id: OTP-11492\n\n- Increase supervision timeouts.\n\n  At diameter application shutdown, DPR could be omitted on open peer\n  connections because of short supervision timeouts.\n\n  Own Id: OTP-12412\n\n- Fix retransmission of messages sent as header/avps list.\n\n  Extracting End-to-End and Hop-by-Hop Identifiers resulted in a function clause\n  error, resulting in a handle_error callback.\n\n  Own Id: OTP-12415\n\n- Fix diameter_avp decode of Grouped AVPs having decode errors.\n\n  Components of such an AVP were not extracted, causing it to be represented by\n  a single diameter_avp record instead of the intended list.\n\n  Dictionary files must be recompiled for the fix to have effect.\n\n  Own Id: OTP-12475\n\n- Fix ordering of AVPs in relayed messages.\n\n  The order was reversed relative to the received order, with a Route-Record AVP\n  prepended.\n\n  Thanks to Andrzej Trawiński.\n\n  Own Id: OTP-12551\n\n- Fix issues with DiameterURI encode/decode.\n\n  RFC 6773 changed the default port and transport, but the RFC 3588 defaults\n  were used even if the RFC 6733 common dictionary was in use. The RFC 3588\n  defaults are now only used when the common dictionary is\n  diameter_gen_base_rfc3588.\n\n  Both RFC 3588 and 6733 disallow transport=udp;protocol=diameter. Encode of the\n  combination now fails.\n\n  Decode of ports numbers outside the range 0-65535 and fully qualified domain\n  names longer than 255 octets now fails.\n\n  Note that RFC 3588 is obsolete, and that there is a diameter_gen_base_rfc6733.\n  The change in defaults is a potential interoperability problem when moving to\n  RFC 6733 with peers that do not send all URI components. The fact that 6733\n  allows 5xxx result codes in answer messages setting the E-bit, which RFC 3588\n  doesn't, is another.\n\n  Own Id: OTP-12589","ref":"notes.html#fixed-bugs-and-malfunctions-31"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Add service_opt() string_decode.\n\n  To disable the decode of potentially large binaries to string. This prevents\n  large strings from being copied when incoming Diameter messages are passed\n  between processes, a vulnerability that can lead to memory being exhausted\n  given sufficiently malicious peers.\n\n  The value is a boolean(), true being the default for backwards compatibility.\n  Setting false causes both diameter_caps records and decoded messages to\n  contain binary() in relevant places that previously had string():\n  diameter_app(3) callbacks need to be prepared for the change.\n\n  The Diameter types affected are OctetString and the derived types UTF8String,\n  DiameterIdentity, DiameterURI, IPFilterRule, and QoSFilterRule. Time and\n  Address are unaffected.\n\n  Own Id: OTP-11952\n\n- Add transport_opt() pool_size.\n\n  To allow for pools of accepting transport processes, which can better service\n  multiple simultaneous peer connections. The option can also be used with\n  connecting transports, to establish multiple connections to the same peer\n  without having to configure multiple transports.\n\n  Own Id: OTP-12428\n\n- Allow DPR to be sent with diameter:call/4.\n\n  It has been possible to send, but the answer was regarded as unsolicited and\n  discarded. DPA now causes the transport process in question to be terminated,\n  as for DPR that diameter itself sends.\n\n  Own Id: OTP-12542\n\n- Discard requests after DPR.\n\n  RFC 6733 is imprecise, but the tone is that messages received after DPR are an\n  exception to be dealt with only because of the possibility of unordered\n  delivery over SCTP. As a consequence, and because a request following DPR is\n  unlikely to be answered due to the impending loss of the peer connection,\n  discard outgoing requests following an outgoing or incoming DPR. Incoming\n  requests are also discarded, with the exception of DPR itself. Answers are\n  sent and received as usual.\n\n  Own Id: OTP-12543\n\n- Add transport_opt() dpr_timeout.\n\n  To cause a peer connection to be closed following an outgoing DPA when the\n  peer fails to do so. It is the recipient of DPA that should close the\n  connection according to RFC 6733.\n\n  Own Id: OTP-12609\n\n- Add service_opt() incoming_maxlen.\n\n  To bound the expected size of incoming Diameter messages. Messages larger than\n  the specified number of bytes are discarded, to prevent a malicious peer from\n  generating excessive load.\n\n  Own Id: OTP-12628","ref":"notes.html#improvements-and-new-features-11"},{"type":"extras","title":"diameter 1.8 - Release Notes","doc":"","ref":"notes.html#diameter-1-8"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix remote diameter_request table leak.\n\n  An outgoing request whose pick_peer callback selected a transport on another\n  node resulted in an orphaned table entry on that node.\n\n  Own Id: OTP-12196\n\n- Fix handling of 3xxx Result-Code without E-bit.\n\n  OTP-12233 broke the population of the errors field of the diameter_packet\n  record when an incoming request with an E-bit/Result-Code mismatch was\n  detected, causing a 4-tuple to be inserted as Result-Code in a diameter_avp\n  record.\n\n  Own Id: OTP-12233\n\n- Fix ignored connect timer.\n\n  There are two timers governing the establishment of peer connections:\n  connect_timer and watchdog_timer. The former is the RFC 6733 Tc timer, and is\n  used at initial connection establishment. The latter is RFC 3539 TwInit, and\n  is used for connection reestablishment. A connecting transport erroneously\n  used watchdog_timer in both cases.\n\n  Own Id: OTP-12281 Aux Id: seq12728","ref":"notes.html#fixed-bugs-and-malfunctions-32"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Order candidate peers in pick_peer callbacks.\n\n  The order of candidate peers presented to a diameter_app(3) pick_peer callback\n  has previously not been documented, but there are use cases that are\n  simplified by an ordering. The order is now determined by the filter.\n\n  Own Id: OTP-12308","ref":"notes.html#improvements-and-new-features-12"},{"type":"extras","title":"diameter 1.7.1 - Release Notes","doc":"","ref":"notes.html#diameter-1-7-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Don't leave extra bit in decoded AVP data.\n\n  An extra bit could be communicated in the data field of a diameter_avp record\n  in the case of length errors. Of no consequence for code using the record\n  encoding of Diameter messages, but code examining diameter_avp records would\n  see this bit.\n\n  Dictionary files must be recompiled for the fix to have effect.\n\n  Own Id: OTP-12074\n\n- Fix counting of outgoing requests and answers setting the E-bit.\n\n  OTP-11721 broke these counters for all outgoing requests except DWR, and\n  caused answers setting the E-bit to be counted as unknown messages.\n\n  Own Id: OTP-12080\n\n- Fix Failed-AVP decode.\n\n  The best-effort decode only worked for AVPs in the common dictionary, not for\n  those in the dictionary of the application identified in the Diameter Header\n  of the answer message in question.\n\n  Failed-AVP in an answer decoded with the RFC 3588 common dictionary\n  (diameter_gen_base_rfc3588) was regarded as an error. The RFC 6733 dictionary\n  was unaffected.\n\n  Dictionary files must be recompiled for the fix to have effect.\n\n  Own Id: OTP-12094","ref":"notes.html#fixed-bugs-and-malfunctions-33"},{"type":"extras","title":"diameter 1.7 - Release Notes","doc":"","ref":"notes.html#diameter-1-7"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Improve robustness.\n\n  Counters returned by diameter:service_info/2 now only count messages known to\n  the dictionary in question, so that an attacker cannot cause arbitrarily many\n  counters to be created.\n\n  Messages to the Erlang log have been minimized, and those related to traffic\n  have been removed entirely since an attacker could cause a node to be logged\n  to death. Consequently, the default answer_errors configuration has been\n  changed from report to discard. A service needs to be restarted for the change\n  in default to take effect.\n\n  Own Id: OTP-11721\n\n- Fix request table leak.\n\n  Outgoing Diameter requests are stored in a table until an answer is received\n  or times out. Calling diameter:stop_service/1 before this took place would\n  orphan the entries, resulting in a memory leak.\n\n  Own Id: OTP-11893\n\n- Fix broken SCTP transport.\n\n  OTP-11593 caused the sending of answer messages over SCTP to fail.\n\n  Own Id: OTP-11901 Aux Id: OTP-11593\n\n- Fix watchdog process leak.\n\n  A failed capabilities exchange on a listening transport would orphan a\n  process, causing a memory leak.\n\n  Own Id: OTP-11934\n\n- Fix incorrect handling of incoming DPR.\n\n  In the case of a listening transport, a reconnection by a peer following DPR\n  could transition the watchdog state to REOPEN instead of OKAY.\n\n  Own Id: OTP-11938\n\n- Fix handling of AVP length errors on unknown AVPs.\n\n  An AVP (Header) length that pointed past the end of the message was not\n  flagged as a 5014 error in this case. Moreover, encoding such an AVP in the\n  Failed-AVP of an answer message as a consequence of other errors (eg. M-bit,\n  resulting in 5001) failed if the AVP contained a complete header.\n\n  Dictionary files must be recompiled for the fix to have effect.\n\n  Own Id: OTP-11946\n\n- Fix broken check in dictionary compilation.\n\n  That an AVP specified in the content of a @codecs or @custom_types section was\n  undefined went undetected, causing compilation to fail when attempting to\n  lookup the AVP's type.\n\n  Own Id: OTP-11958","ref":"notes.html#fixed-bugs-and-malfunctions-34"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Add result code counters for CEA, DWA, and DPA.\n\n  In addition to the existing result code counters on other answer messages.\n\n  Own Id: OTP-11891\n\n- Add best-effort decode of AVPs within Failed-AVP.\n\n  OTP-11007 disabled the decode of AVPs in Failed-AVP since errors could cause\n  the decode of Failed-AVP itself to fail. Component AVPs are now decoded if\n  possible, otherwise not. AVPs of type Grouped are decoded as much as possible,\n  as deeply as possible.\n\n  Dictionary files must be recompiled for the fix to have effect.\n\n  Own Id: OTP-11936 Aux Id: OTP-11007\n\n- Add counters for encode errors in outgoing Diameter messages.\n\n  In addition to the existing counters on decode errors. The latter now count\n  independently of result codes in answer messages since decode errors do not\n  preclude the presence of a result code.\n\n  Own Id: OTP-11937","ref":"notes.html#improvements-and-new-features-13"},{"type":"extras","title":"diameter 1.6 - Release Notes","doc":"","ref":"notes.html#diameter-1-6"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Add missing check at dictionary compilation.\n\n  In particular, that an AVP defined as having type Grouped in an @avp_types\n  section has a corresponding definition in a @grouped section.\n\n  Own Id: OTP-11561\n\n- Correct documentation on the setting of Origin-State-Id\n\n  It was incorrectly stated that the AVP would be set in an outgoing DPR/DPA.\n\n  Own Id: OTP-11583\n\n- Change interface for communicating outbound stream id to diameter_sctp\n\n  The module uses the transport_data field of record diameter_packet to\n  communicate the stream on which the an incoming message is received and on\n  which an outgoing message should be sent, the previous interface being that\n  both are communicated as a tuple of the form \\{stream, Id\\}. However, since\n  diameter retains the value of an incoming request's transport_data unless the\n  corresponding answer message specifies otherwise, the behaviour in this case\n  is to send an answer on the outbound stream with the same identifier as the\n  that of the inbound stream on which the request was received. If the inbound\n  stream id is greater than or equal to the number of outbound streams then this\n  is guaranteed to fail, causing the transport process in question to terminate.\n  There is no relationship between inbound and outbound stream identifiers so\n  diameter_sctp's imposition of one is simply wrong.\n\n  Outbound stream ids are now communicated with a different tuple: \\{outstream,\n  Id\\}, interpreted modulo the number of outbound streams. Thus, retention of an\n  inbound request's transport_data has no effect on the selection of an outbound\n  stream.\n\n  The change in interface is not strictly backwards compatible because of the\n  new atom for the outbound stream. However, as there is currently no documented\n  way of obtaining the available number of outbound streams for a peer\n  connection, there is no way for a client to have known the range of ids from\n  which it could reliably have chosen with the previous interface, so any\n  setting of the outbound stream has probably been unintentional. Not explicitly\n  specifying an outbound stream now results in a round-robin selection.\n\n  Thanks to Sharmila Pillai for reporting the problem.\n\n  \\*** POTENTIAL INCOMPATIBILITY \\***\n\n  Own Id: OTP-11593\n\n- Fix unicode path failure in diameter_make:codec/2.\n\n  A dictionary path containing a unicode codepoint > 255 caused the function to\n  fail.\n\n  Own Id: OTP-11655\n\n- Fix 'accept' config to diameter_sctp.\n\n  OTP-10893 added support for \\{accept, Match\\} tuples to specify addresses or\n  regexps that should be matched against peer addresses to decide whether or not\n  a newly established association should be retained, but this hasn't been\n  functional in the SCTP case because of missing support in inet(3).\n\n  The display of both local and peer addresses in diameter:service_info/2 output\n  has also been corrected.\n\n  Own Id: OTP-11661 Aux Id: OTP-10229\n\n- Be lenient with the M-bit in Grouped AVPs.\n\n  RFC 6733 says this, in 4.4:\n\n  - \\_\\_\\_\\_ -\n    `Receivers of a Grouped AVP that does not have the 'M' (mandatory) bit set and one or more of the encapsulated AVPs within the group has the 'M' (mandatory) bit set MAY simply be ignored if the Grouped AVP itself is unrecognized. The rule applies even if the encapsulated AVP with its 'M' (mandatory) bit set is further encapsulated within other sub-groups, i.e., other Grouped AVPs embedded within the Grouped AVP.`\n\n  The first sentence is mangled but take it to mean this:\n\n  - \\_\\_\\_\\_ -\n    `An unrecognized AVP of type Grouped that does not set the 'M' bit MAY be ignored even if one of its encapsulated AVPs sets the 'M' bit.`\n\n  This is a bit of a non-statement since if the AVP is unrecognized then its\n  type is unknown. We therefore don't know that its data bytes contain\n  encapsulated AVPs, so can't but ignore any of those that set the M-bit. Doing\n  anything else when the type _is_ known would be inconsistent.\n\n  OTP-11087 (R16B03) caused the M-bit on any unrecognized AVP to be regarded as\n  an error, unrecognized being taken to mean \"not explicitly defined as a member\n  of its container\". (That is, an AVP that can't be packed into a dedicated\n  record field, which is slightly stronger than \"not defined\".) This fixed the\n  original intention for top-level AVPs but broke the required leniency for\n  Grouped AVPs whose type is known. This leniency is now restored.\n\n  Note that dictionary files need to be recompiled for the change to have\n  effect.\n\n  Thanks to Rory McKeown for reporting the problem.\n\n  Own Id: OTP-11675 Aux Id: OTP-11087\n\n- Fix pick_peer case clause failure.\n\n  In the case of \\{call_mutates_state, true\\} configuration on the service in\n  question, any peer selection that failed to select a peer resulted in a case\n  clause failure. This was noticed in the case of a peer failover in which an\n  alternate peer wasn't available.\n\n  Own Id: OTP-11789","ref":"notes.html#fixed-bugs-and-malfunctions-35"},{"type":"extras","title":"diameter 1.5 - Release Notes","doc":"","ref":"notes.html#diameter-1-5"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Rename reconnect_timer to connect_timer.\n\n  The former is still accepted for backwards compatibility, but the name is\n  misleading given the semantics of the timer.\n\n  Own Id: OTP-11168\n\n- Extend diameter_make(3).\n\n  Dictionaries can now be compiled from strings, not just filesystem paths, and\n  results can be returned instead of written to the filesystem.\n\n  Own Id: OTP-11348\n\n- Remove hardcoding of diameter_base as @prefix on dictionaries for application\n  id 0.\n\n  Own Id: OTP-11361","ref":"notes.html#improvements-and-new-features-14"},{"type":"extras","title":"diameter 1.4.4 - Release Notes","doc":"","ref":"notes.html#diameter-1-4-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix setting of End-to-End and Hop-by-Hop Identifiers in outgoing DWA.\n\n  Broken by OTP-11184, which caused the identifiers to be set anew, discarding\n  the values from the incoming DWR.\n\n  Own Id: OTP-11367\n\n- Fix handling of 5014, DIAMETER_INVALID_AVP_LENGTH.\n\n  The error was detected as 5004, DIAMETER_INVALID_AVP_VALUE, for some Diameter\n  types, in which case an AVP length that pointed past the end of a message\n  resulted in encode failure.\n\n  Own Id: OTP-11395","ref":"notes.html#fixed-bugs-and-malfunctions-36"},{"type":"extras","title":"diameter 1.4.3 - Release Notes","doc":"","ref":"notes.html#diameter-1-4-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix UTF8String encode.\n\n  Encode now accepts any nested list of codepoints and binaries. A list\n  containing a binary was previously misinterpreted and the documentation was\n  incomplete.\n\n  Own Id: OTP-11172\n\n- Ensure DWR isn't sent immediately after DWA.\n\n  This was possible if the timing was unfortunate. An incoming DWR now properly\n  resets the watchdog timer.\n\n  Own Id: OTP-11184\n\n- Fix faulty encode of Failed-AVP\n\n  Reception of a CER, DWR or DPR that has decode failures caused encode of the\n  corresponding answer message to fail.\n\n  Own Id: OTP-11293\n\n- Fix broken service_opt() spawn_opt.\n\n  The option was ignored.\n\n  Own Id: OTP-11299","ref":"notes.html#fixed-bugs-and-malfunctions-37"},{"type":"extras","title":"diameter 1.4.2 - Release Notes","doc":"","ref":"notes.html#diameter-1-4-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix handling of 5014 (INVALID_AVP_LENGTH) errors.\n\n  This was in some cases reported as 3009 (INVALID_AVP_BITS).\n\n  Note that the correction is partially implemented in modules generated by\n  diameterc(1): a dictionary file must be recompiled for the correction to apply\n  to any messages it defines.\n\n  Own Id: OTP-11007\n\n- Fix faulty capitalization in release notes.\n\n  Diameter = the protocol.  \n  diameter = the Erlang application.\n\n  Own Id: OTP-11014\n\n- Fix watchdog memory leak.\n\n  Entries were not removed from a service-specific ets table, causing them to be\n  orphaned at connection reestablishment for listening transports, and\n  diameter:remove_transport/2 for both listening and connecting transports.\n\n  The fault was introduced by OTP-10692 in diameter-1.4.1 (R16B).\n\n  Own Id: OTP-11019 Aux Id: OTP-10692\n\n- Fix decode failure on AVP Length < 8.\n\n  The failure caused the message in question to be discarded.\n\n  Own Id: OTP-11026\n\n- Respect Host-IP-Address configuration.\n\n  Addresses returned from a transport module were always used to populate\n  Host-IP-Address AVP's in an outgoing CER/CEA, which precluded the sending of a\n  VIP address. Transport addresses are now only used if Host-IP-Address is\n  unspecified.\n\n  Own Id: OTP-11045\n\n- Fix mkdir race.\n\n  Install could fail if examples/code and examples/dict were created in\n  parallel. Noticed on FreeBSD.\n\n  Own Id: OTP-11051\n\n- Fix recognition of 5001 on mandatory AVP's.\n\n  An AVP setting the M-bit was not regarded as erroneous if it was defined in\n  the dictionary in question and its container (message or Grouped AVP) had an\n  'AVP' field. It's now regarded as a 5001 error (AVP_UNSUPPORTED), as in the\n  case that the AVP is not defined.\n\n  Note that the correction is partially implemented in modules generated by\n  diameterc(1): a dictionary file must be recompiled for the correction to apply\n  to any messages it defines.\n\n  Own Id: OTP-11087\n\n- Fix setting of Failed-AVP on handle_request \\{answer_message, 5xxx\\} return.\n\n  Failed-AVP was never in the outgoing answer-message. It is now set with the\n  AVP from the first entry with the specified Result-Code in the errors field of\n  the incoming diameter_packet, if found.\n\n  Own Id: OTP-11092\n\n- Fix watchdog function_clause\n\n  A listening transport on a service that allowed multiple connections to the\n  same peer could result in a function_clause error in module diameter_watchdog.\n  The resulting crash was harmless but unseemly.\n\n  Thanks to Aleksander Nycz.\n\n  Own Id: OTP-11115\n\n- Fix population of Failed-AVP.\n\n  In cases in which diameter populated this AVP, many values were sent instead\n  of one as suggested by RFC 6733. This was partially corrected by OTP-11007.\n\n  Own Id: OTP-11127 Aux Id: OTP-11007\n\n- Fix list-valued Vendor-Specific-Application-Id config\n\n  R16B (specifically, OTP-10760) broke the handling of such configuration,\n  resulting in a function clause error if the list was not of length 3, and\n  faulty interpretation of the list's contents otherwise. Only record-valued\n  configuration was properly interpreted.\n\n  Own Id: OTP-11165","ref":"notes.html#fixed-bugs-and-malfunctions-38"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Allow peer connections to be shared between Erlang nodes for the purpose of\n  sending outgoing requests.\n\n  A diameter_app(3) pick_peer/4 callback gets a list of remote candidates as\n  argument, allowing a callback on one node to select a transport connection\n  established on another node. The service_opt() share_peers controls the extent\n  to which local connections are shared with remote nodes. The service_opt()\n  use_shared_peers controls the extent to which connections shared from remote\n  nodes are utilized on the local node.\n\n  Own Id: OTP-9610\n\n- Allow listening diameter\\_\\{tcp,sctp\\} transports to be configured with remote\n  addresses.\n\n  Option 'accept' allows remote addresses to be configured as tuples or regular\n  expressions. Remote addresses are matched against the configured values at\n  connection establishment, any non-matching address causing the connection to\n  be aborted.\n\n  Own Id: OTP-10893\n\n- Detect more transport_opt() configuration errors at diameter:add_transport/2.\n\n  Many errors would previously not be detected until transport start,\n  diameter:add_transport/2 returning 'ok' but transport connections failing to\n  be established. An error tuple is now returned.\n\n  Own Id: OTP-10972\n\n- Make explicit local address configuration optional in diameter_tcp:start/3.\n\n  The default address (as determined by gen_tcp) is now used when a local\n  address is not explicitly configured.\n\n  Own Id: OTP-10986\n\n- Improve handling of unrecognized service options.\n\n  Such options were silently ignored by diameter:start_service/2. An error tuple\n  is now returned.\n\n  Own Id: OTP-11017\n\n- Don't send default Inband-Security-Id in CER/CEA.\n\n  RFC 6733 recommends against the use of Inband-Security-Id. Only send a value\n  that differs from the default, NO_INBAND_SECURITY = 0.\n\n  Own Id: OTP-11050\n\n- Make spawn options for request processes configurable.\n\n  Own Id: OTP-11060","ref":"notes.html#improvements-and-new-features-15"},{"type":"extras","title":"diameter 1.4.1.1 - Release Notes","doc":"","ref":"notes.html#diameter-1-4-1-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix broken Vendor-Specific-Application-Id configuration.\n\n  RFC 6733 changed the definition of this Grouped AVP, changing the arity of\n  Vendor-Id from 1\\* to 1. A component Vendor-Id can now be either list- or\n  integer-valued in service and transport configuration, allowing it to be used\n  with both RFC 3588 and RFC 6733 dictionaries.\n\n  Own Id: OTP-10942","ref":"notes.html#fixed-bugs-and-malfunctions-39"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Add transport_opt() watchdog_config to allow non-standard behaviour of the\n  watchdog state machine.\n\n  This can be useful during test but should not be used on nodes that must\n  conform to RFC 3539.\n\n  Own Id: OTP-10898","ref":"notes.html#improvements-and-new-features-16"},{"type":"extras","title":"diameter 1.4.1 - Release Notes","doc":"","ref":"notes.html#diameter-1-4-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix erroneous watchdog transition from DOWN to INITIAL.\n\n  This transition took place when a peer connection was reestablished following\n  a failed capabilities exchange. RFC 3539 requires DOWN to transition into\n  REOPEN.\n\n  Own Id: OTP-10692","ref":"notes.html#fixed-bugs-and-malfunctions-40"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Add application_opt() request_errors to make the handling of incoming requests\n  containing decode errors configurable.\n\n  The value 'callback' ensures that a handle_request callback takes place for\n  all such requests, the default being for diameter to answer 3xxx series errors\n  itself.\n\n  Own Id: OTP-10686\n\n- Add transport_opt() length_errors.\n\n  The value determines how messages received over the transport interface with\n  an incorrect Message Length are dealt with.\n\n  Own Id: OTP-10687\n\n- Add commentary on RFC 6733 to Standards Compliance chapter of the User's\n  Guide.\n\n  Own Id: OTP-10688\n\n- Allow a 5xxx result code in an answer-message on peer connections using the\n  RFC 6733 common dictionary.\n\n  RFC 6733 allows this while RFC 3588 does not. A handle_request callback can\n  return \\{answer_message, 3000..3999|5000..5999\\} in the simplest case.\n\n  Own Id: OTP-10759\n\n- Add dictionaries for RFC 6733.\n\n  Both the common and accounting dictionaries differ from their RFC 3588\n  counterparts, which is reflected in generated record definitions. Application\n  configuration on a service or transport determines the dictionary that will be\n  used on a given peer connection.\n\n  Own Id: OTP-10760\n\n- Allow a handle_request callback to control diameter's setting of Result-Code\n  and Failed-AVP.\n\n  Setting errors = false in a returned #diameter_packet\\{\\} disables the\n  setting.\n\n  Own Id: OTP-10761","ref":"notes.html#improvements-and-new-features-17"},{"type":"extras","title":"diameter 1.4 - Release Notes","doc":"","ref":"notes.html#diameter-1-4"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Add registered server names to the app file.\n\n  Own Id: OTP-10442\n\n- Fix #diameter_header\\{\\} handling broken by OTP-10445.\n\n  The fault caused the the header of a \\[Header | Avps] request to be ignored if\n  both end_to_end_id and hop_by_hop_id were undefined.\n\n  Own Id: OTP-10609\n\n- Fix error handling for handle_request callback.\n\n  A callback that returned a #diameter_packet\\{\\} would fail if the incoming\n  request had decode errors.\n\n  Own Id: OTP-10614\n\n- Fix timing of service start event.\n\n  The event did not necessarily precede other events as documented.\n\n  Own Id: OTP-10618\n\n- Fix setting of header T flag at peer failover.\n\n  The flag is now set in the diameter_header record passed to a\n  prepare_retransmit callback.\n\n  Own Id: OTP-10619\n\n- Fix sending of CER/CEA timeout event at capx_timeout.\n\n  The event was not sent as documented.\n\n  Own Id: OTP-10628\n\n- Fix improper setting of Application-ID in the Diameter header of an answer\n  message whose E flag is set.\n\n  The value should be that of the request in question. The fault caused it\n  always to be 0.\n\n  Own Id: OTP-10655\n\n- Fix faulty handling of AVP length errors.\n\n  An incorrect AVP length but no other errors caused an incoming request to\n  fail.\n\n  Own Id: OTP-10693","ref":"notes.html#fixed-bugs-and-malfunctions-41"},{"type":"extras","title":"diameter 1.3.1 - Release Notes","doc":"","ref":"notes.html#diameter-1-3-1"},{"type":"extras","title":"Known Bugs and Problems - Release Notes","doc":"- Fix function clause resulting from use of an eval callback.\n\n  Own Id: OTP-10685","ref":"notes.html#known-bugs-and-problems-1"},{"type":"extras","title":"diameter 1.3 - Release Notes","doc":"","ref":"notes.html#diameter-1-3"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix faulty handling of Origin-State-Id and faulty config values.\n\n  The former was expected in a list despite the documentation requiring\n  (correctly) an integer. A bare value for a list-valued capability was not\n  handled.\n\n  Own Id: OTP-10440\n\n- Fix timing of up/down events.\n\n  Previously, a call to diameter:call/4 following a peer_up callback might\n  incorrectly return \\{error, no_connection\\}, depending on timing. Both events\n  now follow the corresponding callbacks.\n\n  Own Id: OTP-10459\n\n- Make diameter:service_info/2 usable in peer_up, peer_down and pick_peer\n  callbacks.\n\n  Except for in pick_peer when \\{call_mutates_state, false\\}, it would\n  previously hang indefinitely.\n\n  Own Id: OTP-10460\n\n- Verify that End-to-End and Hop-by-Hop Identifiers in an incoming CEA/DPA match\n  those sent in the corresponding CER/DPR.\n\n  The values were previously ignored. Answers whose identifiers do not match are\n  handled as unexpected.\n\n  Own Id: OTP-10565\n\n- Fix formatting problems in PDF documentation.\n\n  In particular, text corresponding to links in HTML was omitted in preformatted\n  blocks. There are still issues with indentation but this is not\n  diameter-specific.\n\n  Own Id: OTP-10583","ref":"notes.html#fixed-bugs-and-malfunctions-42"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Let prepare_request, prepare_retransmit and handle_request callbacks return a\n  function to be invoked on outgoing messages after encode.\n\n  This allows encoded messages to be logged for example.\n\n  Own Id: OTP-10441\n\n- Add service_opt() 'restrict_connections' to allow multiple transport\n  connections with the same peer.\n\n  Own Id: OTP-10443\n\n- Add service_opt() 'sequence' to allow the masking of a constant onto the\n  topmost bits of End-to-End and Hop-by-Hop identifiers.\n\n  This allows the same service on different nodes to use distinct values in\n  outgoing request messages.\n\n  Own Id: OTP-10445\n\n- Add diameter:service_info(PeerRef) to return the transport_ref() and\n  transport_opt() list of the corresponding transport.\n\n  This allows easy access to these from diameter_app callbacks that only get\n  peer_ref() as an argument.\n\n  Own Id: OTP-10470\n\n- Add reference pages diameter_codec(3) and diameter_make(3).\n\n  Own Id: OTP-10471\n\n- Add events for service start and stop.\n\n  Own Id: OTP-10492\n\n- Add transport_opt() 'disconnect_cb' to make the sending of DPR configurable.\n\n  Whether or not DPR should be sent at application stop, service stop or\n  transport removal is determined by the value returned by the callback, as is\n  the Disconnect-Cause and timeout if DPA is not received.\n\n  Own Id: OTP-10493\n\n- Add transport_opt() 'capx_timeout' for the timeout associated with\n  non-reception of CER/CEA.\n\n  Own Id: OTP-10554\n\n- Allow a handle_request callback to return a #diameter_packet\\{\\}.\n\n  This allows an answer to set transport_data and header fields.\n\n  Own Id: OTP-10566\n\n- Update documentation for RFC 6733.\n\n  RFC 3588 is now obsolete.\n\n  Own Id: OTP-10568","ref":"notes.html#improvements-and-new-features-18"},{"type":"extras","title":"diameter 1.2 - Release Notes","doc":"","ref":"notes.html#diameter-1-2"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix broken Result-Code setting and Destination-Host/Realm extraction.\n\n  Result-Code was assumed to have arity 1 when setting this value in an answer\n  to a request containing AVP decode errors. Destination-Host/Realm were only\n  correctly extracted from messages in the common application.\n\n  Own Id: OTP-10202\n\n- Handle insufficient capabilities configuration more gracefully.\n\n  A transport that does not have sufficient capabilities configuration in order\n  to encode CER/CEA will now emit an error report noting the configuration error\n  and exit instead of failing. The error is not detected at\n  diameter:add_transport/2 since there is no requirement that a service be\n  configured before its transports.\n\n  Own Id: OTP-10203\n\n- Ensure a failing peer_up/down callback does not affect transport connections\n  to other peers.\n\n  Such a failure would previously have taken down all of a service's\n  connections.\n\n  Own Id: OTP-10215","ref":"notes.html#fixed-bugs-and-malfunctions-43"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Statistics related to Diameter messages can be retrieved using\n  diameter:service_info/2.\n\n  Both Diameter and socket-level statistics are available, for both incoming and\n  outgoing messages.\n\n  Own Id: OTP-9608\n\n- Allow multiple transport_module/config to diameter:add_transport/2.\n\n  Multiple values are attempted in sequence until one results in an established\n  connection. This provides a way for a connecting transport to specify\n  configuration in order of preference. (For example, SCTP before TCP.)\n\n  Own Id: OTP-9885\n\n- Add events for state transitions in the RFC 3539 watchdog state machine.\n\n  The watchdog state is also available through diameter:service_info/2.\n\n  Own Id: OTP-10212\n\n- Add diameter:service_info(SvcName, connections).\n\n  This provides an alternative to diameter:service_info(SvcName, transport) that\n  presents information per established connection instead of per transport\n  reference.\n\n  Own Id: OTP-10213\n\n- Assorted documentation corrections/improvements.\n\n  Own Id: OTP-10216","ref":"notes.html#improvements-and-new-features-19"},{"type":"extras","title":"diameter 1.1 - Release Notes","doc":"","ref":"notes.html#diameter-1-1"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix fault in sending of 'closed' events.\n\n  The fault made it possible for the 'closed' event not to be sent following a\n  failed capabilities exchange.\n\n  Own Id: OTP-9824\n\n- Fix faulty diameterc -name/-prefix.\n\n  A minor blunder when introducing the new dictionary parser in diameter-1.0\n  broke these options.\n\n  Own Id: OTP-9826","ref":"notes.html#fixed-bugs-and-malfunctions-44"},{"type":"extras","title":"diameter 1.0 - Release Notes","doc":"","ref":"notes.html#diameter-1-0"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Fix faulty cleanup after diameter:remove_transport/2.\n\n  Removing a transport removed the configuration but did not prevent the\n  transport process from being restarted.\n\n  Own Id: OTP-9756","ref":"notes.html#fixed-bugs-and-malfunctions-45"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- Add support for TLS over TCP.\n\n  RFC 3588 requires that a Diameter server support TLS. In practice this seems\n  to mean TLS over SCTP since there are limitations with running over SCTP: see\n  RFC 6083 (DTLS over SCTP), which is a response to RFC 3436 (TLS over SCTP).\n  The current RFC 3588 draft acknowledges this by equating TLS with TLS/TCP and\n  DTLS/SCTP.\n\n  TLS handshaking can take place either following a CER/CEA that negotiates TLS\n  using the Inband-Security-Id AVP (the method documented in RFC 3588) or\n  immediately following connection establishment (the method added to the\n  current draft).\n\n  Own Id: OTP-9605\n\n- Improvements to the dictionary parser.\n\n  The dictionary parser now emits useful error messages in case of faults in the\n  input file, also identifying the line number at which the fault was detected.\n  There are semantic checks that were missing in the previous parser, a fault in\n  the interpretation of vendor id's in combination with @inherits has been fixed\n  and @end can be used to terminate parsing explicitly instead of always parsing\n  to end of file.\n\n  Own Id: OTP-9639\n\n- Improve dictionary reusability.\n\n  Reusing a dictionary just to get a different generated module name or prefix\n  previously required taking a copy of the source, which may consist of several\n  files if inheritance is used, just to edit a couple of lines which don't\n  affect the semantics of the Diameter application being defined. Options\n  --name, --prefix and --inherits have been added to diameterc to allow\n  corresponding values to be set at compile time.\n\n  Own Id: OTP-9641\n\n- Add capabilities_cb transport option.\n\n  Its value is a function that's applied to the transport reference and\n  capabilities record after capabilities exchange. If a callback returns\n  anything but 'ok' then the connection is closed. In the case of an incoming\n  CER, the callback can return a result code with which to answer. Multiple\n  callbacks can be specified and are applied until either all return 'ok' or one\n  doesn't.\n\n  This provides a way to reject a peer connection.\n\n  Own Id: OTP-9654\n\n- Add @codecs to dictionary format.\n\n  The semantics are similar to @custom_types but results in codec functions of\n  the form TypeName(encode|decode, AvpName, Data) rather than\n  AvpName(encode|decode, TypeName, Data). That is, the role of the AVP name and\n  Diameter type name are reversed. This eliminates the need for exporting one\n  function for each AVP sharing a common specialized encode/decode.\n\n  Own Id: OTP-9708 Aux Id: OTP-9639\n\n- Add #diameter_callback\\{\\} for more flexible callback configuration.\n\n  The record allows individual functions to be configured for each of the\n  diameter_app(3) callbacks, as well as a default callback.\n\n  Own Id: OTP-9777","ref":"notes.html#improvements-and-new-features-20"},{"type":"extras","title":"diameter 0.10 - Release Notes","doc":"","ref":"notes.html#diameter-0-10"},{"type":"extras","title":"Fixed Bugs and Malfunctions - Release Notes","doc":"- Handle #sctp_paddr_change and #sctp_pdapi_event from gen_sctp.\n\n  The events are enabled by default but diameter_sctp neither disabled nor dealt\n  with them. Reception of such an event caused a transport process to crash.\n\n  Own Id: OTP-9538\n\n- Fix header folding bug.\n\n  A prepare_request callback from diameter can return a diameter_header record\n  in order to set values in the header of an outgoing request. A fault in\n  diameter_lib:fold_tuple/3 caused the subsequent encode of the outgoing request\n  to fail.\n\n  Own Id: OTP-9577\n\n- Fix bugs in sending of answer-message replies.\n\n  3001 (DIAMETER_COMMAND_UNSUPPORTED) was not sent since the decode placed the\n  AVP list in the wrong field of the diameter_packet, causing the subsequent\n  encode to fail. Session-Id was also set improperly, causing encode to fail\n  even in this case.\n\n  Own Id: OTP-9578\n\n- Fix improper use of error_logger:info_report/2.\n\n  Function doesn't take a format string and arguments as it was called. Instead\n  use error_logger:info_report/1 and use the same report format as used for\n  warning and error reports.\n\n  Own Id: OTP-9579\n\n- Fix and clarify semantics of peer filters.\n\n  An eval filter returning a non-true value caused the call process to fail and\n  the doc was vague on how an exception was treated. Clarify that the non-tuple\n  host/realm filters assume messages of a certain form.\n\n  Own Id: OTP-9580\n\n- Fix and clarify relay behaviour.\n\n  Implicit filtering of the sending peer in relaying a request could cause loop\n  detection to be preempted in a manner not specified by RFC3588. Reply with\n  3002 (DIAMETER_UNABLE_TO_DELIVER) on anything but an answer to a relayed\n  request.\n\n  Own Id: OTP-9583","ref":"notes.html#fixed-bugs-and-malfunctions-46"},{"type":"extras","title":"Improvements and New Features - Release Notes","doc":"- @id required in dictionary files only when @messages is specified.\n\n  @id defines an application identifier and this is used only when sending or\n  receiving messages. A dictionary can define only AVP's however, to be included\n  by other dictionaries using @inherits, in which case it makes no sense to\n  require @id.\n\n  Note that message definitions are not inherited with @inherits, only AVP's\n\n  Own Id: OTP-9467\n\n- Allow @enum when AVP is defined in an inherited dictionary.\n\n  3GPP standards (for one) extend the values allowed for RFC 3588 AVP's of type\n  Enumerated. Previously, extending an AVP was only possible by completely\n  redefining the AVP.\n\n  Own Id: OTP-9469\n\n- Migrate testsuites to pure common test and add both suites and testcases.\n\n  Own Id: OTP-9553\n\n- Requests of arbitrary form.\n\n  diameter:call/4 can be passed anything, as long as the subsequent\n  prepare_request callback returns a term that can be encoded.\n\n  Own Id: OTP-9581","ref":"notes.html#improvements-and-new-features-21"},{"type":"extras","title":"diameter 0.9 - Release Notes","doc":"Initial release of the diameter application.\n\nKnown issues or limitations:\n\n- Some agent-related functionality is not entirely complete. In particular,\n  support for proxy agents, that advertise specific Diameter applications but\n  otherwise relay messages in much the same way as relay agents (for which a\n  handle_request callback can return a `relay` tuple), will be completed in an\n  upcoming release. There may also be more explicit support for redirect agents,\n  although redirect behaviour can be implemented with the current functionality.\n- There is some asymmetry in the treatment of messages sent as\n  `diameter_header/avp` records and those sent in the \"normal\" fashion, and not\n  all of this is documented. This is related to the previous point since this\n  form of sending a message was introduced specifically to handle relay agent\n  behaviour using the same callback interface as for client/server behaviour.\n- The User's Guide is currently quite thin. The introductory chapter followed by\n  the examples (in the application `examples` subdirectory) may be sufficient\n  for those having some familiarity with the Diameter protocol but the intention\n  is to provide more introductory text. The reference documentation is quite\n  complete, although some points could likely be expanded upon.\n- The function diameter:service_info/2 can be used to retrieve information about\n  a started service (statistics, information about connected peers, etc) but\n  this is not yet documented and both the input and output may change in the\n  next release.\n\nSee [Standards Compliance](diameter_soc.md) for standards-related issues.","ref":"notes.html#diameter-0-9"},{"type":"extras","title":"Introduction","doc":"<!--\n%CopyrightBegin%\n\nSPDX-License-Identifier: Apache-2.0\n\nCopyright Ericsson AB 2023-2025. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Introduction\n\nThe diameter application is an implementation of the Diameter protocol as\ndefined by RFC 6733. It supports arbitrary Diameter applications by way of a\n_dictionary_ interface that allows messages and AVPs to be defined and input\ninto diameter as configuration. It has support for all roles defined in the RFC:\nclient, server and agent. This chapter provides a short overview of the\napplication.\n\nA Diameter node is implemented by configuring a _service_ and one or more\n_transports_ using the interface module `m:diameter`. The service configuration\ndefines the Diameter applications to be supported by the node and, typically,\nthe capabilities that it should send to remote peers at capabilities exchange\nupon the establishment of transport connections. A transport is configured on a\nservice and provides protocol-specific send/receive functionality by way of a\ntransport interface defined by diameter and implemented by a transport module.\nThe diameter application provides two transport modules: `m:diameter_tcp` and\n`m:diameter_sctp` for transport over TCP (using `m:gen_tcp`) and SCTP (using\n`m:gen_sctp`) respectively. Other transports can be provided by any module that\nimplements diameter's [transport interface](`m:diameter_transport`).\n\nWhile a service typically implements a single Diameter node (as identified by an\nOrigin-Host AVP), transports can themselves be associated with capabilities AVPs\nso that a single service can be used to implement more than one Diameter node.\n\nEach Diameter application defined on a service is configured with a callback\nmodule that implements the [application interface](`m:diameter_app`) through\nwhich diameter communicates the connectivity of remote peers, requests peer\nselection for outgoing requests, and communicates the reception of incoming\nDiameter request and answer messages. An application using diameter implements\nthese application callback modules to provide the functionality of the Diameter\nnode(s) it implements.\n\nEach Diameter application is also configured with a dictionary module that\nprovide encode/decode functionality for outgoing/incoming Diameter messages\nbelonging to the application. A dictionary module is generated from a\n[dictionary file](diameter_dict.md) using the [diameterc](diameterc_cmd.md)\nutility. Dictionaries for the RFC 6733 Diameter Common Messages, Base Accounting\nand Relay applications are provided with the diameter application.","ref":"diameter_intro.html"},{"type":"extras","title":"Usage","doc":"<!--\n%CopyrightBegin%\n\nSPDX-License-Identifier: Apache-2.0\n\nCopyright Ericsson AB 2023-2025. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Usage\n\nTo be written.","ref":"diameter_using.html"},{"type":"extras","title":"Examples","doc":"<!--\n%CopyrightBegin%\n\nSPDX-License-Identifier: Apache-2.0\n\nCopyright Ericsson AB 2023-2025. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Examples\n\nExample code can be found in the diameter application's `examples` subdirectory.","ref":"diameter_examples.html"},{"type":"extras","title":"Standards Compliance","doc":"<!--\n%CopyrightBegin%\n\nSPDX-License-Identifier: Apache-2.0\n\nCopyright Ericsson AB 2023-2025. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# Standards Compliance\n\nThe table below summarizes the diameter application's compliance with RFC 6733.\nSince the diameter application isn't a Diameter node on its own, compliance is\nstrictly the responsibility of the user in many cases, diameter providing the\nmeans for the user to be compliant rather than being compliant on its own.\n\nThe Compliance column notes _C_ (Compliant) if the required functionality is\nimplemented, _PC_ (Partially Compliant) if there are limitations, _NC_ (Not\nCompliant) if functionality is not implemented, or a dash if text is\ninformational or only places requirements that must be met by the user's\nimplementation.\n\nCapitalized _Diameter_ refers to the protocol, lowercase _diameter_ to the\nErlang application.","ref":"diameter_soc.html"},{"type":"extras","title":"RFC 6733 - Diameter Base Protocol - Standards Compliance","doc":"| Section  | Title                                                 | Compliance | Notes                                                                                                                                                                                                                                                                                                                                                                                 |\n| -------- | ----------------------------------------------------- | ---------- | ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------- |\n| 1        | Introduction                                          | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 1\\.1     | Diameter Protocol                                     | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 1\\.1.1   | Description of the Document Set                       | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 1\\.1.2   | Conventions Used in This Document                     | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 1\\.1.3   | Changes from RFC 3588                                 | —          | It is possible to configure a 3588 dictionary in order to get 3588 semantics, where the differ from 6733.                                                                                                                                                                                                                                                                             |\n| 1\\.2     | Terminology                                           | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 1\\.3     | Approach to Extensibility                             | —          | The dictionary interface documented in [diameter_dict(4)](diameter_dict.md) provides extensibility, allowing the user to defined new AVPs, commands, and applications. Ready dictionaries are provided for the RFC 6733 common message, base accounting, and relay applications, as well as for RFC 7683, Diameter Overload Indicator Conveyance.                                     |\n| 1\\.3.1   | Defining New AVP Values                               | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 1\\.3.2   | Creating New AVPs                                     | —          | New AVPs can be defined using the dictionary interface. Both RFC data formats and extensions are supported.                                                                                                                                                                                                                                                                           |\n| 1\\.3.3   | Creating New Commands                                 | —          | New commands can be defined using the dictionary interface.                                                                                                                                                                                                                                                                                                                           |\n| 1\\.3.4   | Creating New Diameter Applications                    | —          | New applications can be defined using the dictionary interface.                                                                                                                                                                                                                                                                                                                       |\n| 2        | Protocol Overview                                     | —          | Session state is the responsibility of the user.   The role of a Diameter node is determined by the user's implementation.                                                                                                                                                                                                                                                            |\n| 2\\.1     | Transport                                             | PC         | Ports are configured by the user: diameter places no restrictions.   The transport interface documented in `m:diameter_transport` allows the user to implement their own methods. Ready support is provided for TCP, TCP/TLS, and SCTP, but not DTLS/SCTP.   Multiple connections to the same peer is possible. ICMP messages are not interpreted.                                    |\n| 2\\.1.1   | SCTP Guidelines                                       | C          | Unordered sending is configurable in `m:diameter_sctp`. There is no special handling of DPR/DPA: since a user that cares about pending answers should wait for them before initiating DPR.   A PPID can be configured with a a gen_sctp sctp_default_send_param option.                                                                                                               |\n| 2\\.2     | Securing Diameter Messages                            | PC         | DTLS is not supported by `m:diameter_sctp`. See also 2.1.                                                                                                                                                                                                                                                                                                                             |\n| 2\\.3     | Diameter Application Compliance                       | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 2\\.4     | Application Identifiers                               | C          | The user configures diameter with the identifiers to send at capabilities exchange, along with corresponding dictionaries defining the messages of the applications.                                                                                                                                                                                                                  |\n| 2\\.5     | Connections vs. Sessions                              | C          | Connections are realized by configuring transport. Sessions are the responsibility of the user.                                                                                                                                                                                                                                                                                       |\n| 2\\.6     | Peer Table                                            | PC         | Routing is implemented by the user in callbacks documented in `m:diameter_app`. A peer table of the documented form is not exposed to the user.                                                                                                                                                                                                                                       |\n| 2\\.7     | Routing Table                                         | PC         | See 2.6. A routing table of the documented form is not exposed to the user.                                                                                                                                                                                                                                                                                                           |\n| 2\\.8     | Role of Diameter Agents                               | C          | Most role-specific behaviour is implemented by the user. How a node advertises itself at capabilities exchange is determined by user configuration.                                                                                                                                                                                                                                   |\n| 2\\.8.1   | Relay Agents                                          | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 2\\.8.2   | Proxy Agents                                          | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 2\\.8.3   | Redirect Agents                                       | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 2\\.8.4   | Translation Agents                                    | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 2\\.9     | Diameter Path Authorization                           | —          | Authorization is the responsibility of the user.                                                                                                                                                                                                                                                                                                                                      |\n| 3        | Diameter Header                                       | C          | Hop-by-Hop and End-to-End Identifiers are set by diameter when sending outgoing requests.                                                                                                                                                                                                                                                                                             |\n| 3\\.1     | Command Codes                                         | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 3\\.2     | Command Code Format Specification                     | C          | Commands are defined as CCF specifications in dictionary files.                                                                                                                                                                                                                                                                                                                       |\n| 3\\.3     | Diameter Command Naming Conventions                   | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 4        | Diameter AVPs                                         | C          | Any required padding is added by diameter when encoding outgoing messages.                                                                                                                                                                                                                                                                                                            |\n| 4\\.1     | AVP Header                                            | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 4\\.1.1   | Optional Header Elements                              | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 4\\.2     | Basic AVP Data Formats                                | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 4\\.3     | Derived AVP Data Formats                              | C          | Arbitrary derived data formats are supported by the dictionary interface.                                                                                                                                                                                                                                                                                                             |\n| 4\\.3.1   | Common Derived AVP Data Formats                       | C          | Beware that RFC 6733 changed the DiameterURI transport/port defaults specified in RFC3588. Relying on the defaults can result in interoperability problems.                                                                                                                                                                                                                           |\n| 4\\.4     | Grouped AVP Values                                    | C          | The M-bit on a component AVP of a Grouped AVP that does not set M is ignored: such AVPs are not regarded as erroneous at decode.   Grouped AVPs are defined as CCF specifications in dictionary files.                                                                                                                                                                                |\n| 4\\.4.1   | Example AVP with a Grouped Data Type                  | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 4\\.5     | Diameter Base Protocol AVPs                           | C          | The base AVPs are defined in the common dictionary provided by diameter. There are common dictionaries for both RFC 3588 and RFC 6733 since the latter made changes to both syntax and semantics.                                                                                                                                                                                     |\n| 5        | Diameter Peers                                        | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.1     | Peer Connections                                      | PC         | A peer's DiameterIdentity is not required when initiating a connection: the identify is received at capabilities exchange, at which time the connection can be rejected if the identity is objectionable.   The number of connections established depends on the user's configuration. Multiple connections per peer is possible.                                                     |\n| 5\\.2     | Diameter Peer Discovery                               | NC         | No form of peer discovery is implemented. The user can implement this independently of diameter if required.                                                                                                                                                                                                                                                                          |\n| 5\\.3     | Capabilities Exchange                                 | C          | All supported applications are sent in CEA. The user can reject an incoming CER or CEA in a configured callback.   Both transport security at connection establishment and negotiated via an Inband-Security AVP are supported.                                                                                                                                                       |\n| 5\\.3.1   | Capabilities-Exchange-Request                         | C          | CER is sent and received by diameter.                                                                                                                                                                                                                                                                                                                                                 |\n| 5\\.3.2   | Capabilities-Exchange-Answer                          | C          | CEA is sent and received by diameter.                                                                                                                                                                                                                                                                                                                                                 |\n| 5\\.3.3   | Vendor-Id AVP                                         | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.3.4   | Firmware-Revision AVP                                 | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.3.5   | Host-IP-Address AVP                                   | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.3.6   | Supported-Vendor-Id AVP                               | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.3.7   | Product-Name AVP                                      | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.4     | Disconnecting Peer Connections                        | C          | DPA will not be answered with error: a peer that wants to a avoid a race can wait for pending answers before sending DPR.                                                                                                                                                                                                                                                             |\n| 5\\.4.1   | Disconnect-Peer-Request                               | C          | DPR is sent by diameter in response to configuration changes requiring a connection to be broken. The user can also send DPR.                                                                                                                                                                                                                                                         |\n| 5\\.4.2   | Disconnect-Peer-Answer                                | C          | DPR is answered by diameter.                                                                                                                                                                                                                                                                                                                                                          |\n| 5\\.4.3   | Disconnect-Cause AVP                                  | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.5     | Transport Failure Detection                           | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.5.1   | Device-Watchdog-Request                               | C          | DWR is sent and received by diameter. Callbacks notify the user of transitions into and out of the OKAY state.                                                                                                                                                                                                                                                                        |\n| 5\\.5.2   | Device-Watchdog-Answer                                | C          | DWA is sent and received by diameter.                                                                                                                                                                                                                                                                                                                                                 |\n| 5\\.5.3   | Transport Failure Algorithm                           | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.5.4   | Failover and Failback Procedures                      | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.6     | Peer State Machine                                    | PC         | The election process is modified as described in 5.6.4.                                                                                                                                                                                                                                                                                                                               |\n| 5\\.6.1   | Incoming Connections                                  | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.6.2   | Events                                                | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.6.3   | Actions                                               | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 5\\.6.4   | The Election Process                                  | PC         | As documented, the election assumes knowledge of a peer's DiameterIdentity when initiating a connection, which diameter doesn't require. Connections will be accepted if configuration allows multiple connections per peer to be established or there is no existing connection. Note that the election process is only applicable when multiple connections per peer is disallowed. |\n| 6        | Diameter Message Processing                           | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.1     | Diameter Request Routing Overview                     | —          | Routing is performed by the user. A callback from diameter provides a list of available suitable peer connections.                                                                                                                                                                                                                                                                    |\n| 6\\.1.1   | Originating a Request                                 | C          | Requests are constructed by the user; diameter sets header fields as defined in the relevant dictionary.                                                                                                                                                                                                                                                                              |\n| 6\\.1.2   | Sending a Request                                     | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.1.3   | Receiving Requests                                    | C          | Loops are detected by diameter when the return value of a request callback asks that a request be forwarded. Loop detection in other cases is the responsibility of the user.                                                                                                                                                                                                         |\n| 6\\.1.4   | Processing Local Requests                             | C          | The user decides whether or not to process a request locally in the request callback from diameter.                                                                                                                                                                                                                                                                                   |\n| 6\\.1.5   | Request Forwarding                                    | PC         | See 2.6.                                                                                                                                                                                                                                                                                                                                                                              |\n| 6\\.1.6   | Request Routing                                       | PC         | See 2.7.                                                                                                                                                                                                                                                                                                                                                                              |\n| 6\\.1.7   | Predictive Loop Avoidance                             | C          | See 6.1.3.                                                                                                                                                                                                                                                                                                                                                                            |\n| 6\\.1.8   | Redirecting Requests                                  | PC         | See 2.6.                                                                                                                                                                                                                                                                                                                                                                              |\n| 6\\.1.9   | Relaying and Proxying Requests                        | C          | A Route-Record AVP is appended by diameter when the return value of a request callback asks that a request be forwarded. Appending the AVP in other cases is the responsibility of the user.                                                                                                                                                                                          |\n| 6\\.2     | Diameter Answer Processing                            | C          | Answer message are constructed by the user, except in the case of some protocol errors, in which case the procedures are followed.                                                                                                                                                                                                                                                    |\n| 6\\.2.1   | Processing Received Answers                           | C          | Answers with an unknown Hop-by-Hop Identifier are discarded.                                                                                                                                                                                                                                                                                                                          |\n| 6\\.2.2   | Relaying and Proxying Answers                         | —          | Modifying answers is the responsibility of the user in callbacks from diameter.                                                                                                                                                                                                                                                                                                       |\n| 6\\.3     | Origin-Host AVP                                       | C          | The order of AVPs in an encoded message is determined by the CCF of the message in question.   AVPs defined in the RFC are defined in dictionaries provided by diameter. Their proper use in application messages is the responsibility of the user.                                                                                                                                  |\n| 6\\.4     | Origin-Realm AVP                                      | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.5     | Destination-Host AVP                                  | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.6     | Destination-Realm AVP                                 | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.7     | Routing AVPs                                          | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.7.1   | Route-Record AVP                                      | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.7.2   | Proxy-Info AVP                                        | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.7.3   | Proxy-Host AVP                                        | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.7.4   | Proxy-State AVP                                       | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.8     | Auth-Application-Id AVP                               | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.9     | Acct-Application-Id AVP                               | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.10    | Inband-Security-Id AVP                                | C          | See 2.1.                                                                                                                                                                                                                                                                                                                                                                              |\n| 6\\.11    | Vendor-Specific-Application-Id AVP                    | C          | Note that the CCF of this AVP is not the same as in RFC 3588.                                                                                                                                                                                                                                                                                                                         |\n| 6\\.12    | Redirect-Host AVP                                     | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.13    | Redirect-Host-Usage AVP                               | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 6\\.14    | Redirect-Max-Cache-Time AVP                           | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 7        | Error Handling                                        | C          | Answers are formulated by the user in most cases. Answers setting the E-bit can be sent by diameter itself in response to a request that cannot be handled by the user.                                                                                                                                                                                                               |\n| 7\\.1     | Result-Code AVP                                       | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 7\\.1.1   | Informational                                         | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 7\\.1.2   | Success                                               | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 7\\.1.3   | Protocol Errors                                       | C          | Result codes 3001, 3002, 3005, and 3007 can be sent in answers formulated by diameter, if configured to do so.                                                                                                                                                                                                                                                                        |\n| 7\\.1.4   | Transient Failures                                    | C          | Result code 4003 is sent in CEA if there is an existing connection to the peer in question and configuration does not allow more than one.                                                                                                                                                                                                                                            |\n| 7\\.1.5   | Permanent Failures                                    | C          | Message reception detects 5001, 5004, 5005, 5008, 5009, 5010, 5011, 5014, 5015, and 5017 errors. It ignores 5013 errors at the admonition of sections 3 and 4.1.   Note that RFC 3588 did not allow 5xxx result codes in answers setting the E-bit, while RFC 6733 does. This is a potential interoperability problem since the Diameter protocol version has not changed.            |\n| 7\\.2     | Error Bit                                             | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 7\\.3     | Error-Message AVP                                     | C          | The user can include this AVP as required.                                                                                                                                                                                                                                                                                                                                            |\n| 7\\.4     | Error-Reporting-Host AVP                              | C          | The user can include this AVP as required.                                                                                                                                                                                                                                                                                                                                            |\n| 7\\.5     | Failed-AVP AVP                                        | C          | The user constructs application-specific messages, but diameter provides failed AVPs in message callbacks. Failed component AVPs are grouped within the relevant Grouped AVPs.                                                                                                                                                                                                        |\n| 7\\.6     | Experimental-Result AVP                               | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 7\\.7     | Experimental-Result-Code AVP                          | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8        | Diameter User Sessions                                | —          | Authorization and accounting AVPs are defined in provided dictionaries. Their proper use is the responsibility of the user.                                                                                                                                                                                                                                                           |\n| 8\\.1     | Authorization Session State Machine                   | —          | Authorization is the responsibility of the user: diameter does not implement this state machine.                                                                                                                                                                                                                                                                                      |\n| 8\\.2     | Accounting Session State Machine                      | —          | Accounting is the responsibility of the user: diameter does not implement this state machine.                                                                                                                                                                                                                                                                                         |\n| 8\\.3     | Server-Initiated Re-Auth                              | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.3.1   | Re-Auth-Request                                       | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.3.2   | Re-Auth-Answer                                        | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.4     | Session Termination                                   | —          | Session-related messages and AVPs are defined in provided dictionaries. Their proper use is the user's responsibility.                                                                                                                                                                                                                                                                |\n| 8\\.4.1   | Session-Termination-Request                           | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.4.2   | Session-Termination-Answer                            | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.5     | Aborting a Session                                    | —          | Session-related messages and AVPs are defined in provided dictionaries. Their proper use is the user's responsibility.                                                                                                                                                                                                                                                                |\n| 8\\.5.1   | Abort-Session-Request                                 | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.5.2   | Abort-Session-Answer                                  | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.6     | Inferring Session Termination from Origin-State-Id    | —          | Session-related messages and AVPs are defined in provided dictionaries. Their proper use is the user's responsibility.                                                                                                                                                                                                                                                                |\n| 8\\.7     | Auth-Request-Type AVP                                 | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.8     | Session-Id AVP                                        | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.9     | Authorization-Lifetime AVP                            | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.10    | Auth-Grace-Period AVP                                 | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.11    | Auth-Session-State AVP                                | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.12    | Re-Auth-Request-Type AVP                              | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.13    | Session-Timeout AVP                                   | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.14    | User-Name AVP                                         | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.15    | Termination-Cause AVP                                 | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.16    | Origin-State-Id AVP                                   | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.17    | Session-Binding AVP                                   | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.18    | Session-Server-Failover AVP                           | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.19    | Multi-Round-Time-Out AVP                              | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.20    | Class AVP                                             | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 8\\.21    | Event-Timestamp AVP                                   | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9        | Accounting                                            | —          | Accounting-related messages and AVPs are defined in provided dictionaries. Their proper use is the user's responsibility.                                                                                                                                                                                                                                                             |\n| 9\\.1     | Server Directed Model                                 | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.2     | Protocol Messages                                     | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.3     | Accounting Application Extension and Requirements     | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.4     | Fault Resilience                                      | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.5     | Accounting Records                                    | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.6     | Correlation of Accounting Records                     | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.7     | Accounting Command Codes                              | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.7.1   | Accounting-Request                                    | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.7.2   | Accounting-Answer                                     | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.8     | Accounting AVPs                                       | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.8.1   | Accounting-Record-Type AVP                            | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.8.2   | Acct-Interim-Interval AVP                             | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.8.3   | Accounting-Record-Number AVP                          | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.8.4   | Acct-Session-Id AVP                                   | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.8.5   | Acct-Multi-Session-Id AVP                             | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.8.6   | Accounting-Sub-Session-Id AVP                         | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 9\\.8.7   | Accounting-Realtime-Required AVP                      | C          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 10       | AVP Occurrence Tables                                 | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 10\\.1    | Base Protocol Command AVP Table                       | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 10\\.2    | Accounting AVP Table                                  | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11       | IANA Considerations                                   | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.1    | AVP Header                                            | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.1.1  | AVP Codes                                             | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.1.2  | AVP Flags                                             | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.2    | Diameter Header                                       | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.2.1  | Command Codes                                         | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.2.2  | Command Flags                                         |            |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3    | AVP Values                                            | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.1  | Experimental-Result-Code AVP                          | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.2  | Result-Code AVP Values                                | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.3  | Accounting-Record-Type AVP Values                     | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.4  | Termination-Cause AVP Values                          | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.5  | Redirect-Host-Usage AVP Values                        | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.6  | Session-Server-Failover AVP Values                    | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.7  | Session-Binding AVP Values                            | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.8  | Disconnect-Cause AVP Values                           | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.9  | Auth-Request-Type AVP Values                          | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.10 | Auth-Session-State AVP Values                         | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.11 | Re-Auth-Request-Type AVP Values                       | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.12 | Accounting-Realtime-Required AVP Values               | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.3.13 | Inband-Security-Id AVP (code 299)                     | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.4    | \\_diameters Service Name and Port Number Registration | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.5    | SCTP Payload Protocol Identifiers                     | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 11\\.6    | S-NAPTR Parameters                                    | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 12       | Diameter Protocol-Related Configurable Parameters     | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 13       | Security Considerations                               | PC         | See 2.1.   IPsec is transparent to diameter.                                                                                                                                                                                                                                                                                                                                          |\n| 13\\.1    | TLS/TCP and DTLS/SCTP Usage                           | PC         | See 2.1.                                                                                                                                                                                                                                                                                                                                                                              |\n| 13\\.2    | Peer-to-Peer Considerations                           | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 13\\.3    | AVP Considerations                                    | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 14       | References                                            | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 14\\.1    | Normative References                                  | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n| 14\\.2    | Informative References                                | —          |                                                                                                                                                                                                                                                                                                                                                                                       |\n\n_Table: RFC 6733 Compliance_","ref":"diameter_soc.html#rfc-6733-diameter-base-protocol"},{"type":"extras","title":"diameterc","doc":"<!--\n%CopyrightBegin%\n\nSPDX-License-Identifier: Apache-2.0\n\nCopyright Ericsson AB 2023-2025. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# diameterc\n\nCompile a diameter dictionary to Erlang source.","ref":"diameterc_cmd.html"},{"type":"extras","title":"Synopsis - diameterc","doc":"```\ndiameterc [ ]  \n```","ref":"diameterc_cmd.html#synopsis"},{"type":"extras","title":"Description - diameterc","doc":"The diameterc utility is used to compile a diameter\n[dictionary file](diameter_dict.md) into Erlang source. The resulting source\nimplements the interface diameter required to encode and decode the dictionary's\nmessages and AVPs.\n\nThe module `m:diameter_make` provides an alternate compilation interface.\n\n# USAGE\n\nCompile a single dictionary file to Erlang\nsource. Valid options are as follows.\n\n- **\\-i  ** - Prepend the specified directory to the code path. Use to\n  point at beam files compiled from inherited dictionaries,\n  `[@inherits](diameter_dict.md#inherits)` in a dictionary file creating a\n  beam dependency, not an erl/hrl dependency.\n\n  Multiple `-i` options can be specified.\n\n- **\\-o  ** - Write generated source to the specified directory. Defaults\n  to the current working directory.\n\n- **\\-E** - Suppress `.erl` file generation.\n\n- **\\-H** - Suppress `.hrl` file generation.\n\n- **\\--name  ** - Name the output module.\n\n- **\\--prefix  ** - Transform the input dictionary before compilation,\n  setting `[@name](diameter_dict.md#name)` or\n  `[@prefix](diameter_dict.md#prefix)` to the specified string.\n\n- **\\--inherits  ** - Transform the input dictionary before compilation,\n  appending `[@inherits](diameter_dict.md#inherits)` of the specified string.\n\n  Two forms of `--inherits` have special meaning:\n\n  ```text\n  --inherits -\n  --inherits Prev/Mod\n  ```\n\n  The first has the effect of clearing any previous inherits, the second of\n  replacing a previous inherits of `Prev` to one of `Mod`. This allows the\n  semantics of the input dictionary to be changed without modifying the file\n  itself.\n\n  Multiple `--inherits` options can be specified.\n\n# EXIT STATUS\n\nReturns 0 on success, non-zero on failure.\n\n# SEE ALSO\n\n`m:diameter_make`, [diameter_dict(4)](diameter_dict.md)","ref":"diameterc_cmd.html#description"},{"type":"extras","title":"diameter_dict","doc":"<!--\n%CopyrightBegin%\n\nSPDX-License-Identifier: Apache-2.0\n\nCopyright Ericsson AB 2023-2025. All Rights Reserved.\n\nLicensed under the Apache License, Version 2.0 (the \"License\");\nyou may not use this file except in compliance with the License.\nYou may obtain a copy of the License at\n\n    http://www.apache.org/licenses/LICENSE-2.0\n\nUnless required by applicable law or agreed to in writing, software\ndistributed under the License is distributed on an \"AS IS\" BASIS,\nWITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\nSee the License for the specific language governing permissions and\nlimitations under the License.\n\n%CopyrightEnd%\n-->\n# diameter_dict\n\nDictionary interface of the diameter application.","ref":"diameter_dict.html"},{"type":"extras","title":"Description - diameter_dict","doc":"A diameter service, as configured with `diameter:start_service/2`, specifies one\nor more supported Diameter applications. Each Diameter application specifies a\ndictionary module that knows how to encode and decode its messages and AVPs. The\ndictionary module is in turn generated from a file that defines these messages\nand AVPs. The format of such a file is defined in\n[FILE FORMAT](diameter_dict.md#FILE_FORMAT) below. Users add support for their\nspecific applications by creating dictionary files, compiling them to Erlang\nmodules using either [diameterc(1)](diameterc_cmd.md) or `m:diameter_make` and\nconfiguring the resulting dictionaries modules on a service.\n\nDictionary module generation also results in a hrl file that defines records for\nthe messages and Grouped AVPs defined by the dictionary, these records being\nwhat a user of the diameter application sends and receives, modulo other\npossible formats as discussed in `m:diameter_app`. These records and the\nunderlying Erlang data types corresponding to Diameter data formats are\ndiscussed in [MESSAGE RECORDS](diameter_dict.md#MESSAGE_RECORDS) and\n[DATA TYPES](diameter_dict.md#DATA_TYPES) respectively. The generated hrl also\ncontains macro definitions for the possible values of AVPs of type Enumerated.\n\nThe diameter application includes five dictionary modules corresponding to\napplications defined in section 2.4 of RFC 6733: `diameter_gen_base_rfc3588` and\n`diameter_gen_base_rfc6733` for the Diameter Common Messages application with\napplication identifier 0, `diameter_gen_accounting` (for RFC 3588) and\n`diameter_gen_acct_rfc6733` for the Diameter Base Accounting application with\napplication identifier 3 and `diameter_gen_relay` the Relay application with\napplication identifier 0xFFFFFFFF.\n\nThe Common Message and Relay applications are the only applications that\ndiameter itself has any specific knowledge of. The Common Message application is\nused for messages that diameter itself handles: CER/CEA, DWR/DWA and DPR/DPA.\nThe Relay application is given special treatment with regard to encode/decode\nsince the messages and AVPs it handles are not specifically defined.\n\n[](){: #FILE_FORMAT }","ref":"diameter_dict.html#description"},{"type":"extras","title":"FILE FORMAT - diameter_dict","doc":"A dictionary file consists of distinct sections. Each section starts with a tag\nfollowed by zero or more arguments and ends at the the start of the next section\nor end of file. Tags consist of an ampersand character followed by a keyword and\nare separated from their arguments by whitespace. Whitespace separates\nindividual tokens but is otherwise insignificant.\n\nThe tags, their arguments and the contents of each corresponding section are as\nfollows. Each section can occur multiple times unless otherwise specified. The\norder in which sections are specified is unimportant.\n\n- **`@id Number`{: #id }** - Defines the integer Number as the Diameter\n  Application Id of the application in question. Can occur at most once and is\n  required if the dictionary defines `@messages`. The section has empty content.\n\n  The Application Id is set in the Diameter Header of outgoing messages of the\n  application, and the value in the header of an incoming message is used to\n  identify the relevant dictionary module.\n\n  Example:\n\n  ```text\n  @id 16777231\n  ```\n\n- **`@name Mod`{: #name }** - Defines the name of the generated dictionary\n  module. Can occur at most once and defaults to the name of the dictionary file\n  minus any extension. The section has empty content.\n\n  Note that a dictionary module should have a unique name so as not collide with\n  existing modules in the system.\n\n  Example:\n\n  ```text\n  @name etsi_e2\n  ```\n\n- **`@prefix Name`{: #prefix }** - Defines Name as the prefix to be added to\n  record and constant names (followed by a `'_'` character) in the generated\n  dictionary module and hrl. Can occur at most once. The section has empty\n  content.\n\n  A prefix is optional but can be be used to disambiguate between record and\n  constant names resulting from similarly named messages and AVPs in different\n  Diameter applications.\n\n  Example:\n\n  ```text\n  @prefix etsi_e2\n  ```\n\n- **`@vendor Number Name`{: #vendor }** - Defines the integer Number as the the\n  default Vendor-Id of AVPs for which the V flag is set. Name documents the\n  owner of the application but is otherwise unused. Can occur at most once and\n  is required if an AVP sets the V flag and is not otherwise assigned a\n  Vendor-Id. The section has empty content.\n\n  Example:\n\n  ```text\n  @vendor 13019 ETSI\n  ```\n\n- **`@avp_vendor_id Number`{: #avp_vendor_id }** - Defines the integer Number as\n  the Vendor-Id of the AVPs listed in the section content, overriding the\n  `@vendor` default. The section content consists of AVP names.\n\n  Example:\n\n  ```text\n  @avp_vendor_id 2937\n\n  WWW-Auth\n  Domain-Index\n  Region-Set\n  ```\n\n- **`@inherits Mod`{: #inherits }** - Defines the name of a dictionary module\n  containing AVP definitions that should be imported into the current\n  dictionary. The section content consists of the names of those AVPs whose\n  definitions should be imported from the dictionary, an empty list causing all\n  to be imported. Any listed AVPs must not be defined in the current dictionary\n  and it is an error to inherit the same AVP from more than one dictionary.\n\n  Note that an inherited AVP that sets the V flag takes its Vendor-Id from\n  either `@avp_vendor_id` in the inheriting dictionary or `@vendor` in the\n  inherited dictionary. In particular, `@avp_vendor_id` in the inherited\n  dictionary is ignored. Inheriting from a dictionary that specifies the\n  required `@vendor` is equivalent to using `@avp_vendor_id` with a copy of the\n  dictionary's definitions but the former makes for easier reuse.\n\n  All dictionaries should typically inherit RFC 6733 AVPs from\n  `diameter_gen_base_rfc6733`.\n\n  Example:\n\n  ```text\n  @inherits diameter_gen_base_rfc6733\n  ```\n\n- **`@avp_types`{: #avp_types }** - Defines the name, code, type and flags of\n  individual AVPs. The section consists of definitions of the form\n\n  `Name Code Type Flags`\n\n  where Code is the integer AVP code, Type identifies an AVP Data Format as\n  defined in section [DATA TYPES](diameter_dict.md#DATA_TYPES) below, and Flags\n  is a string of V, M and P characters indicating the flags to be set on an\n  outgoing AVP or a single `'-'` (minus) character if none are to be set.\n\n  Example:\n\n  ```text\n  @avp_types\n\n  Location-Information   350  Grouped     MV\n  Requested-Information  353  Enumerated   V\n  ```\n\n  > #### Warning {: .warning }\n  >\n  > The P flag has been deprecated by RFC 6733.\n\n- **`@custom_types Mod`{: #custom_types }** - Specifies AVPs for which module\n  Mod provides encode/decode functions. The section contents consists of AVP\n  names. For each such name, `Mod:Name(encode|decode, Type, Data, Opts)` is\n  expected to provide encode/decode for values of the AVP, where Name is the\n  name of the AVP, Type is it's type as declared in the `@avp_types` section of\n  the dictionary, Data is the value to encode/decode, and Opts is a term that is\n  passed through encode/decode.\n\n  Example:\n\n  ```text\n  @custom_types rfc4005_avps\n\n  Framed-IP-Address\n  ```\n\n- **`@codecs Mod`{: #codecs }** - Like `@custom_types` but requires the\n  specified module to export `Mod:Type(encode|decode, Name, Data, Opts)` rather\n  than `Mod:Name(encode|decode, Type, Data, Opts)`.\n\n  Example:\n\n  ```text\n  @codecs rfc4005_avps\n\n  Framed-IP-Address\n  ```\n\n- **`@messages`{: #messages }** - Defines the messages of the application. The\n  section content consists of definitions of the form specified in section 3.2\n  of RFC 6733, \"Command Code Format Specification\".\n\n  ```text\n  @messages\n\n  RTR ::=  \n           \n          { Auth-Application-Id }\n          { Auth-Session-State }\n          { Origin-Host }\n          { Origin-Realm }\n          { Destination-Host }\n          { SIP-Deregistration-Reason }\n          [ Destination-Realm ]\n          [ User-Name ]\n        * [ SIP-AOR ]\n        * [ Proxy-Info ]\n        * [ Route-Record ]\n        * [ AVP ]\n\n  RTA ::=  \n           \n          { Auth-Application-Id }\n          { Result-Code }\n          { Auth-Session-State }\n          { Origin-Host }\n          { Origin-Realm }\n          [ Authorization-Lifetime ]\n          [ Auth-Grace-Period ]\n          [ Redirect-Host ]\n          [ Redirect-Host-Usage ]\n          [ Redirect-Max-Cache-Time ]\n        * [ Proxy-Info ]\n        * [ Route-Record ]\n        * [ AVP ]\n  ```\n\n- **`@grouped`** - Defines the contents of the AVPs of the\n  application having type Grouped. The section content consists of definitions\n  of the form specified in section 4.4 of RFC 6733, \"Grouped AVP Values\".\n\n  Example:\n\n  ```text\n  @grouped\n\n  SIP-Deregistration-Reason ::=  \n                                { SIP-Reason-Code }\n                                [ SIP-Reason-Info ]\n                              * [ AVP ]\n  ```\n\n  Specifying a Vendor-Id in the definition of a grouped AVP is equivalent to\n  specifying it with `@avp_vendor_id`.\n\n- **`@enum Name`{: #enum }** - Defines values of AVP Name having type\n  Enumerated. Section content consists of names and corresponding integer\n  values. Integer values can be prefixed with 0x to be interpreted as\n  hexadecimal.\n\n  Note that the AVP in question can be defined in an inherited dictionary in\n  order to introduce additional values to an enumeration otherwise defined in\n  another dictionary.\n\n  Example:\n\n  ```text\n  @enum SIP-Reason-Code\n\n  PERMANENT_TERMINATION    0\n  NEW_SIP_SERVER_ASSIGNED  1\n  SIP_SERVER_CHANGE        2\n  REMOVE_SIP_SERVER        3\n  ```\n\n- **`@end`{: #end }** - Causes parsing of the dictionary to terminate: any\n  remaining content is ignored.\n\nComments can be included in a dictionary file using semicolon: characters from a\nsemicolon to end of line are ignored.\n\n[](){: #MESSAGE_RECORDS }","ref":"diameter_dict.html#file-format"},{"type":"extras","title":"MESSAGE RECORDS - diameter_dict","doc":"The hrl generated from a dictionary specification defines records for the\nmessages and grouped AVPs defined in `@messages` and `@grouped` sections. For\neach message or grouped AVP definition, a record is defined whose name is the\nmessage or AVP name, prefixed with any dictionary prefix defined with `@prefix`,\nand whose fields are the names of the AVPs contained in the message or grouped\nAVP in the order specified in the definition in question. For example, the\ngrouped AVP\n\n```text\nSIP-Deregistration-Reason ::=  \n                              { SIP-Reason-Code }\n                              [ SIP-Reason-Info ]\n                            * [ AVP ]\n```\n\nwill result in the following record definition given an empty prefix.\n\n```text\n-record('SIP-Deregistration-Reason', {'SIP-Reason-Code',\n                                      'SIP-Reason-Info',\n                                      'AVP'}).\n```\n\nThe values encoded in the fields of generated records depends on the type and\nnumber of times the AVP can occur. In particular, an AVP which is specified as\noccurring exactly once is encoded as a value of the AVP's type while an AVP with\nany other specification is encoded as a list of values of the AVP's type. The\nAVP's type is as specified in the AVP definition, the RFC 6733 types being\ndescribed below.\n\n[](){: #DATA_TYPES }","ref":"diameter_dict.html#message-records"},{"type":"extras","title":"DATA TYPES - diameter_dict","doc":"The data formats defined in sections 4.2 (\"Basic AVP Data Formats\") and 4.3\n(\"Derived AVP Data Formats\") of RFC 6733 are encoded as values of the types\ndefined here. Values are passed to `diameter:call/4` in a request record when\nsending a request, returned in a resulting answer record and passed to a\n[handle_request/3](`c:diameter_app:handle_request/3`) callback upon reception of\nan incoming request.\n\nIn cases in which there is a choice between string() and binary() types for\nOctetString() and derived types, the representation is determined by the value\nof [diameter:service_opt()](`m:diameter#service_opt`)\n[string_decode](`m:diameter#string_decode`).\n\n_Basic AVP Data Formats_\n\n```erlang\nOctetString() = string() | binary()\nInteger32()   = -2147483647..2147483647\nInteger64()   = -9223372036854775807..9223372036854775807\nUnsigned32()  = 0..4294967295\nUnsigned64()  = 0..18446744073709551615\nFloat32()     = '-infinity' | float() | infinity\nFloat64()     = '-infinity' | float() | infinity\nGrouped()     = record()\n```\n\nOn encode, an OctetString() can be specified as an iolist(), excessively large\nfloats (in absolute value) are equivalent to `infinity` or `'-infinity'` and\nexcessively large integers result in encode failure. The records for grouped\nAVPs are as discussed in the previous section.\n\n_Derived AVP Data Formats_\n\n[](){: #Address }\n\n```text\nAddress() = OctetString()\n          | tuple()\n```\n\nOn encode, an OctetString() IPv4 address is parsed in the usual x.x.x.x format\nwhile an IPv6 address is parsed in any of the formats specified by section 2.2\nof RFC 2373, \"Text Representation of Addresses\". An IPv4 tuple() has length 4\nand contains values of type 0..255. An IPv6 tuple() has length 8 and contains\nvalues of type 0..65535. The tuple representation is used on decode.\n\n[](){: #Time }\n\n```erlang\nTime() = {date(), time()}\n\nwhere\n\n  date() = {Year, Month, Day}\n  time() = {Hour, Minute, Second}\n\n  Year   = integer()\n  Month  = 1..12\n  Day    = 1..31\n  Hour   = 0..23\n  Minute = 0..59\n  Second = 0..59\n```\n\nAdditionally, values that can be encoded are limited by way of their encoding as\nfour octets as required by RFC 6733 with the required extension from RFC 2030.\nIn particular, only values between `{{1968,1,20},{3,14,8}}` and\n`{{2104,2,26},{9,42,23}}` (both inclusive) can be encoded.\n\n[](){: #UTF8String }\n\n```text\nUTF8String() = [integer()] | binary()\n```\n\nList elements are the UTF-8 encodings of the individual characters in the\nstring. Invalid codepoints will result in encode/decode failure. On encode, a\nUTF8String() can be specified as a binary, or as a nested list of binaries and\ncodepoints.\n\n[](){: #DiameterIdentity }\n\n```text\nDiameterIdentity() = OctetString()\n```\n\nA value must have length at least 1.\n\n[](){: #DiameterURI }\n\n```text\nDiameterURI() = OctetString()\n              | #diameter_URI{type = Type,\n                              fqdn = FQDN,\n                              port = Port,\n                              transport = Transport,\n                              protocol  = Protocol}\n\nwhere\n\n  Type = aaa | aaas\n  FQDN = OctetString()\n  Port = integer()\n  Transport = sctp | tcp\n  Protocol  = diameter | radius | 'tacacs+'\n```\n\nOn encode, fields port, transport and protocol default to 3868, sctp and\ndiameter respectively. The grammar of an OctetString-valued DiameterURI() is as\nspecified in section 4.3 of RFC 6733. The record representation is used on\ndecode.\n\n[](){: #Enumerated }\n\n```text\nEnumerated() = Integer32()\n```\n\nOn encode, values can be specified using the macros defined in a dictionary's\nhrl file.\n\n[](){: #IPFilterRule } [](){: #QoSFilterRule }\n\n```erlang\nIPFilterRule()  = OctetString()\nQoSFilterRule() = OctetString()\n```\n\nValues of these types are not currently parsed by diameter.","ref":"diameter_dict.html#data-types"},{"type":"extras","title":"SEE ALSO - diameter_dict","doc":"[diameterc(1)](diameterc_cmd.md), `m:diameter`, `m:diameter_app`,\n`m:diameter_codec`, `m:diameter_make`","ref":"diameter_dict.html#see-also"}],"proglang":"erlang","content_type":"text/markdown","producer":{"name":"ex_doc","version":"0.38.1"}}